/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.net;

import net.hizlab.kagetaka.Debug;

import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

/**
 * Keep-Alive 륯饹Ǥ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.3 $
 */
class KeepAliveManager {
    private static final int MAX_CONNECITONS = 5;
    private static final int SCAN_INTERVAL   = 5000;

    private Hashtable cache = new Hashtable();
    private Monitor   monitor;

    /** ³֤ */
    private static int getMaxConnections() {
        String max;

        if ((max = System.getProperty("kagetaka.http.maxConnections")) == null) {
            max = System.getProperty("http.maxConnections");
        }

        if (max != null) {
            try {
                return Integer.parseInt(max);
            } catch (NumberFormatException e) { }
        }

        return MAX_CONNECITONS;
    }

    /**
     * 󥹥󥹤ޤ
     */
    KeepAliveManager() {
    }

    /**
     * ޤ
     *
     * @param  protocol ץȥ
     * @param  host     ۥ
     * @param  port     ݡ
     *
     * @return HTTP ͥ¸ߤʤ <code>null</code>
     */
    HttpConnection get(String protocol, String host, int port) {
        Target target = (Target) cache.get(createKey(protocol, host, port));
        if (target == null) {
            return null;
        }

        HttpConnection connection = target.get();
        if (connection == null) {
            return null;
        }

        connection.reuse();

        return connection;
    }

    /**
     * ɲäޤ
     *
     * @param  connection HTTP ͥ
     */
    void add(HttpConnection connection) {
        // ˥򳫻
        Monitor nowMonitor = this.monitor;
        if (nowMonitor == null || !nowMonitor.isAlive()) {
            ThreadGroup group  = Thread.currentThread().getThreadGroup();
            ThreadGroup parent = null;
            while ((parent = group.getParent()) != null) {
                group = parent;
            }
            synchronized (this) {
                if (monitor == null || !monitor.isAlive()) {
                    monitor = new Monitor(group);
                    monitor.start();
                }
            }
        }

        String  key = createKey(connection.protocol, connection.serverHost, connection.serverPort);
        Target target;
        synchronized (cache) {
            target = (Target) cache.get(key);
            if (target == null) {
                cache.put(key, target = new Target(key));
            }
        }
        target.add(connection);
    }

    /**
     * ޤ
     *
     * @param  connection HTTP ͥ
     */
    void remove(HttpConnection connection) {
        Target target = (Target) cache.get(createKey(connection.protocol, connection.serverHost, connection.serverPort));
        if (target == null) {
            return;
        }

        target.remove(connection);
    }

    /**  */
    private String createKey(String protocol, String host, int port) {
        StringBuffer sb = new StringBuffer(protocol.length() + host.length() + 9);
        sb.append(protocol);
        sb.append("://"   );
        sb.append(host    );
        sb.append(':'     );
        sb.append(port    );
        return sb.toString();
    }

//### Target
    /** ͥꥹ */
    private final class Target {
        private String         key;
        private int            count;
        private HttpConnection top;
        private HttpConnection last;

        /** 󥹥󥹤 */
        private Target(String key) {
            this.key = key;
        }

        /**  */
        private synchronized HttpConnection get() {
            HttpConnection connection;

            while (top != null) {
                count--;
                connection = top;
                top  = connection.next;
                if (top != null) {
                    top       .prev = null;
                    connection.next = null;
                } else {
                    last = null;
                }

                if (connection.isAlive()) {
Debug.out.println("% " + Integer.toHexString(connection.hashCode()) + ", km.get: " + connection.protocol + "://" + connection.serverHost + ":" + connection.serverPort);
//Debug.out.println("*** get=" + connection + "," + connection.next);
                    return connection;
                }
                connection.close();
            }

            return null;
        }

        /** ɲ */
        private synchronized void add(HttpConnection connection) {
//Debug.out.println("*** add=" + connection + "," + connection.next);
            if (count >= KeepAliveManager.getMaxConnections()) {
Debug.out.println("% " + Integer.toHexString(connection.hashCode()) + ", km.max: " + connection.protocol + "://" + connection.serverHost + ":" + connection.serverPort);
                connection.close();
            } else {
                if (top == null) {
                    top = last = connection;
                } else {
                    last      .next = connection;
                    connection.prev = last;
                    last            = connection;
                }
                count++;
Debug.out.println("% " + Integer.toHexString(connection.hashCode()) + ", km.add: " + connection.protocol + "://" + connection.serverHost + ":" + connection.serverPort);
            }
        }

        /**  */
        private synchronized void remove(HttpConnection connection) {
            if (connection.next == null && connection.prev == null) {
                if (top == connection) {
                    top = last = null;
                    count--;
//Debug.out.println("*** remove=" + connection + "," + connection.next);
                }
                return;
            }

            if (connection.prev == null) {
                top = connection.next;
                if (top != null) {
                    top       .prev = null;
                    connection.next = null;
                } else {
                    last = null;
                }
            } else
            if (connection.next == null) {
                last            = connection.prev;
                last      .next = null;
                connection.prev = null;
            } else {
                connection.prev.next = connection.next;
                connection.next.prev = connection.prev;
                connection.prev      = connection.next = null;
            }
            count--;
//Debug.out.println("*** remove=" + connection + "," + connection.next);
        }

        /** å */
        private synchronized void check() {
//Debug.out.println("*** checkstart=" + this);
            if (top == null) {
                return;
            }
            HttpConnection nextConnection, connection = top;
            do {
//Debug.out.println("*** check=" + connection);
                nextConnection = connection.next;
                if (!connection.isAlive()) {
                    connection.close();
                    remove(connection);
                }
                connection = nextConnection;
            } while (connection != null);
        }

        /** ʸɽ֤ */
        public String toString() {
            return super.toString() + "[" + key + ", " + count + "]";
        }
    }

//### Monitor
    /** ˥ */
    private final class Monitor extends Thread {
        private Vector removeList = new Vector();

        /** 󥹥󥹤 */
        private Monitor(ThreadGroup group) {
            super(group, "Keep-Alive-Timer");
            setDaemon  (true);
            setPriority(MAX_PRIORITY - 2);
        }

        /** ºݤ˼¹Ԥ */
        public void run() {
            try {
                Target target;
                for (;;) {
                    Thread.sleep(SCAN_INTERVAL);
                    if (cache.size() == 0) {
                        continue;
                    }

                    for (Enumeration e = cache.elements(); e.hasMoreElements();) {
                        target = (Target) e.nextElement();
                        target.check();
                        if (target.top == null) {
                            removeList.addElement(target);
                        }
                    }

                    //  Target 
                    if (removeList.size() > 0) {
                        synchronized (cache) {
                            for (Enumeration e = removeList.elements(); e.hasMoreElements();) {
                                target = (Target) e.nextElement();
                                if (target.top == null) {
                                    cache.remove(target.key);
                                }
                            }
                        }
                        removeList.removeAllElements();
                    }
                }
            } catch (InterruptedException e) { }
            synchronized (KeepAliveManager.this) {
                if (monitor == this) {
                    monitor = null;
                }
            }
        }
    }
}
