/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.net;

import net.hizlab.kagetaka.Debug;
import net.hizlab.kagetaka.io.TextOutputStream;

import java.io.BufferedOutputStream;
import java.io.InputStream;
import java.io.IOException;
import java.net.Socket;

/**
 * HTTP ³ݻ뤿Υ饹Ǥ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.1 $
 */
class HttpConnection {
    private static KeepAliveManager keepAliveManager = new KeepAliveManager();

    private static boolean candoHttpKeepAlive;

    /** ƥå */
    static {
        String ka = System.getProperty("kagetaka.http.keepAlive");
        if (ka == null) {
            System.getProperty("http.keepAlive");
        }
        candoHttpKeepAlive = (ka != null ? Boolean.valueOf(ka).booleanValue() : true);
    }

    /** ץȥ */
    final String  protocol;
    /** ̾ */
    final String  serverHost;
    /** ݡֹ */
    final int     serverPort;
    /** ץ⡼ɤɤ */
    final boolean usingProxy;

    /** Хå */
    Socket           serverSocket;
    /** Фؤν */
    TextOutputStream serverOutput;
    /** Фؤ */
    InputStream      serverInput;

    private boolean closed;                 // ĤƤ뤫ɤ
    private long    keepAliveExpires;       // ͭ
    private int     useCount = 1;           // Ѳ


    // KeepAliveManager Ѥ
    HttpConnection next;
    HttpConnection prev;

    /**
     * Keep-Alive  󥹥󥹤ޤ
     *
     * @param  protocol ץȥ
     * @param  host     ФΥۥ
     * @param  port     ФΥݡ
     *
     * @return 󥹥󥹡¸ߤʤ <code>null</code>
     */
    static HttpConnection findInstance(String protocol, String host, int port) {
        HttpConnection connection = keepAliveManager.get(protocol, host, port);
        if (connection == null) {
            return null;
        }

        return connection;
    }

    /**
     * ꤷۥȤȥݡȤ³ HTTP 饤Ȥޤ
     *
     * @param  protocol ץȥ
     * @param  host     ФΥۥ
     * @param  port     ФΥݡ
     * @param  proxy    Фץξ <code>true</code>
     *                  ʳξ <code>false</code>
     */
    HttpConnection(String protocol, String host, int port, boolean proxy) {
        this.protocol     = protocol;
        this.serverHost   = host;
        this.serverPort   = port;
        this.usingProxy   = proxy;
    }

    /**
     * ۥȤ³ޤ
     *
     * @throws IOException IO 顼ȯ
     * @throws java.net.UnknownHostException ʥۥȤꤷ
     */
    void open() throws IOException {
        if (serverSocket != null) {
            throw new IOException("Already opened");
        }

        serverSocket = NetworkClient.doConnect(serverHost, serverPort);
        serverOutput = new TextOutputStream(
                         new BufferedOutputStream(serverSocket.getOutputStream()),
                         NetworkClient.defaultOutputEncoding);
        serverInput  = serverSocket.getInputStream();

        serverSocket.setTcpNoDelay(true);
Debug.out.println("% " + Integer.toHexString(hashCode()) + ", opened: " + protocol + "://" + serverHost + ":" + serverPort);
    }

    /**
     * ³Ĥޤ
     */
    final void close() {
        if (closed) {
            return;
        }

        closed           = true;
        keepAliveExpires = -1;

Debug.out.println("% " + Integer.toHexString(hashCode()) + ", closed: " + protocol + "://" + serverHost + ":" + serverPort);
        try {
            closeImpl();
        } catch (Exception e) { }
    }

    /**
     * ³ºݤĤޤ
     *
     * @throws IOException IO 顼ȯ
     */
    void closeImpl() throws IOException {
        serverSocket.close();
    }

    /**
     * Keep-Alive 򥵥ݡȤƤ뤫ɤ֤ޤ
     *
     * @return Keep-Alive 򥵥ݡȤƤ <code>true</code>
     *         ݡȤƤʤ <code>false</code>
     */
    boolean candoHttpKeepAlive() {
        return candoHttpKeepAlive;
    }

    /**
     *  Keep-Alive ³Ƥ뤫֤ޤ
     *
     * @return Keep-Alive ³Ƥ <code>true</code>
     *         Ƥʤ <code>false</code>
     */
    boolean isKeepingAlive() {
        return candoHttpKeepAlive() && (keepAliveExpires != -1);
    }

    /**
     * Keep-Alive ॢȤޤǤÿꤷޤ
     *
     * @param  seconds ॢȤޤǤÿ
     *                 <code>0</code> ̵¡
     *                 <code>-1</code>  Keep-Alive ̵
     */
    void setKeepAliveTimeout(int seconds) {
        this.keepAliveExpires = (seconds > 0
                                 ? System.currentTimeMillis() + (seconds * 1000)
                                 : seconds);
    }

    /**
     * ߤ³ͭɤ֤ޤ
     *
     * @return ͭξ <code>true</code>
     *         ̵ξ <code>false</code>
     */
    boolean isAlive() {
        return (isKeepingAlive()
             && (keepAliveExpires == 0
              || keepAliveExpires >= System.currentTimeMillis()));
    }

    /**
     * Keep-Alive ˤꥳͥ󤬺ƻѤ˸ƤӽФޤ
     * ˤꡢѥ󥿡åפޤ
     */
    void reuse() {
        useCount++;
    }

    /**
     * ߤѲޤ
     *
     * @return Ѳ
     */
    int getUseCount() {
        return useCount;
    }

    /**
     * ץѤ뤫ɤ֤ޤ
     *
     * @return ץѤ <code>true</code>
     *         Ѥʤ <code>false</code>
     */
    boolean usingProxy() {
        return usingProxy;
    }

    /**
     * ХåȤ֤ޤ
     *
     * @return Хå
     */
    Socket getServerSocket() {
        return serverSocket;
    }

    /**
     * Фνϥȥ꡼֤ޤ
     *
     * @return Фνϥȥ꡼
     */
    TextOutputStream getOutputStream() {
        return serverOutput;
    }

    /**
     * Фϥȥ꡼֤ޤ
     *
     * @return Фϥȥ꡼
     */
    InputStream getInputStream() {
        return serverInput;
    }

    /**
     * ʸɽ֤ޤ
     *
     * @return ʸɽ
     */
    public String toString() {
        return super.toString() + "[" + protocol + "://" + serverHost + ":" + serverPort + "]";
    }
}
