/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.net;

import net.hizlab.kagetaka.Debug;
import net.hizlab.kagetaka.io.TextOutputStream;

import java.io.BufferedInputStream;
import java.io.InputStream;
import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.SocketException;

/**
 * HTTP 饤ȤδŪʵǽ󶡤ޤ
 * Υ饹ϥåɥդǤϤޤ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.5 $
 */
public class HttpClient extends NetworkClient {
    private static final int HTTP_CONTINUE                   = 100;
    private static final int DEFAULT_KEEPALIVE_MAX           =  5;
    private static final int DEFAULT_KEEPALIVE_PROXY_MAX     = 50;
    private static final int DEFAULT_KEEPALIVE_TIMEOUT       = 15;
    private static final int DEFAULT_KEEPALIVE_PROXY_TIMEOUT = 60;

    private static KeepAliveManager keepAliveManager = new KeepAliveManager();

    /** ǥեȥݡ */
    public static final int DEFAULT_PORT = 80;

    /** Хå */
    HttpConnection connection;

    private InputStream serverInput;        // Ф
    private boolean     needRetry;          // ƻԤۤɤɤ
    private boolean     error;              // 顼ȯɤ

    /**
     * HTTP ץȥ³饤ȤΥ󥹥󥹤ޤ
     *
     * @param  protocol ץȥ
     * @param  host     ФΥۥ
     * @param  port     ФΥݡ
     * @param  proxy    Фץξ <code>true</code>
     *                  ʳξ <code>false</code>
     *
     * @return HTTP 饤
     *
     * @throws IOException IO 顼ȯ
     * @throws java.net.UnknownHostException ʥۥȤꤷ
     */
    public static HttpClient getInstance(String protocol, String host, int port, boolean proxy)
            throws IOException {
        if (port == -1) {
            port = DEFAULT_PORT;
        }

        return new HttpClient(getConnection(protocol, host, port, proxy));
    }

    /**
     * Keep-Alive  󥹥󥹤ޤ
     *
     * @param  protocol ץȥ
     * @param  host     ФΥۥ
     * @param  port     ФΥݡ
     *
     * @return 󥹥󥹡¸ߤʤ <code>null</code>
     */
    static HttpConnection findInstance(String protocol, String host, int port) {
        return keepAliveManager.get(protocol, host, port);
    }

    /** ͥ */
    private static HttpConnection getConnection(String protocol, String host, int port,
                                                boolean proxy)
            throws IOException {
        HttpConnection connection;
        if ((connection = findInstance(protocol, host, port)) == null) {
            connection = new HttpConnection(protocol, host, port, proxy);
            connection.open();
        }

        return connection;
    }

    /**
     * ꤷͥ򸵤ˤ HTTP 饤Ȥޤ
     *
     * @param  connection HTTP ͥ
     */
    HttpClient(HttpConnection connection) {
        this.connection = connection;
    }

    /**
     * ³ޤ
     * Keep-Alive ξǥ顼ȯƤʤС
     * ͥϺѥ塼˲󤵤ޤ
     */
    public synchronized void dispose() {
        if (connection == null) {
            return;
        }

        try {
            if (serverInput != null) {
                // serverInput.close() ⤫顢μ¤ release ƤФƲ
                try {
                    serverInput.close();
                } catch (IOException e) { }
            } else if (!error) {
                // ٤ȤƤʤǡ顼Ƥʤкѥ塼᤹
                keepAliveManager.add(connection);
            } else {
                // إå˥顼ȯ
                connection.close();
            }
        } finally {
            connection = null;
        }
    }

    /**
     * Ѥʤʤä˸ƤӽФޤ
     * <code>is</code>  <code>null</code> ǤϤʤξǡ
     * Keep-Alive ͭʤСͥ󤬺ѥ塼˲󤵤ޤ
     * ̵ξϡ<code>is</code> ޤ
     * <code>is</code>  <code>null</code> ξϡ
     * ѤϤޤ
     *
     * @param  is    ͥ󤬺ѤǤʤäˡ
     *               륹ȥ꡼
     * @param  error 顼ȯȤ <code>true</code>
     *               ｪλ <code>false</code>
     */
    synchronized void release(InputStream is, boolean error) {
        if (connection == null) {
            return;
        }

        try {
            // ϥȥ꡼ϲ⤻˲
            if (serverInput != null) {
                serverInput = null;
            }

            // 顼ȯƤʤKeep-Alive ͭʤ顢Ѥ
            if (!error && !this.error && connection.isKeepingAlive()) {
                keepAliveManager.add(connection);
                return;
            }

            // ѤǤʤΤǡȥ꡼򥯥
            if (is != null) {
                try {
                    is.close();
                } catch (IOException e) { }
            }
            connection.close();
        } finally {
            connection = null;
        }
    }

    /**
     * HTTP ꥯȤ򥵡Фꡢ쥹ݥ󥹤֤ޤ
     *
     * @param  requests   ꥯȥإå
     * @param  poster     POST ǡ̵ <code>null</code>
     * @param  responses  쥹ݥ󥹥إåǼ륤󥹥
     * @param  isEmpty    쥹ݥ󥹤Ȥɬξ <code>true</code>
     *                    ʳξ <code>false</code>
     * @param  canRelease 쥹ݥ󥹤ξˡ³Ƥ
     *                    פʾ <code>true</code>
     *                    ʾ <code>false</code>
     *
     * @return 쥹ݥ󥹥ܥǥ뤿ϥȥ꡼
     *
     * @throws IOException IO 顼ȯ
     */
    public synchronized InputStream send(MessageHeader requests,
                                         PosterOutputStream poster,
                                         MessageHeader responses,
                                         boolean isEmpty,
                                         boolean canRelease)
            throws IOException {
        if (connection == null) {
            throw new IOException("Connection closed");
        }

        try {
            // ꥯȤ
Debug.out.println("++ Request : " + requests);
            TextOutputStream out = connection.getOutputStream();
            requests.write(out);
            if (poster != null) {
                poster.writeTo(out);
            }
            out.flush();

            // إåϤơȥ꡼֤
            return serverInput = parseHeader(connection.getInputStream(),
                                             responses, isEmpty, canRelease);
        } catch (SocketException e) {
            error     = true;
            needRetry = (connection.getUseCount() > 1);
Debug.out.println("### WARNING ### HttpClient.send : " + e);
            dispose();
            throw e;
        } catch (IOException e) {
            error = true;
Debug.out.println("### WARNING ### HttpClient.send : " + e);
e.printStackTrace(Debug.out);
            dispose();
            throw e;
        }
    }

    /**
     * 顼ȯˡƻԤۤɤɤ֤ޤ
     *
     * @return ƻԤۤɤ <code>true</code>
     *         ʾ <code>false</code>
     */
    public boolean needRetry() {
        return needRetry;
    }

    /** HTTP 쥹ݥ󥹥إå */
    private InputStream parseHeader(InputStream in,
                                    MessageHeader responses,
                                    boolean isEmpty,
                                    boolean canRelease)
            throws IOException {
        boolean usingProxy = connection.usingProxy;

        // ɬꤹɬפΤѿ
        boolean     keepingAlive;
        InputStream inputStream;

        // Keep-Alive 
        int keepAliveMax;
        int keepAliveTimeout = (usingProxy ? DEFAULT_KEEPALIVE_PROXY_TIMEOUT : DEFAULT_KEEPALIVE_TIMEOUT);

        byte[] b = new byte[8];
        int readLength = 0;

        // Ƭ 8 Хɤ߹
        if (!in.markSupported()) {
            in = new BufferedInputStream(in);
        }
        in.mark(8);
        while (readLength < 8) {
            int length = in.read(b, readLength, 8 - readLength);
            if (length < 0) {
                break;
            }
            readLength += length;
        }
        in.reset();
        in.mark(-1);

        // HTTP/1. ξ
        if (b[0] == 'H' && b[1] == 'T' && b[2] == 'T' && b[3] == 'P'
                && b[4] == '/' && b[5] == '1' && b[6] == '.') {
            // 쥹ݥ󥹤
            responses.parse(in);
Debug.out.println("++ Response : " + responses);

            String conHeader = null;
            if (usingProxy) {
                conHeader = responses.get("Proxy-Connection");
            }
            if (conHeader == null) {
                conHeader = responses.get("Connection");
            }

            if (conHeader != null) {
                // Keep-Alive ͭ
                if (conHeader.toLowerCase().compareTo("keep-alive") == 0) {
                    keepAliveMax = (usingProxy ? DEFAULT_KEEPALIVE_PROXY_MAX : DEFAULT_KEEPALIVE_MAX);

                    HeaderParser p = new HeaderParser(responses.get("Keep-Alive"));
                    if (p != null) {
                        keepAliveMax     = p.getInt("max"    , keepAliveMax    );
                        keepAliveTimeout = p.getInt("timeout", keepAliveTimeout);
                    }
                } else {
                    // Connection: close ʤɤξ
                    keepAliveMax = 0;
                }
            } else if (b[7] != '0') {
                // HTTP/1.0 ǤϤʤ
                keepAliveMax = (conHeader != null ? 1 : DEFAULT_KEEPALIVE_MAX);
            } else {
                // HTTP/1.0  Connection إå̵
                keepAliveMax = -1;
            }
Debug.out.println("% " + Integer.toHexString(connection.hashCode()) + ", kalive: " + connection.protocol + "://" + connection.serverHost + ":" + connection.serverPort + ", " + keepAliveMax + ", " + keepAliveTimeout);
        } else if (readLength != 8) {
            throw new SocketException("Unexpected end of file from server");
        } else {
            responses.set("Content-type", "unknown/unknown");
            keepAliveMax = -1;
        }

        // HTTP 쥹ݥ󥹥ɤ
        int code = -1;
        try {
            String line = responses.get(0);
            int    p    = line.indexOf(' ');
            while (line.charAt(p) == ' ') {
                p++;
            }
            code = Integer.parseInt(line.substring(p, p + 3));
        } catch (Exception e) { }


        // HTTP CONTINUE (100) ξϡαԤ
        if (code == HTTP_CONTINUE) {
            responses.reset();
            return parseHeader(in, responses, isEmpty, canRelease);
        }


        ///// ȥ꡼

        if (isEmpty) {
            keepingAlive = (keepAliveMax > 0);
            inputStream  = new EmptyInputStream  (this, in);
        } else if (isChunked(responses)) {
            keepingAlive = (keepAliveMax > 0);
            inputStream  = new ChunkedInputStream(this, in, responses);
        } else {
            // Content-Length 
            int contentLength;
            if (/*---*/code == HttpURLConnection.HTTP_NOT_MODIFIED
                    || code == HttpURLConnection.HTTP_NO_CONTENT) {
                contentLength = 0;
            } else {
                contentLength = -1;
                String s = responses.get("content-length");
                if (s != null && s.length() > 0) {
                    try {
                        contentLength = Integer.parseInt(s);
                    } catch (Exception e) { }
                }
            }

            // Content-Length λ꤬䡢ƥĤʤϡ
            // Kee-Alive ݻǤ
            keepingAlive = (keepAliveMax > 0 && contentLength >= 0);

            if (contentLength > 0) {
                inputStream = new PartInputStream  (this, in, contentLength);
            } else if (contentLength < 0) {
                inputStream = new DirectInputStream(this, in);
            } else {
                inputStream = new EmptyInputStream (this, in);
                isEmpty     = true;
            }
        }

        // Keep-Alive ¸
        connection.setKeepAliveTimeout(keepingAlive ? keepAliveTimeout : -1);

        // 쥹ݥ󥹥ܥǥξϡͥƤޤ
        if (isEmpty && canRelease) {
            release(in, false);
        }

        return inputStream;
    }

    /** chunked žɤĴ٤֤ */
    private boolean isChunked(MessageHeader responses) {
        try {
            String transferEncoding = responses.get("Transfer-Encoding");

            return (transferEncoding != null
                 && transferEncoding.toLowerCase().compareTo("chunked") == 0);
        } catch (Exception e) { }

        return false;
    }

    /**
     * Keep-Alive 򥵥ݡȤƤ뤫ɤ֤ޤ
     *
     * @return Keep-Alive 򥵥ݡȤƤ <code>true</code>
     *         ݡȤƤʤ <code>false</code>
     */
    public boolean candoHttpKeepAlive() {
        HttpConnection connection = this.connection;
        return (connection != null
                ? connection.candoHttpKeepAlive()
                : false);
    }

    /**
     * ץѤ뤫ɤ֤ޤ
     *
     * @return ץѤ <code>true</code>
     *         Ѥʤ <code>false</code>
     */
    public boolean usingProxy() {
        HttpConnection connection = this.connection;
        return (connection != null
                ? connection.usingProxy
                : false);
    }

    /**
     * ʸɽ֤ޤ
     *
     * @return ʸɽ
     */
    public String toString() {
        HttpConnection connection = this.connection;
        return super.toString() + "[" + (connection != null ? connection.toString() : "") + "]";
    }
}
