/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.net;

import java.io.InputStream;
import java.io.IOException;

/**
 * ȥ꡼फ̵ùɤ߹िΥȥ꡼Ǥ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.1 $
 */
class DirectInputStream extends InputStream implements InputStreamExtension {
    private static final int BUFFER_SIZE = 512;

    /*
     * ȥ꡼νüޤɤ߹硢http ꡼
     * http  null ꤹ롣ΤȤŪˤ in ⥯뤬
     * ޡξ in ϥʤ
     *
     * ʤʤ顢ȥ꡼νüޤãƤ⡢ꥻåȤ
     * ɹǽǤ롣
     *
     * http ꡼졢in 򥯥ϡɬ overOdd 
     * markOffset 롣
     * Ĥޤꡢhttp == null Ǥ markOffset != -1 ξϡ
     * ɤ߹ߡޡꥻåǽʤȤ򼨤
     *
     * ĹǤϤ뤬http == null && overOdd == 0 ξϡ
     * ޡ椫ɤˤ餺ȥ꡼νüãƤ뤳Ȥ򼨤
     *
     * Υ饹 close Ū˸ƤӽФȡin  null ˤʤꡢ
     * ٤Ƥػߤ롣
     */

    /** HTTP 饤 */
    protected HttpClient http;
    /** ؤϥȥ꡼ */
    protected InputStream in;

    /** ɤ߹ߺѤߤ礭 */
    protected int overOdd;
    /** ޡϰ֤Υեå */
    protected int markOffset = -1;

    /** ȥ꡼˥ */
    protected StreamMonitor streamMonitor;

    /**
     * ꤵ줿ϥȥ꡼򸵤ˤ󥹥󥹤ޤ
     *
     * @param  http HTTP 饤
     * @param  in   ϥȥ꡼
     */
    DirectInputStream(HttpClient http, InputStream in) {
        this.http = http;
        this.in   = in;
    }

    /**
     * ΥХϥȥ꡼फ顢ǡμΥХȤɤ߹ߤޤ
     * ΥХȤϡ<code>0</code>  <code>255</code> ϰϤ
     * <code>int</code> Ȥ֤ޤ
     * ȥ꡼νãƥХȤʤϡ
     *  <code>-1</code> ֤ޤ
     * Υ᥽åɤϡϥǡɤ߹ޤ褦ˤʤ뤫
     * ȥ꡼ν꤬Ф뤫
     * ޤ㳰ޤǥ֥åޤ
     *
     * @return ǡμΥХȡ
     *         ȥ꡼νã <code>-1</code>
     *
     * @throws IOException I/O 顼ȯ
     */
    public synchronized int read()
            throws IOException {
        ensureOpen();
        if (http == null && overOdd == 0) {
            return -1;
        }

        int ret;
        try {
            ret = in.read();
        } catch (IOException e) {
            release(StreamMonitor.ERROR);
            throw e;
        }

        if (ret == -1) {
            release(StreamMonitor.END_OF_STREAM);
        } else if (overOdd > 0) {
            overOdd--;
            markOffset++;
        } else {
            // ȥ꡼˥
            if (streamMonitor != null) {
                streamMonitor.pass(ret);
            }

            if (markOffset != -1) {
                markOffset++;
            }
        }
        return ret;
    }

    /**
     * ΥХϥȥ꡼फХȤɤ߼äơ
     * ꤵ줿Х˻ΥեåȤޤ
     *
     * @param  b   žХåե
     * @param  off ХȤγǼϥեå
     * @param  len ɤ߹ХȤκ
     *
     * @return ɤ߹ޤХȿ
     *         ȥ꡼νã <code>-1</code>
     *
     * @throws IOException I/O 顼ȯ
     */
    public synchronized int read(byte[] b, int off, int len)
            throws IOException {
        ensureOpen();
        if ((off | len | (off + len) | (b.length - (off + len))) < 0) {
            throw new IndexOutOfBoundsException();
        } else if (len == 0) {
            return 0;
        } else if (http == null && overOdd == 0) {
            return -1;
        }

        int readlen;
        try {
            readlen = in.read(b, off, len);
        } catch (IOException e) {
            release(StreamMonitor.ERROR);
            throw e;
        }

        if (readlen == -1) {
            release(StreamMonitor.END_OF_STREAM);
        } else if (overOdd > 0) {
            if (overOdd < readlen) {
                // ȥ꡼˥
                if (streamMonitor != null) {
                    streamMonitor.pass(b, off + overOdd, readlen - overOdd);
                }

                overOdd = 0;
            } else {
                overOdd -= readlen;
            }
            markOffset += readlen;
        } else {
            // ȥ꡼˥
            if (streamMonitor != null) {
                streamMonitor.pass(b, off, readlen);
            }

            if (markOffset != -1) {
                markOffset += readlen;
            }
        }
        return readlen;
    }

    /**
     * ϥȥ꡼फΥǡФ
     * <code>n</code> ХȤåפƥǡΤƤޤ
     *
     * @param  n åפХȿ
     *
     * @return ºݤ˥åפ줿Хȿ
     *
     * @throws IOException I/O 顼ȯ硣
     */
    public synchronized long skip(long n)
            throws IOException {
        ensureOpen();
        if (n == 0 || (http == null && overOdd == 0)) {
            return 0;
        }

        int len;

        // ȥ꡼˥Τ뤿˼ºݤɤ߹
        if (streamMonitor != null) {
            len = (n < BUFFER_SIZE ? (int) n : BUFFER_SIZE);
            byte[] b = new byte[len];
            return Math.max(read(b, 0, len), 0);
        }

        // ľܥå
        len = (n <= Integer.MAX_VALUE ? (int) n : Integer.MAX_VALUE);
        try {
            len = (int) in.skip(len);
        } catch (IOException e) {
            release(StreamMonitor.ERROR);
            throw e;
        }

        if (len == 0) {
            return 0;
        } else if (overOdd > 0) {
            if (overOdd < len) {
                overOdd = 0;
            } else {
                overOdd -= len;
            }
            markOffset += len;
        } else {
            if (markOffset != -1) {
                markOffset += len;
            }
        }
        return len;
    }

    /**
     * ֥å󥰤ȯ뤳Ȥʤˡ
     * Ϥɤ߹ळȤΤǤХȿ֤ޤ
     *
     * @return ɤ߹Хȿ
     *
     * @throws IOException IO 顼ȯ
     */
    public synchronized int available()
            throws IOException {
        ensureOpen();
        if (http == null && overOdd == 0) {
            return 0;
        }

        try {
            return in.available();
        } catch (IOException e) {
            release(StreamMonitor.ERROR);
            throw e;
        }
    }

    /**
     * ϥȥ꡼򥯥˴ϢƤΥƥ꥽
     * ޤ
     *
     * @throws IOException I/O 顼ȯ
     */
    public synchronized void close()
            throws IOException {
        if (in == null) {
            return;
        }

        try {
            // äϡĤ򥹥åפƤߤ
            if (streamMonitor != null) {
                int len;
                while ((len = available()) > 0) {
                    skip(len);
                }
            }
            release(StreamMonitor.CLOSE);
        } finally {
            in = null;
        }
    }

    /**
     * HTTP 饤Ȥޤ
     *
     * @param  cause ȥ꡼˥Τͳ
     *
     * @see StreamMonitor
     */
    protected void release(int cause) {
        if (http == null) {
            return;
        }

        try {
            // ˱ in 򥯥롣
            // in 򥯥 overOdd  markOffset ꥻåȤ

            if (cause == StreamMonitor.END_OF_STREAM) {
                if (markOffset != -1) {
                    // ޡξϥꥻåȤǽΤ
                    // in Τϥʤ http ϥ꡼롣
                    // http Ƥ⡢in ΥХåե
                    // ɤ߹ߤ
                    http.release(null, false);
                } else {
                    overOdd    = 0;
                    markOffset = -1;
                    http.release(in, false);
                }
            } else {
                overOdd    = 0;
                markOffset = -1;
                http.release(in, true);
            }
        } finally {
            http = null;
            if (streamMonitor != null) {
                try {
                    streamMonitor.stop(cause);
                } finally {
                    streamMonitor = null;
                }
            }
        }
    }

    /**
     * ϥȥ꡼θ֤߰˥ޡդޤ
     *
     * @param  readlimit ޡ̵֤ˤʤɤ߹߲ǽʺХȿ
     */
    public synchronized void mark(int readlimit) {
        if (in == null) {
            return;
        }

        in.mark(readlimit);
        if (readlimit >= 0) {
            markOffset = 0;
        } else {
            markOffset = -1;
        }
    }

    /**
     * ϥȥ꡼غǸ {@link #mark(int)} ᥽åɤ
     * ƤӽФ줿ȤΥޡ֤ؤΥȥ꡼ΰ֤ꤷޤ
     *
     * @throws IOException ȥ꡼˥ޡդƤʤä硢
     *                     ޤϥޡ̵ˤʤäƤ
     */
    public synchronized void reset()
            throws IOException {
        ensureOpen();
        if (markOffset == -1) {
            throw new IOException("Resetting to an invalid mark");
        }

        try {
            in.reset();
            overOdd    += markOffset;
            markOffset = 0;
        } catch (IOException e) {
            overOdd    = 0;
            markOffset = -1;
            throw e;
        }
    }

    /**
     * ޡ򥵥ݡȤƤ뤫ɤ֤ޤ
     *
     * @return ޡ򥵥ݡȤƤ <code>true</code>
     *         ݡȤƤʤ <code>false</code>
     */
    public boolean markSupported() {
        return (in == null ? false : in.markSupported());
    }

    /**
     * Υȥ꡼बƤ뤫åޤ
     *
     * @throws IOException ȥ꡼बĤƤ
     */
    protected void ensureOpen()
            throws IOException {
        if (in == null) {
            throw new IOException("Stream closed");
        }
    }

    /* ȥ꡼ήǡƻ뤹뤿Υ˥ */
    /** {@inheritDoc} */
    public void setStreamMonitor(StreamMonitor sm) {
        this.streamMonitor = sm;
    }
}
