/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.net;

import java.io.InputStream;
import java.io.IOException;

/**
 * 󥯥ȥ꡼ǧɤ߹ॹȥ꡼Ǥ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.7 $
 */
class ChunkedInputStream extends InputStream implements InputStreamExtension {
    private static final int STATE_END_OF_CHUNK   = 1;     // 󥯤κǸ CRLF ΰ
    private static final int STATE_AWAITING_CHUNK = 2;     // 󥯤γϰ
    private static final int STATE_READING_CHUNK  = 3;     // ɤ߹
    private static final int STATE_DONE           = 4;     // λ

    /** HTTP 饤 */
    protected HttpClient  http;
    /** ؤϥȥ꡼ */
    protected InputStream in;
    /** 쥹ݥ󥹥إå */
    protected MessageHeader responses;

    /** ǡǼХåե */
    protected byte[] buffer;
    /** Хåեθ߰ */
    protected int    bufferPointer;
    /** ХåեλĤͭʥХȿ */
    protected int    bufferOdd;
    /** ޡ줿Ǥ <code>bufferPointer</code> եɤ */
    protected int    markPointer = -1;
    /** ɤ߲ǽʥǡ */
    protected int    markLimitSize;

    private int           state;            // 󥯤Υơ
    private int           chunkOdd;         // ߤΥ󥯤λĤꥵ
    private StreamMonitor streamMonitor;    // ȥ꡼˥

    /**
     * ꤵ줿ϥȥ꡼򸵤ˤ󥹥󥹤ޤ
     *
     * @param  http      HTTP 饤
     * @param  in        ϥȥ꡼
     * @param  responses 쥹ݥ󥹥إå
     */
    ChunkedInputStream(HttpClient http, InputStream in, MessageHeader responses) {
        this.http      = http;
        this.in        = in;
        this.responses = responses;
        this.state     = STATE_AWAITING_CHUNK;
    }

    /**
     * ΥХϥȥ꡼फ顢ǡμΥХȤɤ߹ߤޤ
     * ΥХȤϡ<code>0</code>  <code>255</code> ϰϤ
     * <code>int</code> Ȥ֤ޤ
     * ȥ꡼νãƥХȤʤϡ
     *  <code>-1</code> ֤ޤ
     * Υ᥽åɤϡϥǡɤ߹ޤ褦ˤʤ뤫
     * ȥ꡼ν꤬Ф뤫
     * ޤ㳰ޤǥ֥åޤ
     *
     * @return ǡμΥХȡ
     *         ȥ꡼νã <code>-1</code>
     *
     * @throws IOException I/O 顼ȯ
     */
    public synchronized int read()
            throws IOException {
        ensureOpen();

        try {
            if (bufferOdd == 0) {
                // ޡ̵ξϡľɤ߹
                if (markPointer < 0) {
                    return readImpl();
                }

                // Хåեɤ߹
                if (!fill(1)) {
                    return -1;
                }

                // Хåեɤ߹ʤϡƵľɤ߹ߤ
                if (bufferOdd == 0) {
                    return read();
                }
            }

            // Хåե֤
            int b = buffer[bufferPointer] & 0xFF;

            bufferPointer++;
            bufferOdd--;

            return b;
        } catch (IOException e) {
            release(StreamMonitor.ERROR);
            throw e;
        }
    }

    /**
     * ΥХϥȥ꡼फХȤɤ߼äơ
     * ꤵ줿Х˻ΥեåȤޤ
     *
     * @param  b   žХåե
     * @param  off ХȤγǼϥեå
     * @param  len ɤ߹ХȤκ
     *
     * @return ɤ߹ޤХȿ
     *         ȥ꡼νã <code>-1</code>
     *
     * @throws IOException I/O 顼ȯ
     */
    public synchronized int read(byte[] b, int off, int len)
            throws IOException {
        ensureOpen();
        if ((off | len | (off + len) | (b.length - (off + len))) < 0) {
            throw new IndexOutOfBoundsException();
        } else if (len == 0) {
            return 0;
        }

        try {
            if (bufferOdd == 0) {
                // ޡ̵ξϡľɤ߹
                if (markPointer < 0) {
                    return readImpl(b, off, len);
                }

                // Хåեɤ߹
                if (!fill(len)) {
                    return -1;
                }

                // Хåեɤ߹ʤϡƵľɤ߹ߤ
                if (bufferOdd == 0) {
                    return read(b, off, len);
                }
            }

            // Хåե֤
            int readlen = (len <= bufferOdd ? len : bufferOdd);
            System.arraycopy(buffer, bufferPointer, b, off, readlen);

            bufferPointer += readlen;
            bufferOdd     -= readlen;

            return readlen;
        } catch (IOException e) {
            release(StreamMonitor.ERROR);
            throw e;
        }
    }

    /**
     * ϥȥ꡼फΥǡФ
     * <code>n</code> ХȤåפƥǡΤƤޤ
     *
     * @param  n åפХȿ
     *
     * @return ºݤ˥åפ줿Хȿ
     *
     * @throws IOException I/O 顼ȯ硣
     */
    public synchronized long skip(long n)
            throws IOException {
        ensureOpen();
        if (n == 0) {
            return 0;
        }

        try {
            // ޡ̵ξϡľܥå
            if (markPointer < 0) {
                return skipImpl(n);
            }

            if (bufferOdd == 0) {
                // Хåեɤ߹
                if (!fill((n <= Integer.MAX_VALUE ? (int) n : Integer.MAX_VALUE))) {
                    return 0;
                }

                // Хåեɤ߹ʤϡƵľܥåפ
                if (bufferOdd == 0) {
                    return skip(n);
                }
            }

            // Хåեΰ֤򥹥å
            int skipped = (n <= bufferOdd ? (int) n : bufferOdd);

            bufferPointer += skipped;
            bufferOdd     -= skipped;

            return skipped;
        } catch (IOException e) {
            release(StreamMonitor.ERROR);
            throw e;
        }
    }

    /**
     * ֥å󥰤ȯ뤳Ȥʤˡ
     * Ϥɤ߹ळȤΤǤХȿ֤ޤ
     *
     * @return ɤ߹Хȿ
     *
     * @throws IOException IO 顼ȯ
     */
    public synchronized int available()
            throws IOException {
        ensureOpen();

        try {
            return (bufferOdd > 0
                    ? bufferOdd
                    : (chunkOdd == 0
                       ? 0
                       : Math.min(in.available(), chunkOdd)));
        } catch (IOException e) {
            release(StreamMonitor.ERROR);
            throw e;
        }
    }

    /**
     * ϥȥ꡼򥯥˴ϢƤΥƥ꥽
     * ޤ
     *
     * @throws IOException I/O 顼ȯ
     */
    public synchronized void close()
            throws IOException {
        if (in == null) {
            return;
        }

        try {
            // äϡĤ򥹥åפƤߤ
            if (streamMonitor != null) {
                int len;
                while ((len = available()) > 0) {
                    skip(len);
                }
            }
            release(StreamMonitor.ABORT);
        } finally {
            in     = null;
            buffer = null;
        }
    }

    /** HTTP 饤Ȥ */
    private void release(int cause) {
        if (http == null) {
            return;
        }

        try {
            http.release(in, (cause != StreamMonitor.END_OF_STREAM));
        } finally {
            http = null;
            if (streamMonitor != null) {
                try {
                    streamMonitor.stop(cause);
                } finally {
                    streamMonitor = null;
                }
            }
        }
    }

    /**
     * ϥȥ꡼θ֤߰˥ޡդޤ
     *
     * @param  readlimit ޡ̵֤ˤʤɤ߹߲ǽʺХȿ
     */
    public synchronized void mark(int readlimit) {
        markLimitSize = readlimit;
        if (bufferOdd == 0) {
            bufferPointer = 0;
        }
        markPointer = (readlimit >= 0 ? bufferPointer : -1);
    }

    /**
     * ϥȥ꡼غǸ {@link #mark(int)} ᥽åɤ
     * ƤӽФ줿ȤΥޡ֤ؤΥȥ꡼ΰ֤ꤷޤ
     *
     * @throws IOException ȥ꡼˥ޡդƤʤä硢
     *                     ޤϥޡ̵ˤʤäƤ
     */
    public synchronized void reset()
            throws IOException {
        ensureOpen();

        if (markPointer < 0) {
            throw new IOException("Resetting to an invalid mark");
        }
        bufferOdd     = (bufferPointer - markPointer);
        bufferPointer = markPointer;
    }

    /**
     * ޡ򥵥ݡȤƤ뤫ɤ֤ޤ
     * Υ饹ϥޡ򥵥ݡȤƤΤǡ<code>true</code> ֤ޤ
     *
     * @return ޡ򥵥ݡȤƤ <code>true</code>
     *         ݡȤƤʤ <code>false</code>
     */
    public boolean markSupported() {
        return true;
    }

    /** ǡºݤ˥ȥ꡼फɤ߹ */
    private int readImpl()
            throws IOException {
        // 󥯤Ǥ̵
        if (state != STATE_READING_CHUNK) {
            // 󥯤λƤ
            if (state == STATE_DONE) {
                return -1;
            }

            // Υ󥯤ν򤹤
            if (!setupNextChunk()) {
                return -1;
            }
        }

        // ºݤɤ߹
        int ret;
        if ((ret = in.read()) == -1) {
            throw new IOException("Premature EOF");
        }

        // ȥ꡼˥
        if (streamMonitor != null) {
            streamMonitor.pass(ret);
        }

        // 󥯤ξ֤Ĵ
        if (--chunkOdd == 0) {
            state = STATE_END_OF_CHUNK;
            setupNextChunk();
        }

        return ret;
    }

    /** ǡºݤ˥ȥ꡼फɤ߹ */
    private int readImpl(byte[] b, int off, int len)
            throws IOException {
        // 󥯤Ǥ̵
        if (state != STATE_READING_CHUNK) {
            // 󥯤λƤ
            if (state == STATE_DONE) {
                return -1;
            }

            // Υ󥯤ν򤹤
            if (!setupNextChunk()) {
                return -1;
            }
        }

        // ºݤɤ߹
        int readlen;
        if ((readlen = in.read(b, off, (chunkOdd <= len ? chunkOdd : len))) == -1) {
            throw new IOException("Premature EOF");
        }

        if (readlen > 0) {
            // ȥ꡼˥
            if (streamMonitor != null) {
                streamMonitor.pass(b, off, readlen);
            }

            // 󥯤ξ֤Ĵ
            if ((chunkOdd -= readlen) == 0) {
                state = STATE_END_OF_CHUNK;
                setupNextChunk();
            }
            return readlen;
        }

        return 0;
    }

    /** ǡºݤ˥ȥ꡼फ饹å */
    private long skipImpl(long n)
            throws IOException {
        // 󥯤Ǥ̵
        if (state != STATE_READING_CHUNK) {
            // 󥯤λƤ
            if (state == STATE_DONE) {
                return 0;
            }

            // Υ󥯤ν򤹤
            if (!setupNextChunk()) {
                return 0;
            }
        }

        int len = (int) (chunkOdd <= n ? chunkOdd : n);
        int skiplen;
        if (streamMonitor != null) {
            // ȥ꡼˥Τ뤿˼ºݤɤ߹
            byte[] b = new byte[len];
            if ((skiplen = in.read(b, 0, len)) == -1) {
                throw new IOException("Premature EOF");
            }

            // ȥ꡼˥
            streamMonitor.pass(b, 0, skiplen);
        } else {
            // ºݤ˥åסchunkOdd ʾˤϥåפǤʤΤǡͤ int
            skiplen = (int) in.skip(len);
        }

        if (skiplen > 0) {
            if ((chunkOdd -= skiplen) == 0) {
                state = STATE_END_OF_CHUNK;
                setupNextChunk();
            }
            return skiplen;
        }

        return 0;
    }

    /** Υ󥯥Ĵ٤ʥ󥯤ǤϸƤФʤ */
    private boolean setupNextChunk()
            throws IOException {
        // RFC2616 3.6.1 򻲾
        switch (state) {
        case STATE_END_OF_CHUNK:        // 󥯤κǸ
        {
            // CRLF Ϥ
            if (in.read() != '\r' || in.read() != '\n') {
                throw new IOException("missing CR/LF");
            }

            // ե륹롼
        }
        case STATE_AWAITING_CHUNK:     // 󥯥إå
        {
            StringBuffer sb = new StringBuffer(8);

            // ƬԤɤ߹
            int     b;
            char    c;
            boolean valid = true;
            READ:
            for (;;) {
                switch (b = in.read()) {
                case '\r':
                    if (in.read() == '\n') {
                        // CRLF Ĥä
                        break READ;
                    }
                    // ե륹롼
                case '\n':
                    throw new IOException("missing CR/LF");
                default:
                    if (!valid) {
                        continue;
                    }

                    c = (char) b;

                    // 󥯥إå 16 ʿǤ̵Ϥʹ̵
                    // chunk-size [ *( ";" chunk-ext-name [ "=" chunk-ext-val ] ) ] CRLF
                    if (Character.digit(c, 16) == -1) {
                        valid = false;
                        continue;
                    }

                    sb.append(c);
                    break;
                }
            }

            // 󥯥
            try {
                chunkOdd = Integer.parseInt(sb.toString(), 16);
            } catch (NumberFormatException e) {
                throw new IOException("Bogus chunk size");
            }

            if (chunkOdd > 0) {
               state = STATE_READING_CHUNK;
               return true;
            }

            // ǸΥ󥯤ʤΤǡԤޤɲåإåȤɤ߹
            if (!responses.merge(in)) {
                throw new IOException("Premature EOF");
            }

            state = STATE_DONE;
            release(StreamMonitor.END_OF_STREAM);
            return false;
        }
        default:
            throw new IllegalStateException("invalid state " + state);
        }
    }

    /** ХåեޤʥޡξǡͭʥХåեʤ (bufferOdd == 0) Τ߸ƤФ */
    private boolean fill(int len) throws IOException {
        ensureOpen();

        // ʾɤ߹ʤ
        if (state == STATE_DONE) {
            markPointer = -1;
            return false;
        }

        // Хåեɤ߹߲ǽʻĤΥ򻻽
        int readOdd = Math.max((buffer != null ? buffer.length - bufferPointer : 0),  // ХåեλĤ
                               (markLimitSize - (bufferPointer - markPointer)     )); // ޡλĤ
        int max     = (readOdd <= len ? readOdd : len);

        // ʾХåեȤäƤϤʤ
        if (max == 0) {
            markPointer = -1;
            return true;
        }

        int readlen;

        // Хåեθɤ߹Ǥߤ
        if (buffer != null) {
            int odd = buffer.length - bufferPointer;

            // ˶ʤ硢Ƭ˵ͤ
            if (odd == 0 && markPointer > 0) {
                int length = bufferPointer - markPointer;
                System.arraycopy(buffer, markPointer, buffer, 0, length);
                odd           = markPointer;
                bufferPointer = length;
                markPointer   = 0;
            }

            // ˶ߤ硢ɤ߹
            if (odd > 0) {
                if ((readlen = readImpl(buffer, bufferPointer, (odd <= max ? odd : max))) == -1) {
                    return false;
                }
                bufferOdd = readlen;
                return true;
            }

            // Хåեĥ
            int newSize = bufferPointer + (bufferPointer <= readOdd ? bufferPointer : readOdd);
            byte[] newBuffer = new byte[newSize];
            System.arraycopy(buffer, 0, newBuffer, 0, bufferPointer);
            buffer = newBuffer;
        } else {
            buffer = new byte[(readOdd <= 512 ? readOdd : 512)];
        }

        if ((readlen = readImpl(buffer, bufferPointer, Math.min(buffer.length - bufferPointer, max))) == -1) {
            return false;
        }
        bufferOdd = readlen;
        return true;
    }

    /** Ƥ뤫å */
    private void ensureOpen()
            throws IOException {
        if (in == null) {
            throw new IOException("Stream closed");
        }
    }

    /* ȥ꡼ήǡƻ뤹뤿Υ˥ */
    /** {@inheritDoc} */
    public void setStreamMonitor(StreamMonitor sm) {
        this.streamMonitor = sm;
    }
}
