/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.io;

import java.io.FilterInputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.IOException;

/**
 * ȥ꡼फɤ߹ǡ򡢻ꤵ줿ϥȥ꡼ˤ
 * ¸뤿Υե륿Ǥ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.3 $
 */
public class TeeInputStream extends FilterInputStream {
    private static final int BUFFER_SIZE = 512;

    private int totalLength;
    private int readLength;

    private Object       osLock = new Object();
    private OutputStream os;
    private boolean      osClosed;
    private boolean      isMarked;
    private int          markOffset;
    private int          overOffset;

    /**
     * ꤵ줿ϥȥ꡼򸵤ˤ󥹥󥹤ޤ
     *
     * @param  in ϥȥ꡼
     * @param  os ϥȥ꡼
     */
    public TeeInputStream(InputStream in, OutputStream os) {
        this(in, os, -1);
    }

    /**
     * ꤵ줿ϥȥ꡼򸵤ˡ
     * ꤷɤ߹।󥹥󥹤ޤ
     *
     * @param  in     ϥȥ꡼
     * @param  os     ϥȥ꡼
     * @param  length ɤ߹ॵ
     */
    public TeeInputStream(InputStream in, OutputStream os, int length) {
        super(in);
        this.os          = os;
        this.totalLength = length;
    }

    /**
     * ΥХϥȥ꡼फ顢ǡμΥХȤɤ߹ߤޤ
     * ΥХȤϡ<code>0</code>  <code>255</code> ϰϤ
     * <code>int</code> Ȥ֤ޤ
     * ȥ꡼νãƥХȤʤϡ
     *  <code>-1</code> ֤ޤ
     * Υ᥽åɤϡϥǡɤ߹ޤ褦ˤʤ뤫
     * ȥ꡼ν꤬Ф뤫
     * ޤ㳰ޤǥ֥åޤ
     *
     * @return ǡμΥХȡ
     *         ȥ꡼νã <code>-1</code>
     *
     * @throws IOException I/O 顼ȯ
     */
    public int read()
            throws IOException {
        int b = in.read();

        if (b != -1) {
            if (overOffset > 0) {
                overOffset--;
                markOffset++;
            } else {
                os.write(b);
                if (isMarked) {
                    markOffset++;
                }

                if (totalLength != -1
                        && ++readLength == totalLength) {
                    closeOutputStream(false);
                }
            }
        } else {
            closeOutputStream(false);
        }

        return b;
    }

    /**
     * ΥХϥȥ꡼फХȤɤ߼äơ
     * ꤵ줿Х˻ΥեåȤޤ
     *
     * @param  b   žХåե
     * @param  off ХȤγǼϥեå
     * @param  len ɤ߹ХȤκ
     *
     * @return ɤ߹ޤХȿ
     *         ȥ꡼νã <code>-1</code>
     *
     * @throws IOException I/O 顼ȯ
     */
    public int read(byte[] b, int off, int len)
            throws IOException {
        int length = in.read(b, off, len);

        if (length != -1) {
            if (overOffset > 0) {
                if (overOffset < length) {
                    os.write(b, off + overOffset, length - overOffset);

                    if (totalLength != -1
                            && (readLength += (length - overOffset)) == totalLength) {
                        closeOutputStream(false);
                    }

                    overOffset = 0;
                } else {
                    overOffset -= length;
                }
                markOffset += length;
            } else {
                os.write(b, off, length);
                if (isMarked) {
                    markOffset += length;
                }

                if (totalLength != -1
                        && (readLength += length) == totalLength) {
                    closeOutputStream(false);
                }
            }
        } else {
            closeOutputStream(false);
        }

        return length;
    }

    /**
     * ϥȥ꡼फΥǡФ
     * <code>n</code> ХȤåפƥǡΤƤޤ
     *
     * @param  n åפХȿ
     *
     * @return ºݤ˥åפ줿Хȿ
     *
     * @throws IOException I/O 顼ȯ
     */
    public long skip(long n)
            throws IOException {
        if (n == 0) {
            return 0;
        }

        int    size   = (n < BUFFER_SIZE ? (int) n : BUFFER_SIZE);
        byte[] buffer = new byte[size];
        long   skip   = n;
        int    length;
        do {
            if ((length = read(buffer, 0, (skip < size ? (int) skip : size))) == -1) {
                return n - skip;
            }

            skip -= length;
        } while (skip > 0);

        return n;
    }

    /**
     * ϥȥ꡼򥯥˴ϢƤΥƥ꥽
     * ޤ
     *
     * @throws IOException I/O 顼ȯ
     */
    public void close()
            throws IOException {
        if (in == null) {
            return;
        }

        closeOutputStream(true);
        super.close();
    }

    /**
     * ϥȥ꡼θ֤߰˥ޡդޤ
     *
     * @param   readlimit ޡ̵֤ˤʤɤ߹߲ǽʺХȿ
     */
    public synchronized void mark(int readlimit) {
        super.mark(readlimit);
        if (readlimit > 0) {
            isMarked   = true;
            markOffset = 0;
            overOffset = 0;
        } else {
            isMarked   = false;
            markOffset = 0;
        }
    }

    /**
     * ϥȥ꡼غǸ {@link #mark(int)} ᥽åɤ
     * ƤӽФ줿ȤΥޡ֤ؤΥȥ꡼ΰ֤ꤷޤ
     *
     * @throws IOException I/O 顼ȯ
     */
    public synchronized void reset()
            throws IOException {
        try {
            super.reset();
            overOffset += markOffset;
            markOffset = 0;
        } catch (IOException e) {
            isMarked   = false;
            markOffset = 0;
            overOffset = 0;
            throw e;
        }
    }

    /** ϥȥ꡼Ĥ */
    private void closeOutputStream(boolean read) {
        if (read && (totalLength == -1 || readLength < totalLength)) {
            try {
                // ;꤬äɤ߹Ǥߤ
                int available = available();
                if (available > 0) {
                    if (totalLength > 0) {
                        available = Math.min(available, totalLength - readLength);
                    }
                    byte[] buffer = new byte[available];
                    int    readLength;

                    while ((readLength = read(buffer, 0, available)) != -1) {
                        if ((available -= readLength) <= 0) {
                            break;
                        }
                    }
                }
            } catch (IOException e) { }
        }

        synchronized (osLock) {
            if (!osClosed) {
                try {
                    os.flush();
                    os.close();
                } catch (IOException e) { }
                osClosed = true;
            }
        }
    }
}
