/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.awt.tate;

import net.hizlab.kagetaka.addin.java2.DnDListener;
import net.hizlab.kagetaka.addin.java2.DnDWrapper;
import net.hizlab.kagetaka.awt.Border;
import net.hizlab.kagetaka.awt.ImageCreator;
import net.hizlab.kagetaka.awt.LayoutUtils;
import net.hizlab.kagetaka.awt.Text;
import net.hizlab.kagetaka.awt.image.GrayFilter;
import net.hizlab.kagetaka.rendering.Option;
import net.hizlab.kagetaka.util.Environment;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Frame;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Insets;
import java.awt.Point;
import java.awt.SystemColor;
import java.awt.Window;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.InputEvent;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;

/**
 * ɽΥܥǤ
 *
 * @kagetaka.bugs ľ󲽤ϡꥹʤ¸ʤޤ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.6 $
 */
public abstract class TextComponent extends Component {
    private static final int BLINK_INTERVAL = 700;

    static final int INSET_H = 4;
    static final int INSET_V = 4;

    /** @serial ƥ */
    Text text;
    /** @serial ʸ */
    char echoChar;

    /** @serial å */
    private boolean locked;
    /** @serial ʣ */
    private boolean multiline;
    /** @serial  */
    private int     columns;
    /** @serial Կ */
    private int     rows;
    /** @serial ɥå */
    private boolean wordwrap;
    /** @serial 夫ɽ */
    private Point   viewPosition = new Point(0, 0);
    /** @serial ƥȤμºݤΥ */
    private Insets  frameSize;
    /** @serial Ԥι⤵ */
    private int     lineHeight;
    /** @serial  */
    private Blink   blink;
    /** @serial Υ */
    private boolean viewCursor;
    /** @serial ϥɥƤ뤫 */
    private boolean openWindow;
    /** @serial ǸϥɥĤ */
    private long    lastClose;


    /**
     * ꤵ줿ƥȤޤࡢ󥹥󥹤ޤ
     *
     * @param  option  ץ
     * @param  ic      ᡼ꥨ
     * @param  text    ƥ
     * @param  rows    Կ
     * @param  columns 
     * @param  multiline ʣԲĤξ <code>true</code>
     *                   ʳξ <code>false</code>
     */
    public TextComponent(Option option, ImageCreator ic, String text,
                         int rows, int columns, boolean multiline) {
        super(option, ic);
        this.multiline = multiline;
        setText   (text   );
        setRows   (rows   );
        setColumns(columns);
        setFrameSize(INSET_V, INSET_H, INSET_V, INSET_H);

        addKeyListener(
            new KeyAdapter() {
                /** 줿 */
                public void keyPressed(KeyEvent e) {
                    if (!isEnabled()) {
                        return;
                    }

                    switch (e.getKeyCode()) {
                    case KeyEvent.VK_SPACE:
                        e.consume();
                        showInputBox(0, 0, false);
                        break;
                    default: // AVOID
                    }
                }
            }
        );

        addMouseListener(
            new MouseAdapter() {
                /** ޥå줿 */
                public void mouseClicked(MouseEvent e) {
                    if (!isEnabled()) {
                        return;
                    }

                    e.consume();

                    if ((e.getModifiers() & InputEvent.BUTTON1_MASK) != 0) {
                        showInputBox(e.getX(), e.getY(), true);
                    }
                }
            }
        );

        addFocusListener(
            new FocusListener() {
                /** ե */
                public void focusGained(FocusEvent e) {
                    synchronized (TextComponent.this) {
                        if (blink == null) {
                            blink = new Blink(BLINK_INTERVAL);
                        }
                    }
                }

                /** ե򼺤ä */
                public void focusLost(FocusEvent e) {
                    synchronized (TextComponent.this) {
                        if (blink != null) {
                            blink.interrupt();
                            blink = null;
                        }
                    }
                }
            }
        );
        DnDWrapper dndWrapper = DnDWrapper.getInstance();
        if (dndWrapper != null) {
            dndWrapper.addDnDListener(this,
                new DnDListener() {
                    /** ɥåפ */
                    public boolean isActive() {
                        return (isEnabled() && !locked);
                    }

                    /** ɥåפ줿 */
                    public void drop(Object[] objects) {
                        for (int i = 0; i < objects.length; i++) {
                            Object object = objects[0];
                            String text;
                            if (object instanceof String) {
                                text = (String) object;
                            } else {
                                text = object.toString();
                            }

                            // ե̾Ϣ뤷ʤ
                            if (getText() == null || object instanceof File) {
                                setText(text);
                            } else {
                                setText(getText() + text);
                            }

                            // ʸʳϢ뤷ʤ
                            if (!(object instanceof String)) {
                                break;
                            }
                        }
                    }
                },
                new Class[]{String.class, File.class}
            );
        }

        setForeground(SystemColor.textText);
        if (Environment.javaVersion < 102) {
            setBackground(SystemColor.window  );
        } else {
            setBackground(SystemColor.text    );
        }
    }

    /** {@inheritDoc} */
    public void setEnabled(boolean b) {
        if (!b) {
            synchronized (this) {
                if (blink != null) {
                    blink.interrupt();
                    blink = null;
                }
            }
        }

        super.setEnabled(b);
    }

    /** {@inheritDoc} */
    protected  Dimension createPreferredSize() {
        // '*' ʸʰ̵̣ˤʬξˤ
        StringBuffer sb = new StringBuffer();
        for (int i = 0; i < columns; i++) {
            sb.append('*');
        }

        Text text = getText(sb.toString(), echoChar);
        Dimension size = text.getSize(0);
        lineHeight = size.width;
        if (rows > 0) {
            size.width *= rows;
        }

        Insets frameSize = this.frameSize;
        return new Dimension(size.width  + frameSize.left + frameSize.right,
                             size.height + frameSize.top + frameSize.bottom);

    }

    /** {@inheritDoc} */
    protected void refresh(Image offscreen, Graphics g, Dimension size,
                           int state, boolean focus) {
        Color fg = getForeground();
        Color bg = getBackground();

        // ƥȤ
        Text   text      = this.text;
        Insets frameSize = this.frameSize;

        // طʿɤ
        g.setColor(bg);
        g.fillRect(0, 0, size.width, size.height);

        // ٥
        if (text != null) {
            this.text = text = text.getText(getFont());
            Point p     = this.viewPosition;
            Color color = fg;
            if (state == DISABLE) {
                color = new Color((new GrayFilter(bg)).filterRGB(0, 0, color.getRGB()));
            }
            g.setColor(color);
            g.setFont (text.getFont());
            text.draw(g,
                      size.width - frameSize.right + p.x,
                      frameSize.top - p.y,
                      (wordwrap ? size.height - frameSize.top - frameSize.bottom : 0));
        }

        // Ȥ
        Border border = getBorder();
        if (border != null) {
            border.draw(g, 0, 0, size.width - 1, size.height - 1, fg);
        } else {
            g.setColor(SystemColor.controlShadow);
            g.drawLine(0, 0, size.width - 1,  0);
            g.drawLine(0, 0, 0, size.height - 1);
            g.setColor(SystemColor.controlDkShadow);
            g.drawLine(1, 1, size.width - 2,  1);
            g.drawLine(1, 1, 1, size.height - 2);
            g.setColor(SystemColor.controlHighlight);
            g.drawLine(size.width - 2,  2, size.width - 2, size.height - 2);
            g.drawLine(2, size.height - 2, size.width - 2, size.height - 2);
            g.setColor(SystemColor.controlLtHighlight);
            g.drawLine(size.width - 1,  1, size.width - 1, size.height - 1);
            g.drawLine(1, size.height - 1, size.width - 1, size.height - 1);
        }

        if (focus && viewCursor) {
            paintCursor(g);
        }
    }

    /**  */
    private void paintCursor(Graphics g) {
        Point p = this.viewPosition;
        if (p.x >= lineHeight || p.y >= 1) {
            return;
        }

        g.setColor  (Color.white);
        g.setXORMode(getBackground());

        int x = 0, y = 0;
        Dimension size      = getSize();
        Insets    frameSize = this.frameSize;
        synchronized (this) {
            x = size.width + p.x - frameSize.right - 1;
            y = p.y + frameSize.top;
        }

        g.drawLine(Math.min(size.width - frameSize.right + 1, x), y, x - lineHeight, y);
        g.setPaintMode();
    }

    /**
     * ΥݡͥȤ˴ޤޤƥȤ֤ޤ
     *
     * @return ƥ
     */
    public String getText() {
        Text text = this.text;
        return (text != null ? text.getValue() : "");
    }

    /**
     * ΥݡͥȤ˴ޤޤƥȤꤷޤ
     *
     * @param  text ƥ
     */
    public synchronized void setText(String text) {
        Text oldText = this.text;

        if (text != null) {
            if (!multiline) {
                // ܰʹߤ򥫥å
                int p1 = text.indexOf(0x0D);
                int p2 = text.indexOf(0x0A);
                if (p1 != -1 || p2 != -1) {
                    text = text.substring(0,
                                          (p1 != -1 && p2 != -1
                                           ? Math.min(p1, p2)
                                           : Math.max(p1, p2)));
                }
            }

            if (oldText != null && oldText.getValue().compareTo(text) == 0) {
                return;
            }

            this.text = getText(text, echoChar);
        } else if (oldText == null) {
            return;
        } else {
            this.text = null;
        }

        invalidate();
        repaintForce();
    }

    /**
     * ݡͥȤåƤ뤫ɤ֤ޤ
     *
     * @return åƤ <code>true</code>
     *         ʳξ <code>false</code>
     */
    public boolean isLocked() {
        return locked;
    }

    /**
     * ݡͥȤ˥å򤷡ƤѹǤʤ褦ˤޤ
     *
     * @param  b å <code>true</code>
     *           å̵ˤ <code>false</code>
     */
    public void setLocked(boolean b) {
        this.locked = b;
    }

    /**
     *  TextField ֤ޤ
     *
     * @return 
     */
    int getColumns() {
        return columns;
    }

    /**
     *  TextField ꤷޤ
     *
     * @param  columns 
     *
     * @throws IllegalArgumentException  <code>0</code> ̤ξ
     */
    void setColumns(int columns) {
        int oldColumns = this.columns;
        if (columns <= 0) {
            throw new IllegalArgumentException("columns less than zero.");
        }

        if (oldColumns != columns) {
            this.columns = columns;
            invalidate();
        }
    }

    /**
     *  TextField ֤ޤ
     *
     * @return 
     */
    int getRows() {
        return rows;
    }

    /**
     *  TextField ιԿꤷޤ
     *
     * @param  rows Կ
     *
     * @throws IllegalArgumentException Կ <code>0</code> ̤ξ
     */
    void setRows(int rows) {
        int oldRows = this.rows;
        if (rows <= 0) {
            throw new IllegalArgumentException("rows less than zero.");
        }

        if (oldRows != rows) {
            this.rows = rows;
            invalidate();
        }
    }

    /**
     * ɥåפ̵֤ͭޤ
     *
     * @return ɥåפͭʾ <code>true</code>
     *         ʳξ <code>false</code>
     */
    boolean getWordwrap() {
        return wordwrap;
    }

    /**
     * ɥåפ̵ͭꤷޤ
     *
     * @param  b ɥåפͭˤ <code>true</code>
     *           ʳξ <code>false</code>
     */
    void setWordwrap(boolean b) {
        if (this.wordwrap != b) {
            this.wordwrap = b;
            invalidate();
        }
    }

    /**
     * ɽ֤֤ޤ
     *
     * @return ɽ
     */
    Point getViewPosition() {
        return new Point(viewPosition);
    }

    /**
     * ɽ֤ꤷޤ
     *
     * @param  x 夫 X
     * @param  y 夫 Y
     */
    void setViewPosition(int x, int y) {
        viewPosition = new Point(x, y);
    }

    /**
     * ΥݡͥȤΥե졼ॵ֤ޤ
     *
     * @return ե졼ॵ
     */
    Insets getFrameSize() {
        return (Insets) frameSize.clone();
    }

    /**
     * ΥݡͥȤΥե졼ॵꤷޤ
     *
     * @param  top    
     * @param  left   
     * @param  bottom 
     * @param  right  
     */
    void setFrameSize(int top, int left, int bottom, int right) {
        frameSize = new Insets(top, left, bottom, right);
        invalidate();
    }

    /**
     * Ԥι⤵֤ޤ
     *
     * @return Ԥι⤵
     */
    int getLineHeight() {
        return lineHeight;
    }

    /** ѤΥɥ򳫤 */
    private void showInputBox(int x, int y, boolean mouse) {
        if (!isEnabled()) {
            return;
        }

        Dimension size      = getSize();
        Insets    frameSize = this.frameSize;
        if (/*---*/mouse
                && (x < frameSize.left - 2
                 || y < frameSize.top  - 2
                 || size.width  - frameSize.right  + 2 <= x
                 || size.height - frameSize.bottom + 2 <= y)) {
            return;
        }

        Frame owner = LayoutUtils.getFrame(this);
        if (owner == null) {
            return;
        }

        synchronized (this) {
            if (openWindow || lastClose + 200 > System.currentTimeMillis()) {
                return;
            }
            openWindow = true;
            lastClose  = 0;
        }

        Window window = showInputBox(owner, x, y);
        if (window == null) {
            openWindow = false;
            return;
        }

        // ꥹʤϿ
        window.addWindowListener(
            new WindowAdapter() {
                /** ɥĤ줿 */
                public void windowClosed(WindowEvent e) {
                    ((Window) e.getComponent()).removeWindowListener(this);
                    lastClose  = System.currentTimeMillis();
                    openWindow = false;
                }
            }
        );
    }

    /**
     * ѤΥɥ򳫤ޤ
     *
     * @param  owner 
     * @param  x     X
     * @param  y     Y
     *
     * @return ɥ
     */
    abstract Window showInputBox(Frame owner, int x, int y);

//### Blink
    /** ǤԤ */
    private final class Blink extends Thread {
        private int interval;

        /** 󥹥󥹤 */
        private Blink(int interval) {
            this.interval = interval;
            viewCursor = true;
            start();
        }

        /** ¹ */
        public void run() {
            try {
                for (;;) {
                    if (isInterrupted()) {
                        break;
                    }
                    sleep(interval);

                    // ľ
                    Graphics g = getGraphics();
                    if (g != null) {
                        try {
                            paintCursor(g);
                        } finally {
                            g.dispose();
                        }
                    }
                    viewCursor = !viewCursor;
                }
            } catch (InterruptedException e) { }
        }
    }
}
