/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.awt.tate;

import net.hizlab.kagetaka.Resource;
import net.hizlab.kagetaka.awt.ImageCreator;
import net.hizlab.kagetaka.awt.InnerScrollbar;
import net.hizlab.kagetaka.awt.Text;
import net.hizlab.kagetaka.rendering.Option;

import java.awt.BorderLayout;
import java.awt.Dialog;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.GridLayout;
import java.awt.Frame;
import java.awt.Image;
import java.awt.Insets;
import java.awt.Panel;
import java.awt.Point;
import java.awt.SystemColor;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.InputEvent;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;

/**
 * ɽΥܥǤ
 *
 * @kagetaka.bugs ľ󲽤ϡꥹʤ¸ʤޤ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.7 $
 */
public class TextArea extends TextComponent {
    private static final String RESOURCE    = "net.hizlab.kagetaka.awt.tate.Resources";
    private static final String BASE        = "textarea";
    private static final int    INPUTFIX    = 3;
    private static       int    nameCounter = 0;

    private static final int SCROLLBAR_SIZE = 16;

    /** ξ */
    public static final int SCROLLBARS_BOTH            = java.awt.TextArea.SCROLLBARS_BOTH;
    /** ĤΤ */
    public static final int SCROLLBARS_VERTICAL_ONLY   = java.awt.TextArea.SCROLLBARS_HORIZONTAL_ONLY;
    /** Τ */
    public static final int SCROLLBARS_HORIZONTAL_ONLY = java.awt.TextArea.SCROLLBARS_VERTICAL_ONLY;
    /** ̵ */
    public static final int SCROLLBARS_NONE            = java.awt.TextArea.SCROLLBARS_NONE;

    /** @serial СβĻ° */
    private int       scrollbarVisibility;
    /** @serial С */
    private Scrollbar scrollbarHorizontal;
    /** @serial ĥС */
    private Scrollbar scrollbarVertical;

    /**
     * 󥹥󥹤ޤ
     *
     * @param  option ץ
     * @param  ic     ᡼ꥨ
     */
    public TextArea(Option option, ImageCreator ic) {
        this(option, ic, null, 1, 20, SCROLLBARS_BOTH);
    }

    /**
     * ꤵ줿ƥȤޤ।󥹥󥹤ޤ
     *
     * @param  option  ץ
     * @param  ic      ᡼ꥨ
     * @param  text    ƥ
     */
    public TextArea(Option option, ImageCreator ic, String text) {
        this(option, ic, text, 1, 20, SCROLLBARS_BOTH);
    }

    /**
     * ꤵ줿Υ󥹥󥹤ޤ
     *
     * @param  option  ץ
     * @param  ic      ᡼ꥨ
     * @param  rows    Կ
     * @param  columns 
     */
    public TextArea(Option option, ImageCreator ic, int rows, int columns) {
        this(option, ic, null, rows, columns, SCROLLBARS_BOTH);
    }

    /**
     * ꤵ줿ƥȤΥ󥹥󥹤ޤ
     *
     * @param  option  ץ
     * @param  ic      ᡼ꥨ
     * @param  text    ƥ
     * @param  rows    Կ
     * @param  columns 
     */
    public TextArea(Option option, ImageCreator ic, String text,
                    int rows, int columns) {
        this(option, ic, text, rows, columns, SCROLLBARS_BOTH);
    }

    /**
     * ꤵ줿ƥȤΥ󥹥󥹤ޤ
     *
     * @param  option  ץ
     * @param  ic      ᡼ꥨ
     * @param  text    ƥ
     * @param  rows    Կ
     * @param  columns 
     * @param  scrollbars СβĻ°
     */
    public TextArea(Option option, ImageCreator ic, String text,
                    int rows, int columns, int scrollbars) {
        super(option, ic, text, rows, columns, true);
        this.scrollbarVisibility = scrollbars;

        setFrameSize(INSET_V,
                     INSET_H,
                     INSET_V + (scrollbars == SCROLLBARS_BOTH || scrollbars == SCROLLBARS_HORIZONTAL_ONLY ? SCROLLBAR_SIZE : 0),
                     INSET_H + (scrollbars == SCROLLBARS_BOTH || scrollbars == SCROLLBARS_VERTICAL_ONLY   ? SCROLLBAR_SIZE : 0));
    }

    /**
     * ݡͥȤƥʤɲä줿ȤΤޤ
     */
    public void addNotify() {
        synchronized (getTreeLock()) {
            super.addNotify();
            if (/*---*/scrollbarVisibility == SCROLLBARS_BOTH
                    || scrollbarVisibility == SCROLLBARS_HORIZONTAL_ONLY) {
                scrollbarHorizontal = new Scrollbar(Scrollbar.HORIZONTAL);
            }
            if (/*---*/scrollbarVisibility == SCROLLBARS_BOTH
                    || scrollbarVisibility == SCROLLBARS_VERTICAL_ONLY  ) {
                scrollbarVertical   = new Scrollbar(Scrollbar.VERTICAL  );
            }
        }
    }

    /**
     * ݡͥȤƥʤ줿ȤΤޤ
     */
    public void removeNotify() {
        synchronized (getTreeLock()) {
            super.removeNotify();
            if (scrollbarHorizontal != null) {
                scrollbarHorizontal.dispose();
                scrollbarHorizontal = null;
            }
            if (scrollbarVertical   != null) {
                scrollbarVertical  .dispose();
                scrollbarVertical  = null;
            }
        }
    }

    /**
     * ΥƥȥꥢΥѥ᡼ʸ֤ޤ
     *
     * @return ѥ᡼ʸ
     */
    protected String paramString() {
        String str = super.paramString();
        String sbVisStr;
        switch (scrollbarVisibility) {
        case SCROLLBARS_BOTH:
            sbVisStr = "both";
            break;
        case SCROLLBARS_VERTICAL_ONLY:
            sbVisStr = "vertical-only";
            break;
        case SCROLLBARS_HORIZONTAL_ONLY:
            sbVisStr = "horizontal-only";
            break;
        case SCROLLBARS_NONE:
            sbVisStr = "none";
            break;
        default:
            sbVisStr = "invalid display policy";
        }
        return str + ",rows=" + getRows()
                   + ",columns=" + getColumns()
                   + ", scrollbarVisibility=" + sbVisStr;
    }
    /** {@inheritDoc} */
    String getComponentName() {
        return BASE + nameCounter++;
    }

    /** {@inheritDoc} */
    protected void refresh(Image offscreen, Graphics g, Dimension size,
                           int state, boolean focus) {
        super.refresh(offscreen, g, size, state, focus);
        Text text = this.text;

        Point     viewPosition = getViewPosition();
        Insets    frameSize    = getFrameSize();
        Dimension textSize     = (text != null
                                  ? text.getSize((getWordwrap()
                                                  ? size.height - frameSize.top - frameSize.bottom
                                                  : 0))
                                  : new Dimension(0, 0));
        if (scrollbarHorizontal != null) {
            int page = size.width  - frameSize.left - frameSize.right;
            scrollbarHorizontal.setBounds(2,
                                          size.height - SCROLLBAR_SIZE - 2,
                                          size.width  - (scrollbarVertical   != null ? SCROLLBAR_SIZE : 0) - 4,
                                          SCROLLBAR_SIZE);
//Debug.out.println(viewPosition.x + "," + page + "," + 0 + "," + textSize.width);
            scrollbarHorizontal.setReverseValues (viewPosition.x, page, 0, textSize.width);
            scrollbarHorizontal.setBlockIncrement(Math.max(10, page - 10));
            scrollbarHorizontal.setUnitIncrement (10);
            scrollbarHorizontal.setEnabled       (scrollbarHorizontal.canScroll());
            scrollbarHorizontal.paint(g, true);
        }
        if (scrollbarVertical != null) {
            int page = size.height - frameSize.top - frameSize.bottom;
            scrollbarVertical  .setBounds(size.width  - SCROLLBAR_SIZE - 2,
                                          2,
                                          SCROLLBAR_SIZE,
                                          size.height - (scrollbarHorizontal != null ? SCROLLBAR_SIZE : 0) - 4);
//Debug.out.println(viewPosition.y + "," + page + "," + 0 + "," + textSize.height);
            scrollbarVertical  .setValues        (viewPosition.y, page, 0, textSize.height);
            scrollbarVertical  .setBlockIncrement(Math.max(getLineHeight(), page - getLineHeight()));
            scrollbarVertical  .setUnitIncrement (getLineHeight());
            scrollbarVertical  .setEnabled       (scrollbarVertical.canScroll());
            scrollbarVertical  .paint(g, true);
        }
        if (scrollbarHorizontal != null && scrollbarVertical != null) {
            g.setColor(SystemColor.control);
            g.fillRect(size.width  - SCROLLBAR_SIZE - 2,
                       size.height - SCROLLBAR_SIZE - 2,
                       SCROLLBAR_SIZE, SCROLLBAR_SIZE);
        }
    }

    /** {@inheritDoc} */
    public int getColumns() {
        return super.getColumns();
    }

    /** {@inheritDoc} */
    public void setColumns(int columns) {
        super.setColumns(columns);
    }

    /** {@inheritDoc} */
    public int getRows() {
        return super.getRows();
    }

    /** {@inheritDoc} */
    public void setRows(int rows) {
        super.setRows(rows);
    }

    /** {@inheritDoc} */
    public boolean getWordwrap() {
        return super.getWordwrap();
    }

    /** {@inheritDoc} */
    public void setWordwrap(boolean b) {
        super.setWordwrap(b);
    }

    /** {@inheritDoc} */
    Window showInputBox(Frame owner, int x, int y) {
        Input input = new Input(owner);

        Dimension inputSize  = input.getSize();
        Dimension screenSize = getToolkit().getScreenSize();
        Point     point      = getLocationOnScreen();
        x += point.x;
        y += point.y;
        x -= inputSize.width  * x / screenSize.width;
        y -= inputSize.height * y / screenSize.height;
        input.setLocation(x, y);
        input.show();
        input.text.requestFocus();
        input.toFront();

        return input;
    }

//### Input
    /** ϥ */
    private final class Input extends Dialog implements ActionListener {
        private static final String COMMAND_OK     = "ok";
        private static final String COMMAND_CANCEL = "cancel";
        private java.awt.TextArea text;
        private boolean isClosed;

        /** 󥹥󥹤 */
        private Input(Frame parent) {
            super(parent, Resource.getMessage(RESOURCE, "input.title", null), false);

            text = new java.awt.TextArea(getText(), getRows(), getColumns(), scrollbarVisibility);
            text.setForeground(TextArea.this.getForeground());
            text.setBackground(TextArea.this.getBackground());
            text.setFont      (TextArea.this.getFont      ());
            if (isLocked()) {
                text.setEditable(false);
                setTitle(Resource.getMessage(RESOURCE, "view.title", null));
            }
            add(text);

            Panel panel;
            add(panel = new Panel(), BorderLayout.SOUTH);
            panel.setLayout(new GridLayout(1, 2, 10, 5));

            java.awt.Button button;
            panel.add(button = new java.awt.Button(Resource.getMessage(RESOURCE, "textarea.button.ok"    , null)));
            button.setActionCommand (COMMAND_OK    );
            button.addActionListener(this          );
            panel.add(button = new java.awt.Button(Resource.getMessage(RESOURCE, "textarea.button.cancel", null)));
            button.setActionCommand (COMMAND_CANCEL);
            button.addActionListener(this          );

            pack();
            // 礭򾯤
            Dimension size = getSize();
            setSize(size.width, size.height + INPUTFIX);

            // ꥹʤϿ
            addWindowListener(
                new WindowAdapter() {
                    /** ɥĤ */
                    public void windowClosing(WindowEvent e) {
                        close(false);
                    }

                    /** ƥ֤ǤϤʤʤä */
                    public void windowDeactivated(WindowEvent e) {
                        close(true);
                    }
                }
            );
            text.addKeyListener(
                new KeyAdapter() {
                    /** 줿 */
                    public void keyPressed(KeyEvent e) {
                        switch (e.getKeyCode()) {
                        case KeyEvent.VK_ESCAPE:
                            e.consume();
                            close(false);
                            break;
                        case KeyEvent.VK_ENTER:
                            if ((e.getModifiers() & InputEvent.CTRL_MASK) != 0) {
                                e.consume();
                                close(true);
                            }
                            break;
                        default: // AVOID
                        }
                    }
                }
            );
        }

        /** Ĥ */
        private void close(boolean commit) {
            synchronized (this) {
                if (isClosed) {
                    return;
                }
                isClosed = true;
            }

            if (commit) {
                setText(text.getText());
            }
            dispose();
        }

        /**  */
        public void update(Graphics g) {
            paint(g);
        }

        /**  */
        public void actionPerformed(ActionEvent e) {
            close((COMMAND_OK.compareTo(e.getActionCommand()) == 0));
        }
    }

//### Scrollbar
    /** С */
    private final class Scrollbar extends InnerScrollbar {
        private int orientation;

        /** 󥹥󥹤 */
        private Scrollbar(int orientation) {
            super(orientation, TextArea.this);
            this.orientation = orientation;
        }

        /**  */
        public void repaint() {
            repaintForce();
        }

        /** ͤѹ줿 */
        public void changedValue() {
            Point viewPosition = getViewPosition();
            if (orientation == Scrollbar.HORIZONTAL) {
                int x = getReverseValue();
                if (x == viewPosition.x) {
                    return;
                }
                setViewPosition(x, viewPosition.y);
            } else {
                int y = getValue();
                if (y == viewPosition.y) {
                    return;
                }
                setViewPosition(viewPosition.x, y);
            }
            repaintForce();
        }
    }
}
