/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.awt.tate;

import net.hizlab.kagetaka.awt.Border;
import net.hizlab.kagetaka.awt.GraphicsUtils;
import net.hizlab.kagetaka.awt.ImageCreator;
import net.hizlab.kagetaka.awt.InnerScrollbar;
import net.hizlab.kagetaka.awt.Text;
import net.hizlab.kagetaka.awt.image.GrayFilter;
import net.hizlab.kagetaka.rendering.Option;
import net.hizlab.kagetaka.util.Environment;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.SystemColor;
import java.awt.event.InputEvent;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionListener;
import java.util.Vector;

/**
 * ɽΥܥܥåǤ
 *
 * @kagetaka.bugs ľ󲽤ϡꥹʤ¸ʤޤ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.8 $
 */
public class List extends Component {
    private static final String BASE        = "list";
    private static       int    nameCounter = 0;

    private static final int   INSET_LIST_H   = 2;
    private static final int   INSET_LIST_V   = 2;
    private static final int   INSET_NONE_H   = 1;
    private static final int   INSET_NONE_V   = 1;
    private static final int   INSET_S        = 1;
    private static final int   SB_H           = 16;
    private static final int   MOVE_CLICK     = 1;
    private static final int   MOVE_DRAG      = 2;
    private static final int   MOVE_KEY       = 3;

    /** @serial Υڡ */
    private int insetWidth;
    /** @serial ĤΥڡ */
    private int insetHeight;

    /** @serial Կ */
    private int        rows;
    /** @serial ʣ⡼ɤ */
    private boolean    multipleMode;
    /** @serial ꥹȥ⡼ɤ */
    private boolean    listMode;
    /** @serial ٥ʸ */
    private Vector     texts         = new Vector();
    /** @serial 򤵤Ƥ륤ǥå */
    private int        selectedIndex = -1;
    /** @serial ƥ */
    private Vector     selections    = null;
    /** @serial եΤ륤ǥå */
    private int        currentIndex  = 0;
    /** @serial ϥեΤ륤ǥå */
    private int        currentIndex2 = 0;
    /** @serial ޥ󤷤ǥå */
    private int        currentIndex3 = 0;
    /** @serial ɽϥǥå */
    private int        viewStart     = 0;
    /** @serial ɽλǥå */
    private int        viewEnd       = 0;
    /** @serial ɽ륤ǥå */
    private int        visibleIndex  = -1;
    /** @serial ɽ륤ǥå׵ */
    private int        visibleReq    = -1;
    /** @serial 1 Ԥ */
    private int        rowWidth      = -1;
    /** @serial С */
    private Scrollbar  scrollbar     = null;
    /** @serial Сɽɬפ뤫 */
    private boolean    visibleScrollbar = true;
    /** @serial Ǹʣ򤬡֤ɤ */
    private boolean    lastMultipleSelected = true;
    /** @serial Ԥå */
    private AutoScroll autoScroll;


    /**
     * ɽԤʣʤ
     * ꥹȤޤ
     *
     * @param  option ץ
     * @param  ic     ᡼ꥨ
     */
    public List(Option option, ImageCreator ic) {
        this(option, ic, 0, false, true);
    }

    /**
     * ꤵ줿ɽԤʣʤ
     * ꥹȤ롣
     *
     * @param  option ץ
     * @param  ic     ᡼ꥨ
     * @param  rows   Կ
     */
    public List(Option option, ImageCreator ic, int rows) {
        this(option, ic, rows, false, true);
    }

    /**
     * ꤵ줿ɽԤʣɤ򼨤
     * ֡ͤǽ줿ꥹȤ롣
     *
     * @param  option ץ
     * @param  ic     ᡼ꥨ
     * @param  rows   Կ
     * @param  multipleMode <code>true</code> ξ硢ʣ򤬲ǽ
     */
    public List(Option option, ImageCreator ic, int rows, boolean multipleMode) {
        this(option, ic, rows, multipleMode, true);
    }

    /**
     * ꥹȥ⡼ɤꤷƥꥹȤޤ
     *
     * @param  option ץ
     * @param  ic     ᡼ꥨ
     * @param  rows   Կ
     * @param  multipleMode <code>true</code> ξ硢ʣ򤬲ǽ
     * @param  listMode     ꥹȥ⡼ɤ <code>true</code>
     *                      ܥ⡼ɤ <code>false</code>
     */
    List(Option option, ImageCreator ic, int rows, boolean multipleMode, boolean listMode) {
        super(option, ic);
        this.rows     = rows;
        setMultipleMode(multipleMode);
        this.listMode = listMode;
        if (listMode) {
            insetWidth  = INSET_LIST_H;
            insetHeight = INSET_LIST_V;
        } else {
            insetWidth  = INSET_NONE_H;
            insetHeight = INSET_NONE_V;
        }

        addKeyListener(
            new KeyAdapter() {
                /** 줿 */
                public void keyPressed(KeyEvent e) {
                    if (!isEnabled()) {
                        return;
                    }

                    boolean isShift = ((e.getModifiers() & InputEvent.SHIFT_MASK) != 0);
                    boolean isCtrl  = ((e.getModifiers() & InputEvent.CTRL_MASK ) != 0);

                    switch (e.getKeyCode()) {
                    case KeyEvent.VK_LEFT:
                    case KeyEvent.VK_DOWN:
                    {
                        e.consume();
                        int index = currentIndex;
                        if (index + 1 >= texts.size()) {
                            return;
                        }

                        index++;
                        moveCurrent(index, MOVE_KEY, isShift, isCtrl);

                        break;
                    }
                    case KeyEvent.VK_RIGHT:
                    case KeyEvent.VK_UP:
                    {
                        e.consume();
                        int index = currentIndex;
                        if (index <= 0) {
                            return;
                        }

                        index--;
                        moveCurrent(index, MOVE_KEY, isShift, isCtrl);

                        break;
                    }
                    case KeyEvent.VK_SPACE:
                    {
                        e.consume();
                        boolean selected = false;
                        synchronized (List.this) {
                            if (currentIndex == -1 || currentIndex >= texts.size()) {
                                return;
                            }
                            if (List.this.multipleMode) {
                                selected = isIndexSelected(currentIndex);
                            }
                        }
                        if (selected) {
                            deselect(currentIndex);
                        } else {
                            select(currentIndex);
                        }
                        break;
                    }
                    default: // AVOID
                    }
                }
            }
        );

        addMouseListener(
            new MouseAdapter() {
                /** ޥ줿 */
                public void mousePressed(MouseEvent e) {
                    if (!isEnabled()) {
                        return;
                    }

                    e.consume();

                    if (scrollbar.contains(e.getX(), e.getY(), false)) {
                        return;
                    }

                    moveCurrentFromPoint(e.getX(), e.getY(), MOVE_CLICK, true,
                                         ((e.getModifiers() & InputEvent.SHIFT_MASK) != 0),
                                         ((e.getModifiers() & InputEvent.CTRL_MASK ) != 0));
                }

                /** ޥ줿 */
                public void mouseReleased(MouseEvent e) {
                    if (!isEnabled()) {
                        return;
                    }

                    e.consume();

                    if (scrollbar.contains(e.getX(), e.getY(), false)) {
                        return;
                    }

                    synchronized (List.this) {
                        if (autoScroll != null) {
                            autoScroll.interrupt();
                            autoScroll = null;
                        }
                    }
                }
            }
        );

        addMouseMotionListener(
            new MouseMotionListener() {
                /** ޥɥå줿 */
                public void mouseDragged(MouseEvent e) {
                    if (!isEnabled()) {
                        return;
                    }

                    e.consume();

                    if (scrollbar.contains(e.getX(), e.getY(), true)) {
                        return;
                    }

                    moveCurrentFromPoint(e.getX(), e.getY(), MOVE_DRAG, true,
                                         ((e.getModifiers() & InputEvent.SHIFT_MASK) != 0),
                                         ((e.getModifiers() & InputEvent.CTRL_MASK ) != 0));
                }

                /** ޥư */
                public void mouseMoved(MouseEvent e) {
                    if (!isEnabled()) {
                        return;
                    }

                    e.consume();

                    if (scrollbar.contains(e.getX(), e.getY(), false)) {
                        return;
                    }

                    if (!List.this.listMode) {
                        moveCurrentFromPoint(e.getX(), e.getY(), MOVE_DRAG, false, false, false);
                    }
                }
            }
        );

        setForeground(SystemColor.textText);
        if (Environment.javaVersion < 102) {
            setBackground(SystemColor.window  );
        } else {
            setBackground(SystemColor.text    );
        }
    }

    /**
     * Ǿ֤ޤ
     *
     * @return Ǿ
     */
    public Dimension getMinimumSize() {
        return getPreferredSize();
    }

    /** {@inheritDoc} */
    public void setEnabled(boolean b) {
        if (!b) {
            synchronized (this) {
                if (autoScroll != null) {
                    autoScroll.interrupt();
                    autoScroll = null;
                }
            }
        }

        super.setEnabled(b);
    }

    /**
     * ݡͥȤƥʤɲä줿ȤΤޤ
     */
    public void addNotify() {
        synchronized (getTreeLock()) {
            super.addNotify();
            scrollbar = new Scrollbar();
            scrollbar.setVisible(visibleScrollbar);
        }
    }

    /**
     * ݡͥȤƥʤ줿ȤΤޤ
     */
    public void removeNotify() {
        synchronized (getTreeLock()) {
            super.removeNotify();
            if (scrollbar != null) {
                scrollbar.dispose();
                scrollbar = null;
            }
        }
    }

    /**
     * ΥꥹȤΥѥ᡼ʸ֤ޤ
     *
     * @return ѥ᡼ʸ
     */
    protected String paramString() {
        String str = super.paramString();
        return str + ",selected=" + getSelectedItem();
    }

    /** {@inheritDoc} */
    String getComponentName() {
        return BASE + nameCounter++;
    }

    /** {@inheritDoc} */
    protected Dimension createPreferredSize() {
        synchronized (this) {
            // Ǿͤ
            Dimension size = getText("*", (char) 0).getSize(0);
            int w = size.width;
            int h = size.height;

            Font font = getFont();
            Text text;
            for (int i = texts.size() - 1; i >= 0; i--) {
                texts.setElementAt(text = ((Text) texts.elementAt(i)).getText(font), i);
                size = text.getSize(0);
                w = Math.max(w, size.width );
                h = Math.max(h, size.height);
            }

            int row = (listMode
                       ? (rows > 0 ? rows : texts.size())
                       : Math.min(rows, texts.size()));

            rowWidth = w + INSET_S * 2;
            // Сɽ/ɽڤؤ
            visibleScrollbar = (row < texts.size());
            Scrollbar scrollbar = this.scrollbar;
            if (scrollbar != null) {
                scrollbar.setVisible(visibleScrollbar);
            }

            return new Dimension((w + INSET_S * 2) * row + insetWidth * 2,
                                 h + INSET_S * 2 + insetHeight * 2 + SB_H);
        }
    }

    /** {@inheritDoc} */
    protected void refresh(Image offscreen, Graphics g, Dimension size,
                           int state, boolean focus) {
        Color fg = getForeground();
        Color bg = getBackground();
        Font font = getFont();

        // ƥȤ
        Vector texts;
        int lineNum = 0, viewNum = 0;

        synchronized (this) {
            texts = this.texts;
            Text text;
            for (int i = texts.size() - 1; i >= 0; i--) {
                texts.setElementAt(((Text) texts.elementAt(i)).getText(font), i);
            }

            texts = (Vector) texts.clone();
            if (multipleMode) {
                selections = (Vector) this.selections.clone();
            }

            // Ԥ
            lineNum = texts.size();
            viewNum = size.width / rowWidth;
            if (lineNum == 0) {
                return;
            }

            if (lineNum <= viewNum) {
                viewStart = 0;
            } else if (visibleReq != -1 && visibleReq < viewStart) {
                viewStart = visibleReq;
            } else if (visibleReq != -1 && visibleReq > viewEnd) {
                viewStart = Math.max(0, visibleReq - viewNum + 1);
            } else if (viewStart + viewNum > lineNum) {
                viewStart = Math.max(0, lineNum - viewNum);
            }
            viewEnd = Math.min(viewStart + viewNum - 1, lineNum - 1);
            visibleReq = -1;
        }

        // طʿɤ
        g.setColor(bg);
        g.fillRect(0, 0, size.width, size.height);

        Color color = fg;
        if (state == DISABLE) {
            color = new Color((new GrayFilter(bg)).filterRGB(0, 0, color.getRGB()));
        }
        g.setFont(font);
        int x = size.width - insetWidth - 1;
        int y = insetHeight;
        int w = rowWidth;
        int h = size.height - insetHeight * 2;
        if (scrollbar.isVisible()) {
            h -= SB_H;
        }

        for (int i = viewStart; i <= viewEnd; i++) {
            // 
            if (isIndexSelected(i)) {
                g.setColor(SystemColor.textHighlight);
                g.fillRect(x - w + 1, y, w, h);
                g.setColor(SystemColor.textHighlightText);
            } else {
                g.setColor(color);
            }

            // ٥
            ((Text) texts.elementAt(i)).draw(g, x + INSET_S, y + INSET_S, 0);

            if (focus && i == currentIndex) {
                GraphicsUtils.drawDashed(g, x - w + 1, y, w, 1, 1, 1, GraphicsUtils.HORIZONTAL);
                GraphicsUtils.drawDashed(g, x - w + 1, y, 1, h, 1, 1, GraphicsUtils.VERTICAL  );
                GraphicsUtils.drawDashed(g, x - w + 1, y + h - 1, w, 1, 1, 1, GraphicsUtils.HORIZONTAL);
                GraphicsUtils.drawDashed(g, x, y, 1, h, 1, 1, GraphicsUtils.VERTICAL  );
            }

            x -= rowWidth;
        }

        // Ȥ
        Border border = getBorder();
        if (border != null) {
            border.draw(g, 0, 0, size.width - 1, size.height - 1, fg);
        } else {
            if (listMode) {
                g.setColor(SystemColor.controlShadow);
                g.drawLine(0, 0, size.width - 1,  0);
                g.drawLine(0, 0, 0, size.height - 1);
                g.setColor(SystemColor.controlDkShadow);
                g.drawLine(1, 1, size.width - 2,  1);
                g.drawLine(1, 1, 1, size.height - 2);
                g.setColor(SystemColor.controlHighlight);
                g.drawLine(size.width - 2,  2, size.width - 2, size.height - 2);
                g.drawLine(2, size.height - 2, size.width - 2, size.height - 2);
                g.setColor(SystemColor.controlLtHighlight);
                g.drawLine(size.width - 1,  1, size.width - 1, size.height - 1);
                g.drawLine(1, size.height - 1, size.width - 1, size.height - 1);
            } else {
                g.setColor(SystemColor.controlDkShadow);
                g.drawRect(0, 0, size.width - 1, size.height - 1);
            }
        }

        // С
        if (scrollbar.isVisible()) {
            scrollbar.setBounds(insetWidth,
                                size.height - SB_H - insetHeight,
                                size.width - insetWidth * 2,
                                SB_H);
            scrollbar.setReverseValues (viewStart, viewNum, 0, lineNum);
            scrollbar.setBlockIncrement(Math.max(1, viewNum - 1));
            scrollbar.paint(g, true);
        }
    }

    /**
     *  Choice ˹ܤɲäޤ
     *
     * @param  item ɲä
     */
    public synchronized void add(String item) {
        addItem(item);
    }

    /**
     *  Choice ˹ܤɲäޤ
     *
     * @param  item ɲä
     */
    public synchronized void addItem(String item) {
        addItem(getText(item, (char) 0));
    }

    /**
     *  Choice ˹ܤɲäޤ
     *
     * @param  item ɲä
     */
    public synchronized void addItem(Text item) {
        addItem(item, texts.size());
    }

    /**
     *  Choice λꤵ줿֤˹ܤɲäޤ
     *
     * @param  item  ɲä
     * @param  index ɲä
     */
    public synchronized void add(String item, int index) {
        addItem(item, index);
    }

    /**
     *  Choice λꤵ줿֤˹ܤɲäޤ
     *
     * @param  item  ɲä
     * @param  index ɲä
     */
    public synchronized void addItem(String item, int index) {
        addItem(getText(item, (char) 0), index);
    }

    /**
     *  Choice λꤵ줿֤˹ܤɲäޤ
     *
     * @param  item  ɲä
     * @param  index ɲä
     */
    public synchronized void addItem(Text item, int index) {
        texts.insertElementAt(item, index);
        if (multipleMode) {
            selections.insertElementAt(Boolean.FALSE, index);
        }

        if (viewStart <= index && index <= viewEnd) {
            repaintForce();
        }
    }

    /**
     *  Choice λꤵ줿֤ιܤ֤ޤ
     *
     * @param  item  ܤ
     * @param  index ֤
     */
    public synchronized void replaceItem(String item, int index) {
        texts.setElementAt(getText(item, (char) 0), index);

        if (viewStart <= index && index <= viewEnd) {
            repaintForce();
        }
    }

    /**
     * Choice λꤵ줿źʸ֤ޤ
     *
     * @param  index ܤź
     *
     * @return ʸ
     */
    public String getItem(int index) {
        return (String) texts.elementAt(index);
    }

    /**
     * ꥹȤˤ륢ƥ֤ޤ
     *
     * @return ƥꥹ
     */
    public synchronized String[] getItems() {
        String[] lists = new String[texts.size()];
        for (int i = texts.size() - 1; i >= 0; i--) {
            lists[i] = ((Text) texts.elementAt(i)).getValue();
        }
        return lists;
    }

    /**
     *  Choice ꤷܤޤ
     *
     * @param  index ܤź
     */
    public synchronized void delItem(int index) {
        remove(index);
    }

    /**
     *  Choice ꤷܤޤ
     *
     * @param  index ܤź
     */
    public synchronized void remove(int index) {
        texts.removeElementAt(index);
        if (multipleMode) {
            selections.removeElementAt(index);
        }

        if (index == selectedIndex) {
            select(-1);
        }

        if (viewStart <= index && index <= viewEnd) {
            repaintForce();
        }
    }

    /**
     *  Choice ꤷܤޤ
     *
     * @param  item ܤʸ
     */
    public synchronized void remove(String item) {
        for (int i = texts.size() - 1; i >= 0; i--) {
            if (((Text) texts.elementAt(i)).getValue().compareTo(item) == 0) {
                remove(i);
                return;
            }
        }
    }

    /**
     *  Choice 餹٤Ƥιܤޤ
     */
    public synchronized void removeAll() {
        texts.removeAllElements();
        if (multipleMode) {
            selections.removeAllElements();
        }

        selectedIndex = -1;
        currentIndex  =  0;
        currentIndex2 =  0;
        currentIndex3 =  0;
        visibleReq    = -1;
        repaintForce();
    }

    /**
     *  Choice ι֤ܿޤ
     *
     * @return ܿ
     */
    public int getItemCount() {
        return texts.size();
    }

    /**
     * 򤵤Ƥܤź֤ޤ
     *
     * @return ιܤź
     */
    public synchronized int getSelectedIndex() {
        return selectedIndex;
    }

    /**
     * 򤵤Ƥܤź֤ޤ
     *
     * @return ιܤź
     */
    public synchronized int[] getSelectedIndexs() {
        if (!multipleMode) {
            return (selectedIndex == -1
                    ? new int[0]
                    : new int[]{selectedIndex});
        }

        int count = 0;
        for (int i = selections.size() - 1; i >= 0; i--) {
            if (((Boolean) selections.elementAt(i)).booleanValue()) {
                count++;
            }
        }

        int[] ret = new int[count];
        if (count > 0) {
            for (int i = selections.size() - 1; i >= 0; i--) {
                if (((Boolean) selections.elementAt(i)).booleanValue()) {
                    ret[--count] = i;
                }
            }
        }

        return ret;
    }

    /**
     * 򤵤Ƥܤʸ֤ޤ
     *
     * @return ιܤʸ
     */
    public synchronized String getSelectedItem() {
        if (selectedIndex == -1) {
            return null;
        }

        return getItem(selectedIndex);
    }

    /**
     * 򤵤Ƥܤʸ֤ޤ
     *
     * @return ιܤʸ
     */
    public synchronized String[] getSelectedItems() {
        if (!multipleMode) {
            return (selectedIndex == -1
                    ? new String[0]
                    : new String[]{((Text) texts.elementAt(selectedIndex)).getValue()});
        }

        int count = 0;
        for (int i = selections.size() - 1; i >= 0; i--) {
            if (((Boolean) selections.elementAt(i)).booleanValue()) {
                count++;
            }
        }

        String[] ret = new String[count];
        if (count > 0) {
            for (int i = selections.size() - 1; i >= 0; i--) {
                if (((Boolean) selections.elementAt(i)).booleanValue()) {
                    ret[--count] = ((Text) texts.elementAt(i)).getValue();
                }
            }
        }

        return ret;
    }

    /**
     * ꤵ줿ǥåˤ륢ƥब򤵤Ƥ뤫֤ޤ
     *
     * @param  index ܤΰ
     *
     * @return 򤵤Ƥ <code>true</code>
     *         ʳξ <code>false</code>
     */
    public synchronized boolean isIndexSelected(int index) {
        return (multipleMode
                ? ((Boolean) selections.elementAt(index)).booleanValue()
                : (selectedIndex == index));
    }

    /**
     * ꤵ줿֤ιܤ򤷤ޤ
     *
     * @param  index ܤΰ
     *
     * @throws IllegalArgumentException ܤΰ̵֤ξ
     */
    public synchronized void select(int index) {
        if (selectedIndex == index) {
            return;
        }

        boolean moveCurrent = (currentIndex == -1
                            || !multipleMode
                            || (multipleMode
                             && currentIndex < selections.size()
                             && !isIndexSelected(currentIndex)));

        selectedIndex = index;
        if (moveCurrent) {
            currentIndex  = index;
            currentIndex2 = index;
            currentIndex3 = index;
            visibleReq    = index;
        }

        if (multipleMode) {
            selections.setElementAt(Boolean.TRUE, index);
        }

        if (isVisible()) {
            repaintForce();
        }
    }

    /**
     * ꤵ줿֤ιܤޤ
     *
     * @param  index ܤΰ
     */
    public synchronized void deselect(int index) {
        if (selectedIndex == index) {
            selectedIndex = -1;
        }

        if (multipleMode) {
            selections.setElementAt(Boolean.FALSE, index);
        }

        if (isVisible()) {
            repaintForce();
        }
    }

    /**
     * ΥꥹȤɽԿ֤ޤ
     *
     * @return ɽԿ
     */
    public int getRows() {
        return rows;
    }

    /**
     * ΥꥹȤʣǤ뤫ɤ֤ޤ
     *
     * @return ʣĤξ <code>true</code>
     *         ʳξ <code>false</code>
     */
    public boolean getMultipleMode() {
        return multipleMode;
    }

    /**
     * ΥꥹȤʣǽˤ뤫ɤꤷޤ
     *
     * @param  b ʣĤˤ <code>true</code>
     *           ʳξ <code>false</code>
     */
    public synchronized void setMultipleMode(boolean b) {
        this.multipleMode = b;
        if (multipleMode) {
            selections = new Vector();
            for (int i = texts.size() - 1; i >= 0; i--) {
                selections.addElement(Boolean.FALSE);
            }
            if (selectedIndex != -1) {
                selections.setElementAt(Boolean.TRUE, selectedIndex);
            }
        } else {
            selections = null;
        }
    }

    /**
     * {@link #makeVisible(int)} ᥽åɤˤäƺǸɽ줿
     * ƥΥǥå֤ޤ
     *
     * @return ɽǥå
     */
    public int getVisibleIndex() {
        return visibleIndex;
    }

    /**
     * ꤵ줿ǥåˤ륢ƥŪɽޤ
     *
     * @param  index ɽ륢ƥΰ
     */
    public synchronized void makeVisible(int index) {
        this.visibleIndex = index;

        if (/*---*/(visibleIndex < viewStart || viewEnd < visibleIndex)
                && isVisible()) {
            visibleReq = index;
            repaintForce();
        }
    }

    /**
     * ꤵ줿ɸСξ夫ɤ֤ޤ
     *
     * @param  x X
     * @param  y Y
     *
     * @return Сξξ <code>true</code>
     *         ʳξ <code>false</code>
     */
    public boolean containsScrollbar(int x, int y) {
        Scrollbar scrollbar = this.scrollbar;
        return (scrollbar != null && scrollbar.contains(x, y, false));
    }

    /** ֤ˤ륤ǥå */
    private void moveCurrentFromPoint(int x, int y, int source, boolean auto,
                                      boolean isShift, boolean isCtrl) {
        int index = 0;

        synchronized (this) {
            Dimension size = getSize();

            if (source != MOVE_DRAG) {
                if (/*---*/x <= insetWidth  || size.width  - insetWidth  <= x
                        || y <= insetHeight || size.height - insetHeight <= y) {
                    return;
                }
            } else {
                // List ؤΥɥåǥ
                if (x <= insetWidth) {
                    if (auto && viewEnd + 1 < texts.size()) {
                        int d = -1 * (x - insetWidth);
                        if (autoScroll == null) {
                            (autoScroll = new AutoScroll(true, d, isShift, isCtrl)).start();
                        } else {
                            autoScroll.setDistance(true, d, isShift, isCtrl);
                        }
                    }
                    return;
                } else if (size.width  - insetWidth <= x) {
                    if (auto && viewStart > 0) {
                        int d = x - (size.width - insetWidth);
                        if (autoScroll == null) {
                            (autoScroll = new AutoScroll(false, d, isShift, isCtrl)).start();
                        } else {
                            autoScroll.setDistance(false, d, isShift, isCtrl);
                        }
                    }
                    return;
                } else {
                    if (autoScroll != null) {
                        autoScroll.interrupt();
                        autoScroll = null;
                    }
                }
            }

            int w = size.width - insetWidth * 2;
            int d = w % rowWidth;
            int last  = viewStart + w / rowWidth - 1;
            x -= insetWidth;

            if (x <= d) {
                index = last;
            } else {
                index = last - ((x - d) / rowWidth);
            }
//Debug.out.println("x=" + x + ",w=" + w + ",d=" + d + "vs=" + viewStart + ",last=" + last + ",index=" + index);
        }

        moveCurrent(index, source, isShift, isCtrl);
    }

    /** ꤷǥåإȤư */
    private void moveCurrent(int index, int source, boolean isShift, boolean isCtrl) {
        synchronized (this) {
            int size = texts.size();
            if (/*---*/index >= size
                    || (index == currentIndex && index == selectedIndex && !multipleMode)) {
                return;
            }

            currentIndex = index;
            if (!isShift) {
                currentIndex2 = index;
            }
            if (source == MOVE_CLICK) {
                currentIndex3 = index;
            }
            visibleReq   = index;

            if (multipleMode) {
                if (isShift) {
                    // Shift 򲡤줿֤ǡɥåå줿
                    int start  = Math.min(currentIndex, currentIndex2);
                    int end    = Math.max(currentIndex, currentIndex2);
                    for (int i = 0; i < start; i++) {
                        selections.setElementAt(Boolean.FALSE, i);
                    }
                    for (int i = start; i <= end; i++) {
                        selections.setElementAt(Boolean.TRUE , i);
                    }
                    for (int i = end + 1; i < size; i++) {
                        selections.setElementAt(Boolean.FALSE, i);
                    }
                } else if (isCtrl) {
                    switch (source) {
                    case MOVE_CLICK:
                        // Ctrl 򲡤줿֤ǡå줿
                        if (lastMultipleSelected = isIndexSelected(index)) {
                            selections.setElementAt(Boolean.FALSE, index);
                        } else {
                            selections.setElementAt(Boolean.TRUE , index);
                        }
                        break;
                    case MOVE_DRAG:
                        // Ctrl 򲡤줿֤ǡɥå줿
                        if (lastMultipleSelected) {
                            selections.setElementAt(Boolean.FALSE, index);
                        } else {
                            selections.setElementAt(Boolean.TRUE , index);
                        }
                        break;
                    case MOVE_KEY:
                        break;
                    default: // AVOID
                    }
                } else {
                    switch (source) {
                    case MOVE_DRAG:
                    {
                        // 򲡤줺ˡɥå줿
                        int start  = Math.min(currentIndex, currentIndex3);
                        int end    = Math.max(currentIndex, currentIndex3);
                        for (int i = 0; i < start; i++) {
                            selections.setElementAt(Boolean.FALSE, i);
                        }
                        for (int i = start; i <= end; i++) {
                            selections.setElementAt(Boolean.TRUE , i);
                        }
                        for (int i = end + 1; i < size; i++) {
                            selections.setElementAt(Boolean.FALSE, i);
                        }

                        break;
                    }
                    default:
                        // 򲡤줺ˡå줿
                        for (int i = 0; i < size; i++) {
                            selections.setElementAt(Boolean.FALSE, i);
                        }
                        selections.setElementAt(Boolean.TRUE, index);
                        break;
                    }
                }
            } else {
                selectedIndex = index;
            }
        }
        repaintForce();
    }

//### Scrollbar
    /** С */
    private final class Scrollbar extends InnerScrollbar {

        /** 󥹥󥹤 */
        private Scrollbar() {
            super(HORIZONTAL, List.this);
        }

        /**  */
        public void repaint() {
            repaintForce();
        }

        /** ͤѹ줿 */
        public void changedValue() {
            synchronized (List.this) {
                int v = getReverseValue();
                if (v == viewStart) {
                    return;
                }

                viewStart = v;
            }

            repaintForce();
        }
    }

//### AutoScroll
    /** ư */
    private final class AutoScroll extends Thread {
        private boolean forward;
        private int     interval;
        private boolean isShift;
        private boolean isCtrl;

        /** 󥹥󥹤 */
        private AutoScroll(boolean forward, int distance, boolean isShift, boolean isCtrl) {
            setDistance(forward, distance, isShift, isCtrl);
        }

        /** Υ */
        private void setDistance(boolean forward, int n, boolean isShift, boolean isCtrl) {
            this.forward  = forward;
            this.isShift  = isShift;
            this.isCtrl   = isCtrl;

            if (n < 4) {
                interval = 200;
            } else if (n < 8) {
                interval =  80;
            } else {
                interval =  20;
            }
//Debug.out.println(n + "," + interval);
        }

        /** ¹ */
        public void run() {
            try {
                for (;;) {
                    if (isInterrupted()) {
                        break;
                    }
                    sleep(interval);

                    int index = 0;
                    if (forward) {
                        if ((index = viewEnd + 1) >= texts.size()) {
                            break;
                        }
                    } else {
                        if ((index = viewStart - 1) < 0) {
                            break;
                        }
                    }

                    moveCurrent(index, MOVE_DRAG, isShift, isCtrl);
                }
            } catch (InterruptedException e) { }
        }
    }
}
