/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.awt.panel;

import net.hizlab.kagetaka.awt.Drawer;
import net.hizlab.kagetaka.awt.ImageCreator;
import net.hizlab.kagetaka.awt.InnerToolTip;
import net.hizlab.kagetaka.rendering.Background;
import net.hizlab.kagetaka.rendering.FormItem;
import net.hizlab.kagetaka.rendering.HawkContext;
import net.hizlab.kagetaka.rendering.ItemMap;

import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Insets;
import java.awt.Point;
import java.awt.Rectangle;
import java.util.Vector;

/**
 * 襹꡼󶡤ޤ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.9 $
 */
class Screen extends Container {
    private static final int TIP_BOTTOM = 22;

    /**  */
    static final int ITEM_LINK = 0;
    /** ֥Ȥ URL */
    static final int ITEM_URL  = 1;
    /**  */
    static final int ITEM_TIP  = 2;
    /**  */
    static final int ITEM_INFO = 3;

    /** @serial 륳ƥ */
    private HawkContext context;

    /** @serial ե꡼ */
    private Object    lock            = new Object();
    /** @serial ꡼礭 */
    private Dimension screenSize      = new Dimension(0, 0);
    /** @serial 褵Ƥ뱦ΰ */
    private Point     positionCurrent = new Point(0, 0);
    /** @serial 褹뱦ΰ */
    private Point     positionNext    = new Point(0, 0);

    /** @serial ᡼ */
    private Tile[] tiles = new Tile[10];
    /** @serial ᡼ο */
    private int    tilesCount;

    /** @serial طʾ */
    private Background background;
    /** @serial ġå */
    private ToolTip    toolTip;
    /** @serial եॢƥ */
    private Vector     formItems = new Vector();

    /**
     * 襹꡼ޤ
     *
     * @param  context 륳ƥ
     */
    Screen(HawkContext context) {
        this.context = context;
        setLayout(null);
    }

    /**
     * ֥å᡼ɲäޤ
     *
     * @param  index     ǥåɲäȤ <code>-1</code>
     * @param  image     ֥å᡼
     * @param  width     ֥å᡼
     * @param  height    ֥å᡼ι⤵
     * @param  itemMap   ƥޥå
     * @param  totalSize ٤ƤΥ֥å᡼碌礭ꤹ
     *
     * @return ǥå
     */
    int setImage(int index, Image image, int width, int height, ItemMap itemMap, Dimension totalSize) {
        Tile tile;
        int  tilesIndex;
        synchronized (lock) {
            boolean resize;

            if (index == -1) {
                tilesIndex = tilesCount++;

                // ĥ
                if (tilesIndex == tiles.length) {
                    int    newSize  = (int) (tilesIndex * (tilesIndex > 300 ? 1.5 : 2));
                    Tile[] newTiles = new Tile[newSize];
                    System.arraycopy(tiles, 0, newTiles, 0, tilesIndex);
                    tiles = newTiles;
                }

                tile   = tiles[tilesIndex] = new Tile((tilesIndex > 0 ? tiles[tilesIndex - 1] : null));
                resize = false;
            } else {
                tilesIndex = index;
                tile   = tiles[tilesIndex];
                resize = (tile.width != width);
                if (tile.image != null) {
                    tile.image.flush();
                }
            }

            // 
            tile.reset(image, width, height, itemMap);

            // ʬʹߤΰ֤Ĵ
            if (resize) {
                for (int i = tilesIndex + 1; i < tilesCount; i++) {
                    tiles[i].move();
                }
            }

            // ץ֥å򹹿
            if (index == -1) {
                totalSize.width  += width;
                totalSize.height =  Math.max(height, totalSize.height);
            } else {
                width = height = 0;
                for (int i = 0; i < tilesCount; i++) {
                    tile = tiles[i];
                    width  += tile.width;
                    height =  Math.max(tile.height, height);
                }
                totalSize.width  = width;
                totalSize.height = height;
            }
        }

        // ɽԤ
        if (isShowing()) {
            repaintImage(tile.right, 0, screenSize.width, screenSize.height);
        }

        return tilesIndex;
    }

    /**
     * եॢƥɲäޤ
     *
     * @param  item եॢƥ
     */
    void addFormItem(FormItem item) {
        Component c = item.getComponent();

        c.setVisible(false);
        add(c);

        synchronized (lock) {
            // ݡͥȤꥹȤϿ
            formItems.addElement(item);
            allocateFormItem(item,
                             positionCurrent.x, positionCurrent.y,
                             screenSize.width, screenSize.height);
        }
    }

    /**
     * ꡼ϰϤꤷƺɽޤ
     * ɸϱ夫εΥǤ
     *
     * @param  x ꡼α顢ɽ֤αüεΥ
     * @param  y ꡼ξ夫顢ɽ֤ξüεΥ
     * @param  width  ɽϰϤ
     * @param  height ɽϰϤι⤵
     */
    void repaintImage(int x, int y, int width, int height) {
//Debug.out.println(x + "," + y+"," + width + "," + height);
        if (isShowing()) {
            synchronized (lock) {
                repaint(positionCurrent.x + screenSize.width - (x + width), y, width, height);
            }
        }
    }

    /**
     * 򥯥ꥢơطʤꤷޤ
     *
     * @param  background طʾ
     */
    void setupScreen(Background background) {
        synchronized (lock) {
            positionCurrent.x = 0;
            positionCurrent.y = 0;
            positionNext.x    = 0;
            positionNext.y    = 0;

            for (int i = 0; i < tilesCount; i++) {
                if (tiles[i].image != null) {
                    tiles[i].image.flush();
                    tiles[i] = null;
                }
            }

            tilesCount = 0;

            if (formItems.size() > 0) {
                for (int i = formItems.size() - 1; i >= 0; i--) {
                    ((FormItem) formItems.elementAt(i)).dispose();
                }
                formItems.removeAllElements();
                removeAll();
            }

            this.background = background;
            super.setBackground((background != null
                                 ? background.getColor()
                                 : null));
        }
    }

    /**
     * 褹륤᡼ΰ֤ѹޤ
     *
     * @param  x 夫εΥ
     * @param  y 夫εΥ
     */
    void movePosition(int x, int y) {
        synchronized (lock) {
            positionNext.x = x;
            positionNext.y = y;
        }

        if (isShowing()) {
            repaint();
        }
    }

    /**
     * ꤵ줿֤Υ衢 URLᡢ֤ޤ
     *
     * @param  type 륿
     * @param  x    X ɸ
     * @param  y    Y ɸ
     *
     * @return פΥ֥ȡ
     *         ¸ߤʤ <code>null</code>
     *
     * @see    ItemMap
     */
    Object getItem(int type, int x, int y) {
        Tile tile;

        synchronized (lock) {
            if (tilesCount == 0) {
                return null;
            }

            x =  positionCurrent.x + screenSize.width - x;
            y += positionCurrent.y;

            int index = getTileIndex(x);
            if (index == -1) {
                return null;
            }

            tile =  tiles[index];
            x    -= tile.right;
        }

        switch (type) {
        case ITEM_LINK: return tile.itemMap.getLink(x, y);
        case ITEM_URL : return tile.itemMap.getURL (x, y);
        case ITEM_TIP : return tile.itemMap.getTip (x, y);
        case ITEM_INFO: return tile.itemMap.getInfo(x, y);
        default: // AVOID
            return null;
        }
    }

    /**
     * ġåפɽޤ
     *
     * @param  tip   ʸ
     * @param  x     X
     * @param  y     Y
     * @param  delay ɽޤǤԵ֡ʥߥá
     * @param  term  ɽ֡ʥߥá
     */
    void showToolTip(String tip, int x, int y, long delay, long term) {
        synchronized (lock) {
            if (toolTip == null) {
                toolTip = new ToolTip(context, delay, term, new Insets(0, 0, TIP_BOTTOM * 2, 0));
            } else {
                toolTip.setDelay(delay);
                toolTip.setTerm (term );
            }

            toolTip.showTip(tip, x, y + TIP_BOTTOM, screenSize.width, screenSize.height, getFont());
        }
    }

    /**
     * ġåפäޤ
     */
    void clearToolTip() {
        synchronized (lock) {
            if (toolTip != null) {
                toolTip.cancel();
            }
        }
    }

    /**
     * ѹбޤ
     */
    void componentResized() {
        synchronized (lock) {
            Dimension size = getSize();
            if (screenSize.equals(size)) {
                return;
            }

            // 岼˿Ӥ˾ΰ֤Ĵ
            if (size.height > screenSize.height) {
                positionNext.y = Math.max(positionNext.y - (size.height - screenSize.height), 0);
            }
        }

        if (isShowing()) {
            repaint();
        }
    }

    /**
     * ꥽ޤ
     */
    void cleanup() {
        synchronized (lock) {
            if (toolTip != null) {
                toolTip.dispose();
                toolTip = null;
            }

            setupScreen(null);
        }
    }

//### Override
    /**
     * 褷ޤ
     *
     * @param  g Graphics
     */
    public void update(Graphics g) {
        paint(g);
    }

    /**
     * 褷ޤ
     *
     * @param  g Graphics
     */
    public void paint(Graphics g) {
        synchronized (lock) {
            Dimension size = getSize();

            int oldX = positionCurrent.x;
            int oldY = positionCurrent.y;
            int posX = positionNext.x;
            int posY = positionNext.y;

            // եॢƥ֤
            if (formItems.size() > 0
                    && (!size.equals(screenSize)
                     || posX != oldX
                     || posY != oldY)) {
                allocateFormItems(posX, posY, size.width, size.height);
            }
            // ͤ¸
            screenSize        = size;
            positionCurrent.x = posX;
            positionCurrent.y = posY;

            int distX = size.width;
            int index;
            if (tilesCount > 0
                    && (index = getTileIndex(posX)) >= 0) {
                Tile tile = tiles[index];
                int  h    = size.height;
                int x1, x2, y2;

                // 뤬¦ˤϤ߽ФƤ
                distX -= (tile.right - posX);

                // 褹륿αüΰ֤Ĵ
                do {
                    if (tile.image != null) {
                        x2 = distX;
                        y2 = Math.min(tile.height - posY, h);
                        x1 = (distX -= tile.width);
                        g.drawImage(tile.image,
                                    x1, 0, x2, y2,
                                    0, posY, tile.width, Math.min(tile.height, h + posY),
                                    this);
                        // ¦;򤬤硢طʤԤ
                        if (y2 < h && background != null) {
                            drawBackground(g, x1, y2, x2, h, size.width, posX, posY);
                        }
                    }

                    index++;
                    tile = tile.next;
                } while (index < tilesCount && distX > 0);
            }

            // ¦;򤬤硢طʤԤ
            if (distX > 0 && background != null) {
                drawBackground(g, 0, 0, distX, size.height, size.width, posX, posY);
            }

            // ƥ
            paintItem(g);
        }
    }

    /**
     * ե뤳Ȥ뤫֤ޤ
     * ŪˤΥڥϡեܡɤˤǽʤΤǡ
     *  <code>true</code> ֤ޤ
     *
     * @return ե뤳ȤǤ뤫
     */
    public boolean isFocusTraversable() {
        return true;
    }

    /**
     * Υ꡼Υѥ᡼ʸ֤ޤ
     *
     * @return ѥ᡼ʸ
     */
    protected String paramString() {
        String str = super.paramString();
        return str;
    }

//### private
    /** ֤Υ֤֤ lock */
    private int getTileIndex(int x) {
        Tile[] tiles      = this.tiles;
        int    tilesCount = this.tilesCount;

        if (tiles[tilesCount - 1].left <= x) {
            return -1;
        }

        int start  = 0;
        int end    = tilesCount - 1;
        int middle;
        int right;

        while (start + 1 < end) {
            middle = (end + start) / 2;
            if ((right = tiles[middle].right) == x) {
                start = middle;
                break;
            }

            if (right < x) {
                start = middle;
            } else {
                end   = middle;
            }
        }

        //  0 Υξ硢Ʊ֤κǸΥ֤
        // Υ᥽åɤκǽǡlast.left <= x ǥåƤΤǡ
        // ΥޤǤδ֤ˤɬ width > 0 Υ뤬¸ߤ
        while (tiles[start].width == 0) {
            start++;
        }

        return start;
    }

    /** ϰϤطʿطʲԤ - ºݤ x,y ǻ( 0,0) lock */
    private void drawBackground(Graphics g, int x1, int y1, int x2, int y2,
                                int width, int posX, int posY) {
//Debug.out.println("a=" + x1 + "," + y1 + "," + x2 + "," + y2 + "," + posX + "," + posY);
        background.draw(new DirectDrawer(g),
                        x1, y1, x2 - x1, y2 - y1,
                        width,
                        posX + width - x2,
                        posY + y1);
    }

    /** եॢƥෲ֡ lock */
    private void allocateFormItems(int posX, int posY, int width, int height) {
        for (int i = formItems.size() - 1; i >= 0; i--) {
            allocateFormItem((FormItem) formItems.elementAt(i),
                             posX, posY, width, height);
        }
    }

    /** եॢƥ֡ lock */
    private void allocateFormItem(FormItem item,
                                  int posX, int posY,
                                  int width, int height) {
        Point     pos  = item.getPosition ();
        Dimension size = item.getSize     ();
        Component c    = item.getComponent();

        if (/*---*/posX <= pos.x + size.width
                && posY <= pos.y + size.height
                && posX + width  >= pos.x
                && posY + height >= pos.y) {
            c.setBounds(width - pos.x + posX - size.width,
                        pos.y - posY,
                        size.width,
                        size.height);
            c.setVisible(true);
        } else {
            c.setVisible(false);
        }
    }

    /** ƥʥեࡢġåסˤ lock */
    private void paintItem(Graphics g) {
        // ե
        Graphics  cg;
        Component c;
        Rectangle r;
        for (int i = formItems.size() - 1; i >= 0; i--) {
            if ((c = ((FormItem) formItems.elementAt(i)).getComponent()) != null) {
                if (!c.isVisible()) {
                    continue;
                }
                r  = c.getBounds();
                cg = g.create(r.x, r.y, r.width, r.height);

                try {
                    c.paintAll(cg);
                } finally {
                    cg.dispose();
                }
            }
        }

        // ġåפ
        if (toolTip != null) {
            toolTip.paint(g);
        }
    }

//### Tile
    /** ᡼ */
    private final class Tile {
        private Image   image;
        private ItemMap itemMap;
        private int     width;
        private int     height;
        private int     right;
        private int     left;

        private Tile  next;
        private Tile  prev;

        /** 󥹥󥹤 */
        private Tile(Tile prev) {
            this.prev = prev;
            if (prev != null) {
                prev.next = this;
            }
        }

        /**  */
        private void reset(Image image, int width, int height, ItemMap itemMap) {
            this.image   = image;
            this.width   = width;
            this.height  = height;
            this.itemMap = itemMap;
            move();
        }

        /** ֤ */
        private void move() {
            this.right = (prev != null ? prev.left : 0);
            this.left  = right + width;
        }
    }

//### ToolTip
    /** ġå */
    private final class ToolTip extends InnerToolTip {
        /** 󥹥󥹤 */
        private ToolTip(ImageCreator creator, long delay, long term, Insets fix) {
            super(creator, delay, term, fix);
        }

        /** ɽ */
        public void repaint(int x, int y, int width, int height) {
            Screen.this.repaint(x, y, width, height);
        }
    }

//### DirectDrawer
    /** @serial 襤󥿡ե */
    private final class DirectDrawer implements Drawer {
        private Graphics g;

        /** 󥹥󥹤 */
        private DirectDrawer(Graphics g) {
            this.g = g;
        }

        /**  */
        public void setColor(Color color) {
            g.setColor(color);
        }

        /** ɤĤ֤ */
        public void fillRect(int x, int y, int width, int height) {
            g.fillRect(x, y, width, height);
        }

        /**  */
        public void drawImage(Image image,
                              int dx1, int dy1, int dx2, int dy2,
                              int sx1, int sy1, int sx2, int sy2) {
            g.drawImage(image, dx1, dy1, dx2, dy2, sx1, sy1, sx2, sy2, Screen.this);
        }
    };
}
