/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.awt.panel;

import net.hizlab.kagetaka.Resource;
import net.hizlab.kagetaka.rendering.Document;
import net.hizlab.kagetaka.rendering.FormItem;
import net.hizlab.kagetaka.rendering.IdMap;
import net.hizlab.kagetaka.rendering.ItemMap;
import net.hizlab.kagetaka.rendering.HawkContext;
import net.hizlab.kagetaka.rendering.Link;
import net.hizlab.kagetaka.rendering.Option;
import net.hizlab.kagetaka.rendering.Reporter;
import net.hizlab.kagetaka.rendering.Request;
import net.hizlab.kagetaka.token.Value;

import java.awt.CheckboxMenuItem;
import java.awt.Component;
import java.awt.Container;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Image;
import java.awt.MediaTracker;
import java.awt.Menu;
import java.awt.MenuItem;
import java.awt.Point;
import java.awt.PopupMenu;
import java.awt.SystemColor;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.StringSelection;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.AdjustmentEvent;
import java.awt.event.AdjustmentListener;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.awt.event.InputEvent;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionAdapter;
import java.net.URL;
import java.util.Vector;

/**
 * 襹꡼ѥڥǤ
 * ϲ̱νĽɽʤΤǡ
 * ꥵʤɤԤޤ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.12 $
 */
public class HawkPanel extends Container {
    private static final long RESIZE_WAIT =  5000;
    private static final long TIP_DELAY   =   700;
    private static final long TIP_TERM    =  5000;
    private static final long EVENT_DELAY =    30;
    private static final long EVENT_SPAN  =    20;

    private static final int MENU_MENU  = 0;
    private static final int MENU_ITEM  = 1;
    private static final int MENU_CHECK = 2;

    private static final int    MENU_SEPARATOR                  = -1;
    /** ᥤ˥塼 */
    public  static final String MENU_MAIN_BACK                  =  "screen.menu.main.back";
    private static final int    MENU_MAIN_BACK_INDEX            =  2;
    /** ᥤ󡦼ؿʤ˥塼 */
    public  static final String MENU_MAIN_FORWARD               =  "screen.menu.main.forward";
    private static final int    MENU_MAIN_FORWARD_INDEX         =  3;
    /** ᥤߥ˥塼 */
    public  static final String MENU_MAIN_STOP                  =  "screen.menu.main.stop";
    private static final int    MENU_MAIN_STOP_INDEX            =  4;
    /** ᥤ󡦺ɹ˥塼 */
    public  static final String MENU_MAIN_RELOAD                =  "screen.menu.main.reload";
    private static final int    MENU_MAIN_RELOAD_INDEX          =  5;
    /** ᥤ󡦥ץ˥塼 */
    public  static final String MENU_MAIN_OPTION                =  "screen.menu.main.option";
    private static final int    MENU_MAIN_OPTION_INDEX          =  7;
    /** ᥤ󡦥ץ󡦲ɤ߹ޤʤ˥塼 */
    public  static final String MENU_MAIN_OPTION_IMAGE          =  "screen.menu.main.option.image";
    private static final int    MENU_MAIN_OPTION_IMAGE_INDEX    =  8;
    /** ᥤ󡦥ץ󡦥Сɽ˥塼 */
    public  static final String MENU_MAIN_OPTION_BAR            =  "screen.menu.main.option.bar";
    private static final int    MENU_MAIN_OPTION_BAR_INDEX      =  9;
    /** ᥤ󡦥ץ󡦲ž˥塼 */
    public  static final String MENU_MAIN_OPTION_SPIN           = "screen.menu.main.option.spin";
    private static final int    MENU_MAIN_OPTION_SPIN_INDEX     = 10;
    /** ᥤ󡦾˥塼 */
    public  static final String MENU_MAIN_INFO                  = "screen.menu.main.info";
    private static final int    MENU_MAIN_INFO_INDEX            = 11;
    /** 󥯡 URL 򥳥ԡ˥塼 */
    public  static final String MENU_LINK_COPY                  = "screen.menu.link.copy";
    private static final int    MENU_LINK_COPY_INDEX            = 24;
    /** ɽ˥塼 */
    public  static final String MENU_IMAGE_OPEN                 = "screen.menu.image.open";
    private static final int    MENU_IMAGE_OPEN_INDEX           = 32;
    /**  URL 򥳥ԡ˥塼 */
    public  static final String MENU_IMAGE_COPY                 = "screen.menu.image.copy";
    private static final int    MENU_IMAGE_COPY_INDEX           = 33;

    private static final int    MENU_SIZE                       = 34;

    /** Υ */
    public  static final int SCROLL_HORIZONTAL = 1;
    /** Υ */
    public  static final int SCROLL_VERTICAL   = 2;
    /** ˥åñ̤ǤΥ */
    public  static final int SCROLL_UNIT       = 0;
    /** ֥åñ̤ǤΥ */
    public  static final int SCROLL_BLOCK      = 1;

    /** Сɬɽ */
    public  static final int SCROLLBAR_YES     = 0;
    /** СɬפξΤɽ */
    public  static final int SCROLLBAR_AUTO    = 1;
    /** Сɬɽ */
    public  static final int SCROLLBAR_NO      = 2;


    //### 
    /** @serial 륳ƥ */
    private HawkContext    context;
    /** @serial ץ */
    private Option         option;
    /** @serial 쥤 */
    private GridBagLayout  gbl;
    /** @serial 륹꡼ */
    private Screen         screen;
    /** @serial С */
    private PanelScrollbar horizontal;
    /** @serial ĥС */
    private PanelScrollbar vertical;
    /** @serial Сɽ⡼ */
    private int            scrollbarMode;
    /** @serial Сɲäɤ */
    private boolean        horizontalVisible;
    /** @serial ĥСɲäɤ */
    private boolean        verticalVisible;
    /** @serial СѰդ򤹤뤫ɤ */
    private boolean        readyScrollbar;
    /** @serial ġåפȤ */
    private boolean        toolTipUse   = true;
    /** @serial ġåפɽޤǤԤ */
    private long           toolTipDelay = TIP_DELAY;
    /** @serial ġå׾äޤǤԤ */
    private long           toolTipTerm  = TIP_TERM;

    //### 
    /** @serial å */
    private Object     lock = new Object();;
    /** @serial ꡼󥵥 */
    private Dimension  screenSize;
    /** @serial ǥեȥӥ塼ݡȥ */
    private Dimension  defaultSize;
    /** @serial ǥեȥӥ塼ݡȥǲСɽƤ뤫 */
    private boolean    defaultSizeHorizontal;
    /** @serial ӥ塼ݡȥ */
    private Dimension  viewportSize;
    /** @serial 褹뱦ΰ */
    private Point      position;

    //### ˥塼
    /** @serial ƥȥ˥塼 */
    private PopupMenu  popupMenu;
    /** @serial ƥȥ˥塼ƥ */
    private MenuItem[] menuItems;
    /** @serial ᥤΥ˥塼ƥ */
    private Vector     mainMenuItems  = new Vector();
    /** @serial 󥯤Υ˥塼ƥ */
    private Vector     linkMenuItems  = new Vector();
    /** @serial ᡼Υ˥塼ƥ */
    private Vector     imageMenuItems = new Vector();

    //### 
    /** @serial ʸȹ⤵ */
    private Dimension     charScale;
    /** @serial ꥹ */
    private PanelListener listener;

    //### 
    /** @serial ǥեȤΥ륿 */
    private int    defaultCursorType   = Cursor.DEFAULT_CURSOR;
    /** @serial Ūʥ륿 */
    private int    temporaryCursorType = Cursor.DEFAULT_CURSOR;
    /** @serial ɽΥ륿 */
    private int    cursorType          = Cursor.DEFAULT_CURSOR;

    //### 󥯡ݥåץåץƥ
    /** @serial ɥ */
    private Document document;
    /** @serial Υ */
    private Link     currentLink;
    /** @serial Υ֥Ȥ URL */
    private URL      currentUrl;
    /** @serial ID ޥå */
    private IdMap    idMap;
    /** @serial ꥵ٥ȤΤ륿 */
    private int      resizeRequestTag;

    //### ¾
    /** @serial 롼 */
    private boolean callRequestFocus;

    /**
     * 襹꡼ѥڥޤ
     *
     * @param  context 륳ƥ
     * @param  h       ʿС
     * @param  v       ľС
     * @param  mode    Сɽ⡼ɡ
     *                 {@link Value#SCROLLING_AUTO}
     *                 {@link Value#SCROLLING_YES}
     *                 {@link Value#SCROLLING_NO} 
     */
    public HawkPanel(HawkContext context, PanelScrollbar h, PanelScrollbar v, int mode) {
        this.context       = context;
        this.option        = context.getOption();
        this.gbl           = new GridBagLayout();
        this.screen        = new Screen       (context);
        this.horizontal    = h;
        this.vertical      = v;
        this.scrollbarMode = mode;
        this.screenSize    = new Dimension(0, 0);
        this.viewportSize  = new Dimension(0, 0);
        this.position      = new Point    (0, 0);
        this.charScale     = new Dimension(0, 0);

        h.addAdjustmentListener(new ScrollbarListener(ScrollbarListener.HORIZONTAL, h));
        v.addAdjustmentListener(new ScrollbarListener(ScrollbarListener.VERTICAL  , v));

        // ̤
        setLayout(gbl);
        horizontal.getComponent().setVisible(false);
        vertical  .getComponent().setVisible(false);
        GridBagConstraints gbc = new GridBagConstraints();
        addComponent(screen                   , gbl, gbc, 1.0, 0, 0, GridBagConstraints.BOTH      );
        addComponent(horizontal.getComponent(), gbl, gbc, 0.0, 0, 1, GridBagConstraints.HORIZONTAL);
        addComponent(vertical  .getComponent(), gbl, gbc, 0.0, 1, 0, GridBagConstraints.VERTICAL  );
        fixScrollbar((scrollbarMode == Value.SCROLLING_YES), false, true);

        super.setBackground(SystemColor.control);

        // ˥塼
        initMenu();

        // ꥹʡ
        addFocusListener(
            new FocusAdapter() {
                public void focusGained(FocusEvent e) {
                    screen.requestFocus();
                }
            }
        );

        addComponentListener(new HawkPanelComponentListener());
        screen.addKeyListener        (new HawkScreenKeyListener        ());
        screen.addMouseListener      (new HawkScreenMouseListener      ());
        screen.addMouseMotionListener(new HawkScreenMouseMotionListener());
        screen.addComponentListener  (new HawkScreenComponentListener  ());
    }

    /** ȥɲäޤ */
    private void addComponent(Component c, GridBagLayout gbl, GridBagConstraints gbc,
                              double weight, int x, int y, int fill) {
        gbc.gridx   = x;
        gbc.gridy   = y;
        gbc.weightx = weight;
        gbc.weighty = weight;
        gbc.fill    = fill;
        gbl.setConstraints(c, gbc);
        add(c);
    }

    /** ˥塼 */
    private void initMenu() {
        Menu menu = null;
        menuItems = new MenuItem[MENU_SIZE];
        popupMenu = new PopupMenu(Resource.getMessage("screen.menu", null));
        screen.add(popupMenu);

        // ᥤ˥塼
        createMenuItem(MENU_MAIN_BACK_INDEX           , MENU_ITEM , null, mainMenuItems );
        createMenuItem(MENU_MAIN_FORWARD_INDEX        , MENU_ITEM , null, mainMenuItems );
        createMenuItem(MENU_SEPARATOR                 , MENU_ITEM , null, mainMenuItems );
        createMenuItem(MENU_MAIN_STOP_INDEX           , MENU_ITEM , null, mainMenuItems );
        createMenuItem(MENU_MAIN_RELOAD_INDEX         , MENU_ITEM , null, mainMenuItems );
        createMenuItem(MENU_SEPARATOR                 , MENU_ITEM , null, mainMenuItems );
        menu = (Menu)
        createMenuItem(MENU_MAIN_OPTION_INDEX         , MENU_MENU , null, mainMenuItems );
        createMenuItem(MENU_MAIN_OPTION_IMAGE_INDEX   , MENU_CHECK, menu, null          );
        createMenuItem(MENU_MAIN_OPTION_BAR_INDEX     , MENU_CHECK, menu, null          );
        createMenuItem(MENU_MAIN_OPTION_SPIN_INDEX    , MENU_CHECK, menu, null          );
        //### TODO
//        createMenuItem(MENU_MAIN_INFO_INDEX           , MENU_ITEM , null, mainMenuItems );

        // 󥯥˥塼
        createMenuItem(MENU_LINK_COPY_INDEX           , MENU_ITEM , null, linkMenuItems );

        // ˥塼
        createMenuItem(MENU_IMAGE_OPEN_INDEX          , MENU_ITEM , null, imageMenuItems);
        createMenuItem(MENU_IMAGE_COPY_INDEX          , MENU_ITEM , null, imageMenuItems);
    }

    /** ˥塼ܤ */
    private MenuItem createMenuItem(int index, int type, Menu parent, Vector items) {
        String menuKey = null;

        /// MENU
        switch (index) {
        case MENU_SEPARATOR                 : break;
        case MENU_MAIN_BACK_INDEX           : menuKey = MENU_MAIN_BACK           ; break;
        case MENU_MAIN_FORWARD_INDEX        : menuKey = MENU_MAIN_FORWARD        ; break;
        case MENU_MAIN_STOP_INDEX           : menuKey = MENU_MAIN_STOP           ; break;
        case MENU_MAIN_RELOAD_INDEX         : menuKey = MENU_MAIN_RELOAD         ; break;
        case MENU_MAIN_OPTION_INDEX         : menuKey = MENU_MAIN_OPTION         ; break;
        case MENU_MAIN_OPTION_IMAGE_INDEX   : menuKey = MENU_MAIN_OPTION_IMAGE   ; break;
        case MENU_MAIN_OPTION_BAR_INDEX     : menuKey = MENU_MAIN_OPTION_BAR     ; break;
        case MENU_MAIN_OPTION_SPIN_INDEX    : menuKey = MENU_MAIN_OPTION_SPIN    ; break;
        case MENU_MAIN_INFO_INDEX           : menuKey = MENU_MAIN_INFO           ; break;
        case MENU_LINK_COPY_INDEX           : menuKey = MENU_LINK_COPY           ; break;
        case MENU_IMAGE_OPEN_INDEX          : menuKey = MENU_IMAGE_OPEN          ; break;
        case MENU_IMAGE_COPY_INDEX          : menuKey = MENU_IMAGE_COPY          ; break;
        default:
            Reporter reporter = context.getReporter();
            if (reporter != null) {
                reporter.report(Reporter.RENDERING, Reporter.ERROR, Reporter.NONE,
                                null, 0, 0,
                                "Panel", Resource.getMessage("internal.error", new String[]{"HawkPanel.createMenuItem", "panel menu", ""}));
            }
            return null;
        }

        String label = (menuKey == null ? "-" : Resource.getMessage(menuKey, null));
        MenuItem mi = null;
        switch (type) {
        case MENU_MENU:
            mi = new Menu(label);
            break;
        case MENU_ITEM:
            mi = new MenuItem(label);
            if (index != MENU_SEPARATOR) {
                mi.addActionListener(new MenuActionListner(index));
            }
            break;
        case MENU_CHECK:
            mi = new CheckboxMenuItem(label);
            ((CheckboxMenuItem) mi).addItemListener(new MenuActionListner(index));
            break;
        default: // AVOID
        }

        mi.setActionCommand(menuKey != null ? menuKey : "-");

        if (index != MENU_SEPARATOR) {
            menuItems[index] = mi;
        }

        if (parent != null) {
            parent.add(mi);
        }
        if (items != null) {
            items.addElement(mi);
        }

        return mi;
    }

//### Original Method
    /**
     * 뤷ڥΥӥ塼ݡȤ 0, 0 ֤ɽθߤ
     * 夫 x, y ֤֤ޤ
     *
     * @return ߤΥ֤α夫κɸ
     */
    public Point getScrollPosition() {
        return new Point(position.x, position.y);
    }

    /**
     * ҥݡͥλꤵ줿֤إ뤷ޤ
     * ֤ϱ夫εΥˤʤޤ
     *
     * @param  x  x 
     * @param  y  y 
     * @param  force ֤˥ǤʤƤ⥹뤹 <code>true</code>
     *               ʳξ <code>false</code>
     *
     * @return 뤵줿 <code>true</code>
     *         Ǥʤä <code>false</code>
     */
    public boolean setScrollPosition(int x, int y, boolean force) {
        synchronized (lock) {
            if (!force
                    && ((x > 0 && x + viewportSize.width  > screenSize.width )
                     || (y > 0 && y + viewportSize.height > screenSize.height))) {
                return false;
            }

            setScrollPosition(x, y);
        }
        return true;
    }

    /**
     * ҥݡͥλꤵ줿ե󥹰֤إ뤷ޤ
     * ե󥹤¸ߤʤ䡢ɤ߹ޤƤʤϰưޤ
     *
     * @param  reference ե
     * @param  force ֤˥ǤʤƤ⥹뤹 <code>true</code>
     *               ʳξ <code>false</code>
     *
     * @return 뤵줿 <code>true</code>
     *         Ǥʤä <code>false</code>
     */
    public boolean setScrollPosition(String reference, boolean force) {
        if (reference == null || idMap == null) {
            return force;
        }

        Point position = idMap.getIdPoint(reference);
        if (position == null) {
            return force;
        }

        return setScrollPosition(position.x, position.y, force);
    }

    /**
     * ᥤΥ˥塼ƥ֤ޤ
     *
     * @return ᥤ˥塼ƥ
     */
    public Vector getMainMenuItems() {
        return mainMenuItems;
    }

    /**
     * 󥯤Υ˥塼ƥ֤ޤ
     *
     * @return 󥯥˥塼ƥ
     */
    public Vector getLinkMenuItems() {
        return linkMenuItems;
    }

    /**
     * ᡼Υ˥塼ƥ֤ޤ
     *
     * @return ᡼˥塼ƥ
     */
    public Vector getImageMenuItems() {
        return imageMenuItems;
    }

    /**
     * ѥͥꥹʤϿޤ
     *
     * @param  listener ѥͥꥹ
     */
    public void setPanelListener(PanelListener listener) {
        this.listener = listener;
    }

    /**
     * ǥȥå֤ޤ
     *
     * @return ǥȥå
     */
    public MediaTracker getMediaTracker() {
        return new MediaTracker(screen);
    }

    /**
     * ֥å᡼ꤷޤ
     *
     * @param  index   ǥåɲäȤ <code>-1</code>
     * @param  reason  ͳ
     * @param  tag     ꥯȤ̤륿
     *                 ꥵ٥ȤΰϤ줿ͤ
     *                  <code>0</code>
     * @param  image   ֥å᡼
     * @param  width   ֥å᡼
     * @param  height  ֥å᡼ι⤵
     * @param  itemMap ƥޥå
     *
     * @return ǥå
     */
    public int setImage(int index, int reason, int tag,
                        Image image, int width, int height, ItemMap itemMap) {
        synchronized (lock) {
            index = screen.setImage(index, image, width, height, itemMap, screenSize);

            int hmax = Math.max(screenSize.width , viewportSize.width );
            int vmax = Math.max(screenSize.height, viewportSize.height);
            horizontal.setMaximum(hmax);
            vertical  .setMaximum(vmax);
            horizontal.setValue(Math.max(0, hmax - horizontal.getVisibleAmount() - position.x));
            vertical  .setValue(Math.min(vmax, position.y));

//Debug.out.println(resizeRequestTag+","+tag+","+index+","+height+","+width+","+screenSize.height+","+viewportSize.height);
            if (resizeRequestTag == tag) {
//Debug.out.println("> panel.setImage : adjust");
                // СɽĴ
                if (scrollbarMode != Value.SCROLLING_NO) {
                    boolean h = (horizontalVisible
                              || screenSize.width > viewportSize.width);
                    boolean v = (verticalVisible
                              || height > viewportSize.height);
                    if (horizontalVisible != h || verticalVisible != v) {
                        fixScrollbar(h, v, false);
                    }
                }
            }
        }
        return index;
    }

    /**
     * եॢƥɲäޤ
     *
     * @param  item եॢƥ
     */
    public void addFormItem(FormItem item) {
        screen.addFormItem(item);
    }

    /**
     * ꤵ줿ꥯȤƤɽ뤿ˡѥͥޤ
     *
     * @param  document       ɥ
     * @param  readyScrollbar СѰդ <code>true</code>
     *                        Ѱդʤ <code>false</code>
     */
    public void setup(Document document, boolean readyScrollbar) {
        synchronized (lock) {
            this.document       = document;
            this.readyScrollbar = readyScrollbar;
            screen.setupScreen(document.getBackground());
            idMap = document.idMap;
            screenSize.width  = 0;
            screenSize.height = 0;
            position.x = 0;
            position.y = 0;
            horizontal.setMaximum(viewportSize.width );
            vertical  .setMaximum(viewportSize.height);

            charScale = document.getCharScale();
            horizontal.setUnitIncrement (charScale.width );
            vertical  .setUnitIncrement (charScale.height);
            horizontal.setBlockIncrement(viewportSize.width  - charScale.width );
            vertical  .setBlockIncrement(viewportSize.height - charScale.height);
            resizeRequestTag = 0;

            // ˱ƲСΤɽ
            boolean h = (readyScrollbar
                      && scrollbarMode == Value.SCROLLING_YES);
            if (horizontalVisible != h || verticalVisible) {
                // С֤Ʊ
                fixScrollbar(h, false, true);
                if (defaultSizeHorizontal != h) {
                    defaultSizeHorizontal = h;
                    defaultSize = null;
                }
                try {
                    lock.wait(RESIZE_WAIT);
                } catch (InterruptedException e) { }
            }
        }

        resetStatus();
    }

    /**
     * ѥͥꤷޤ
     *
     * @param  document ɥ
     * @param  tag      ꥯȤ̤륿
     *                  ꥵ٥ȤΰϤ줿ͤ
     *                   <code>0</code>
     */
    public void commit(Document document, int tag) {
        synchronized (lock) {
//Debug.out.println("> panel.commit: " + (this.document != document) + "," + resizeRequestTag + "," + tag);
            if (this.document != document
                    || resizeRequestTag != tag) {
                return;
            }

            // ѹ⤵ʤäˡ֤
            if (position.x > 0 || position.y > 0) {
                int x = Math.min(screenSize.width  - viewportSize.width , position.x);
                int y = Math.min(screenSize.height - viewportSize.height, position.y);
                if (x != position.x || y != position.y) {
                    setScrollPosition(x, y);
                }
            }

            if (readyScrollbar && scrollbarMode != Value.SCROLLING_NO) {
                // С;פʾɽˤ
                boolean h = (scrollbarMode == Value.SCROLLING_YES
                          || screenSize.width > viewportSize.width);
                boolean v = (screenSize.height > viewportSize.height);
                if (horizontalVisible != h || verticalVisible != v) {
//Debug.out.println("> panel.commit : fixed : " + h + "," + v + " : screen=" + screenSize.width + "x" + screenSize.height + ", default=" + defaultSize.width + "x" + defaultSize.width + "," + defaultSize.height);
                    fixScrollbar(h, v, false);
                }
            }
        }
    }

    /**
     * ꡼ϰϤꤷƺɽޤ
     * ɸϱ夫εΥǤ
     *
     * @param  x ꡼α顢ɽ֤αüεΥ
     * @param  y ꡼ξ夫顢ɽ֤ξüεΥ
     * @param  width  ɽϰϤ
     * @param  height ɽϰϤι⤵
     */
    public void repaintScreen(int x, int y, int width, int height) {
        synchronized (lock) {
            if (screenSize.width < x || screenSize.height < y) {
                return;
            }

            screen.repaintImage(x, y, width, height);
        }
    }

    /**
     * ꤷޤ
     *
     * @param  cursor 
     */
    public void setCursor(int cursor) {
        setCursor(Cursor.getPredefinedCursor(cursor));
    }

    /**
     * ڥΥӥ塼ݡȤθߤΥ֤ޤ
     * ѹξϡꤹޤԵΤǡ
     * AWT ٥椫餳Υ᥽åɤƤӽФƤϤޤ
     *
     * @param  def ǥեȥ <code>true</code>
     *             ߤΥ <code>false</code>
     *
     * @return ԥñ̤Υӥ塼ݡȤΥ
     */
    public Dimension getViewportSize(boolean def) {
        synchronized (lock) {
            // ޤ֤λƤʤϡִλԤ
            if ((def && defaultSize == null)
                    || (!def
                     && viewportSize.width  == 0
                     && viewportSize.height == 0
                     && getParent() != null)) {
                try {
                    lock.wait(RESIZE_WAIT);
                } catch (InterruptedException e) { }
            }
            if (defaultSize == null
                    || (viewportSize.width  == 0
                     && viewportSize.height == 0)) {
                return new Dimension(1, 1);
            }
            return (def
                    ? new Dimension(defaultSize .width, defaultSize .height)
                    : new Dimension(viewportSize.width, viewportSize.height));
        }
    }

    /**
     * 򤵤Ƥ󥯤֤ޤ
     *
     * @return 򤵤Ƥ󥯡
     *         򤵤Ƥʤ <code>null</code>
     */
    public Link getSelectedLink() {
        return currentLink;
    }

//### protected
    /**
     * ºݤΥݡͥȤ֤ޤ
     * Υ᥽åɤϡľܥˡ󶡤ޤ
     * λѤˤϽʬդƤ
     *
     * @return Υݡͥ
     */
    protected Component getScreen() {
        return screen;
    }

    /**
     * ꤷ̤ΥԤޤ
     *
     * @param  sense 
     * @param  type  ñ
     * @param  value 
     */
    protected void scroll(int sense, int type, int value) {
        synchronized (lock) {
            int x = position.x;
            int y = position.y;

            if (type == SCROLL_UNIT) {
                if (sense == SCROLL_HORIZONTAL) {
                    x += charScale.width  * value;
                } else {
                    y += charScale.height * value;
                }
            } else {
                if (sense == SCROLL_HORIZONTAL) {
                    x += (viewportSize.width  - charScale.width )  * value;
                } else {
                    y += (viewportSize.height - charScale.height) * value;
                }
            }

            setScrollPosition(x, y);
        }
    }

//### Override
    /**
     * 褷ޤ
     *
     * @param  g Graphics
     */
    public void update(Graphics g) {
        paint(g);
    }

    /**
     * 褷ޤ
     *
     * @param  g Graphics
     */
    public void paint(Graphics g) {
        super.paint(g);

        synchronized (lock) {
            paintCorner(g);
        }
    }

    /**
     * ݡͥȤƥʤ줿ȤΤޤ
     */
    public void removeNotify() {
        super.removeNotify();
        screen.cleanup();
    }

    /**
     * ꤷޤ
     *
     * @param  cursor 
     */
    public void setCursor(Cursor cursor) {
        synchronized (lock) {
            defaultCursorType = cursor.getType();

            if (/*---*/temporaryCursorType == Cursor.DEFAULT_CURSOR
                    && cursorType != defaultCursorType) {
                cursorType = defaultCursorType;
                super .setCursor(cursor);
                screen.setCursor(cursor);
            }
        }
    }

    /**
     * ġåפ򤷤ޤ
     *
     * @param  use   Ѥ <code>true</code>
     *               ʳ <code>false</code>
     * @param  delay ɽޤǤԤ֡ʥߥá
     * @param  term  ɽƤ֡ʥߥá
     */
    public void setToolTipOption(boolean use, long delay, long term) {
        toolTipUse   = use;
        toolTipDelay = delay;
        toolTipTerm  = term;
    }

    /**
     *  ϥե׵ᤷޤ
     */
    public void requestFocus() {
        if (callRequestFocus) {
            return;
        }
        callRequestFocus = true;

        super .requestFocus();
        screen.requestFocus();

        callRequestFocus = false;
    }

    /**
     * ΥڥΥѥ᡼ʸ֤ޤ
     *
     * @return ѥ᡼ʸ
     */
    public String paramString() {
        String str = super.paramString();
        return str;
    }

//### private
    /** ҥݡͥλꤵ줿֤إץå */
    private void setScrollPosition(int x, int y) {
        x = Math.max(Math.min(x, screenSize.width  - viewportSize.width ), 0);
        y = Math.max(Math.min(y, screenSize.height - viewportSize.height), 0);

        if (position.x == x && position.y == y) {
            return;
        }

        resetStatus();

        position.x = x;
        position.y = y;

        horizontal.setValue(horizontal.getMaximum() - horizontal.getVisibleAmount() - x);
        vertical  .setValue(y);

        screen.movePosition(x, y);
    }

    /** ˥塼˥ƥɲ */
    private void appendPopupMenuItem(Vector src) {
        if (src.size() == 0) {
            return;
        }

        if (popupMenu.getItemCount() > 0) {
            popupMenu.addSeparator();
        }

        MenuItem mi;
        for (int i = 0; i < src.size(); i++) {
            mi = (MenuItem) src.elementAt(i);
            if (mi.getLabel() != null && mi.getLabel().compareTo("-") != 0) {
                popupMenu.remove(mi);
            }
            popupMenu.add(mi);
        }
    }

    /** 򸵤᤹ */
    private void setTemporaryCursor(int type) {
        temporaryCursorType = type;

        if (type == Cursor.DEFAULT_CURSOR) {
            type = defaultCursorType;
        }

        if (cursorType != type) {
            cursorType = type;
            Cursor cursor = Cursor.getPredefinedCursor(cursorType);
            super .setCursor(cursor);
            screen.setCursor(cursor);
        }
    }

    /** СĴץå */
    private void fixScrollbar(boolean h, boolean v, boolean first) {
        if (horizontalVisible == h && verticalVisible == v) {
            return;
        }

//Debug.out.println("> panel.fixScrollbar : fixed : " + h + "," + v + "," + first);
        if (horizontalVisible != h) {
            // ɲäˤʤä硢ĥСɲä̵뤵
            if (!first && h) {
                resizeRequestTag++;
            }
            horizontal.getComponent().setVisible(h);
            horizontalVisible = h;
        }

        if (verticalVisible != v) {
            vertical.getComponent().setVisible(v);
            verticalVisible = v;
        }

        validate();
    }

    /** ζ */
    private void paintCorner(Graphics g) {
        if (horizontalVisible && verticalVisible) {
            Dimension size = getSize();
            Point     posH = horizontal.getComponent().getLocation();
            Point     posV = vertical  .getComponent().getLocation();
            g.setColor(getBackground());
            g.fillRect(posV.x, posH.y, size.width - posH.x, size.height - posV.y);
        }
    }

    /** ġåפץå */
    private int setupToolTip(int cursor, int x, int y) {
        if (toolTipUse) {
            String tip = null;
            if (/*---*/(tip = (String) screen.getItem(Screen.ITEM_INFO, x, y)) == null
                    && (tip = (String) screen.getItem(Screen.ITEM_TIP , x, y)) != null) {
                cursor = Cursor.CROSSHAIR_CURSOR;
            }

            if (tip != null && tip.length() > 0) {
                screen.showToolTip(tip, x, y, toolTipDelay, toolTipTerm);
                return cursor;
            }
        }

        screen.clearToolTip();
        return cursor;
    }

    /** 򸵤᤹ */
    private void resetStatus() {
        setTemporaryCursor(Cursor.DEFAULT_CURSOR);
        context.setTemporaryStatus(null);
        screen.clearToolTip();
    }

//### ޥ
    /** åץܡɤ˥ԡ */
    private void copyClipboard(String value) {
        Clipboard       clipboard = getToolkit().getSystemClipboard();
        StringSelection data      = new StringSelection(value);
        clipboard.setContents(data, data);
    }

//### HawkScreenKeyListener
    /** ϥɥ */
    private final class HawkScreenKeyListener extends KeyAdapter {
        private KeyEvent oldEvent;

        /** 󥹥󥹤 */
        private HawkScreenKeyListener() {
        }

        /**  */
        public void keyPressed(KeyEvent e) {
            boolean isAlt = ((e.getModifiers() & InputEvent.ALT_MASK) != 0);

            // ֤äƤ륤٥Ȥ̵뤹
            if (e.getWhen() + EVENT_DELAY < System.currentTimeMillis()
                    && oldEvent != null
                    && oldEvent.getKeyCode  () == e.getKeyCode  ()
                    && oldEvent.getModifiers() == e.getModifiers()
                    && oldEvent.getWhen() + EVENT_SPAN > e.getWhen()) {
                return;
            }
            oldEvent = e;

            if (isAlt) {
                switch (e.getKeyCode()) {
                case KeyEvent.VK_LEFT : e.consume(); listener.back   (); break;
                case KeyEvent.VK_RIGHT: e.consume(); listener.forward(); break;
                default: // AVOID
                }
            } else {
                switch (e.getKeyCode()) {
                case KeyEvent.VK_LEFT     :
                case KeyEvent.VK_RIGHT    :
                case KeyEvent.VK_UP       :
                case KeyEvent.VK_DOWN     :
                case KeyEvent.VK_PAGE_DOWN:
                case KeyEvent.VK_PAGE_UP  :
                case KeyEvent.VK_HOME     :
                case KeyEvent.VK_END      :
                case KeyEvent.VK_SPACE    :
                    e.consume();
                    synchronized (lock) {
                        int x = position.x;
                        int y = position.y;
                        switch (e.getKeyCode()) {
                        case KeyEvent.VK_LEFT     : x += charScale.width ; break;
                        case KeyEvent.VK_RIGHT    : x -= charScale.width ; break;
                        case KeyEvent.VK_UP       : y -= charScale.height; break;
                        case KeyEvent.VK_DOWN     : y += charScale.height; break;
                        case KeyEvent.VK_PAGE_DOWN: x += (viewportSize.width - charScale.width); break;
                        case KeyEvent.VK_PAGE_UP  : x -= (viewportSize.width - charScale.width); break;
                        case KeyEvent.VK_HOME     : x = 0; y = 0; break;
                        case KeyEvent.VK_END      : x = screenSize.width - viewportSize.width; y = 0; break;
                        case KeyEvent.VK_SPACE    : x += (viewportSize.width - charScale.width) * (!e.isShiftDown() ? 1 : -1); break;
                        default: // AVOID
                        }
                        setScrollPosition(x, y);
                    }
                    break;
                case KeyEvent.VK_ESCAPE:
                    e.consume();
                    if (listener != null) {
                        listener.stop();
                    }
                    break;
                default: // AVOID
                }
            }
        }
    }

//### HawkPanelComponentListener
    /** ѥͥѥݡͥȥϥɥ */
    private final class HawkPanelComponentListener extends ComponentAdapter {
        /** 󥹥󥹤 */
        private HawkPanelComponentListener() {
        }

        /** ѹ */
        public void componentResized(ComponentEvent e) {
            // ѥͥΥѹ줿顢ǥեȥѹ
            synchronized (lock) {
                if (defaultSize == null) {
                    return;
                }

                int bottom = horizontal.getComponent().getPreferredSize().height;
                Dimension size = getSize();

                if (readyScrollbar && scrollbarMode == Value.SCROLLING_YES) {
                    size.height -= Math.min(size.height, bottom);
                }

                defaultSize = size;
            }
        }
    }


//### HawkScreenComponentListener
    /** ꡼ѥݡͥȥϥɥ */
    private final class HawkScreenComponentListener extends ComponentAdapter {
        /** 󥹥󥹤 */
        private HawkScreenComponentListener() {
        }

        /** ѹ */
        public void componentResized(ComponentEvent e) {
            int           tag      = -1;
            PanelListener listener = null;

            synchronized (lock) {

                Dimension size = screen.getSize();

                // ⤵ѤäΤߥ
                if (viewportSize.height != size.height
                        && HawkPanel.this.listener != null) {
                    resizeRequestTag++;
                    if (resizeRequestTag < 0) {
                        resizeRequestTag = 1;
                    }
                    tag      = resizeRequestTag;
                    listener = HawkPanel.this.listener;
                }
//Debug.out.println("> panel.screenResized : tag=" + resizeRequestTag);

                if (defaultSize == null) {
                    defaultSize = size;
                }
                viewportSize = size;
                lock.notifyAll();

                horizontal.setMaximum(Math.max(screenSize.width , viewportSize.width ));
                vertical  .setMaximum(Math.max(screenSize.height, viewportSize.height));
                horizontal.setVisibleAmount (viewportSize.width );
                vertical  .setVisibleAmount (viewportSize.height);
                horizontal.setBlockIncrement(viewportSize.width  - charScale.width );
                vertical  .setBlockIncrement(viewportSize.height - charScale.height);

                screen.componentResized();
            }

            if (listener != null) {
//Debug.out.println("> panel.screenResized : call : tag=" + tag + ", size=" + viewportSize.width + "x" + viewportSize.height + ", url=" + document.content.url.getFile());
                listener.screenResized(tag);
            }
        }

        /** 줿Ȥ */
        public void componentHidden(ComponentEvent e) {
            screen.cleanup();
        }
    }

//### HawkScreenMouseListener
    /** ꡼ѥޥ٥ȥϥɥ */
    private final class HawkScreenMouseListener extends MouseAdapter {
        /** 󥹥󥹤 */
        private HawkScreenMouseListener() {
        }

        /** å */
        public void mouseClicked(MouseEvent e) {
            e.getComponent().requestFocus();

            // ġå
            if (document != null) {
                setupToolTip(0, e.getX(), e.getY());
            }

            if (listener == null) {
                return;
            }

            currentLink = (Link) screen.getItem(Screen.ITEM_LINK, e.getX(), e.getY());
            currentUrl  = (URL ) screen.getItem(Screen.ITEM_URL , e.getX(), e.getY());

            if (e.isPopupTrigger()
                    || ((e.getModifiers() & InputEvent.BUTTON3_MASK) == InputEvent.BUTTON3_MASK)) {

                popupMenu.removeAll();

                if (currentLink == null && currentUrl == null) {
                    menuItems[MENU_MAIN_BACK_INDEX   ].setEnabled(listener.canBack   ());
                    menuItems[MENU_MAIN_FORWARD_INDEX].setEnabled(listener.canForward());
                    menuItems[MENU_MAIN_STOP_INDEX   ].setEnabled(listener.canStop   ());
                    menuItems[MENU_MAIN_RELOAD_INDEX ].setEnabled(listener.canReload ());
                    ((CheckboxMenuItem) menuItems[MENU_MAIN_OPTION_IMAGE_INDEX]).setState(!option.getLoadImage());
                    ((CheckboxMenuItem) menuItems[MENU_MAIN_OPTION_BAR_INDEX  ]).setState(scrollbarMode != Value.SCROLLING_NO);
                    ((CheckboxMenuItem) menuItems[MENU_MAIN_OPTION_SPIN_INDEX ]).setState(option.getSpinImage());

                    appendPopupMenuItem(mainMenuItems);
                }
                if (currentLink != null) {
                    appendPopupMenuItem(linkMenuItems);
                }
                if (currentUrl != null) {
                    appendPopupMenuItem(imageMenuItems);
                }

                popupMenu.show(e.getComponent(), e.getX(), e.getY());

                e.consume();
                return;
            }

            if (currentLink != null) {
                // 󥯾򥯥å줿
                if (/*---*/(e.getModifiers() & InputEvent.BUTTON1_MASK) == InputEvent.BUTTON1_MASK
                        || (e.getModifiers() & InputEvent.BUTTON2_MASK) == InputEvent.BUTTON2_MASK) {
                    resetStatus();

                    listener.openHawk(new Request(currentLink.url, null,
                                                  document.content.url,
                                                  currentLink.target,
                                                  (e.isShiftDown()
                                                   ? Request.OPEN_NEWWINDOW
                                                   : ((e.getModifiers() & InputEvent.BUTTON2_MASK) == InputEvent.BUTTON2_MASK
                                                      ? Request.OPEN_NEWTAB
                                                      : Request.OPEN_DEFAULT)),
                                                  Request.CACHE_NORMAL));
                }

                e.consume();
            }
        }

        /** ޥФ */
        public void mouseExited(MouseEvent e) {
            resetStatus();
        }
    }

//### HawkScreenMouseMotionListener
    /** ꡼ѥޥ⡼ϥɥ */
    private final class HawkScreenMouseMotionListener extends MouseMotionAdapter {
        /** 󥹥󥹤 */
        private HawkScreenMouseMotionListener() {
        }

        /** ޥư */
        public void mouseMoved(MouseEvent e) {
            int    cursor = Cursor.DEFAULT_CURSOR;
            String status = null;

            synchronized (lock) {
                if (document == null) {
                    return;
                }

                // ġå
                cursor = setupToolTip(cursor, e.getX(), e.getY());

                // 󥯾夫ɤå
                Link link = (Link) screen.getItem(Screen.ITEM_LINK, e.getX(), e.getY());
                if (link != null) {
                    cursor = Cursor.HAND_CURSOR;
                    status = link.url.toString();
                }
            }

            e.consume();
            setTemporaryCursor(cursor);
            context.setTemporaryStatus(status);
        }
    }

//### MenuActionListner
    /** ꡼ѥ˥塼ѤΥꥹ */
    private final class MenuActionListner implements ActionListener, ItemListener {
        private int index = 0;

        /** ˥塼ꥹ  */
        private MenuActionListner(int index) {
            this.index = index;
        }

        /** ϥɥ */
        public void actionPerformed(ActionEvent e) {
            if (listener != null) {
                boolean isShift = ((e.getModifiers() & InputEvent.SHIFT_MASK) != 0);
                boolean isCtrl  = ((e.getModifiers() & InputEvent.CTRL_MASK ) != 0);

                /// MENU
                switch (index) {
                case MENU_MAIN_BACK_INDEX   : listener.back   (); break;
                case MENU_MAIN_FORWARD_INDEX: listener.forward(); break;
                case MENU_MAIN_STOP_INDEX   : listener.stop   (); break;
                case MENU_MAIN_RELOAD_INDEX : listener.reload ((isCtrl ? PanelListener.RELOAD_FORCE : PanelListener.RELOAD_CHECK)); break;
//                case MENU_MAIN_INFO_INDEX   : ; break;
                case MENU_LINK_COPY_INDEX   : copyClipboard(currentLink.url.toString()); break;
                case MENU_IMAGE_OPEN_INDEX  : listener.openHawk(new Request(currentUrl, null,
                                                                            document.content.url, null,
                                                                            (isShift
                                                                             ? Request.OPEN_NEWWINDOW
                                                                             : Request.OPEN_DEFAULT),
                                                                            Request.CACHE_NORMAL)); break;
                case MENU_IMAGE_COPY_INDEX  : copyClipboard(currentUrl.toString()); break;
                default: // AVOID
                }
            }
        }

        /** ϥɥ */
        public void itemStateChanged(ItemEvent e) {
            boolean selected = (e.getStateChange() == ItemEvent.SELECTED);

            /// MENU
            switch (index) {
            case MENU_MAIN_OPTION_IMAGE_INDEX   :
                option.setLoadImage(!selected);
                break;
            case MENU_MAIN_OPTION_BAR_INDEX     :
                synchronized (lock) {
                    scrollbarMode = (!selected ? SCROLLBAR_YES : SCROLLBAR_NO);
                    if (!selected) {
                        fixScrollbar(true, verticalVisible, false);
                    } else {
                        fixScrollbar(false, false, false);
                    }
                }
                break;
            case MENU_MAIN_OPTION_SPIN_INDEX :
                option.setSpinImage(selected);
                break;
            default: // AVOID
            }
        }
    }

//### ScrollbarListener
    /** Сꥹ */
    private final class ScrollbarListener implements AdjustmentListener {
        private static final int HORIZONTAL = 0;
        private static final int VERTICAL   = 1;

        private boolean        isHorizontal;
        private PanelScrollbar scrollbar;

        /** 󥹥󥹤 */
        private ScrollbarListener(int orientation, PanelScrollbar scrollbar) {
            this.isHorizontal = (orientation == HORIZONTAL);
            this.scrollbar    = scrollbar;
        }

        /** ͤѹ */
        public void adjustmentValueChanged(AdjustmentEvent e) {
            // ٥ȤͤǤϤʤߤͤ򸵤˰֤ꤹ
            // ϡ٥ȯ塢ƤӽФޤǤ˥饰뤿
            synchronized (lock) {
                if (isHorizontal) {
                    position.x = scrollbar.getMaximum() - scrollbar.getVisibleAmount() - scrollbar.getValue();
                } else {
                    position.y = scrollbar.getValue();
                }

                screen.movePosition(position.x, position.y);
            }
        }
    }
}
