/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.awt.event;

import java.awt.TextComponent;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.TextEvent;
import java.awt.event.TextListener;
import java.util.Hashtable;

/**
 * Ϥåꤵ줿ʸϤǤʤ褦ʥե륿
 * 󶡤ޤƥȥݡͥȤ˥ե륿Ԥϡ
 * Υ饹Υ󥹥󥹤򡢥ꥹʤȤϿޤ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.1.1.1 $
 */
public class KeyFilter {
    /** <code>int</code> ͤĤ */
    public static final int INT    = 4;
    /** <code>long</code> ͤĤ */
    public static final int LONG   = 5;
    /** <code>float</code> ͤĤ */
    public static final int FLOAT  = 6;
    /** <code>double</code> ͤĤ */
    public static final int DOUBLE = 7;

    private static final int INT_LEN     =  9;
    private static final int LONG_LEN    = 18;
    private static final int FLOAT_LEN   =  7;               // Ŭ
    private static final int DOUBLE_LEN  = 14;               // Ŭ

    /** @serial  */
    private int       type;
    /** @serial ꥹ */
    private Listener  listener      = new Listener ();
    /** @serial ͥơ֥ */
    private Hashtable oldValueTable = new Hashtable();

    /**
     * <code>int</code> ͤĤե륿ޤ
     */
    public KeyFilter() {
        this.type = INT;
    }

    /**
     * ꤵ줿ͤĤե륿ޤ
     *
     * @param  type 
     */
    public KeyFilter(int type) {
        this.type = type;
    }

    /**
     * ꤵ줿ݡͥȤˡե륿ӤĤޤ
     *
     * @param  c ƥȥݡͥ
     */
    public void attach(TextComponent c) {
        oldValueTable.put(c, c.getText());
        c.addKeyListener (listener);
        c.addTextListener(listener);
    }

    /**
     * ꤵ줿ݡͥȤ顢ե륿ȼޤ
     *
     * @param  c ƥȥݡͥ
     */
    public void detach(TextComponent c) {
        oldValueTable.remove(c);
        c.removeKeyListener (listener);
        c.removeTextListener(listener);
    }

//### Listener
    /** ꥹ */
    private final class Listener extends KeyAdapter implements TextListener {
        /** 󥹥󥹤 */
        private Listener() {
        }

        /** 줿 */
        public void keyPressed(KeyEvent e) {
            if (!oldValueTable.containsKey(e.getSource())) {
                return;
            }

            if (e.getModifiers() != 0) {
                return;
            }

            int code = e.getKeyCode();

            if (/*---*/code < 0x20                       // control char
                    || (0x21 <= code && code <= 0x28)    // move key
                    || (0x70 <= code && code <= 0x7F)    // F1 - F12, DELETE
                    || code == KeyEvent.VK_NUM_LOCK      // NUM_LOCK
                    || code == KeyEvent.VK_SCROLL_LOCK   // SCROLL_LOCK
                    || (0x9A <= code && code <= 0x9D)) { // PRINTSCREEN, INSERT, HELP, META
                return;
            }

            TextComponent text = (TextComponent) e.getSource();
            int    select = Math.abs(text.getSelectionStart() - text.getSelectionEnd());
            String value  = text.getText();

            switch (type) {
            case INT:
                if (isNumber(code) && value.length() <= INT_LEN + select) {
                    return;
                }
                break;
            case LONG:
                if (isNumber(code) && value.length() <= LONG_LEN + select) {
                    return;
                }
                break;
            case FLOAT:
                if (/*---*/(isNumber(code) || isPeriod(code, value))
                        && value.length() <= FLOAT_LEN + select) {
                    return;
                }
                break;
            case DOUBLE:
                if (/*---*/(isNumber(code) || isPeriod(code, value))
                        && value.length() <= DOUBLE_LEN + select) {
                    return;
                }
                break;
            default:
                return;
            }
            e.consume();
        }

        /** ͤѹ줿 */
        public void textValueChanged(TextEvent e) {
            Object source   = e.getSource();
            String oldValue = (String) oldValueTable.get(source);
            if (oldValue == null) {
                return;
            }

            TextComponent text     = (TextComponent) source;
            String        newValue = text.getText();
            if (oldValue.compareTo(newValue) == 0) {
                return;
            }

            int times;
            for (times = 0; times < 2; times++) {
                try {
                    if (newValue.length() > 0) {
                        switch (type) {
                        case INT:
                            String.valueOf(Integer.parseInt(newValue));
                            if (newValue.length() > INT_LEN) {
                                throw new NumberFormatException();
                            }
                            break;
                        case LONG:
                            String.valueOf(Long  .parseLong(newValue));
                            if (newValue.length() > LONG_LEN) {
                                throw new NumberFormatException();
                            }
                            break;
                        case FLOAT:
                            Float .valueOf("0" + newValue).toString();
                            if (newValue.length() > FLOAT_LEN) {
                                throw new NumberFormatException();
                            }
                            break;
                        case DOUBLE:
                            Double.valueOf("0" + newValue).toString();
                            if (newValue.length() > DOUBLE_LEN) {
                                throw new NumberFormatException();
                            }
                            break;
                        default: // AVOID
                        }
                    }
                } catch (NumberFormatException ex) {
                    if (times == 0) {
                        newValue = oldValue;
                    } else {
                        newValue = "";
                    }
                }
            }
            if (times > 0) {
                int pos = text.getCaretPosition();
                text.setText(newValue);
                text.setCaretPosition(Math.min(pos, newValue.length()));
            }

            oldValueTable.put(source, newValue);
        }

        /** ͤå */
        private boolean isNumber(int code) {
            return ((KeyEvent.VK_0       <= code && code <= KeyEvent.VK_9)
                 || (KeyEvent.VK_NUMPAD0 <= code && code <= KeyEvent.VK_NUMPAD9));
        }

        /** ԥꥪɤϤǤ뤫å */
        private boolean isPeriod(int code, String value) {
            return (value.indexOf('.') == -1
                 && (code == KeyEvent.VK_DECIMAL
                  || code == KeyEvent.VK_PERIOD));

        }
    }
}
