/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.awt;

import net.hizlab.kagetaka.addin.java2.WheelListener;
import net.hizlab.kagetaka.addin.java2.WheelWrapper;
import net.hizlab.kagetaka.awt.event.ItemEvent;
import net.hizlab.kagetaka.awt.image.GrayFilter;
import net.hizlab.kagetaka.awt.image.OffscreenImage;
import net.hizlab.kagetaka.awt.image.SyncObserver;

import java.awt.AWTEvent;
import java.awt.AWTEventMulticaster;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.ItemSelectable;
import java.awt.SystemColor;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.ItemListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.awt.image.MemoryImageSource;
import java.util.Vector;

/**
 * ĥ꡼ΥꥹȤǤ
 *
 * @kagetaka.bugs ľ󲽤ϡꥹʤ¸ʤޤ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.8 $
 */
public class Tree extends Component implements ItemSelectable {
    private static final String BASE        = "tree";
    private static       int    nameCounter = 0;

    private static final int  INSET        =  2;
    private static final int  SPACE        =  2;
    private static final int  BOX          = 13;
    private static final int  SB_W         = 16;

    private static final int[] BOX_OPEN =
    {
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0,
        0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0,
        0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0,
        0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0,
        0, 0, 1, 0, 2, 2, 2, 2, 2, 0, 1, 0, 0,
        0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0,
        0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0,
        0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0,
        0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    };

    private static final int[] BOX_CLOSE =
    {
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0,
        0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0,
        0, 0, 1, 0, 0, 0, 2, 0, 0, 0, 1, 0, 0,
        0, 0, 1, 0, 0, 0, 2, 0, 0, 0, 1, 0, 0,
        0, 0, 1, 0, 2, 2, 2, 2, 2, 0, 1, 0, 0,
        0, 0, 1, 0, 0, 0, 2, 0, 0, 0, 1, 0, 0,
        0, 0, 1, 0, 0, 0, 2, 0, 0, 0, 1, 0, 0,
        0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0,
        0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    };

    /** ̾ξ */
    public  static final int NORMAL       = 0;
    /** ̵ʾ */
    public  static final int DISABLE      = 4;

    /** ֤Υեƥ */
    public static final int ITEM_FOLDER_OPEN  = 1;
    /** Ĥ֤Υեƥ */
    public static final int ITEM_FOLDER_CLOSE = 2;
    /** ҤΤʤ֤Υեƥ */
    public static final int ITEM_FOLDER_NONE  = 3;
    /** ̤Υƥ */
    public static final int ITEM_NORMAL       = 4;
    /** ¸ߤʤƥ */
    public static final int ITEM_INVALID      = -1;

    private static final int TOGGLE = 1;
    private static final int OPEN   = 2;
    private static final int CLOSE  = 3;

    /** @serial å֥ */
    private Object lock = new Object();

    /** @serial ե꡼ */
    private OffscreenImage offscreenImage;
    /** @serial 侩 */
    private Dimension      preferredSize;

    /** @serial ̾ */
    private String  name;
    /** @serial ̾ꤵ줿ɤ */
    private boolean nameExplicitlySet = false;

    /** @serial ܥå */
    private Dimension boxSize;
    /** @serial ᡼ */
    private Image     openImage;
    /** @serial Ĥ륤᡼ */
    private Image     closeImage;
    /** @serial γ᡼ */
    private Image     openSelectedImage;
    /** @serial Ĥ륤᡼ */
    private Image     closeSelectedImage;

    /** @serial Կ */
    private int    rows;
    /** @serial ɽ */
    private int    columns;
    /** @serial ܡ */
    private Border border;
    /** @serial 롼ȥƥ */
    private Item   rootItem;

    /** @serial  */
    private int     state;
    /** @serial ե */
    private boolean focus;

    /** @serial ɽ楢ƥ */
    private Vector    viewItems = new Vector();
    /** @serial С */
    private Scrollbar scrollbar;

    /** @serial 򤵤Ƥ륤ǥå */
    private int selectedIndex    = -1;
    /** @serial 򤵤Ƥǥå */
    private int selectedOldIndex = -1;
    /** @serial ɽϥǥå */
    private int viewStart        = 0;
    /** @serial ɽλǥå */
    private int viewEnd          = 0;
    /** @serial ɽ륤ǥå׵ */
    private int visibleReq       = -1;
    /** @serial 1 Ԥι⤵ */
    private int rowHeight        = -1;

    /** @serial ޥꥹ */
    private MouseListener treeMouseListener;

    private transient Vector       listenerWrappers = new Vector();
    private transient ItemListener itemListener;

    /**
     * ꤵ줿ɽԤĥĥ꡼ۤޤ
     *
     * @param  rows    ɽ륢ƥο
     * @param  columns ĥ꡼ɽ
     */
    public Tree(int rows, int columns) {
        setRows   (rows   );
        setColumns(columns);

        addFocusListener(
            new FocusListener() {
                /** ե */
                public void focusGained(FocusEvent e) {
                    if (!focus) {
                        focus = true;
                        if (isShowing()) {
                            repaint();
                        }
                    }
                }

                /** ե򼺤ä */
                public void focusLost(FocusEvent e) {
                    if (focus) {
                        focus = false;
                    }

                    setViewState(NORMAL);
                }
            }
        );

        addKeyListener(
            new KeyAdapter() {
                /** 줿 */
                public void keyPressed(KeyEvent e) {
                    if (!isEnabled()) {
                        return;
                    }

                    switch (e.getKeyCode()) {
                    case KeyEvent.VK_LEFT:
                        e.consume();
                        synchronized (Tree.this) {
                            if (selectedIndex >= 0) {
                                Item item = (Item) viewItems.elementAt(selectedIndex);
                                if (item.isOpen) {
                                    openItem(selectedIndex, CLOSE);
                                } else if (item.parent != null) {
                                    for (int i = selectedIndex - 1; i >= 0; i--) {
                                        if (((Item) viewItems.elementAt(i)).level < item.level) {
                                            if (!moveCurrent(i, false, true)) {
                                                return;
                                            }
                                            break;
                                        }
                                    }
                                } else {
                                    return;
                                }
                            }
                        }
                        repaintForce();
                        break;
                    case KeyEvent.VK_RIGHT:
                        e.consume();
                        synchronized (Tree.this) {
                            if (selectedIndex >= 0) {
                                Item item = (Item) viewItems.elementAt(selectedIndex);
                                if (item.isOpen || !item.hasChild()) {
                                    return;
                                }
                                openItem(selectedIndex, OPEN);
                            }
                        }
                        repaintForce();
                        break;
                    case KeyEvent.VK_DOWN:
                        e.consume();
                        synchronized (Tree.this) {
                            int index = (selectedIndex == -1 && selectedOldIndex != -1)
                                        ? Math.min(selectedOldIndex, viewItems.size() - 1)
                                        : selectedIndex + 1;
                            if (index >= viewItems.size()) {
                                return;
                            }

                            if (!moveCurrent(index, false, true)) {
                                return;
                            }
                        }
                        repaintForce();
                        break;
                    case KeyEvent.VK_UP:
                        e.consume();
                        synchronized (Tree.this) {
                            int index = (selectedIndex == -1 && selectedOldIndex != -1)
                                        ? Math.min(Math.max(selectedOldIndex - 1, 0), viewItems.size() - 1)
                                        : selectedIndex - 1;
                            if (index < 0) {
                                return;
                            }

                            if (!moveCurrent(index, false, true)) {
                                return;
                            }
                        }
                        repaintForce();
                        break;
                    case KeyEvent.VK_PAGE_DOWN:
                        e.consume();
                        synchronized (Tree.this) {
                            int index = (selectedIndex == -1 && selectedOldIndex != -1)
                                        ? Math.min(selectedOldIndex, viewItems.size() - 1)
                                        : selectedIndex;
                            if (index + 1 >= viewItems.size()) {
                                return;
                            }

                            int viewNum = (getSize().height - INSET * 2) / rowHeight;
                            int request = Math.min(viewEnd + viewNum, viewItems.size() - 1);

                            index = (request > viewEnd ? index + (request - viewEnd) : viewItems.size() - 1);

                            if (!moveCurrent(index, false, true)) {
                                return;
                            }

                            visibleReq = request;
                        }
                        repaintForce();
                        break;
                    case KeyEvent.VK_PAGE_UP:
                        e.consume();
                        synchronized (Tree.this) {
                            int index = (selectedIndex == -1 && selectedOldIndex != -1)
                                        ? Math.min(Math.max(selectedOldIndex - 1, 0), viewItems.size() - 1)
                                        : selectedIndex;
                            if (index == 0) {
                                return;
                            }

                            int viewNum = (getSize().height - INSET * 2) / rowHeight;
                            int request = Math.max(viewStart - viewNum, 0);

                            index = (request < viewStart ? index - (viewStart - request) : 0);

                            if (!moveCurrent(index, false, true)) {
                                return;
                            }

                            visibleReq = request;
                        }
                        repaintForce();
                        break;
                    default: // AVOID
                    }
                }
            }
        );

        addMouseListener(
            treeMouseListener = new MouseAdapter() {
                /** ޥ줿 */
                public void mousePressed(MouseEvent e) {
                    if (!isEnabled()) {
                        return;
                    }

                    e.consume();

                    if (isEnabled()) {
                        requestFocus();
                    }

                    if (scrollbar.contains(e.getX(), e.getY(), false)) {
                        return;
                    }

                    moveCurrentFromPoint(e.getX(), e.getY());
                }

                /** ޥå줿 */
                public void mouseClicked(MouseEvent e) {
                    if (!isEnabled()) {
                        return;
                    }

                    e.consume();

                    if (e.getClickCount() == 2) {
                        int x = e.getX();
                        int y = e.getY();
                        synchronized (Tree.this) {
                            int index = getIndexFromPosition(x, y);
                            if (index < 0) {
                                return;
                            }

                            if (isBoxFromPosition(index, x, y)) {
                                return;
                            }

                            if (!openItem(index, TOGGLE)) {
                                return;
                            }
                        }

                        repaintForce();
                    }
                }
            }
        );
        // ޥۥ
        WheelWrapper wheelWrapper = WheelWrapper.getInstance();
        if (wheelWrapper != null) {
            wheelWrapper.addWheelListener(this, new WheelListener() {
                    /** ޥۥ뤬ž줿 */
                    public void spin(int type, int value, int modifiers) {
                        synchronized (Tree.this) {
                            int request;
                            if (type == WheelListener.UNIT_SCROLL) {
                                if (value > 0) {
                                    if ((request = Math.min(viewEnd   + value, viewItems.size() - 1)) <= viewEnd  ) {
                                        return;
                                    }
                                } else {
                                    if ((request = Math.max(viewStart + value, 0                   )) >= viewStart) {
                                        return;
                                    }
                                }
                            } else {
                                int viewNum = (getSize().height - INSET * 2) / rowHeight;
                                if (value > 0) {
                                    if ((request = Math.min(viewEnd   + value * viewNum, viewItems.size() - 1)) <= viewEnd  ) {
                                        return;
                                    }
                                } else {
                                    if ((request = Math.max(viewStart + value * viewNum, 0                   )) >= viewStart) {
                                        return;
                                    }
                                }
                            }

                            visibleReq = request;
                        }
                        repaintForce();
                    }
                }
            );
        }

        setForeground(SystemColor.controlText);
        setBackground(SystemColor.control    );
    }

//### Original
    /**
     * ݡͥȤ֤̾ޤ
     *
     * @return ̾
     */
    public String getName() {
        String nowName = this.name;
        if (nowName != null) {
            return nowName;
        }
        synchronized (this) {
            if (name == null && !nameExplicitlySet) {
                name = BASE + nameCounter++;
            }
        }
        return name;
    }

    /**
     * ݡͥȤ̾ꤷޤ
     *
     * @param  name ̾
     */
    public void setName(String name) {
        synchronized (this) {
            super.setName(name);
            this.name              = name;
            this.nameExplicitlySet = true;
        }
    }

//### Override
    /**
     * طʿꤷޤ
     *
     * @param  c طʿ
     */
    public void setBackground(Color c) {
        super.setBackground(c);

        initDefaultImage();
        repaintForce();
    }

    /**
     * եȤꤷޤ
     *
     * @param  font ե
     */
    public void setFont(Font font) {
        Font oldFont = getFont();
        if (oldFont != font) {
            super.setFont(font);

            invalidate();
            repaintForce();
        }
    }

    /**
     * 侩֤ޤ
     *
     * @return 侩
     */
    public Dimension getPreferredSize() {
        Dimension preferredSize = this.preferredSize;
        if (preferredSize != null && isValid()) {
            return preferredSize;
        }

        synchronized (getTreeLock()) {
            FontMetrics fm = getFontMetrics(getFont());
            this.preferredSize = new Dimension(
                fm.charWidth('0') * columns + INSET * 2 + (boxSize != null ? boxSize.width : 0) + SPACE * 4,
                (rowHeight = (fm.getHeight() + SPACE * 2)) * rows + INSET * 2
            );
            return this.preferredSize;
        }
    }

    /**
     * ݡͥȤ̵ˤޤ
     */
    public void invalidate() {
        preferredSize = null;
        super.invalidate();
    }

    /**
     * ե뤳Ȥ뤫֤ޤ
     *
     * @return ΥݡͥȤϥեΤǡ
     *         <code>true</code> ֤ޤ
     */
    public boolean isFocusTraversable() {
        return true;
    }

    /**
     * ̤򥢥åץǡȤޤ
     *
     * @param  g եå
     */
    public void update(Graphics g) {
        paint(g);
    }

    /**
     * ᡼ºݤ褷ޤ
     *
     * @param  g եå
     */
    public void paint(Graphics g) {
        offscreenImage.paint(g);
    }

    /**
     * ݡͥȤƥʤɲä줿ȤΤޤ
     */
    public void addNotify() {
        synchronized (getTreeLock()) {
            super.addNotify();
            scrollbar = new Scrollbar();

            // ե꡼
            offscreenImage = new OffscreenImage(this) {
                public void update(Image offscreen, Graphics g, Dimension size) {
                    refresh(offscreen, g, size);
                }
            };
        }
    }

    /**
     * ݡͥȤƥʤ줿ȤΤޤ
     */
    public void removeNotify() {
        synchronized (getTreeLock()) {
            super.removeNotify();
            if (scrollbar != null) {
                scrollbar.dispose();
                scrollbar = null;
            }

            // ե꡼
            if (offscreenImage != null) {
                offscreenImage.dispose();
                offscreenImage = null;
            }
        }
    }

    /**
     * ΥơСΥѥ᡼ʸ֤ޤ
     *
     * @return ѥ᡼ʸ
     */
    protected String paramString() {
        String str = super.paramString();
        return str;
    }

//### Listener
    /**
     * ޥꥹʤϿޤ
     *
     * @param  l Ͽޥꥹ
     */
    public synchronized void addMouseListener(MouseListener l) {
        if (scrollbar != null && !scrollbar.isOwnListener(l) && treeMouseListener != l) {
            l = new ListenerWrapper(l);
            listenerWrappers.addElement(l);
        }
        super.addMouseListener(l);
    }

    /**
     * ޥꥹʤޤ
     *
     * @param  l ޥꥹ
     */
    public synchronized void removeMouseListener(MouseListener l) {
        ListenerWrapper lw;
        for (int i = listenerWrappers.size() - 1; i >= 0; i--) {
            lw = (ListenerWrapper) listenerWrappers.elementAt(i);
            if (lw.mouseListener == l) {
                listenerWrappers.removeElementAt(i);
                removeMouseListener(lw.mouseListener);
            }
        }
        super.removeMouseListener(l);
    }

    /**
     * ޥ⡼ꥹʤϿޤ
     *
     * @param  l Ͽޥ⡼ꥹ
     */
    public synchronized void addMouseMotionListener(MouseMotionListener l) {
        if (scrollbar != null && !scrollbar.isOwnListener(l)) {
            l = new ListenerWrapper(l);
            listenerWrappers.addElement(l);
        }
        super.addMouseMotionListener(l);
    }

    /**
     * ޥ⡼ꥹʤޤ
     *
     * @param  l ޥ⡼ꥹ
     */
    public synchronized void removeMouseMotionListener(MouseMotionListener l) {
        ListenerWrapper lw;
        for (int i = listenerWrappers.size() - 1; i >= 0; i--) {
            lw = (ListenerWrapper) listenerWrappers.elementAt(i);
            if (lw.mouseMotionListener == l) {
                listenerWrappers.removeElementAt(i);
                removeMouseMotionListener(lw.mouseMotionListener);
            }
        }
        super.removeMouseMotionListener(l);
    }

    /**
     * ɽԿ֤ޤ
     *
     * @return ɽԿ
     */
    public int getRows() {
        return rows;
    }

    /**
     * ɽԿꤷޤ
     *
     * @param  rows ɽԿ
     */
    public void setRows(int rows) {
        int oldRows = this.rows;
        if (rows < 0) {
            throw new IllegalArgumentException("rows less than zero.");
        }
        if (oldRows != rows) {
            this.rows = rows;
            invalidate();
        }
    }

    /**
     * ɽ֤ޤ
     *
     * @return ɽ
     */
    public int getColumns() {
        return columns;
    }

    /**
     * ɽꤷޤ
     *
     * @param  columns ɽ
     */
    public void setColumns(int columns) {
        int oldColumns = this.columns;
        if (columns < 0) {
            throw new IllegalArgumentException("columns less than zero.");
        }
        if (oldColumns != columns) {
            this.columns = columns;
            invalidate();
        }
    }

    /**
     * ꤵƤܡ֤ޤ
     *
     * @return ܡ
     */
    public Border getBorder() {
        return border;
    }

    /**
     * ܡꤷޤ
     *
     * @param  border ܡ
     */
    public void setBorder(Border border) {
        Border oldBorder = this.border;
        if (oldBorder != border) {
            this.border = border;
            repaintForce();
        }
    }

    /**
     * 롼ȥƥꤷޤ
     *
     * @param  root 롼ȥƥ
     */
    public synchronized void setRootTreeItem(TreeItem root) {
        rootItem = new Item(null, root, true);

        loadItems();

        repaintForce();
    }

    /**
     * ꤷ֤Υƥ֤ޤ
     *
     * @param  index 륢ƥβ̾Υǥå
     *
     * @return ꤷ֤Υƥࡣ
     *         ꤷ֤˥ƥबʤ <code>null</code>
     */
    public synchronized TreeItem getTreeItem(int index) {
        if (index < 0 || index >= viewItems.size()) {
            return null;
        }

        return ((Item) viewItems.elementAt(index)).data;
    }

    /**
     * ꤷ֤ΥƥΥ٥֤ޤ
     *
     * @param  index 륢ƥβ̾Υǥå
     *
     * @return ꤷ֤Υƥ٥롢
     *         Ǿ̤ <code>1</code>
     *         ƥब¸ߤʤ <code>-1</code>
     */
    public synchronized int getTreeItemLevel(int index) {
        if (index < 0 || index >= viewItems.size()) {
            return -1;
        }

        return ((Item) viewItems.elementAt(index)).level + 1;
    }

    /**
     * ꤷ֤Υƥξ֤֤ޤ
     *
     * @param  index 륢ƥβ̾Υǥå
     *
     * @return ꤷ֤Υƥξ
     */
    public int getTreeItemState(int index) {
        Item item = null;

        synchronized (this) {
            if (index < 0 || index >= viewItems.size()) {
                return ITEM_INVALID;
            }

            item = (Item) viewItems.elementAt(index);
        }

        return (item.hasChild()
                ? (item.isOpen ? ITEM_FOLDER_OPEN : ITEM_FOLDER_CLOSE)
                : (item.child != null ? ITEM_FOLDER_NONE : ITEM_NORMAL));
    }

    /**
     * ĥ꡼餹٤ƤΥƥޤ
     */
    public synchronized void removeAll() {
        viewItems.removeAllElements();

        repaintForce();
    }

    /**
     * ĥ꡼ɹơɽޤ
     */
    public void refresh() {
        synchronized (this) {
            TreeItem selectedTreeItem = getSelectedTreeItem();

            rootItem.reset();
            loadItems();

            // 򤵤Ƥƥ򸡺
            int size = viewItems.size();
            for (int i = 0; i < size; i++) {
                if (((Item) viewItems.elementAt(i)).data == selectedTreeItem) {
                    visibleReq       = i;
                    selectedIndex    = i;
                    selectedTreeItem = null;
                    break;
                }
            }

            if (selectedTreeItem != null) {
                selectedOldIndex = selectedIndex;
                moveCurrent(-1, false, false);
            }
        }
        repaintForce();
    }

    /**
     * 򤵤줿ƥ֤ޤ
     *
     * @return 򤵤Ƥ륢ƥ
     */
    public synchronized TreeItem getSelectedTreeItem() {
        if (selectedIndex < 0 || selectedIndex >= viewItems.size()) {
            return null;
        }

        return ((Item) viewItems.elementAt(selectedIndex)).data;
    }

    /**
     * ꤷǥåΥƥ֤ˤޤ
     *
     * @param  index ܤΰ
     */
    public void selecteIndex(int index) {
        synchronized (this) {
            moveCurrent(index, true, true);
        }
    }

    /**
     * ꤷؤΥƥ֤ˤޤ
     * indices ϡ롼ȥƥफγؤν֤Ǥ
     *
     * @param  indices ǥå
     */
    public void selectTreeItem(int[] indices) {
        synchronized (this) {
            int index = -1;

            Item   item = rootItem;
            Vector child;

            for (int i = 0; ; i++) {
                if (!item.hasChild() || (child = item.child).size() <= indices[i]) {
                    break;
                }

                for (int j = 0; j < indices[i]; j++) {
                    index += countOpenItem((Item) child.elementAt(j));
                }

                index++;

                if (i + 1 >= indices.length || !item.hasChild()) {
                    break;
                }

                item = (Item) child.elementAt(indices[i]);
                item.isOpen = true;
            }

            loadItems();
            moveCurrent(index, false, true);
        }

        repaintForce();
    }

    /** Ƥ륢ƥץå */
    private int countOpenItem(Item item) {
        if (!item.hasChild() || !item.isOpen) {
            return 1;
        }

        int    num   = 1;
        Vector child = item.child;
        for (int i = child.size() - 1; i >= 0; i--) {
            num += countOpenItem((Item) child.elementAt(i));
        }

        return num;
    }

    /**
     * ꤵ줿ǥåˤ륢ƥब򤵤Ƥ뤫֤ޤ
     *
     * @param  index ܤΰ
     *
     * @return 򤵤Ƥ <code>true</code>
     *         ʳξ <code>false</code>
     */
    public synchronized boolean isIndexSelected(int index) {
        return (selectedIndex == index);
    }

    /**
     * 򤵤Ƥ륤ǥå֤ޤ
     *
     * @return 򤵤ƤϤΥǥå
     *         򤵤Ƥʤ <code>-1</code>
     */
    public synchronized int getSelectedIndex() {
        return selectedIndex;
    }

    /**
     * 򤵤줿ƥ֤ޤ
     *
     * @return 򤵤Ƥ륢ƥࡣ
     *         򤵤Ƥʤ <code>null</code>
     */
    public synchronized Object[] getSelectedObjects() {
        if (selectedIndex < 0 || selectedIndex >= viewItems.size()) {
            return null;
        }

        return new TreeItem[]{((Item) viewItems.elementAt(selectedIndex)).data};
    }

    /**
     * ɽƤ륢ƥο֤ޤ
     *
     * @return ɽƤ륢ƥο
     */
    public int getViewItemCount() {
        return viewItems.size();
    }

    /**
     * ꤵ줿֤Υƥ򳫤ޤ
     *
     * @param  index ǥå
     *
     * @return  <code>true</code>
     *         ʳξ <code>false</code>
     */
    public boolean openTreeItem(int index) {
        synchronized (this) {
            if (index < 0 || index >= viewItems.size()) {
                return false;
            }

            if (!openItem(index, OPEN)) {
                return false;
            }
        }

        repaintForce();
        return true;
    }

    /**
     * ꤵ줿֤ΥƥĤޤ
     *
     * @param  index ǥå
     *
     * @return Ĥ줿 <code>true</code>
     *         ʳξ <code>false</code>
     */
    public boolean closeTreeItem(int index) {
        synchronized (this) {
            if (index < 0 || index >= viewItems.size()) {
                return false;
            }

            if (!openItem(index, CLOSE)) {
                return false;
            }
        }

        repaintForce();
        return true;
    }

//### private
    /**
     * ߤΥơˤä֤ˡե꡼褷ޤ
     *
     * @param  offscreen ե꡼
     * @param  g         եå
     * @param  size      礭
     */
    private void refresh(Image offscreen, Graphics g, Dimension size) {
        int     state;
        boolean focus;
        synchronized (lock) {
            state = this.state;
            focus = this.focus;
        }

        Color fg = getForeground();
        Color bg = getBackground();

        // ƥȤ
        int selectedIndex = -1;
        int lineNum = 0, viewNum = 0, drawNum = 0;

        synchronized (this) {
            selectedIndex = this.selectedIndex;

            // Ԥ
            lineNum = viewItems.size();
            viewNum = (size.height - INSET * 2) / rowHeight;

            if (lineNum <= viewNum) {
                viewStart = 0;
            } else if (visibleReq != -1 && visibleReq < viewStart) {
                viewStart = visibleReq;
            } else if (visibleReq != -1 && visibleReq > viewEnd) {
                viewStart = Math.max(0, visibleReq - viewNum + 1);
            } else if (viewStart + viewNum > lineNum) {
                viewStart = Math.max(0, lineNum - viewNum);
            }
            drawNum = viewEnd = Math.min(viewStart + viewNum - 1, lineNum - 1);
            if (drawNum + 1 < lineNum && (size.height - INSET * 2) % rowHeight > 0) {
                drawNum++;
            }
            visibleReq = -1;
        }
//Debug.out.println(viewStart + "-" + viewEnd);

        Font        font = getFont();
        FontMetrics fm   = getFontMetrics(font);


        // Сɽ/ɽڤؤ
        scrollbar.setVisible((viewStart > 0) || (viewEnd < lineNum - 1));

        // طʿɤ
        g.setColor(bg);
        g.fillRect(0, 0, size.width, size.height);

        Color color = fg;
        if (state == DISABLE) {
            color = new Color((new GrayFilter(bg)).filterRGB(0, 0, color.getRGB()));
        }
        g.setFont (font);
        int x = INSET;
        int y = INSET;
        int w = size.width - INSET * 2;
        int h = rowHeight;
        if (scrollbar.isVisible()) {
            w -= SB_W;
        }

        Item    item;
        int     indent;
        boolean selected;
        int     descent = fm.getDescent();
        int     half    = boxSize.width  / 2 + SPACE;
        Image   image;
        int     imageWidth = 0, imageHeight = 0;
        String  label;
        for (int i = viewStart; i <= drawNum; i++) {
            item     = (Item) viewItems.elementAt(i);
            indent   = item.level * (SPACE * 2 + boxSize.width);
            selected = isIndexSelected(i);

            // 
            if (isIndexSelected(i)) {
                g.setColor(SystemColor.textHighlight);
                g.fillRect(x, y, w, h);
            }

            // ĥ꡼¤
            g.setColor(selected ? SystemColor.textHighlightText : SystemColor.textInactiveText);
            GraphicsUtils.drawDashed(g, x + indent + half + 1, y + rowHeight / 2, half, 1 , 1, 1, GraphicsUtils.HORIZONTAL);
            GraphicsUtils.drawDashed(g, x + indent + half, y, 1, (item.isLast ? h / 2 : h), 1, 1, GraphicsUtils.VERTICAL  );
            if (item.hasChild()) {
                drawImage(g,
                          (selected ? (item.isOpen ? openSelectedImage : closeSelectedImage)
                                    : (item.isOpen ? openImage         : closeImage        )),
                          x + indent + SPACE,
                          y + (rowHeight - boxSize.height) / 2,
                          boxSize.width, boxSize.height);
            }
            if (item.level > 0) {
                drawItemTree(g, item.parent, x, y, h, half);
            }

            // 
            if (/*---*/(image = item.data.getTreeIcon(item.isOpen)) != null
                    && ImageUtils.load(image, this)) {
                imageWidth  = image.getWidth (this);
                imageHeight = image.getHeight(this);
                drawImage(g, image, x + indent + boxSize.width + SPACE * 3, y, imageWidth, imageHeight);
            } else {
                imageWidth = imageHeight = 0;
            }

            //### BUGS ­ʤνɬ
            // ٥
            g.setColor(isIndexSelected(i) ? SystemColor.textHighlightText : color);
            if ((label = item.data.getTreeLabel()) != null) {
                g.drawString(label,
                             x + indent + boxSize.width + SPACE * 4 + imageWidth,
                             y + h - SPACE - descent);
            } else {
                g.drawLine(x + indent + boxSize.width + SPACE * 4 + imageWidth,
                           y + rowHeight / 2,
                           size.width - INSET - SB_W - SPACE - 1,
                           y + rowHeight / 2);
            }

            // եȤ
            if (focus && i == selectedIndex) {
                GraphicsUtils.drawDashed(g, x, y, w, 1, 1, 1, GraphicsUtils.HORIZONTAL);
                GraphicsUtils.drawDashed(g, x, y, 1, h, 1, 1, GraphicsUtils.VERTICAL  );
                GraphicsUtils.drawDashed(g, x, y + h - 1, w, 1, 1, 1, GraphicsUtils.HORIZONTAL);
                GraphicsUtils.drawDashed(g, x + w - 1, y, 1, h, 1, 1, GraphicsUtils.VERTICAL  );
            }

            y += rowHeight;
        }

        // Ȥ
        Border border = getBorder();
        if (border != null) {
            border.draw(g, 0, 0, size.width - 1, size.height - 1, fg);
        } else {
            g.setColor(SystemColor.controlShadow);
            g.drawLine(0, 0, size.width - 1,  0);
            g.drawLine(0, 0, 0, size.height - 1);
            g.setColor(SystemColor.controlDkShadow);
            g.drawLine(1, 1, size.width - 2,  1);
            g.drawLine(1, 1, 1, size.height - 2);
            g.setColor(SystemColor.controlHighlight);
            g.drawLine(size.width - 2,  2, size.width - 2, size.height - 2);
            g.drawLine(2, size.height - 2, size.width - 2, size.height - 2);
            g.setColor(SystemColor.controlLtHighlight);
            g.drawLine(size.width - 1,  1, size.width - 1, size.height - 1);
            g.drawLine(1, size.height - 1, size.width - 1, size.height - 1);
        }

        // С
        if (scrollbar.isVisible()) {
            scrollbar.setBounds(size.width - INSET - SB_W,
                                INSET,
                                SB_W,
                                size.height - INSET * 2);
            scrollbar.setValues(viewStart, viewNum, 0, lineNum);
            scrollbar.setBlockIncrement(Math.max(1, viewNum - 1));
            scrollbar.paint(g, true);
        }
    }

    /** ĥ꡼ʬ */
    private void drawItemTree(Graphics g, Item item, int x, int y, int h, int half) {
        if (!item.isLast) {
            int indent = item.level * (SPACE * 2 + boxSize.width);
            GraphicsUtils.drawDashed(g, x + indent + half, y , 1, h, 1, 1, GraphicsUtils.VERTICAL);
        }

        if (item.level > 0) {
            drawItemTree(g, item.parent, x, y, h, half);
        }
    }

    /**
     * ɽ֤ѹޤ
     *
     * @param  state 
     */
    private void setViewState(int state) {
        synchronized (lock) {
            // ݡͥȤ̵ξϡɬ DISABLE ˤ
            if (!isEnabled()) {
                state = DISABLE;
            }

            if (this.state != state) {
                this.state = state;

                repaintForce();
            }
        }
    }

    /** ֤ˤ륤ǥå */
    private void moveCurrentFromPoint(int x, int y) {
        synchronized (this) {
            int index = getIndexFromPosition(x, y);
            if (index < 0) {
                return;
            }

            if (isBoxFromPosition(index, x, y)) {
                if (!openItem(index, TOGGLE)) {
                    return;
                }
            } else if (!moveCurrent(index, true, true)) {
                return;
            }
        }

        repaintForce();
    }

    /** ꤷǥåإȤưץå */
    private boolean moveCurrent(int index, boolean check, boolean move) {
        int size = viewItems.size();
        if (check && (index >= size || index == selectedIndex)) {
            return false;
        }

        if (selectedIndex >= 0 && selectedIndex < size) {
            postItemEvent(selectedIndex, ItemEvent.DESELECTED);
        }

        if (move) {
            visibleReq = index;
        }
        selectedIndex = index;

        if (index >= 0) {
            postItemEvent(index, ItemEvent.SELECTED);
        }

        return true;
    }

    /** ֤饤ǥåץå */
    private int getIndexFromPosition(int x, int y) {
        Dimension size = getSize();

        if (/*---*/x <= INSET || size.width  - INSET <= x
                || y <= INSET || size.height - INSET <= y) {
            return -1;
        }

        int index = viewStart + ((y - INSET) / rowHeight);
        if (index >= viewItems.size()) {
            return -1;
        }

        return index;
    }

    /** ֤ץܥåɤåץå */
    private boolean isBoxFromPosition(int index, int x, int y) {
        int size = viewItems.size();
        if (index >= size) {
            return false;
        }

        Item item = (Item) viewItems.elementAt(index);
        if (!item.hasChild()) {
            return false;
        }

        int left  = INSET + item.level * (SPACE * 2 + boxSize.width) + SPACE;
        int itemY = (y - INSET) % rowHeight;
//Debug.out.println("[" + left + "<" + x + "<" + (left + boxSize.width) + "],[" + SPACE + "<" + itemY + "<" + (rowHeight - SPACE) + "]");
        return (left  <= x     && x     <= left + boxSize.width
             && SPACE <= itemY && itemY <= rowHeight - SPACE);
    }

    /** ɽƥƹۡץå */
    private void loadItems() {
        viewItems.removeAllElements();

        if (!rootItem.hasChild()) {
            return;
        }

        Vector child = rootItem.child;
        int    size  = child.size();
        for (int i = 0; i < size; i++) {
            scanOpenItem((Item) child.elementAt(i), viewItems);
        }
    }

    /** ֤򳫤ץå */
    private boolean openItem(int index, int mode) {
        Item item = (Item) viewItems.elementAt(index);
        if (item.child == null) {
            return false;
        }

        if (/*---*/(mode == OPEN  &&  item.isOpen)
                || (mode == CLOSE && !item.isOpen)) {
            return false;
        }

        int newSelectedIndex = selectedIndex;

        int size    = viewItems.size();
        int newSize = size;
        if (item.isOpen = !item.isOpen) {
            newSize += item.child.size() * 2;
        }

        Vector v = new Vector(newSize);

        // index μޤǤ򥳥ԡ
        for (int i = 0; i < index; i++) {
            v.addElement(viewItems.elementAt(i));
        }

        if (item.isOpen) {
            // 
            int start = v.size();
            scanOpenItem(item, v);
            if (selectedIndex > index) {
                newSelectedIndex += v.size() - start - 1;
            }
        } else {
            // Ĥ
            v.addElement(item);
            int level = item.level;
            int del   = 0;

            while ((++del + index) < size) {
                if (((Item) viewItems.elementAt(del + index)).level <= level) {
                    break;
                }
            }

            del--;
            if (selectedIndex > index) {
                if (index + del < selectedIndex) {
                    newSelectedIndex -= del;
                } else {
                    newSelectedIndex = index;
                }
            }
            index += del;
        }

        // index ʹߤ򥳥ԡ
        for (int i = index + 1; i < size; i++) {
            v.addElement(viewItems.elementAt(i));
        }

        viewItems = v;
        if (newSelectedIndex != selectedIndex) {
            moveCurrent(newSelectedIndex, false, false);
        }

        return true;
    }

    /** ƵŪ˥ƥɲ */
    private void scanOpenItem(Item item, Vector v) {
        v.addElement(item);

        if (item.isOpen && item.hasChild()) {
            int size = item.child.size();
            for (int i = 0; i < size; i++) {
                scanOpenItem((Item) item.child.elementAt(i), v);
            }
        }
    }

    /** Ū˺ɽ */
    private void repaintForce() {
        OffscreenImage oi = offscreenImage;
        if (oi != null) {
            oi.repaint();
        }
    }

    /**  */
    private void initDefaultImage() {
        Color color1 = SystemColor.textInactiveText;
        Color color2 = SystemColor.textText;
        Color color3 = getBackground();
        boxSize    = new Dimension(BOX, BOX);
        openImage  = createImage(BOX_OPEN , boxSize.width, boxSize.height, color1, color2, color3);
        closeImage = createImage(BOX_CLOSE, boxSize.width, boxSize.height, color1, color2, color3);

        color1 = SystemColor.textHighlightText;
        color2 = SystemColor.textHighlightText;
        color3 = SystemColor.textHighlight;
        openSelectedImage  = createImage(BOX_OPEN , boxSize.width, boxSize.height, color1, color2, color3);
        closeSelectedImage = createImage(BOX_CLOSE, boxSize.width, boxSize.height, color1, color2, color3);
    }

    /** 󤫤饤᡼ */
    private Image createImage(int[] base, int width, int height,
                              Color color1, Color color2, Color color3) {
        int fg1 = color1.getRGB() | 0xff000000;
        int fg2 = color2.getRGB() | 0xff000000;
        int bg  = color3.getRGB() | 0xff000000;
        int[] buffer = new int[base.length];

        for (int i = 0; i < buffer.length; i++) {
            if (base[i] == 1) {
                buffer[i] = fg1;
            } else if (base[i] == 2) {
                buffer[i] = fg2;
            } else {
                buffer[i] = bg;
            }
        }

        MemoryImageSource mis = new MemoryImageSource(width, height, buffer, 0, width);
        mis.setAnimated(false);

        return getToolkit().createImage(mis);
    }

    /** ᡼ */
    private void drawImage(Graphics g, Image image,
                           int x, int y, int width, int height) {

        SyncObserver.drawImage(g, image, x, y, width, height, 10000);
    }

//### ٥
    /**
     * ƥꥹʤϿޤ
     *
     * @param  l Ͽ륢ƥꥹ
     */
    public synchronized void addItemListener(ItemListener l) {
        enableEvents(AWTEvent.ITEM_EVENT_MASK);
        itemListener = AWTEventMulticaster.add(itemListener, l);
    }

    /**
     * ƥꥹʤޤ
     *
     * @param  l 륢ƥꥹ
     */
    public synchronized void removeItemListener(ItemListener l) {
        itemListener = AWTEventMulticaster.remove(itemListener, l);
    }

    /**
     * ΥݡͥȤȯ륳ݡͥȥ٥Ȥޤ
     *
     * @param  e ٥
     */
    protected void processEvent(AWTEvent e) {
        if (e instanceof ItemEvent) {
            processItemEvent((ItemEvent) e);
            return;
        }
        super.processEvent(e);
    }

    /**
     * ΥݡͥȤȯ륢󥤥٥Ȥ
     * ϿƤ뤹٤Ƥ {@link ItemListener} 뤳Ȥˤꡢ
     * 󥤥٥Ȥޤ
     *
     * @param  e ٥
     */
    protected void processItemEvent(ItemEvent e) {
        if (itemListener != null) {
            itemListener.itemStateChanged(e);
        }
    }

    /** 󥤥٥Ȥȯޤץå + index å */
    private void postItemEvent(int index, int stateChange) {
        ItemEvent e = new ItemEvent(this,
                                    ItemEvent.ITEM_STATE_USER_CHANGED,
                                    ((Item) viewItems.elementAt(index)).data,
                                    stateChange);
        try {
            getToolkit().getSystemEventQueue().postEvent(e);
        } catch (SecurityException ex) {
            processItemEvent(e);
        }
    }

//### Item
    /** ƥ */
    private final class Item {
        private int      level;
        private Item     parent;
        private TreeItem data;
        private Vector   child;
        private boolean  checkChild;
        private boolean  isOpen;
        private boolean  isLast;

        /** 󥹥󥹤 */
        private Item(Item parent, TreeItem treeItem, boolean isLast) {
            if (parent == null) {
                this.level = -1;
            } else {
                this.level = parent.level + 1;
            }
            this.parent = parent;
            this.data   = treeItem;
            this.isLast = isLast;
        }

        /** ҥƥब뤫ɤ */
        private boolean hasChild() {
            if (!checkChild) {
                Vector v = data.getTreeItems();
                if (v != null) {
                    int    size = v.size();
                    Vector old  = child;
                    child = new Vector(size);

                    if (old == null) {
                        for (int i = 0; i < size; i++) {
                            child.addElement(new Item(this, (TreeItem) v.elementAt(i), (i + 1 == size)));
                        }
                    } else {
                        int oldSize = old.size();
                        TreeItem treeItem;
                        Item     newItem, oldItem;
                        for (int i = 0; i < size; i++) {
                            treeItem = (TreeItem) v.elementAt(i);
                            newItem     = new Item(this, treeItem, (i + 1 == size));
                            child.addElement(newItem);

                            // λҾˡƱҤϡΥơ򥳥ԡ
                            for (int j = 0; j < oldSize; j++) {
                                if ((oldItem = (Item) old.elementAt(j)).data == treeItem) {
                                    newItem.child  = oldItem.child;
                                    newItem.isOpen = oldItem.isOpen;
                                }
                            }
                        }
                    }
                }

                checkChild = true;
            }

            return (child != null && child.size() > 0);
        }

        /** Ҿ򥯥ꥢ */
        private void reset() {
            checkChild = false;

            if (child == null) {
                return;
            }

            for (int i = child.size() - 1; i >= 0; i--) {
                ((Item) child.elementAt(i)).reset();
            }
        }
    }

//### Scrollbar
    /** С */
    private final class Scrollbar extends InnerScrollbar {

        /** 󥹥󥹤 */
        private Scrollbar() {
            super(VERTICAL, Tree.this);
        }

        /**  */
        public void repaint() {
            repaintForce();
        }

        /** ͤѹ줿 */
        public void changedValue() {
            synchronized (Tree.this) {
                int v = getValue();
                if (v == viewStart) {
                    return;
                }

                viewStart = v;
            }

            repaintForce();
        }
    }

//### ListenerWrapper
    /** ꥹʥåѡ */
    private final class ListenerWrapper implements MouseListener, MouseMotionListener {
        private MouseListener       mouseListener;
        private MouseMotionListener mouseMotionListener;

        /** 󥹥󥹤 */
        private ListenerWrapper(MouseListener l) {
            mouseListener = l;
        }

        /** 󥹥󥹤 */
        private ListenerWrapper(MouseMotionListener l) {
            mouseMotionListener = l;
        }

        /** ޥå줿 */
        public void mouseClicked(MouseEvent e) {
            if (!scrollbar.contains(e.getX(), e.getY(), false)) {
                mouseListener.mouseClicked(e);
            }
        }

        /** ޥ줿 */
        public void mousePressed(MouseEvent e) {
            if (!scrollbar.contains(e.getX(), e.getY(), false)) {
                mouseListener.mousePressed(e);
            }
        }

        /** ޥ줿 */
        public void mouseReleased(MouseEvent e) {
            if (!scrollbar.contains(e.getX(), e.getY(), false)) {
                mouseListener.mouseReleased(e);
            }
        }

        /** ޥä */
        public void mouseEntered(MouseEvent e) {
            if (!scrollbar.contains(e.getX(), e.getY(), false)) {
                mouseListener.mouseEntered(e);
            }
        }

        /** ޥ줿 */
        public void mouseExited(MouseEvent e) {
            if (!scrollbar.contains(e.getX(), e.getY(), false)) {
                mouseListener.mouseExited(e);
            }
        }

        /** ޥɥå줿 */
        public void mouseDragged(MouseEvent e) {
            if (!scrollbar.contains(e.getX(), e.getY(), true)) {
                mouseMotionListener.mouseDragged(e);
            }
        }

        /** ޥư */
        public void mouseMoved(MouseEvent e) {
            if (!scrollbar.contains(e.getX(), e.getY(), false)) {
                mouseMotionListener.mouseMoved(e);
            }
        }
    }
}
