/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.awt;

import net.hizlab.kagetaka.awt.image.OffscreenImage;
import net.hizlab.kagetaka.util.Environment;

import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Point;
import java.awt.SystemColor;
import java.awt.Window;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseMotionAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.util.Hashtable;

/**
 * ̾Υġåפ󶡤ޤ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.4 $
 */
public class ToolTip {
    private static final int MARGIN       = 3;
    private static final int SPACE        = 15;
    private static final int SCREEN_SPACE = 100;

    private volatile long delay;
    private volatile long term;
    private Font      font;
    private Color     colorFore;
    private Color     colorBack;
    private Color     colorEdge;
    private Hashtable attached  = new Hashtable();
    private ToolTipWindow       window;
    private ToolTipThread       timer;
    private FocusListener       focusListener;
    private KeyListener         keyListener;
    private MouseListener       mouseListener;
    private MouseMotionListener mouseMotionListener;


    /**
     * ꤵ줿ٱɽġåפޤ
     *
     * @param  delay ٱʥߥá
     * @param  term  ɽ֡ʥߥá
     */
    public ToolTip(long delay, long term) {
        this.delay = delay;
        this.term  = term;

        timer = new ToolTipThread("Kankana-ToolTip");
        timer.start();

        mouseListener = new MouseAdapter() {
            /** ޥä */
            public void mouseEntered(MouseEvent e) {
                timer.reset(e);
            }

            /** ޥФ */
            public void mouseExited(MouseEvent e) {
                timer.cancel();
            }

            /** ޥ줿 */
            public void mousePressed(MouseEvent e) {
                timer.reset(e);
            }

            /** ޥ줿 */
            public void mouseReleased(MouseEvent e) {
                timer.reset(e);
            }
        };
        mouseMotionListener = new MouseMotionAdapter() {
            /** ޥư */
            public void mouseMoved(MouseEvent e) {
                timer.reset(e);
            }
        };
        focusListener = new FocusAdapter() {
            /** ե줿 */
            public void focusLost(FocusEvent e) {
                timer.cancel();
            }
        };
        keyListener = new KeyAdapter() {
            /** 줿 */
            public void keyPressed(KeyEvent e) {
                timer.cancel();
            }
        };

        setForeground(SystemColor.infoText    );
        setBackground(SystemColor.info        );
        setEdgeColor (SystemColor.windowBorder);
    }

    /**
     * եȤ֤ޤ
     *
     * @return ե
     */
    public Font getFont() {
        return font;
    }

    /**
     * եȤꤷޤ
     *
     * @param  font ե
     */
    public synchronized void setFont(Font font) {
        this.font = font;
    }

    /**
     * ʿ֤ޤ
     *
     * @return ʿ
     */
    public Color getForeground() {
        return colorFore;
    }

    /**
     * ʿꤷޤ
     *
     * @param  c ʿ
     */
    public synchronized void setForeground(Color c) {
        colorFore = c;
        ToolTipWindow w = window;
        if (w != null) {
            w.repaintForce();
        }
    }

    /**
     * طʿ֤ޤ
     *
     * @return طʿ
     */
    public Color getBackground() {
        return colorBack;
    }

    /**
     * طʿꤷޤ
     *
     * @param  c طʿ
     */
    public synchronized void setBackground(Color c) {
        colorBack = c;
        ToolTipWindow w = window;
        if (w != null) {
            w.repaintForce();
        }
    }

    /**
     * ο֤ޤ
     *
     * @return ο
     */
    public Color getEdgeColor() {
        return colorEdge;
    }

    /**
     * οꤷޤ
     *
     * @param  c ο
     */
    public synchronized void setEdgeColor(Color c) {
        colorEdge = c;
        ToolTipWindow w = window;
        if (w != null) {
            w.repaintForce();
        }
    }

    /**
     * ɽåäޤ
     */
    public void cancel() {
        timer.cancel();
    }

    /**
     * ɽٱ֤ޤ
     *
     * @return ٱʥߥá
     */
    public long getDelay() {
        return delay;
    }

    /**
     * ɽٱ֤ߥäǻꤷޤ
     *
     * @param  delay ٱʥߥá
     */
    public void setDelay(long delay) {
        this.delay = delay;
    }

    /**
     * ɽ֤ޤ
     *
     * @return ɽ֡ʥߥá
     */
    public long getTerm() {
        return term;
    }

    /**
     * ɽ֤ߥäǻꤷޤ
     *
     * @param  term ɽ֡ʥߥá
     */
    public void setTerm(long term) {
        this.term = term;
    }

    /**
     * ݡͥȤϿޤ
     *
     * @param  c Ͽ륳ݡͥ
     * @param  tip ΥݡͥȤɽå
     */
    public synchronized void attachComponent(Component c, String tip) {
        if (attached.containsKey(c)) {
            return;
        }

        attached.put(c, tip);
        c.addKeyListener        (keyListener        );
        c.addMouseListener      (mouseListener      );
        c.addMouseMotionListener(mouseMotionListener);
        if (Environment.javaVersion >= 104) {
            c.addFocusListener(focusListener);
        }

        if (c instanceof Container) {
            Component[] cs = ((Container) c).getComponents();
            for (int i = 0; i < cs.length; i++) {
                attachComponent(cs[i], tip);
            }
        }
    }

    /**
     * ݡͥȤޤ
     *
     * @param  c 륳ݡͥ
     */
    public synchronized void detachComponent(Component c) {
        if (!attached.containsKey(c)) {
            return;
        }

        attached.remove(c);
        c.removeKeyListener        (keyListener        );
        c.removeMouseListener      (mouseListener      );
        c.removeMouseMotionListener(mouseMotionListener);
        if (Environment.javaVersion >= 104) {
            c.removeFocusListener(focusListener);
        }

        if (c instanceof Container) {
            Component[] cs = ((Container) c).getComponents();
            for (int i = 0; i < cs.length; i++) {
                detachComponent(cs[i]);
            }
        }
    }

    /**
     * ˴Ƥ٤ƤΥ꥽ޤ
     */
    public void dispose() {
        timer.interrupt();
        ToolTipWindow w = window;
        if (w != null) {
            w.dispose();
        }
    }

//### ToolTipWindow
    /** ɥ */
    private final class ToolTipWindow extends Window implements ImageCreator {
        private OffscreenImage offscreenImage;

        private String   text;
        private FontData fontData;

        private int width;
        private int height;

        /** 󥹥󥹤 */
        private ToolTipWindow() {
            super(HiddenFrame.getInstance());

            offscreenImage = new OffscreenImage(this) {
                public void update(Image offscreen, Graphics g, Dimension size) {
                    refresh(offscreen, g, size);
                }
            };

            pack();
        }

        /**  */
        public void update(Graphics g) {
            paint(g);
        }

        /**  */
        public void paint(Graphics g) {
            offscreenImage.paint(g);
        }

        /** ե꡼ */
        private void refresh(Image offscreen, Graphics g, Dimension size) {
            synchronized (getTreeLock()) {
                g.setColor(colorBack);
                g.fillRect(1, 1, width - 2, height - 2);
                g.setColor(colorEdge);
                g.drawRect(0, 0, width - 1, height - 1);

                g.setColor(colorFore);
                g.setFont (fontData.getFont());

                FontMetrics fm       = fontData.getFontMetrics();
                int         baseFull = fontData.getFullBase();
                int         baseHalf = fontData.getHalfBase();

                int x = MARGIN + 1;
                int y = MARGIN + 1;
                char[] chars  = text.toCharArray();
                int    length = chars.length;
                char c;
                for (int i = 0; i < length; i++) {
                    c = chars[i];
                    g.drawChars(chars,
                                i, 1,
                                x,
                                y + ((c <= 0xFF) || (0xFF61 <= c && c <= 0xFF9F)
                                     ? baseHalf
                                     : baseFull));
                    x += fm.charWidth(c);
                }
            }
        }

        /** ˴ */
        public void dispose() {
            if (offscreenImage != null) {
                offscreenImage.dispose();
                offscreenImage = null;
            }
            super.dispose();
        }

        /** ʸɽ */
        private void show(String text, int x, int y) {
            synchronized (getTreeLock()) {
                Font font = ToolTip.this.font;
                if (font == null) {
                    font = getFont();
                }
                // ʸѹ줿
                if (this.text == null || this.text.compareTo(text) != 0
                        || this.fontData.getFont() != font) {
                    this.text     = text;
                    this.fontData = FontData.getInstance((ImageCreator) this, font);

                    FontMetrics fm = fontData.getFontMetrics();

                    int width  = fm.stringWidth(text)
                               + MARGIN * 2 + 2;
                    int height = Math.max(fontData.getFullSize().height,
                                          fontData.getHalfHeight())
                               + MARGIN * 2 + 2;

                    // Ѥä
                    if (this.width != width || this.height != height) {
                        this.width  = width;
                        this.height = height;
                        setSize(width, height);
                    }

                    repaintForce();
                }

                // ֤
                Dimension screenSize = getToolkit().getScreenSize();

                if (x + width + SCREEN_SPACE > screenSize.width) {
                    x = screenSize.width - SCREEN_SPACE - width;
                }
                y += SPACE;
                if (y + height + SCREEN_SPACE > screenSize.height) {
                    y -= height + SPACE * 2;
                }

                setLocation(x, y);
                toFront();
                super.show();
            }
        }

        /** Ū˺ɽ */
        private void repaintForce() {
            OffscreenImage oi = offscreenImage;
            if (oi != null) {
                oi.repaint();
            }
        }
    }

//### ToolTipThread
    /** ɽԤΥå */
    private final class ToolTipThread extends Thread {
        private Component component;
        private boolean   reset;
        private int x, y;

        /** 󥹥󥹤 */
        private ToolTipThread(String name) {
            super(name);
        }

        /** ¹Ԥ */
        public void run() {
            try {
                synchronized (this) {
                    Component component;
                    String    tip;

                    for (;;) {
                        // ɽΤ̵Ԥ
                        wait();

                        while ((component = this.component) != null) {
                            reset = false;

                            // ɽޤǰԤ
                            wait(delay);
                            if (reset) {
                                continue;
                            }

                            // ɽʸ
                            if ((tip = (String) attached.get(component)) == null) {
                                continue;
                            }

                            if (window == null) {
                                window = new ToolTipWindow();
                            }

                            // 
                            window.show(tip, x, y);

                            // ɽ
                            wait(term);

                            // õ
                            this.component = null;
                            window.setVisible(false);
                        }
                    }
                }
            } catch (InterruptedException e) {
            }
        }

        /**  */
        private synchronized void reset(MouseEvent e) {
            component = e.getComponent();
            reset     = true;

            // ֤¸
            Point p = e.getComponent().getLocationOnScreen();
            x = p.x + e.getX();
            y = p.y + e.getY();

            notify();
        }

        /** λ */
        private synchronized void cancel() {
            component = null;
            reset     = true;
            notify();
        }
    }
}
