/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.awt;

import net.hizlab.kagetaka.Debug;
import net.hizlab.kagetaka.awt.image.SyncObserver;
import net.hizlab.kagetaka.util.CharList;
import net.hizlab.kagetaka.util.IntBuffer;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Toolkit;
import java.awt.image.MemoryImageSource;
import java.awt.image.PixelGrabber;
import java.util.Hashtable;
import java.util.Vector;

/**
 * ʸĽɽ뤿Υ饹Ǥ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.8 $
 */
public class Text {
    private static final char TYPE_MASK  = 0x00FF;           // ץޥ
    private static final char TYPE_NONE  = 0x0031;           // ž̵
    private static final char TYPE_RIGHT = 0x0032;           //  90 ٲž
    private static final char TYPE_LTRB  = 0x0033;           // 屦ȿž
    private static final char TYPE_BR    = 0x0034;           // 
    private static final char TYPE_TAB   = 0x0035;           // 

    private static final char KK_NG      = 0x0100;           // Բ
    private static final char KK_SY      = 0x0200;           // ¥ٹ

    private static final int  MT_WAIT    = 10000;            // ǥȥåԤ

    // 󥹥ȥ饯
    private TextManager manager;
    private String      value;
    private char        echoChar;
    private Font        font;
    private CharList    charsSpinRight;
    private CharList    charsSpinLtrb;
    private CharList    charsKinsokuHead;
    private CharList    charsKinsokuTail;

    // 
    private ImageCreator imageCreator;
    private Toolkit      toolkit;
    private FontData     fd;
    private Dimension    fs;
    private FontMetrics  fm;
    private int          ff;                  // fontFullBase
    private int          fh;                  // fontHalfBase
    private int          fx;                  // fontHalfHeight
    private int          fz;                  // fontMaxHeight
    private int          fn;                  // fontMiniBase
    private int          fl;                  // fontMiniLeft

    private StringBuffer values;
    private StringBuffer types;
    private IntBuffer    widths;
    private IntBuffer    heights;
    private int nowLength;
    private int nowWidth;
    private int nowHeight;
    private boolean hasTab;
    private boolean lastPending;            // Ǹ夬ڥǥ󥰤
    private boolean lastHankakuEisu;        // ǸʸȾѱѿ
    private int p = -1;
    private boolean mastHeight = true;      // ɬ礭뤫

    private SyncObserver syncObserver;      // Ԥ

    // å
    private Hashtable cacheSize;
    private Detail    cacheZero;

    /**
     * 󥹥󥹤ޤ
     *
     * @param  manager  ޥ͡
     * @param  value    ʸ
     * @param  echoChar ɽʸ
     * @param  font     եȥǡ
     */
    Text(TextManager manager, String value, char echoChar, Font font) {
        this.manager          = manager;
        this.value            = value;
        this.echoChar         = echoChar;
        this.font             = font;
        this.charsSpinRight   = manager.charsSpinRight;
        this.charsSpinLtrb    = manager.charsSpinLtrb;
        this.charsKinsokuHead = manager.charsKinsokuHead;
        this.charsKinsokuTail = manager.charsKinsokuTail;
    }

    /**
     * ʸ֤ޤ
     *
     * @return ʸ
     */
    public String getValue() {
        return value;
    }

    /**
     * ɽʸ֤ޤ
     *
     * @return ɽʸ
     */
    public char getEchoChar() {
        return echoChar;
    }

    /**
     * եȤ֤ޤ
     *
     * @return ե
     */
    public Font getFont() {
        return font;
    }

    /**
     * ꤵ줿ɽʸ褹륤󥹥󥹤֤ޤ
     *
     * @param  value ʸ
     *
     * @return ߤξ֤ǻꤵ줿ʸΥ󥹥
     */
    public Text getText(String value) {
        return (this.value != null && this.value.compareTo(value) == 0
                ? this
                : manager.getText(value, echoChar, font));
    }

    /**
     * ꤵ줿ɽʸ褹륤󥹥󥹤֤ޤ
     *
     * @param  echoChar ɽʸ
     *
     * @return ߤʸǻꤵ줿ɽʸΥ󥹥
     */
    public Text getText(char echoChar) {
        return (this.echoChar == echoChar
                ? this
                : manager.getText(value, echoChar, font));
    }

    /**
     * ꤵ줿եȤ褹륤󥹥󥹤֤ޤ
     *
     * @param  font ե
     *
     * @return ߤʸǻꤵ줿եȤΥ󥹥
     */
    public Text getText(Font font) {
        return (this.font != null && this.font.equals(font)
                ? this
                : manager.getText(value, echoChar, font));
    }

    /** ٤ */
    private synchronized void ensureInitialize() {
        if (values != null) {
            return;
        }

        values  = new StringBuffer();
        types   = new StringBuffer();
        widths  = new IntBuffer   ();
        heights = new IntBuffer   ();

        syncObserver = new SyncObserver();

        cacheSize = new Hashtable();
        cacheZero = new Detail   ();

        this.imageCreator = manager.imageCreator;
        this.toolkit      = imageCreator.getToolkit();
        this.fd           = FontData.getInstance(imageCreator, font);
        this.fs           = fd.getFullSize   ();
        this.fm           = fd.getFontMetrics();
        this.ff           = fd.getFullBase   ();
        this.fh           = fd.getHalfBase   ();
        this.fx           = fd.getHalfHeight ();
        this.fz           = fd.getMaxHeight  ();
        this.fn           = fd.getMiniBase   ();
        this.fl           = fd.getMiniLeft   ();

        if (value == null) {
            return;
        }

        if (echoChar == 0) {
            append(value, 0, value.length());
        } else {
            // ɽʸꤵƤ
            char c = echoChar;
            StringBuffer sb = new StringBuffer();
            for (int i = value.length() - 1; i >= 0; i--) {
                sb.append(c);
            }
            append(sb.toString(), 0, value.length());
        }
        if (nowLength > 0) {
            newLine();
        }
    }

    /** ʸɲ */
    private void append(String text, int begin, int end) {
        char c, s;
        for (int i = begin; i < end; i++) {
            c = text.charAt(i);

            // Ԥξ
            if (c == 0x0A) {
                commit();
                append(c, TYPE_BR, false);
                newLine();
                continue;
            }

            // ֤ξʰöϥڡȤϿ
            if (c == 0x09) {
                commit();
                append(' ', TYPE_TAB, false);
                hasTab = true;
                continue;
            }

            // Ⱦѱѿʸ
            if (/*---*/(0x30 <= c && c <= 0x39)
                    || (0x41 <= c && c <= 0x5A)
                    || (0x61 <= c && c <= 0x7A)) {
                if (!lastHankakuEisu) {
                    commit();
                }

                append(c, TYPE_RIGHT, true);
                lastHankakuEisu = true;
                continue;
            }

            lastHankakuEisu = false;
            s = TYPE_NONE;

            // žη
            if (/*---*/(c <= 0xFF)
                    || (0xFF61 <= c && c <= 0xFF9F)
                    || charsSpinRight.contains(c)) {
                s = TYPE_RIGHT;
            } else if (charsSpinLtrb.contains(c)) {
                s = TYPE_LTRB;
            }

            // Ƭ§ʸξϡ̤Хåե
            if (charsKinsokuHead.contains(c)) {
                append(c, s, true);
                continue;
            }

            // §ʸξϡߥåȤƳХåե
            if (charsKinsokuTail.contains(c)) {
                commit();
                append(c, s, false);
                continue;
            }

            // ̤ʸξ硢ߥåȤ̤Хåե
            commit();
            append(c, s, true);
        }
    }

    /** Хåեɲ */
    private void append(char c, char s, boolean pending) {
        if (CharList.SOKUYOU.contains(c)) {
            s |= KK_SY;
        }
        if (pending) {
            if (lastPending) {
                types.setCharAt(p, (char) (types.charAt(p) | KK_NG));
            } else {
                lastPending = true;
            }
        }

        int w = calculateWidth (c, s);
        int h = calculateHeight(c, s);

        values .append(c);
        types  .append(s);
        widths .append(w);
        heights.append(h);
        p++;

        nowLength++;
        nowWidth  =  Math.max(nowWidth, w);
        nowHeight += h;
    }

    /** ̤ʬ */
    private void commit() {
        if (lastPending) {
            lastPending = false;
        }
    }

    /** եȤ֤ޤ */
    private int calculateWidth(char c, char s) {
        switch (s) {
        case TYPE_RIGHT:
        case TYPE_LTRB :
        case TYPE_TAB  :
            return ((c <= 0xFF) || (0xFF61 <= c && c <= 0xFF9F) ? fx : fs.width);
        default:
            return fm.charWidth(c);
        }
    }

    /** եȤι⤵֤ޤ */
    private int calculateHeight(char c, char s) {
        switch (s) {
        case TYPE_RIGHT:
        case TYPE_LTRB :
        case TYPE_TAB  :
            return fm.charWidth(c);
        default:
            return fs.height;
        }
    }

    /** ԤԤ */
    private void newLine() {
        // Ԥ˥֤ͭä硢Ԥι⤵
        if (hasTab) {
            StringBuffer types   = this.types;
            IntBuffer    heights = this.heights;
            int length = nowLength;
            int total  = 0;
            int h;
            for (int i = 0; i < length; i++) {
                h = heights.get(i);
                if ((types.charAt(i) & TYPE_MASK) == TYPE_TAB) {
                    int target = 0;
                    do {
                        target += h * 8;
                    } while (total >= target);
                    total = target;
                } else {
                    total += h;
                }
            }
            nowHeight = total;
            hasTab    = false;
        }
        cacheZero.addLine(nowLength, nowWidth, nowHeight);

        nowLength = 0;
        nowWidth  = 0;
        nowHeight = 0;
    }

    /**
     * ʸλꤵ줿⤵褹ɬפ礭֤ޤ
     *
     * @param  maxHeight ι⤵⤵¤ʤ <code>0</code>
     *
     * @return ɬפ礭
     */
    public Dimension getSize(int maxHeight) {
        ensureInitialize();

        return new Dimension(getDatail(maxHeight).size);
    }

    /** ǥƥ */
    private Detail getDatail(int maxHeight) {
        if (maxHeight <= 0) {
            return cacheZero;
        }

        synchronized (cacheSize) {
            Detail detail;
            Integer key = new Integer(maxHeight);

            // å¸ߤФ֤
            if ((detail = (Detail) cacheSize.get(key)) != null) {
                return detail;
            }

            // åɲ
            cacheSize.put(key, (detail = new Detail()));

            int w  = 0, h  = 0;
            int pl = 0, pw = 0, ph = 0;     // ڥǥ礭
            int cl = 0, cw = 0, ch = 0;     // ߤ礭
            char s;

            for (int i = 0; i <= p; i++) {
                s = types  .charAt(i);
                w = widths .get   (i);
                h = heights.get   (i);

                // ꤭ʤ
                if (mastHeight) {
                    if (ch + ph + h > maxHeight) {
                        // ڥǥ󥰤μǲԤƤߤ
                        if (cl > 0) {
                            detail.addLine(cl, cw, ch);
                            cl = cw = ch = 0;
                        }

                        // ڥǥ󥰤Ǥ꤭ʤ
                        if (ph + h > maxHeight) {
                            detail.addLine(pl, pw, ph);
                            pl = pw = ph = 0;
                        }
                    }
                }

                // §ʸξ
                if ((s & KK_NG) != 0) {
                    pl++;
                    pw =  Math.max(pw, w);
                    ph += h;
                    continue;
                }

                // ʸξ硢⤵
                if (s == TYPE_TAB) {
                    int total  = ch + ph;
                    int target = 0;
                    do {
                        target += h * 8;
                    } while (total >= target);
                    if (!mastHeight && target > maxHeight) {
                        target = maxHeight;
                    }
                    h = target - total;
                }

                if (pl > 0) {
                    // ڥǥʸä
                    pl++;
                    pw =  Math.max(pw, w);
                    ph += h;

                    if (!mastHeight && ch + ph > maxHeight) {
                        // ʸɲäǤʤϡڥǥ󥰤μǲ
                        if (cl > 0) {
                            detail.addLine(cl, cw, ch);
                        }

                        cl = pl;
                        cw = pw;
                        ch = ph;
                    } else {
                        // ڥǥ󥰤ꤹ
                        cl += pl;
                        cw =  Math.max(cw, pw);
                        ch += ph;
                    }
                    pl = pw = ph = 0;
                } else {
                    // Ѥߤʸξ

                    if (!mastHeight && ch + h > maxHeight) {
                        // ʸɲäǤʤϡľǲ
                        if (cl > 0) {
                            detail.addLine(cl, cw, ch);
                        }

                        cl = 1;
                        cw = w;
                        ch = h;
                    } else {
                        cl++;
                        cw =  Math.max(cw, w);
                        ch += h;
                    }
                }

                // ¤ϲʸä
                if (s == TYPE_BR) {
                    detail.addLine(cl, cw, ch);
                    cl = cw = ch = 0;
                }
            }

            if (cl > 0) {
                detail.addLine(cl, cw, ch);
            }

            return detail;
        }
    }

    /**
     * ʸ褷ޤ
     *
     * @param  g եå
     * @param  x 夫 X
     * @param  y 夫 Y
     * @param  height ι⤵⤵¤ʤ <code>0</code>
     */
    public void draw(Graphics g, int x, int y, int height) {
        ensureInitialize();

        Detail detail = getDatail(height);

        g.setFont(font);

        int by = y;

        int  begin = 0;
        Line line = null;
        int  ln   = -1, lc = 0;
        int w, h;
        char s;
        for (int i = 0; i <= p;) {
            // 
            if (lc == 0) {
                if (x <= 0) {
                    break;
                }
                line = (Line) detail.lines.elementAt(++ln);
                lc   = line.length;
                x -= line.width; y = by;
            }

            s = (char) (types.charAt(i) & TYPE_MASK);
            w = widths .get(i);
            h = heights.get(i);

            // 
            switch (s) {
            case TYPE_NONE:
                // Ľ񤭤ϡĤ
                char[] tateCs = new char[1];
                tateCs[0] = values.charAt(i);
                if ((types.charAt(i) & KK_SY) != 0) {
                    g.drawChars(tateCs, 0, 1,
                                (fs.width > w ? x + ((fs.width - w) / 2) : x) + fl,
                                y + fn);
                } else {
                    g.drawChars(tateCs, 0, 1, (fs.width > w ? x + ((fs.width - w) / 2) : x), y + ff);
                }
//g.drawRect(x, y, w - 1, h - 1);
                y += h;
                --lc; ++i;
                break;
            case TYPE_RIGHT:
            case TYPE_LTRB :
                // Ʊžʤ顢ޤȤƲž
                begin = i;
                for (;;) {
                    --lc; ++i;
                    if ((lc == 0) || ((types.charAt(i) & TYPE_MASK) != s)) {
                        break;
                    }
                    w =  Math.max(w, widths.get(i));
                    h += heights.get(i);
                }
                drawTextWithSpin(g, begin, i, x, y, w, h, s);
//g.drawRect(x, y, w - 1, h - 1);
                y += h;
                break;
            case TYPE_BR:
                --lc; ++i;
                break;
            case TYPE_TAB:
                {
                    int target = by;
                    do {
                        target += h * 8;
                    } while (y >= target);
                    y = target;
                    --lc; ++i;
                    break;
                }
            default: // AVOID
            }
        }
    }

    /** ʸž */
    private void drawTextWithSpin(Graphics g, int begin, int end,
                                  int x, int y, int width, int height, char type) {
        // եȤΤʤʸž褦Ȥ
        if (height == 0) {
            return;
        }

        int scrapX, scrapY;
        Color backColor = Color.white;
        Color foreColor = Color.black;

        Image scrap;
        try {
            scrap = imageCreator.createImage(height, width);
        } catch (IllegalArgumentException e) {
Debug.out.println("e=[" + values + "],[" + begin + "],[" + end + "],[" + height + "],[" + width + "]");
            return;
        } catch (NegativeArraySizeException e) {
Debug.out.println("e=[" + values + "],[" + begin + "],[" + end + "],[" + height + "],[" + width + "]");
            return;
        }

        Graphics sg = scrap.getGraphics();
        try {
            // طʿ white ˤơwhite Ʃˤ
            sg.setColor(backColor);
            sg.fillRect(0, 0, height, width);

            // ʿ򤳤ꤷʤΤϡƩѤοȽŤʤǽΤȡ
            // JDK 1.1 ΰǡǤʤХб뤿
            sg.setColor(foreColor   );
            sg.setFont (fd.getFont());

            ///// åפ˲žʸ

            int base = 0;
            // TYPE_LTRB ξΥ١֤׻
            if (type == TYPE_LTRB) {
                base = ff + (fs.height - ff) / 2;
            }

            char c;
            scrapX = 0;
            char[] tateCs = new char[1];
            for (int i = begin; i < end; i++) {
                tateCs[0] = c = values.charAt(i);

                // TYPE_RIGHT ξΥ١֤׻
                if (type == TYPE_RIGHT) {
                    base = ((c <= 0xFF) || (0xFF61 <= c && c <= 0xFF9F) ? fh : ff);
                }

                sg.drawChars(tateCs, 0, 1, scrapX, base);
                scrapX += heights.get(i);
            }
        } finally {
            sg.dispose();
        }

        ///// ʸž
        int[] bitmap1 = new int[width * height];
        int[] bitmap2 = new int[width * height];
        PixelGrabber pg = new PixelGrabber(scrap, 0, 0, height, width, bitmap1, 0, height);
        try {
            // ѴǼԤƤ⡢̵뤷Ʒ³ƹʤ
            pg.grabPixels(MT_WAIT);
        } catch (InterruptedException e) { }

        switch (type) {
        case TYPE_RIGHT:
            for (scrapX = 0; scrapX < height; scrapX++) {
                for (scrapY = 0; scrapY < width; scrapY++) {
                    bitmap2[scrapX * width + scrapY] = bitmap1[scrapX + (width - scrapY - 1) * height];
                }
            }
            break;
        case TYPE_LTRB:
            for (scrapX = 0; scrapX < height; scrapX++) {
                for (scrapY = 0; scrapY < width; scrapY++) {
                    bitmap2[scrapX * width + scrapY] = bitmap1[scrapX + scrapY * height];
                }
            }
            break;
        default: // AVOID
        }

        // ʸοꤷطʤƩˤ
        int bcc = backColor.getRGB();
        int fcc = g.getColor().getRGB() | 0xff000000;
        for (int i = 0; i < bitmap2.length; i++) {
            if (bitmap2[i] == bcc) {
                bitmap2[i] &= 0xffffff;             // Ʃ
            } else {
                bitmap2[i] = fcc;                   // ʿˤ
            }
        }

        // ᡼Ǥϡwidth  height ͤϸ򴹤Ƥ

        MemoryImageSource mis = new MemoryImageSource(width, height, bitmap2, 0, width);
        mis.setAnimated(false);
        Image image = toolkit.createImage(mis);

        // 褹褦ˤ
        drawImageSync(g, image, x, y, width, height);

        // ꥽
        scrap.flush();
        image.flush();
    }

    /** Ʊ */
    private void drawImageSync(Graphics g, Image image, int x, int y, int width, int height) {
        synchronized (syncObserver) {
            syncObserver.init(g, x, y, width, height);
            if (!g.drawImage(image, x, y, width, height, syncObserver)) {
                try {
                    syncObserver.wait(MT_WAIT);
                } catch (InterruptedException e) {
Debug.out.println("wait to draw a text");
                }
            }
        }
    }

    /**
     * ʸ˴ơΥ꥽ޤ
     */
    void dispose() {
    }

//### Detail
    /** ǥƥ */
    private final class Detail {
        private Dimension size  = new Dimension();
        private Vector    lines = new Vector();

        /** 󥹥󥹤 */
        private Detail() {
        }

        /** Ԥɲ */
        private void addLine(int length, int width, int height) {
            width = Math.max(width, fz);
            lines.addElement(new Line(length, width, height));
            size.width  += width;
            size.height =  Math.max(size.height, height);
        }
    }

//### Line
    /**  */
    private final class Line {
        private int length;
        private int width;
        private int height;

        /** 󥹥󥹤 */
        private Line(int length, int width, int height) {
            this.length = length;
            this.width  = width;
            this.height = height;
        }
    }
}
