/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.awt;

import net.hizlab.kagetaka.awt.event.TabEvent;
import net.hizlab.kagetaka.awt.event.TabListener;
import net.hizlab.kagetaka.awt.image.ImageContainer;
import net.hizlab.kagetaka.awt.image.WrappedObserver;

import java.awt.AWTEvent;
import java.awt.Color;
import java.awt.CardLayout;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Image;
import java.awt.Insets;
import java.awt.SystemColor;
import java.awt.event.ComponentListener;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.InputEvent;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.awt.image.MemoryImageSource;
import java.util.Vector;

/**
 * ڤؤѥͥǤ
 *
 * @kagetaka.bugs ľ󲽤ϡꥹʤ¸ʤޤ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.6 $
 */
public class TabPanel extends Container {
    private static final int TABBAR_LEFT_FIX   = 4;
    private static final int TABBAR_RIGHT_FIX  = 5;
    private static final int TABBAR_IMAGE_FIX  = 2;
    private static final int TABBAR_HEIGHT_FIX = 4;

    private static final String TITLE_CONTINUE = "...";

    private static final int[] LEFT_IMAGE =
    {
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    };
    private static final int[] RIGHT_IMAGE =
    {
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    };

    /** @serial ĤΥ֤κǾ */
    private int      minWidth =  50;
    /** @serial ĤΥ֤κ */
    private int      maxWidth = 300;
    /** @serial եΤե졼࿧ */
    private Color    focusFrameColor;
    /** @serial ƥ֥ȥե */
    private Font     activeTitleFont;
    /** @serial ƥ֥ȥʸ */
    private Color    activeTitleForeground;
    /** @serial ƥ֥ȥʸ */
    private Color    activeTitleBackground;
    /** @serial 󥢥ƥ֥ȥե */
    private Font     unactiveTitleFont;
    /** @serial 󥢥ƥ֥ȥʸ */
    private Color    unactiveTitleForeground;
    /** @serial 󥢥ƥ֥ȥʸ */
    private Color    unactiveTitleBackground;

    /** @serial ֥С */
    private Tabbar        tabbar;
    /** @serial ֥С */
    private TabScrollbar  tabScrollbar;
    /** @serial ᥤѥͥ */
    private MainPanel     main;
    /** @serial ѥͥΥꥹ */
    private Vector        tabs = new Vector();
    /** @serial ƥ֥ */
    private Tab           activeTab;
    /** @serial ֤եäƤ뤫ɤ */
    private boolean       hasFocus;

    // ꥹʡ
    private transient TabListener         tabListener;
    private transient ComponentListener   tabComponentListener;
    private transient FocusListener       tabFocusListener;
    private transient KeyListener         tabKeyListener;
    private transient MouseListener       tabMouseListener;
    private transient MouseMotionListener tabMouseMotionListener;

    /**
     * ֥ѥͥޤ
     * Dialog եȤѤƥ५顼ǽޤ
     */
    public TabPanel() {
        // ֥С
        tabbar       = new Tabbar      ();
        tabScrollbar = new TabScrollbar();
        main         = new MainPanel   ();

        // ̤
        GridBagLayout gbl    = new GridBagLayout();
        Insets        insets = new Insets(0, 0, 0, 0);
        setLayout(gbl);
        LayoutUtils.addGridBag(this, tabbar      , gbl, 0, 0, 1, 1, 1, 0, GridBagConstraints.NONE, GridBagConstraints.WEST     , insets);
        insets.bottom = 2;
        LayoutUtils.addGridBag(this, tabScrollbar, gbl, 1, 0, 1, 1, 0, 0, GridBagConstraints.NONE, GridBagConstraints.SOUTHEAST, insets);
        insets.bottom = 0;
        LayoutUtils.addGridBag(this, main        , gbl, 0, 1, 2, 1, 1, 1, GridBagConstraints.BOTH, GridBagConstraints.CENTER   , insets);

        // ꥹʤϿ
        addFocusListener(
            new FocusListener() {
                /** ե */
                public void focusGained(FocusEvent e) {
                    synchronized (getTreeLock()) {
                        if (!hasFocus) {
                            hasFocus = true;
                            if (activeTab != null && activeTab.isShowing()) {
                                activeTab.repaint();
                            }
                        }
                    }
                }

                /** ե򼺤ä */
                public void focusLost(FocusEvent e) {
                    synchronized (getTreeLock()) {
                        if (hasFocus) {
                            hasFocus = false;
                            if (activeTab != null && activeTab.isShowing()) {
                                activeTab.repaint();
                            }
                        }
                    }
                }
            }
        );

        addKeyListener(
            new KeyAdapter() {
                /** 줿 */
                public void keyPressed(KeyEvent e) {
                    switch (e.getKeyCode()) {
                    case KeyEvent.VK_LEFT:
                    case KeyEvent.VK_PAGE_UP:
                        synchronized (getTreeLock()) {
                            if (activeTab != null) {
                                int index = tabs.indexOf(activeTab);
                                if (index <= 0) {
                                    return;
                                }
                                setActiveTab((Tab) tabs.elementAt(index - 1));
                            }
                        }
                        break;
                    case KeyEvent.VK_RIGHT:
                    case KeyEvent.VK_PAGE_DOWN:
                        synchronized (getTreeLock()) {
                            if (activeTab != null) {
                                int index = tabs.indexOf(activeTab);
                                if (index + 1 >= tabs.size()) {
                                    return;
                                }
                                setActiveTab((Tab) tabs.elementAt(index + 1));
                            }
                        }
                        break;
                    case KeyEvent.VK_HOME:
                        synchronized (getTreeLock()) {
                            if (tabs.size() > 0) {
                                setActiveTab((Tab) tabs.elementAt(0));
                            }
                        }
                        break;
                    case KeyEvent.VK_END:
                        synchronized (getTreeLock()) {
                            if (tabs.size() > 0) {
                                setActiveTab((Tab) tabs.elementAt(tabs.size() - 1));
                            }
                        }
                        break;
                    default: // AVOID
                    }
                }
            }
        );

        // ƥȥΥǥեȤ
        setForeground             (SystemColor.controlText );
        setBackground             (SystemColor.control     );
        setFocusFrameColor        (SystemColor.windowBorder);
        setActiveTitleForeground  (SystemColor.controlText );
        setActiveTitleBackground  (SystemColor.control     );
        setUnActiveTitleForeground(SystemColor.controlText );
        setUnActiveTitleBackground(SystemColor.control     );
/*
setForeground  (Color.green);
setBackground  (Color.orange);
setFocusFrameColor        (Color.red);
setActiveTitleFont        (Font.decode("Dialog"      ));
setActiveTitleForeground  (Color.black);
setActiveTitleBackground  (Color.blue);
setUnActiveTitleFont      (Font.decode("Dialog"      ));
setUnActiveTitleForeground(Color.gray);
setUnActiveTitleBackground(Color.lightGray);
*/
    }

    /**
     * ֥ѥͥɲäޤ
     * <p>
     * <code>active</code>  <code>true</code> ξ硢ɲä֤
     * ƥ֤ˤʤޤܤΥ֤ξϡͤ˴طʤ
     * Υ֤ƥ֤ˤʤޤ
     *
     * @param  title     ȥ
     * @param  image     ᡼ɽʤ <code>null</code>
     * @param  component ݡͥ
     * @param  active    ɲä֤򥢥ƥ֤ˤ <code>true</code>
     *                   ʳξ <code>false</code>
     */
    public void addPanel(String title, Image image, Component component,
                         boolean active) {
        addPanel(title, image, component, active, -1);
    }

    /**
     * ֥ѥͥꤷǥå֤ɲäޤ
     * <p>
     * <code>active</code>  <code>true</code> ξ硢ɲä֤
     * ƥ֤ˤʤޤܤΥ֤ξϡͤ˴طʤ
     * Υ֤ƥ֤ˤʤޤ
     *
     * @param  title     ȥ
     * @param  image     ᡼ɽʤ <code>null</code>
     * @param  component ݡͥ
     * @param  active    ɲä֤򥢥ƥ֤ˤ <code>true</code>
     *                   ʳξ <code>false</code>
     * @param  index     ɲä륿֥ѥͥΥǥå
     *                   Ǹɲä <code>-1</code>
     *
     * @return ɲä줿֥ݡͥ
     *
     * @throws IllegalArgumentException ʣ
     */
    public Tab addPanel(String title, Image image, Component component,
                        boolean active, int index) {
        Tab tab;

        synchronized (getTreeLock()) {
            tab = new Tab(title, image, component);

            // index λ꤬ Tab.addNotify ˰ѤʤΤǤ tabs ɲ
            if (index == -1) {
                tabs.addElement(tab);
                index = tabs.size() - 1;
            } else {
                tabs.insertElementAt(tab, index);
            }

            tabbar.add(tab, new Integer(index));

            // ƥ֤ꤵƤꡢǽΰܤΥ֤ϥƥ֤ˤ
            if (active || tabs.size() == 1) {
                setActiveTab(tab);
            }
        }

        validate();
        return tab;
    }

    /**
     * ƥ֤ʥ֤Υǥå֤ޤ
     *
     * @return ǥå
     *         ֤Ĥʤ <code>-1</code>
     */
    public int getActiveTabIndex() {
        synchronized (getTreeLock()) {
            if (activeTab == null) {
                return -1;
            }

            return tabs.indexOf(activeTab);
        }
    }

    /**
     * ĤΥ֤κǾ֤ޤ
     *
     * @return Ǿ
     */
    public int getTabMinWidth() {
        return minWidth;
    }

    /**
     * ĤΥ֤κǾꤷޤ
     *
     * @param  width Ǿ
     */
    public void setTabMinWidth(int width) {
        synchronized (getTreeLock()) {
            minWidth = width;
            invalidateAllTabs();
        }
        validate();
    }

    /**
     * ĤΥ֤κ֤ޤ
     *
     * @return 
     */
    public int getTabMaxWidth() {
        return maxWidth;
    }

    /**
     * ĤΥ֤κꤷޤ
     *
     * @param  width 
     */
    public void setTabMaxWidth(int width) {
        synchronized (getTreeLock()) {
            maxWidth = width;
            invalidateAllTabs();
        }
        validate();
    }

    /**
     * եΤե졼࿧֤ޤ
     *
     * @return 
     */
    public Color getFocusFrameColor() {
        return focusFrameColor;
    }

    /**
     * եΤե졼࿧ꤷޤ
     *
     * @param  color 
     */
    public void setFocusFrameColor(Color color) {
        synchronized (getTreeLock()) {
            focusFrameColor = color;
        }
    }

    /**
     * ƥ֥ȥѤեȤ֤ޤ
     *
     * @return ե
     */
    public Font getActiveTitleFont() {
        return activeTitleFont;
    }

    /**
     * ƥ֥ȥѤեȤꤷޤ
     *
     * @param  font ե
     */
    public void setActiveTitleFont(Font font) {
        synchronized (getTreeLock()) {
            activeTitleFont = font;
            invalidateAllTabs();
        }
        validate();
    }

    /**
     * ƥ֥ȥʸ֤ޤ
     *
     * @return 
     */
    public Color getActiveTitleForeground() {
        return activeTitleForeground;
    }

    /**
     * ƥ֥ȥʸꤷޤ
     *
     * @param  color 
     */
    public void setActiveTitleForeground(Color color) {
        synchronized (getTreeLock()) {
            activeTitleForeground = color;
        }
    }

    /**
     * ƥ֥ȥطʿ֤ޤ
     *
     * @return 
     */
    public Color getActiveTitleBackground() {
        return activeTitleBackground;
    }

    /**
     * ƥ֥ȥطʿꤷޤ
     *
     * @param  color 
     */
    public void setActiveTitleBackground(Color color) {
        synchronized (getTreeLock()) {
            activeTitleBackground = color;
        }
    }

    /**
     * 󥢥ƥ֥ȥѤեȤ֤ޤ
     *
     * @return ե
     */
    public Font getUnActiveTitleFont() {
        return unactiveTitleFont;
    }

    /**
     * 󥢥ƥ֥ȥѤեȤꤷޤ
     *
     * @param  font ե
     */
    public void setUnActiveTitleFont(Font font) {
        synchronized (getTreeLock()) {
            unactiveTitleFont = font;
            invalidateAllTabs();
        }
        validate();
    }

    /**
     * 󥢥ƥ֥ȥʸ֤ޤ
     *
     * @return 
     */
    public Color getUnActiveTitleForeground() {
        return unactiveTitleForeground;
    }

    /**
     * 󥢥ƥ֥ȥʸꤷޤ
     *
     * @param  color 
     */
    public void setUnActiveTitleForeground(Color color) {
        synchronized (getTreeLock()) {
            unactiveTitleForeground = color;
        }
    }

    /**
     * 󥢥ƥ֥ȥطʿ֤ޤ
     *
     * @return 
     */
    public Color getUnActiveTitleBackground() {
        return unactiveTitleBackground;
    }

    /**
     * 󥢥ƥ֥ȥطʿꤷޤ
     *
     * @param  color 
     */
    public void setUnActiveTitleBackground(Color color) {
        synchronized (getTreeLock()) {
            unactiveTitleBackground = color;
        }
    }

    /**
     * ֤򥹥뤵ܥꤷޤ
     *
     * @param  left  Υܥ
     * @param  right Υܥ
     */
    public void setTabScrollButton(ImageButton left, ImageButton right) {
        synchronized (getTreeLock()) {
            tabScrollbar.setButton(left, right);
        }
    }

    /**
     * ֤򥹥뤵ܥǥեȤꤷޤ
     */
    public void resetTabScrollButton() {
        synchronized (getTreeLock()) {
            tabScrollbar.createDefaultButton();
        }
    }

    /**
     * ֥ꥹʤϿޤ
     *
     * @param  l Ͽ륿֥ꥹ
     */
    public void addTabListener(TabListener l) {
        enableEvents(0);
        tabListener = AWTEventMulticaster.add(tabListener, l);
    }

    /**
     * ֥ꥹʤޤ
     *
     * @param  l 륿֥ꥹ
     */
    public void removeTabListener(TabListener l) {
        tabListener = AWTEventMulticaster.remove(tabListener, l);
    }

    /**
     * ֤Ф륳ݡͥȥꥹʤϿޤ
     *
     * @param  l Ͽ륳ݡͥȥꥹ
     */
    public void addTabComponentListener(ComponentListener l) {
        synchronized (getTreeLock()) {
            tabComponentListener = AWTEventMulticaster.add(tabComponentListener, l);
            for (int i = 0; i < tabs.size(); i++) {
                ((Tab) tabs.elementAt(i)).addComponentListener(l);
            }
        }
    }

    /**
     * ֤Ф륳ݡͥȥꥹʤޤ
     *
     * @param  l 륳ݡͥȥꥹ
     */
    public void removeTabComponentListener(ComponentListener l) {
        synchronized (getTreeLock()) {
            tabComponentListener = AWTEventMulticaster.remove(tabComponentListener, l);
            for (int i = 0; i < tabs.size(); i++) {
                ((Tab) tabs.elementAt(i)).removeComponentListener(l);
            }
        }
    }

    /**
     * ֤ФեꥹʤϿޤ
     *
     * @param  l Ͽեꥹ
     */
    public void addTabFocusListener(FocusListener l) {
        synchronized (getTreeLock()) {
            tabFocusListener = AWTEventMulticaster.add(tabFocusListener, l);
            for (int i = 0; i < tabs.size(); i++) {
                ((Tab) tabs.elementAt(i)).addFocusListener(l);
            }
        }
    }

    /**
     * ֤Фեꥹʤޤ
     *
     * @param  l եꥹ
     */
    public void removeTabFocusListener(FocusListener l) {
        synchronized (getTreeLock()) {
            tabFocusListener = AWTEventMulticaster.remove(tabFocusListener, l);
            for (int i = 0; i < tabs.size(); i++) {
                ((Tab) tabs.elementAt(i)).removeFocusListener(l);
            }
        }
    }

    /**
     * ֤Ф륭ꥹʤϿޤ
     *
     * @param  l Ͽ륭ꥹ
     */
    public void addTabKeyListener(KeyListener l) {
        synchronized (getTreeLock()) {
            tabKeyListener = AWTEventMulticaster.add(tabKeyListener, l);
            for (int i = 0; i < tabs.size(); i++) {
                ((Tab) tabs.elementAt(i)).addKeyListener(l);
            }
        }
    }

    /**
     * ֤Ф륭ꥹʤޤ
     *
     * @param  l 륭ꥹ
     */
    public void removeTabKeyListener(KeyListener l) {
        synchronized (getTreeLock()) {
            tabKeyListener = AWTEventMulticaster.remove(tabKeyListener, l);
            for (int i = 0; i < tabs.size(); i++) {
                ((Tab) tabs.elementAt(i)).removeKeyListener(l);
            }
        }
    }

    /**
     * ֤ФޥꥹʤϿޤ
     *
     * @param  l Ͽޥꥹ
     */
    public void addTabMouseListener(MouseListener l) {
        synchronized (getTreeLock()) {
            tabMouseListener = AWTEventMulticaster.add(tabMouseListener, l);
            for (int i = 0; i < tabs.size(); i++) {
                ((Tab) tabs.elementAt(i)).addMouseListener(l);
            }
        }
    }

    /**
     * ֤Фޥꥹʤޤ
     *
     * @param  l ޥꥹ
     */
    public void removeTabMouseListener(MouseListener l) {
        synchronized (getTreeLock()) {
            tabMouseListener = AWTEventMulticaster.remove(tabMouseListener, l);
            for (int i = 0; i < tabs.size(); i++) {
                ((Tab) tabs.elementAt(i)).removeMouseListener(l);
            }
        }
    }

    /**
     * ֤Фޥ⡼ꥹʤϿޤ
     *
     * @param  l Ͽޥ⡼ꥹ
     */
    public void addTabMouseMotionListener(MouseMotionListener l) {
        synchronized (getTreeLock()) {
            tabMouseMotionListener = AWTEventMulticaster.add(tabMouseMotionListener, l);
            for (int i = 0; i < tabs.size(); i++) {
                ((Tab) tabs.elementAt(i)).addMouseMotionListener(l);
            }
        }
    }

    /**
     * ֤Фޥ⡼ꥹʤޤ
     *
     * @param  l ޥ⡼ꥹ
     */
    public void removeTabMouseMotionListener(MouseMotionListener l) {
        synchronized (getTreeLock()) {
            tabMouseMotionListener = AWTEventMulticaster.remove(tabMouseMotionListener, l);
            for (int i = 0; i < tabs.size(); i++) {
                ((Tab) tabs.elementAt(i)).removeMouseMotionListener(l);
            }
        }
    }

//### Override
    /**
     * ե뤳Ȥ뤫ɤ֤ޤ
     *
     * @return ե <code>true</code>
     *         ʤ <code>false</code>
     */
    public boolean isFocusTraversable() {
        return true;
    }

    /**
     * ֤֤ޤ
     */
    public void doLayout() {
        // ֥Сɽ뤫ɤꤹ
        boolean visible = (tabs.size() > 1
                        && getSize().width < tabbar.getPreferredSize().width);

        if (tabScrollbar.isVisible() != visible) {
            tabScrollbar.setVisible(visible);
        }

        // 
        super.doLayout();
    }

    /**
     * ֥ѥͥκԤޤ
     *
     * @param  g Graphics ɥ
     */
    public void update(Graphics g) {
        paint(g);
    }

    /**
     * ֥ѥͥԤޤ
     *
     * @param  g Graphics ɥ
     */
    public void paint(Graphics g) {
        Dimension size = getSize();

        size.height = tabbar.getSize().height;

        // طʿɤ
        g.setColor(getBackground());
        g.fillRect(0, 0, size.width, size.height);

        // 
        g.setColor(activeTitleBackground);
        g.drawLine(0, size.height - 1, size.width - 1, size.height - 1);
        g.setColor(unactiveTitleBackground.brighter());
        g.drawLine(0, size.height - 2, size.width - 1, size.height - 2);

        // ݡͥȤ
        paintComponents(g);
    }

    /**
     * Υ֤Υѥ᡼ʸ֤ޤ
     *
     * @return ѥ᡼ʸ
     */
    protected String paramString() {
        String str = super.paramString()
                   + ",tabcount=" + tabs.size();
        return str;
    }

    /**
     * ΥݡͥȤȯ륳ݡͥȥ٥Ȥޤ
     *
     * @param  e ٥
     */
    protected void processEvent(AWTEvent e) {
        if (e instanceof TabEvent) {
            processTabEvent((TabEvent) e);
            return;
        }
        super.processEvent(e);
    }

    /**
     * ΥݡͥȤȯ륿֥٥Ȥ
     * ϿƤ뤹٤Ƥ {@link TabListener} 뤳Ȥˤꡢ
     * ֥٥Ȥޤ
     *
     * @param  e ٥
     */
    protected void processTabEvent(TabEvent e) {
        if (tabListener == null) {
            return;
        }

        switch (e.getID()) {
        case TabEvent.ACTIVE_CHANGED: tabListener.activeChanged(e); break;
        case TabEvent.TAB_ADDED     : tabListener.tabAdded     (e); break;
        case TabEvent.TAB_REMOVED   : tabListener.tabRemoved   (e); break;
        case TabEvent.TAB_CHANGED   : tabListener.tabChanged   (e); break;
        default: // AVOID
        }
    }

//### private
    /** ƥ֤ lock */
    private void setActiveTab(Tab tab) {
        if (tab == activeTab) {
            return;
        }

        if (activeTab != null) {
            activeTab.setActive(false);
        }

        tab.setActive(true);

        activeTab = tab;
    }

    /** ٤ƤΥ֤礭̵ˤ lock */
    private void invalidateAllTabs() {
        for (int i = tabs.size() - 1; i >= 0; i--) {
            ((Tab) tabs.elementAt(i)).invalidate();
        }
        tabbar.invalidate();
    }

    /** ٥Ȥȯ */
    private void postEvent(Tab tab, int id) {
        getToolkit().getSystemEventQueue().postEvent(new TabEvent(this, tab, id));
    }

//### MainPanel
    /** ᥤѥͥ */
    private final class MainPanel extends Container {
        private CardLayout layout;

        /** 󥹥󥹤 */
        private MainPanel() {
            setLayout(layout = new CardLayout());
        }
    }

//### Tabbar
    /** ֥С */
    private final class Tabbar extends Container {
        private AlignLayout layout;         // 쥤ȥޥ͡
        private int         height;         // ⤵

        // ʥå
        private Dimension minimumSize;
        private Dimension maximumSize;
        private Dimension preferredSize;

        /** 󥹥󥹤 */
        private Tabbar() {
            setLayout(layout = new AlignLayout(AlignLayout.HORIZONTAL));
        }

        /** ǾΥ */
        public Dimension getMinimumSize() {
            synchronized (getTreeLock()) {
                if (minimumSize == null || !isValid()) {
                    calculateHeight();
                    minimumSize = adjustHeight(super.getMinimumSize());
                }
                return minimumSize;
            }
        }

        /** 祵 */
        public Dimension getMaximumSize() {
            synchronized (getTreeLock()) {
                if (maximumSize == null || !isValid()) {
                    calculateHeight();
                    maximumSize = adjustHeight(super.getMaximumSize());
                }
                return maximumSize;
            }
        }

        /** 侩 */
        public Dimension getPreferredSize() {
            synchronized (getTreeLock()) {
                if (preferredSize == null || !isValid()) {
                    calculateHeight();
                    preferredSize = adjustHeight(super.getPreferredSize());
                }
                return preferredSize;
            }
        }

        /** ̵ˤ */
        public void invalidate() {
            synchronized (getTreeLock()) {
                minimumSize   = null;
                maximumSize   = null;
                preferredSize = null;
            }
            super.invalidate();
        }

        /** ֤ */
        public void doLayout() {
            super.doLayout();

            // ֥Сξ֤ڤؤ
            AlignLayout.Status status = layout.getStatus();
            tabScrollbar.setValues(status.getTop(), status.getLast(), status.getTotal());
        }

        /**  */
        public void update(Graphics g) {
            paint(g);
        }

        /** ⤵ξĴ lock */
        private Dimension adjustHeight(Dimension size) {
            size.height = Math.max(size.height, height);
            return size;
        }

        /** ֥Сι⤵Ʒ׻ lock */
        private void calculateHeight() {
            // ⤵Ʒ׻
            int h = Math.max((  activeTitleFont != null ? getFontMetrics(  activeTitleFont).getHeight() : 0),
                             (unactiveTitleFont != null ? getFontMetrics(unactiveTitleFont).getHeight() : 0));

            // βι⤵׻
            for (int i = tabs.size() - 1; i >= 0; i--) {
                h = Math.max(h, ((Tab) tabs.elementAt(i)).imageContainer.height);
            }

            h += TABBAR_HEIGHT_FIX * 2;

            height = h;
        }
    }

//### Tab
    /**
     * ֤ɽݡͥȤǤ
     */
    public final class Tab extends Component {
        /** @serial  */
        private String          key;              // ᥤѥͥѤΥ
        /** @serial ݡͥ */
        private Component       component;
        /** @serial ȥ */
        private String          title;            // ȥ
        /** @serial ȥ char */
        private char[]          titleChars;
        /** @serial  */
        private ImageContainer  imageContainer;   // 
        /** @serial  */
        private int             width;            // 
        /** @serial  */
        private WrappedObserver observer;

        // ʥå
        /** @serial Ǿ */
        private Dimension minimumSize;
        /** @serial 祵 */
        private Dimension maximumSize;
        /** @serial 侩 */
        private Dimension preferredSize;

        /** 󥹥󥹤 */
        private Tab(String title, Image image, Component component) {
            this.key       = Integer.toHexString(hashCode());
            this.component = component;
            setLabel(title, image);

            addMouseListener(
                new MouseAdapter() {
                    /** å줿 */
                    public void mouseClicked(MouseEvent e) {
                        if (((e.getModifiers() & InputEvent.BUTTON1_MASK) == InputEvent.BUTTON1_MASK)
                                && e.getClickCount() == 1) {
                            setActiveTab(Tab.this);
                            TabPanel.this.requestFocus();
                        }
                    }
                }
            );

            if (tabComponentListener   != null) { addComponentListener  (tabComponentListener  ); }
            if (tabFocusListener       != null) { addFocusListener      (tabFocusListener      ); }
            if (tabKeyListener         != null) { addKeyListener        (tabKeyListener        ); }
            if (tabMouseListener       != null) { addMouseListener      (tabMouseListener      ); }
            if (tabMouseMotionListener != null) { addMouseMotionListener(tabMouseMotionListener); }
        }

//### original
        /**
         * ֤Υ٥ɽ륿ȥʸꤷޤ
         *
         * @param  title ȥ
         */
        public void setLabel(String title) {
            synchronized (getTreeLock()) {
                setLabel(title,
                         (imageContainer != null ? imageContainer.image : null));
            }
        }

        /**
         * ֤Υ٥ɽꤷޤ
         *
         * @param  image 
         */
        public void setLabel(Image image) {
            setLabel(title, image);
        }

        /**
         * ֤Υ٥ɽ륿ȥʸȲꤷޤ
         *
         * @param  title ȥ
         * @param  image 
         */
        public void setLabel(String title, Image image) {
            boolean sizeChanged = false;
            synchronized (getTreeLock()) {
                boolean dataChanged = false;

                if (this.title != title
                        && (this.title == null
                         || title == null
                         || this.title.compareTo(title) != 0)) {
                    dataChanged = true;
                    sizeChanged = true;     // Ȥ㤦ϥ㤦Ƚ

                    this.title      = title;
                    this.titleChars = (title != null
                                       ? title.toCharArray()
                                       : null);
                }

                ImageContainer newIc = ImageContainer.setupImageContainer(imageContainer, image);
                if (imageContainer != newIc) {
                    dataChanged = true;
                    sizeChanged = sizeChanged || !imageContainer.equalsSize(newIc);

                    // ξ򥯥ꥢ
                    if (imageContainer != null) {
                        releaseObserver();
                    }

                    // ¸
                    imageContainer = newIc;
                    if (newIc != null) {
                        observer = new WrappedObserver(this);
                    }
                }

                // ѹ̵ä
                if (!dataChanged) {
                    return;
                }

                // ѹ줿
                if (sizeChanged) {
                    invalidate();
                    tabbar.invalidate();
                    TabPanel.this.validate();
                }

            }

            if (isShowing() && !sizeChanged) {
                repaint();
            }
        }

        /**
         * Υ֤򥢥ƥ֤ˤޤ
         */
        public void requestActive() {
            synchronized (getTreeLock()) {
                setActiveTab(this);
            }
        }

//### Override
        /**
         * ݡͥȤƥʤɲä줿ΤΤޤ
         */
        public void addNotify() {
            synchronized (getTreeLock()) {
                super.addNotify();

                // ǥեȤΥꤷƤ
                component.setSize(main.getSize());

                // ᥤ󥳥ݡͥȤɲ
                main.add(component, key);
            }
        }

        /**
         * ݡͥȤƥʤ줿ΤΤޤ
         */
        public void removeNotify() {
            synchronized (getTreeLock()) {
                super.removeNotify();

                // ᥤ󥳥ݡͥȤ
                main.remove(component);

                int index = tabs.indexOf(this);

                tabs.removeElementAt(index);

                if (activeTab == this) {
                    activeTab = null;

                    if (tabs.size() > 0) {
                        if (index > 0) {
                            index--;
                        }

                        setActiveTab((Tab) tabs.elementAt(index));
                    }
                }
            }
        }

        /**
         * Ǿ֤ޤ
         *
         * @return Ǿ
         */
        public Dimension getMinimumSize() {
            synchronized (getTreeLock()) {
                if (minimumSize == null || !isValid()) {
                    calculateWidth();
                    minimumSize = new Dimension(minWidth, tabbar.height);
                }
                return minimumSize;
            }
        }

        /**
         * 祵֤ޤ
         *
         * @return 祵
         */
        public Dimension getMaximumSize() {
            synchronized (getTreeLock()) {
                if (maximumSize == null || !isValid()) {
                    calculateWidth();
                    maximumSize = getPreferredSize();
                }
                return maximumSize;
            }
        }

        /**
         * 侩֤ޤ
         *
         * @return 侩
         */
        public Dimension getPreferredSize() {
            synchronized (getTreeLock()) {
                if (preferredSize == null || !isValid()) {
                    calculateWidth();
                    preferredSize = new Dimension(Math.max(Math.min(width, maxWidth),
                                                           minWidth),
                                                  tabbar.height);
                }
                return preferredSize;
            }
        }

        /**
         * ʤɤ̵ˤơƥ쥤Ȥ褦ˤޤ
         */
        public void invalidate() {
            synchronized (getTreeLock()) {
                minimumSize   = null;
                maximumSize   = null;
                preferredSize = null;
            }
            super.invalidate();
        }

        /**
         * ֤κԤޤ
         *
         * @param  g եå
         */
        public void update(Graphics g) {
            paint(g);
        }

        /**
         * ֤Ԥޤ
         *
         * @param  g եå
         */
        public void paint(Graphics g) {
            synchronized (getTreeLock()) {
                boolean   isActive = (activeTab == this);
                Dimension size = getSize();
                Color     bg;

                // طʿɤ
                g.setColor(TabPanel.this.getBackground());
                g.fillRect(0, 0, size.width, size.height);
                if (isActive) {
                    g.setColor(bg = activeTitleBackground);
                    g.fillRect(0, 1, size.width - 2, size.height - 2);
                } else {
                    g.setColor(bg = unactiveTitleBackground);
                    g.fillRect(1, 3, size.width - 3, size.height - 5);
                }

                int fix = (isActive ? 1 : 0);
                int x   = TABBAR_LEFT_FIX + fix;

                // ɽ
                if (imageContainer != null) {
                    g.drawImage(imageContainer.image,
                                x,
                                ((int) ((tabbar.height - imageContainer.height) / 2.0 + 0.5)) - fix,
                                observer);
                    x += imageContainer.width;
                    x += TABBAR_IMAGE_FIX;
                }

                // ʸɽ
                if (title != null) {
                    Font font;
                    if (isActive) {
                        font = activeTitleFont;
                        g.setColor(activeTitleForeground );
                    } else {
                        font = unactiveTitleFont;
                        g.setColor(unactiveTitleForeground );
                    }
                    if (font == null) {
                        font = getFont();
                    }
                    g.setFont(font);
                    FontData    fd = FontData.getInstance(this, font);
                    FontMetrics fm = fd.getFontMetrics();
                    int offset = (int) ((tabbar.height - fd.getFullSize().height) / 2.0 + 0.5);
                    int y1 = fd.getHalfBase() + offset - fix;
                    int y2 = fd.getFullBase() + offset - fix;

                    // ­ʤɤå
                    int     end = size.width - TABBAR_RIGHT_FIX;
                    boolean cut = false;
                    if (fm.stringWidth(title) + x > end) {
                        end -= fm.stringWidth(TITLE_CONTINUE);
                        cut = true;
                    }

                    char[] chars  = titleChars;
                    int    length = chars.length;
                    char   c;
                    int    w;
                    for (int i = 0; i < length; i++) {
                        c = chars[i];
                        w = fm.charWidth(c);
                        if (x + w > end) {
                            break;
                        }
                        g.drawChars(chars, i, 1, x, ((c <= 0xFF) || (0xFF61 <= c && c <= 0xFF9F) ? y1 : y2));
                        x += w;
                    }
                    if (cut) {
                        g.drawString(TITLE_CONTINUE, x, y1);
                    }
                }

                // Ȥɽ
                if (isActive) {
                    g.setColor(bg.brighter());
                    g.drawLine(1, 1, 1, 1);
                    g.drawLine(0, 2, 0, size.height - 1);
                    g.drawLine(2, 0, size.width - 3,  0);

                    g.setColor(bg.darker());
                    g.drawLine(size.width - 2, 2, size.width - 2, size.height - 2);

                    g.setColor(bg.darker().darker());
                    g.drawLine(size.width - 2, 1, size.width - 2, 1);
                    g.drawLine(size.width - 1, 2, size.width - 1, size.height - 2);
                } else {
                    int index = tabs.indexOf(this);
                    if (index + 1 < tabs.size()
                            && tabs.elementAt(index + 1) == activeTab
                            && activeTab.isVisible()) {
                        // ƥ֥֤κ¦ξ
                        g.setColor(bg.brighter());
                        g.drawLine(1, 3, 1, 3);
                        g.drawLine(2, 2, size.width - 1,  2);
                        g.drawLine(0, 4, 0, size.height - 1);
                        g.drawLine(0, size.height - 2, size.width - 1, size.height - 2);
                    } else if (index > 0
                            && tabs.elementAt(index - 1) == activeTab
                            && activeTab.isVisible()) {
                        // ƥ֥֤α¦ξ
                        g.setColor(bg.brighter());
                        g.drawLine(0, 2, size.width - 3,  2);
                        g.drawLine(0, size.height - 2, size.width - 1, size.height - 2);

                        g.setColor(bg.darker());
                        g.drawLine(size.width - 2, 4, size.width - 2, size.height - 3);

                        g.setColor(bg.darker().darker());
                        g.drawLine(size.width - 2, 3, size.width - 2, 3);
                        g.drawLine(size.width - 1, 4, size.width - 1, size.height - 3);
                    } else {
                        // ̤󥢥ƥ֤ξ
                        g.setColor(bg.brighter());
                        g.drawLine(1, 3, 1, 3);
                        g.drawLine(2, 2, size.width - 3,  2);
                        g.drawLine(0, 4, 0, size.height - 1);
                        g.drawLine(0, size.height - 2, size.width - 1, size.height - 2);

                        g.setColor(bg.darker());
                        g.drawLine(size.width - 2, 4, size.width - 2, size.height - 3);

                        g.setColor(bg.darker().darker());
                        g.drawLine(size.width - 2, 3, size.width - 2, 3);
                        g.drawLine(size.width - 1, 4, size.width - 1, size.height - 3);
                    }
                }

                // ǲɽ
                g.setColor(activeTitleBackground);
                g.drawLine(0, size.height - 1, size.width - 1, size.height - 1);

                // եȤɽ
                if (isActive && hasFocus) {
                    g.setColor(focusFrameColor);
                    GraphicsUtils.drawDashed(g, 3, 3, size.width  - 8, 1, 1, 1, GraphicsUtils.HORIZONTAL);
                    GraphicsUtils.drawDashed(g, 3, 3, 1, size.height - 5, 1, 1, GraphicsUtils.VERTICAL  );
                    GraphicsUtils.drawDashed(g, 3, size.height - 3, size.width - 8, 1, 1, 1, GraphicsUtils.HORIZONTAL);
                    GraphicsUtils.drawDashed(g, size.width - 5, 3, 1, size.height - 5, 1, 1, GraphicsUtils.VERTICAL  );
                }
            }
        }

        /** ƥ֤ѹ */
        private void setActive(boolean isActive) {
            if (isActive) {
                int index = tabs.indexOf(this);

                tabbar.layout.show(tabbar, index);
                main  .layout.show(main  , key  );

                TabPanel.this.postEvent(this, TabEvent.ACTIVE_CHANGED);
            }

            if (isShowing()) {
                if (isActive) {
                    TabPanel.this.validate();
                } else {
                    repaint();
                }
            }
        }

        /** ʬƷ׻ lock */
        private void calculateWidth() {
            int w = (activeTab == this ? 1 : 0);

            if (imageContainer != null) {
                w += imageContainer.width;
                w += TABBAR_IMAGE_FIX;
            }

            if (title != null) {
                Font   activeTitleFont = TabPanel.this.activeTitleFont;
                Font unactiveTitleFont = TabPanel.this.unactiveTitleFont;
                if (  activeTitleFont == null) {   activeTitleFont = getFont(); }
                if (unactiveTitleFont == null) { unactiveTitleFont = getFont(); }

                w += Math.max(getFontMetrics(  activeTitleFont).stringWidth(title),
                              getFontMetrics(unactiveTitleFont).stringWidth(title));
            }
            w += TABBAR_LEFT_FIX + TABBAR_RIGHT_FIX;

            width = w;
        }

        /**
         * ֤꥽ޤ
         * <p>
         * Υ֤ƥ֤ä硢Υ֤κ¦Υ֤
         * ƥ֤ˤʤޤ֤Ƭξϡ
         * ƬΥ֤ƥ֤ˤʤޤ
         */
        public void dispose() {
            tabbar.remove(this);
            releaseObserver();
            TabPanel.this.validate();
        }

        /** ֥Фγ */
        private void releaseObserver() {
            if (observer != null) {
                observer.dispose();
                observer = null;
            }
        }
    }

//### TabScrollbar
    /** ֥С */
    private final class TabScrollbar extends Container {
        private ImageButton defaultLeft ;
        private ImageButton defaultRight;

        private ImageButton left;
        private ImageButton right;

        /** 󥹥󥹤 */
        private TabScrollbar() {
            setLayout(new FlowLayout(FlowLayout.CENTER, 0, 0));
            setVisible(false);
        }

        /**  */
        public void update(Graphics g) {
            this.paint(g);
        }

        /** ܥ */
        private void setButton(ImageButton left, ImageButton right) {
            synchronized (getTreeLock()) {
                if (this.left  != null && left  != null) { left .setEnabled(this.left .isEnabled()); }
                if (this.right != null && right != null) { right.setEnabled(this.right.isEnabled()); }

                this.left  = left;
                this.right = right;

                Color bg = getBackground();
                left .setBackground(bg);
                right.setBackground(bg);

                left .addMouseListener(
                    new MouseAdapter() {
                        private boolean pressed;

                        /** ޥ줿 */
                        public void mousePressed(MouseEvent e) {
                            if (check(e)) {
                                pressed = true;
                            }
                        }

                        /** ޥ줿 */
                        public void mouseReleased(MouseEvent e) {
                            if (check(e)) {
                                tabbar.layout.previous(tabbar);
                            }
                            pressed = false;
                        }
                    }
                );
                right.addMouseListener(
                    new MouseAdapter() {
                        private boolean pressed;

                        /** ޥ줿 */
                        public void mousePressed(MouseEvent e) {
                            if (check(e)) {
                                pressed = true;
                            }
                        }

                        /** ޥ줿 */
                        public void mouseReleased(MouseEvent e) {
                            if (check(e)) {
                                tabbar.layout.next(tabbar);
                            }
                            pressed = false;
                        }
                    }
                );

                removeAll();
                add(left );
                add(right);
            }
        }

        /** ֤ */
        private void setValues(int top, int last, int total) {
            synchronized (getTreeLock()) {
                if (left  != null) {
                    left .setEnabled((top > 0));
                }
                if (right != null) {
                    right.setEnabled((last + 1 < total));
                }
            }
        }

        /** Ǿ֤ */
        public Dimension getMinimumSize() {
            return getPreferredSize();
        }

        /** 祵֤ */
        public Dimension getMaximumSize() {
            return getPreferredSize();
        }

        /** 侩֤ */
        public Dimension getPreferredSize() {
            Dimension leftSize  = null, rightSize = null;

            if (left  != null) {
                leftSize  = left .getPreferredSize();
            }
            if (right != null) {
                rightSize = right.getPreferredSize();
            }

            return new Dimension((leftSize  != null ? leftSize .width : 0)
                               + (rightSize != null ? rightSize.width : 0),
                                 Math.max((leftSize  != null ? leftSize .height : 0),
                                          (rightSize != null ? rightSize.height : 0)));
        }

        /** ե뤳Ȥ뤫֤ */
        public boolean isFocusTraversable() {
            return false;
        }

        /** طʿ */
        public void setBackground(Color c) {
            super.setBackground(c);

            if (left  != null) {
                left .setBackground(c);
            }
            if (right != null) {
                right.setBackground(c);
            }
        }

        /** ʬɲä줿 */
        public void addNotify() {
            super.addNotify();

            if (left == null || right == null) {
                createDefaultButton();
            }
        }

        /** ǥեȥܥ */
        private void createDefaultButton() {
            if (defaultLeft == null) {
                ImageButton left  = defaultLeft  = new ImageButton(null);
                ImageButton right = defaultRight = new ImageButton(null);

                MemoryImageSource mis;
                int[] bitmap;
                int fg = getForeground().getRGB() | 0xff000000;
                //int bg = getBackground().getRGB() | 0xff000000;
                int bg = 0;

                int w = 13;
                int h = 13;

                // left
                bitmap = new int[w * h];
                for (int i = 0; i < bitmap.length; i++) {
                    if (LEFT_IMAGE[i] == 1) {
                        bitmap[i] = fg;
                    } else {
                        bitmap[i] = bg;
                    }
                }

                mis = new MemoryImageSource(w, h, bitmap, 0, w);
                mis.setAnimated(false);
                left.setImage(left.createImage(mis));

                // right
                bitmap = new int[w * h];
                for (int i = 0; i < bitmap.length; i++) {
                    if (RIGHT_IMAGE[i] == 1) {
                        bitmap[i] = fg;
                    } else {
                        bitmap[i] = bg;
                    }
                }

                mis = new MemoryImageSource(w, h, bitmap, 0, w);
                mis.setAnimated(false);
                right.setImage(right.createImage(mis));
            }

            setButton(defaultLeft, defaultRight);
        }

        /** ܥ夫å */
        private boolean check(MouseEvent e) {
            Component c = e.getComponent();

            if (!c.isEnabled()) {
                return false;
            }
            if ((e.getModifiers() & InputEvent.BUTTON1_MASK) == 0) {
                return false;
            }
            int x = e.getX();
            int y = e.getY();
            Dimension size = c.getSize();
            return (0 <= x && x < size.width && 0 <= y && y <= size.height);
        }
    }
}
