/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.awt;

import net.hizlab.kagetaka.Resource;
import net.hizlab.kagetaka.awt.image.OffscreenImage;
import net.hizlab.kagetaka.awt.image.SyncObserver;
import net.hizlab.kagetaka.util.Environment;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.MenuItem;
import java.awt.PopupMenu;
import java.awt.SystemColor;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.StringSelection;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.InputEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.util.Enumeration;
import java.util.Vector;

/**
 * ơɽ뤿ΥơСݡͥȤǤ
 * ̤ϡWindow κǲˤꡢץफûå䡢
 * ʤɤɽ뤿ѤޤΤᡢΥݡͥȤ
 * 岼ϸˤʤäƤޤWindow ǲ֤뤿ˡ
 * {@link java.awt.BorderLayout} ѤǤ
 * <pre>
 * StatusBar s = new StatusBar();
 * setLayout(new BorderLayout());
 * add(s, "South");
 * s.setText("Hello");
 * </pre>
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.6 $
 */
public class StatusBar extends Component {
    private static final String RESOURCE = "net.hizlab.kagetaka.awt.Resources";
    private static final int IMAGE_SIZE = 16;
    private static final int MARGIN_H   = 2;
    private static final int MARGIN_W   = 3;
    private static final int MARGIN_TOP = 2;
    private static final int MIN_HEIGHT = IMAGE_SIZE + MARGIN_TOP + 4;

    private static final int MENU_COPY = 0;

    /** @serial ե꡼ */
    private OffscreenImage offscreenImage;
    /** @serial 侩 */
    private Dimension      preferredSize;

    /** @serial ơɽʸ */
    private String          statusText;
    /** @serial ơɽ륢 */
    private StatusBarIcon[] statusIcons;
    /** @serial ݥåץåץ˥塼 */
    private PopupMenu       popupMenu;
    /** @serial ᡼ꥨ */

    /** @serial ƥѥꥹ */
    private Vector listeners = new Vector();

    /**
     * ơСޤ
     * Dialog եȤѤƥ५顼ǽޤ
     */
    public StatusBar() {
        setForeground(SystemColor.controlText);
        setBackground(SystemColor.control    );

        // ˥塼Ѱ
        popupMenu = new PopupMenu(Resource.getMessage(RESOURCE, "statusbar.menu", null));
        MenuItem mi = new MenuItem(Resource.getMessage(RESOURCE, "statusbar.menu.copy", null));
        mi.addActionListener(new MenuActionListener(MENU_COPY));
        popupMenu.add(mi);
        add(popupMenu);
    }

//### Original
    /**
     * ơСɽʸꤷޤ
     *
     * @param  value ơСɽʸ
     */
    public void setText(String value) {
        String old = statusText;
        if (old == value
                || (old   != null
                 && value != null
                 && old.compareTo(value) == 0)) {
            return;
        }

        this.statusText = value;
        repaintForce();
    }

    /**
     * ơСɽ륢ꤷޤ
     *
     * @param  icons ơСɽ륢
     */
    public void setIcons(StatusBarIcon[] icons) {
        StatusBarIcon[] old = statusIcons;
        if (old == icons) {
            return;
        }

        // Ʊå
        if (old != null && icons != null
                && old.length == icons.length) {
            boolean same = true;
            for (int i = icons.length - 1; i >= 0; i--) {
                if (old[i] != icons[i]) {
                    same = false;
                    break;
                }
            }
            if (same) {
                return;
            }
        }

        this.statusIcons = icons;
        repaintForce();
    }

//### Override
    /**
     * ɽѤեȤꤷޤ
     *
     * @param  font ե
     */
    public void setFont(Font font) {
        Font oldFont = getFont();
        if (oldFont != font) {
            super.setFont(font);

            invalidate();
            repaintForce();
        }
    }

    /**
     * ơСκǾΥ֤ޤ
     * ⤵Τ¤Ƥޤ
     *
     * @return Ǿ
     */
    public Dimension getMinimumSize() {
        return getPreferredSize();
    }

    /**
     * ơСκ祵֤ޤ
     * ⤵Τ¤Ƥޤ
     *
     * @return 祵
     */
    public Dimension getMaximumSize() {
        return getPreferredSize();
    }

    /**
     * ơСο侩֤ޤ
     * ⤵Τ߻ꤵƤޤ
     *
     * @return 侩
     */
    public Dimension getPreferredSize() {
        Dimension preferredSize = this.preferredSize;
        if (preferredSize != null && isValid()) {
            return preferredSize;
        }

        synchronized (getTreeLock()) {
            FontMetrics fm = getFontMetrics(getFont());
            this.preferredSize = new Dimension(
                0,
                Math.max(fm.getHeight() + 5, MIN_HEIGHT)
            );
            return this.preferredSize;
        }
    }

    /**
     * ݡͥȤ̵ˤޤ
     */
    public void invalidate() {
        preferredSize = null;
        super.invalidate();
    }

    /**
     * ơСκԤޤ
     *
     * @param  g Graphics ɥ
     */
    public void update(Graphics g) {
        paint(g);
    }

    /**
     * ơСԤޤ
     *
     * @param  g Graphics ɥ
     */
    public void paint(Graphics g) {
        offscreenImage.paint(g);
    }

    /**
     * ݡͥȤƥʤɲä줿ȤΤޤ
     */
    public void addNotify() {
        synchronized (getTreeLock()) {
            super.addNotify();

            // ե꡼
            offscreenImage = new OffscreenImage(this) {
                public void update(Image offscreen, Graphics g, Dimension size) {
                    refresh(offscreen, g, size);
                }
            };
        }
    }

    /**
     * ݡͥȤƥʤ줿ȤΤޤ
     */
    public void removeNotify() {
        synchronized (getTreeLock()) {
            super.removeNotify();

            // ե꡼
            if (offscreenImage != null) {
                offscreenImage.dispose();
                offscreenImage = null;
            }
        }
    }

    /**
     * ΥơСΥѥ᡼ʸ֤ޤ
     *
     * @return ѥ᡼ʸ
     */
    protected String paramString() {
        String str = super.paramString();

        String status = this.statusText;
        if (status != null) {
            str += ",message=" + statusText;
        }

        return str;
    }

    /** ե꡼ */
    private void refresh(Image offscreen, Graphics g, Dimension size) {
        // طʿɤ
        Color bg = getBackground();
        g.setColor(bg);
        g.fillRect(0, 0, size.width, size.height);

        int x = 0;
        int y = MARGIN_TOP;
        MouseListener listener;

        // ꥹʤ
        if (listeners.size() > 0) {
            for (Enumeration e = listeners.elements(); e.hasMoreElements();) {
                removeMouseListener((MouseListener) e.nextElement());
            }
            listeners.removeAllElements();
        }

        // ɽ
        StatusBarIcon[] icons = statusIcons;
        if (icons != null) {
            Image image;
            for (int i = 0; i < icons.length; i++) {
                if ((image = icons[i].getIcon()) != null) {
                    SyncObserver.drawImage(g, image, x + 2, y + 2, IMAGE_SIZE, IMAGE_SIZE, 10000);

                    // ꥹʤϿ
                    if ((listener = icons[i].getMouseListener()) != null) {
                        listener = new ItemMouseListener(listener,
                                                         x,
                                                         y,
                                                         x + 4 + IMAGE_SIZE,
                                                         size.height);
                        listeners.addElement(listener);
                        addMouseListener(listener);
                    }

                    // Ȥɽ
                    g.setColor(SystemColor.controlShadow);
                    g.drawLine(x, y, x + IMAGE_SIZE + 2, y);
                    g.drawLine(x, y, x, size.height - 1   );
                    g.setColor(SystemColor.controlLtHighlight);
                    g.drawLine(x, size.height - 1   , x + IMAGE_SIZE + 3, size.height - 1);
                    g.drawLine(x + IMAGE_SIZE + 3, y, x + IMAGE_SIZE + 3, size.height - 1);

                    x += IMAGE_SIZE + 4 + MARGIN_H; //  + 
                }
            }
        }

        // ʸɽ
        String text = this.statusText;
        if (text != null) {
            Font font = getFont();
            char[] chars = statusText.toCharArray();
            g.setFont (font);
            g.setColor(getForeground());

            FontData    fontData = FontData.getInstance(this, font);
            FontMetrics fm       = fontData.getFontMetrics();

            int p   = x + MARGIN_W;
            int fix = (size.height - fontData.getMaxHeight()) / 2 + 1;
            int y1  = fontData.getHalfBase() + fix;
            int y2  = fontData.getFullBase() + fix;
            char c;
            for (int i = 0; i < chars.length; i++) {
                c = chars[i];
                g.drawChars(chars, i, 1, p, ((c <= 0xFF) || (0xFF61 <= c && c <= 0xFF9F) ? y1 : y2));
                p += fm.charWidth(chars[i]);
            }

            // ꥹʤϿ
            listener = new TextMouseListener(x, y, size.width, size.height);
            listeners.addElement(listener);
            addMouseListener(listener);
        }

        // Ȥɽ
        g.setColor(bg.darker());
        g.drawLine(x, y, size.width,  y);
        g.drawLine(x, y, x, size.height);
        g.setColor(bg.brighter());
        g.drawLine(x, size.height - 1, size.width - 1, size.height - 1);
        g.drawLine(size.width - 1,  y, size.width - 1, size.height - 1);

        // ξɽ
        if (Environment.isWindows) {
            g.setColor(bg);
            g.drawLine(size.width -  2, size.height - 2, size.width - 2, size.height -  2);
            g.drawLine(size.width -  6, size.height - 2, size.width - 2, size.height -  6);
            g.drawLine(size.width - 10, size.height - 2, size.width - 2, size.height - 10);

            g.setColor(bg.brighter());
            g.drawLine(size.width -  5, size.height - 2, size.width - 2, size.height -  5);
            g.drawLine(size.width -  9, size.height - 2, size.width - 2, size.height -  9);
            g.drawLine(size.width - 13, size.height - 2, size.width - 2, size.height - 13);

            g.setColor(bg.darker());
            g.drawLine(size.width -  3, size.height - 2, size.width - 2, size.height -  3);
            g.drawLine(size.width -  4, size.height - 2, size.width - 2, size.height -  4);
            g.drawLine(size.width -  7, size.height - 2, size.width - 2, size.height -  7);
            g.drawLine(size.width -  8, size.height - 2, size.width - 2, size.height -  8);
            g.drawLine(size.width - 11, size.height - 2, size.width - 2, size.height - 11);
            g.drawLine(size.width - 12, size.height - 2, size.width - 2, size.height - 12);
        }
    }

    /** Ū˺ɽ */
    private void repaintForce() {
        OffscreenImage oi = offscreenImage;
        if (oi != null) {
            oi.repaint();
        }
    }

    /** åץܡɤ˥ԡ */
    private void copyClipboard(String value) {
        Clipboard       clipboard = getToolkit().getSystemClipboard();
        StringSelection data      = new StringSelection(value);
        clipboard.setContents(data, data);
    }

//### MenuActionListener
    /** ˥塼ѤΥꥹ */
    private final class MenuActionListener implements ActionListener {
        private int index = 0;

        /** ˥塼ꥹ  */
        private MenuActionListener(int index) {
            this.index = index;
        }

        /** ϥɥ */
        public void actionPerformed(ActionEvent e) {
            switch (index) {
            case MENU_COPY: copyClipboard(statusText); break;
            default: // AVOID
            }
        }
    }

//### AbstractMouseListener
    /** ѥޥꥹ */
    private abstract class AbstractMouseListener extends MouseAdapter {
        private MouseListener listener;
        private int x1;
        private int y1;
        private int x2;
        private int y2;

        /** ƥѥޥꥹ  */
        private AbstractMouseListener(MouseListener listener,
                                      int x1, int y1, int x2, int y2) {
            this.listener = listener;
            this.x1 = x1;
            this.y1 = y1;
            this.x2 = x2;
            this.y2 = y2;
        }

        /** ޥå줿 */
        public void mouseClicked(MouseEvent e) {
            int x = e.getX();
            int y = e.getY();
            if (x1 <= x && x < x2 && y1 <= y && y < y2) {
                mouseClickedOnItem(e);
            }
        }

        /** ƥ¦ޥǥå줿 */
        public void mouseClickedOnItem(MouseEvent e) {
            listener.mouseClicked(new MouseEvent(e.getComponent(),
                                                 e.getID(),
                                                 e.getWhen(),
                                                 e.getModifiers(),
                                                 e.getX() - x1,
                                                 e.getY() - y1,
                                                 e.getClickCount(),
                                                 e.isPopupTrigger()));
        }
    }

//### ItemMouseListener
    /** ƥѥޥꥹ */
    private final class ItemMouseListener extends AbstractMouseListener {
        /** ƥѥޥꥹʤ */
        private ItemMouseListener(MouseListener listener,
                                  int x1, int y1, int x2, int y2) {
            super(listener, x1, y1, x2, y2);
        }
    }

//### TextMouseListener
    /** ƥѥޥꥹ */
    private final class TextMouseListener extends AbstractMouseListener {
        /** ƥѥޥꥹ  */
        private TextMouseListener(int x1, int y1, int x2, int y2) {
            super(null, x1, y1, x2, y2);
        }

        /** ƥ¦ޥǥå줿 */
        public void mouseClickedOnItem(MouseEvent e) {
            if (e.isPopupTrigger()
                    || ((e.getModifiers() & InputEvent.BUTTON3_MASK) == InputEvent.BUTTON3_MASK)) {

                popupMenu.show(e.getComponent(), e.getX(), e.getY());
            }
        }
    }
}
