/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.awt;

import net.hizlab.kagetaka.Debug;
import net.hizlab.kagetaka.awt.image.GrayFilter;
import net.hizlab.kagetaka.awt.image.SyncObserver;

import java.awt.Color;
import java.awt.Component;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.SystemColor;
import java.awt.event.InputEvent;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.awt.image.MemoryImageSource;
import java.util.EventListener;

/**
 * ݡͥ褹륹СǤ
 *
 * @kagetaka.bugs ľ󲽤ϡꥹʤ¸ʤޤ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.2 $
 */
public abstract class InnerScrollbar {
    private static final int NONE   = -1;
    private static final int ARROW1 =  1;
    private static final int ARROW2 =  2;
    private static final int SPACE1 =  3;
    private static final int SPACE2 =  4;
    private static final int HANDLE =  5;

    /** ʿ */
    public static final int HORIZONTAL = java.awt.Scrollbar.HORIZONTAL;
    /** ľ */
    public static final int VERTICAL   = java.awt.Scrollbar.VERTICAL;

    private static final int ARROW_SIZE = 15;

    private static final int[] ARROW_LEFT =
    {
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0,
        0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0,
        0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0,
        0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    };

    private static final int[] ARROW_RIGHT =
    {
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0,
        0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0,
        0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0,
        0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    };

    private static final int[] ARROW_TOP =
    {
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0,
        0, 0, 0, 1, 1, 1, 1, 1, 0, 0, 0,
        0, 0, 1, 1, 1, 1, 1, 1, 1, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    };

    private static final int[] ARROW_BOTTOM =
    {
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 1, 1, 1, 1, 1, 1, 1, 0, 0,
        0, 0, 0, 1, 1, 1, 1, 1, 0, 0, 0,
        0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    };

    /** @serial ꥹʡ */
    private Listener  listener;
    /** @serial  */
    private int       orientation;
    /** @serial ġ륭å */
    private Component component;
    /** @serial  */
    private boolean   isHorizontal;
    /** @serial ͭɤ */
    private boolean   isEnabled = true;
    /** @serial ɽƤ뤫ɤ */
    private boolean   isVisible = true;
    /** @serial  */
    private Color     sliderColor;

    /** @serial å֥ */
    private Object lock = new Object();

    /**  @serial  */
    private int value          = 0;
    /** ɽ @serial ɽ */
    private int visible        = 1;
    /** Ǿ @serial Ǿ */
    private int minimum        = 0;
    /**  @serial  */
    private int maximum        = 1;
    /** ˥åȥ󥯥 @serial ˥åȥ󥯥 */
    private int unitIncrement  = 1;
    /** ֥å󥯥 @serial ֥å󥯥 */
    private int blockIncrement = 1;

    /** @serial ޥ줿 */
    private int   pressedArea   = NONE;
    /** @serial ϥɥ򲡤ȤΥեå */
    private Point pressedHandle = new Point(0, 0);

    /** @serial ȥ */
    private AutoScroll autoScroll;

    /** @serial ͤѹƤ뤫 */
    private boolean rectChanged;

    private Rectangle rectangle  = new Rectangle(0, 0, 0, 0);
    private Rectangle rectArrow1 = new Rectangle(0, 0, 0, 0);
    private Rectangle rectArrow2 = new Rectangle(0, 0, 0, 0);
    private Rectangle rectSlider = new Rectangle(0, 0, 0, 0);
    private Rectangle rectHandle = new Rectangle(0, 0, 0, 0);

    /**
     * 󥹥󥹤ޤ
     *
     * @param  orientation Сθ
     * @param  component   С褹륳ݡͥ
     */
    public InnerScrollbar(int orientation, Component component) {
        this.orientation  = orientation;
        this.component    = component;
        this.isHorizontal = (orientation == HORIZONTAL);
        calculateColor();
    }

    /**
     * Ȱ֤ꤷޤ
     *
     * @param  x X
     * @param  y Y
     * @param  width  
     * @param  height ⤵
     */
    public void setBounds(int x, int y, int width, int height) {
        synchronized (lock) {
            rectangle .setBounds(x, y, width, height);
            if (isHorizontal) {
                rectArrow1.setBounds(x, y, ARROW_SIZE, height);
                rectArrow2.setBounds(x + width  - ARROW_SIZE, y, ARROW_SIZE, height);
                rectSlider.setBounds(x + ARROW_SIZE, y, width - ARROW_SIZE * 2, height);
            } else {
                rectArrow1.setBounds(x, y, width , ARROW_SIZE);
                rectArrow2.setBounds(x, y + height - ARROW_SIZE, width , ARROW_SIZE);
                rectSlider.setBounds(x, y + ARROW_SIZE, width, height - ARROW_SIZE * 2);
            }
            rectChanged = true;
        }
    }

    /**
     * Ȱ֤֤ޤ
     *
     * @return Ȱ
     */
    public Rectangle getBounds() {
        return new Rectangle(rectangle);
    }

    /**
     * ߤ֤ͤޤ
     *
     * @return 
     */
    public int getValue() {
        return value;
    }

    /**
     * ͤꤷޤ
     * Υ᥽åɤϹ®Τˡͤͭϰϥå򤷤Ƥޤ
     *
     * @param  value   
     */
    public void setValue(int value) {
        synchronized (lock) {
            if (this.value == value) {
                return;
            }

            this.value = value;
            rectChanged = true;
        }
    }

    /**
     * ߤͤ򡢵դθȤƤȤ館֤ͤޤ
     *
     * @return ոȤƤȤ館
     */
    public int getReverseValue() {
        synchronized (lock) {
            return maximum - visible - value + minimum;
        }
    }

    /**
     * ɽ֤ͤޤ
     *
     * @return ɽ
     */
    public int getVisibleAmount() {
        return visible;
    }

    /**
     * Ǿ֤ͤޤ
     *
     * @return Ǿ
     */
    public int getMinimum() {
        return minimum;
    }

    /**
     * ֤ͤޤ
     *
     * @return 
     */
    public int getMaximum() {
        return maximum;
    }

    /**
     * ͤꤷޤ
     *
     * @param  value   
     * @param  visible  ɽ
     * @param  minimum Ǿ
     * @param  maximum 
     *
     * @return ͤѹ줿 <code>true</code>
     *         ѹʤä <code>false</code>
     */
    public boolean setValues(int value, int visible, int minimum, int maximum) {
        if (maximum <= minimum) {
            maximum = minimum + 1;
        }
        if (visible > maximum - minimum) {
            visible = maximum - minimum;
        }
        if (visible < 1) {
            visible = 1;
        }
        if (value < minimum) {
            value = minimum;
        }
        if (value > maximum - visible) {
            value = maximum - visible;
        }

        synchronized (lock) {
            if (/*---*/this.value   == value
                    && this.visible == visible
                    && this.minimum == minimum
                    && this.maximum == maximum) {
                return false;
            }

            this.value   = value;
            this.visible  = visible;
            this.minimum = minimum;
            this.maximum = maximum;
            rectChanged = true;
        }
        return true;
    }

    /**
     * ոȤƤȤ館ͤꤷޤ
     *
     * @param  value   ոȤƤȤ館
     * @param  visible  ɽ
     * @param  minimum Ǿ
     * @param  maximum 
     */
    public void setReverseValues(int value, int visible, int minimum, int maximum) {
        setValues(maximum - visible - value + minimum, visible, minimum, maximum);
    }

    /**
     * ˥åȥ󥯥Ȥ֤ޤ
     *
     * @return ˥åȥ󥯥
     */
    public int getUnitIncrement() {
        return unitIncrement;
    }

    /**
     * ˥åȥ󥯥Ȥ֤ޤ
     *
     * @param  unit ˥åȥ󥯥
     */
    public void setUnitIncrement(int unit) {
        unitIncrement = unit;
    }

    /**
     * ˥åȥ󥯥Ȥ֤ޤ
     *
     * @return ˥åȥ󥯥
     */
    public int getBlockIncrement() {
        return blockIncrement;
    }

    /**
     * ֥å󥯥Ȥ֤ޤ
     *
     * @param  block ֥å󥯥
     */
    public void setBlockIncrement(int block) {
        blockIncrement = block;
    }

    /**
     * ݡͥȤ˴ޤ
     */
    public void dispose() {
        synchronized (lock) {
            if (listener != null) {
                component.removeMouseListener      (listener);
                component.removeMouseMotionListener(listener);
                listener = null;
            }

            if (autoScroll != null) {
                autoScroll.interrupt();
                autoScroll = null;
            }
        }
    }

    /**
     * ݡͥȤѲǽˤޤ
     *
     * @param  b Ѳǽˤ <code>true</code>
     *           ԲĤˤ <code>false</code>
     */
    public void setEnabled(boolean b) {
        isEnabled = b;
        if (!b) {
            pressedArea = NONE;
            synchronized (lock) {
                if (autoScroll != null) {
                    autoScroll.pause();
                }
            }
        }
    }

    /**
     * ݡͥȤɽƤ뤫֤ޤ
     *
     * @return ɽƤ <code>true</code>
     *         ɽξ <code>false</code>
     */
    public boolean isVisible() {
        return isVisible;
    }

    /**
     * ݡͥȤɽޤ
     *
     * @param  b ɽ <code>true</code>
     *           ɽˤ <code>false</code>
     */
    public void setVisible(boolean b) {
        isVisible = b;
        if (!b) {
            pressedArea = NONE;
            synchronized (lock) {
                if (autoScroll != null) {
                    autoScroll.pause();
                }
            }
        }
    }

    /**
     * 뤬ǽɤ֤ޤ
     *
     * @return 뤬ǽʾ <code>true</code>
     *         ʳξ <code>false</code>
     */
    public boolean canScroll() {
        synchronized (lock) {
            return (maximum - minimum > visible);
        }
    }

    /**
     * С褷ޤ
     *
     * @param  g     եå
     * @param  arrow 褹뤫ɤ
     */
    public void paint(Graphics g, boolean arrow) {
        if (!isVisible) {
            return;
        }

        synchronized (lock) {
            if (listener == null) {
                listener = new Listener();
                component.addMouseListener      (listener);
                component.addMouseMotionListener(listener);
            }

            // ϥɥΰ֤ư
            if (rectChanged) {
                int total       = maximum - minimum;
                int denominator = maximum - visible - minimum;

                if (isHorizontal) {
                    int w = rectSlider.width;
                    if (visible < total && total > 0) {
                        w = (int) Math.round(((double) w * visible / total) + 0.5);
                    }
                    int x = rectSlider.width  - w;
                    if (denominator > 0) {
                        x = x * (value - minimum) / denominator;
                    }
                    rectHandle.setBounds(rectSlider.x + x, rectSlider.y, w, rectSlider.height);
                } else {
                    int h = rectSlider.height;
                    if (visible < total && total > 0) {
                        h = (int) Math.round(((double) h * visible / total) + 0.5);
                    }
                    int y = rectSlider.height - h;
                    if (denominator > 0) {
                        y = y * (value - minimum) / denominator;
                    }
                    rectHandle.setBounds(rectSlider.x, rectSlider.y + y, rectSlider.width , h);
                }
            }

            Rectangle r;

            // طʿɤ
            r = rectSlider;
            g.setColor(sliderColor);
            g.fillRect(r.x, r.y, r.width, r.height);
/*
            g.setColor(SystemColor.controlLtHighlight);
            for (int i = 0; i < r.width; i++) {
                for (int j = i % 2; j < r.height; j += 2) {
                    g.drawLine(r.x + i, r.y + j, r.x + i, r.y + j);
                }
            }
*/

            // 
            if (arrow) {
                int      []   types = {ARROW1    , ARROW2    };
                Rectangle[]   rects = {rectArrow1, rectArrow2};
                int           fix;
                int      [][] images;
                if (isHorizontal) {
                    images = new int[][]{ARROW_LEFT, ARROW_RIGHT};
                } else {
                    images = new int[][]{ARROW_TOP, ARROW_BOTTOM};
                }

                Color color = SystemColor.controlText;
                if (!isEnabled) {
                    color = new Color((new GrayFilter(SystemColor.control)).filterRGB(0, 0, color.getRGB()));
                }
                for (int i = 0; i < 2; i++) {
                    r = rects[i];

                    // ط
                    g.setColor(SystemColor.control);
                    g.fillRect(r.x, r.y, r.width, r.height);

                    // 
                    fix = (pressedArea == types[i] ? 3 : 2);
                    if (!isEnabled) {
                        drawArray(g, images[i],
                                  (isHorizontal ? r.x : r.x + (r.width  - ARROW_SIZE) / 2) + fix + 1,
                                  (isHorizontal ? r.y + (r.height - ARROW_SIZE) / 2 : r.y) + fix + 1,
                                  ARROW_SIZE - 4, ARROW_SIZE - 4, SystemColor.controlLtHighlight);
                    }
                    drawArray(g, images[i],
                              (isHorizontal ? r.x : r.x + (r.width  - ARROW_SIZE) / 2) + fix,
                              (isHorizontal ? r.y + (r.height - ARROW_SIZE) / 2 : r.y) + fix,
                              ARROW_SIZE - 4, ARROW_SIZE - 4, color);

                    // 
                    if (pressedArea == types[i]) {
                        g.setColor(SystemColor.controlShadow);
                        g.drawRect(r.x, r.y, r.width - 1, r.height - 1);
                    } else {
                        g.setColor(SystemColor.controlHighlight);
                        g.drawLine(r.x, r.y, r.x + r.width  - 1, r.y);
                        g.drawLine(r.x, r.y, r.x, r.y + r.height - 1);
                        g.setColor(SystemColor.controlLtHighlight);
                        g.drawLine(r.x + 1, r.y + 1, r.x + r.width  - 2, r.y + 1);
                        g.drawLine(r.x + 1, r.y + 1, r.x + 1, r.y + r.height - 2);
                        g.setColor(SystemColor.controlShadow);
                        g.drawLine(r.x + r.width  - 2, r.y + 2, r.x + r.width - 2, r.y + r.height - 2);
                        g.drawLine(r.x + 2, r.y + r.height - 2, r.x + r.width - 2, r.y + r.height - 2);
                        g.setColor(SystemColor.controlDkShadow);
                        g.drawLine(r.x + r.width  - 1, r.y + 1, r.x + r.width - 1, r.y + r.height - 1);
                        g.drawLine(r.x + 1, r.y + r.height - 1, r.x + r.width - 1, r.y + r.height - 1);
                    }
                }
            }

            // ϥɥ
            if (isEnabled && rectHandle.width > 0 && rectHandle.height > 0) {
                r = rectHandle;

                // ط
                g.setColor(SystemColor.control);
                g.fillRect(r.x, r.y, r.width, r.height);

                g.setColor(SystemColor.controlHighlight);
                g.drawLine(r.x, r.y, r.x + r.width  - 1, r.y);
                g.drawLine(r.x, r.y, r.x, r.y + r.height - 1);
                g.setColor(SystemColor.controlLtHighlight);
                g.drawLine(r.x + 1, r.y + 1, r.x + r.width  - 2, r.y + 1);
                g.drawLine(r.x + 1, r.y + 1, r.x + 1, r.y + r.height - 2);
                g.setColor(SystemColor.controlShadow);
                g.drawLine(r.x + r.width  - 2, r.y + 2, r.x + r.width - 2, r.y + r.height - 2);
                g.drawLine(r.x + 2, r.y + r.height - 2, r.x + r.width - 2, r.y + r.height - 2);
                g.setColor(SystemColor.controlDkShadow);
                g.drawLine(r.x + r.width  - 1, r.y + 1, r.x + r.width - 1, r.y + r.height - 1);
                g.drawLine(r.x + 1, r.y + r.height - 1, r.x + r.width - 1, r.y + r.height - 1);
            }

            // ڡ򲡤Ƥ翧Ѥ
            switch (pressedArea) {
            case SPACE1:
            case SPACE2:
                g.setColor(SystemColor.controlShadow);
                r = rectSlider;
                if (pressedArea == SPACE1) {
                    if (isHorizontal) {
                        g.fillRect(r.x, r.y, rectHandle.x - r.x, r.height);
                    } else {
                        g.fillRect(r.x, r.y, r.width , rectHandle.y - r.y);
                    }
                } else {
                    if (isHorizontal) {
                        g.fillRect(r.x + (rectHandle.x - r.x) + rectHandle.width , r.y, r.width  - (rectHandle.x - r.x) - rectHandle.width , r.height);
                    } else {
                        g.fillRect(r.x, r.y + (rectHandle.y - r.y) + rectHandle.height, r.width , r.height - (rectHandle.y - r.y) - rectHandle.height);
                    }
                }
                break;
            default: // AVOID
            }
        }
    }

    /** 󤫤 */
    private void drawArray(Graphics g, int[] base,
                           int x, int y, int width, int height, Color color) {
        int fg = color.getRGB() | 0xff000000;
        int bg = 0;
        int[] buffer = new int[base.length];

        for (int i = 0; i < buffer.length; i++) {
            if (base[i] == 1) {
                buffer[i] = fg;
            } else {
                buffer[i] = bg;
            }
        }

        MemoryImageSource mis = new MemoryImageSource(width, height, buffer, 0, width);
        mis.setAnimated(false);
        Image image = component.getToolkit().createImage(mis);

        SyncObserver so = new SyncObserver();
        synchronized (so) {
            so.init(g, x, y, width, height);
            if (!g.drawImage(image, x, y, width, height, so)) {
                try {
                    so.wait(10000);
                } catch (InterruptedException e) {
                    Debug.out.println("wait to draw a icon");
                }
            }
        }
    }

    /**
     * ꤵ줿ɸΥСξ夫ɤ֤ޤ
     *
     * @param  x X
     * @param  y Y
     * @param  drag ɥåξ <code>true</code>
     *               ʳξ <code>false</code>
     *
     * @return ΥСξξ <code>true</code>
     *         ʳξ <code>false</code>
     */
    public boolean contains(int x, int y, boolean drag) {
        if (!isVisible) {
            return false;
        }

        if (pressedArea != NONE) {
            return true;
        }

        if (drag) {
            return false;
        }

        return rectangle.contains(x, y);
    }

    /**
     * ꥹʤСΥꥹʤɤ֤ޤ
     *
     * @param  l åꥹ
     *
     * @return ʬΥꥹʤξ <code>true</code>
     *         ʳξ <code>false</code>
     */
    public boolean isOwnListener(EventListener l) {
        return (l == listener);
    }

    /**
     * Υ᥽åɤƤӽФ줿硢¹Ԥɬפޤ
     */
    public abstract void repaint();

    /**
     * ͤѹ줿˸ƤӽФޤ
     */
    public void changedValue() {
    }

    /** طʿ׻ */
    private void calculateColor() {
        int rgb1 = SystemColor.controlHighlight  .getRGB();
        int rgb2 = SystemColor.controlLtHighlight.getRGB();
        int r1 = ((rgb1 & 0x00ff0000) >> 16);
        int g1 = ((rgb1 & 0x0000ff00) >>  8);
        int b1 =  (rgb1 & 0x000000ff);
        int r2 = ((rgb2 & 0x00ff0000) >> 16);
        int g2 = ((rgb2 & 0x0000ff00) >>  8);
        int b2 =  (rgb2 & 0x000000ff);

        sliderColor = new Color((r1 + r2) / 2,
                                (g1 + g2) / 2,
                                (b1 + b2) / 2);
    }

//### Listener
    /** ꥹʡ */
    private final class Listener implements MouseListener, MouseMotionListener {
        /** 󥹥󥹤 */
        private Listener() {
        }

        /** ޥå줿 */
        public void mouseClicked(MouseEvent e) {
        }

        /** ޥ줿 */
        public void mousePressed(MouseEvent e) {
            mousePressed(e, true);
        }

        /** ޥ줿 */
        private void mousePressed(MouseEvent e, boolean mouse) {
            int x  = e.getX();
            int y  = e.getY();

            if (/*---*/!isVisible || !isEnabled || !e.getComponent().isEnabled()
                    || !rectangle.contains(x, y)) {
                return;
            }

            e.consume();

            if ((e.getModifiers() & InputEvent.BUTTON1_MASK) == 0) {
                return;
            }

            boolean repaint = false;
            boolean change  = false;

            synchronized (lock) {
                int pa = pressedArea;

                // СΤɤΰ֤򲡤줿å
                if (rectArrow1.contains(x, y)) { pressedArea = ARROW1; } else
                if (rectArrow2.contains(x, y)) { pressedArea = ARROW2; } else
                if (rectHandle.contains(x, y)) {
                    pressedArea = HANDLE;
                    pressedHandle.x = x - rectHandle.x;
                    pressedHandle.y = y - rectHandle.y;
                } else
                if (rectSlider.contains(x, y)) {
                    if (isHorizontal) {
                        if (x < rectHandle.x) {
                            pressedArea = SPACE1;
                        } else {
                            pressedArea = SPACE2;
                        }
                    } else {
                        if (y < rectHandle.y) {
                            pressedArea = SPACE1;
                        } else {
                            pressedArea = SPACE2;
                        }
                    }
                } else {
                    pressedArea = NONE;
                }

                // ֤ѲƤʤå
                if (/*---*/pa != pressedArea
                        && (pressedArea == ARROW1 || pressedArea == ARROW2
                         || pressedArea == SPACE1 || pressedArea == SPACE2)) {
                    repaint = true;
                }

                // ͤѹ
                int v = value;
                switch (pressedArea) {
                case ARROW1: v = Math.max(minimum, v - unitIncrement ); break;
                case ARROW2: v = Math.min(maximum, v + unitIncrement ); break;
                case SPACE1: v = Math.max(minimum, v - blockIncrement); break;
                case SPACE2: v = Math.min(maximum, v + blockIncrement); break;
                default: // AVOID
                }
                v = Math.max(minimum, Math.min(v, maximum - visible));
                if (v != value) {
                    value = v;
                    rectChanged = true;
                    repaint = true;
                    change  = true;
                }
                if (!mouse) {
                    pressedArea = pa;
                }

                // ȥͭ
                if (mouse && pressedArea != HANDLE) {
                    if (autoScroll == null) {
                        (autoScroll = new AutoScroll(e)).start();
                    } else {
                        autoScroll.resume(e);
                    }
                }
            }

            if (change) {
                changedValue();
            }

            // 
            if (repaint) {
                repaint();
            }

            return;
        }

        /** ޥ줿 */
        public void mouseReleased(MouseEvent e) {
            synchronized (lock) {
                // ȥ̵
                if (autoScroll != null) {
                    autoScroll.pause();
                }
            }

            if (/*---*/!isVisible || !isEnabled || !e.getComponent().isEnabled()
                    || pressedArea == NONE) {
                return;
            }

            e.consume();

            if ((e.getModifiers() & InputEvent.BUTTON1_MASK) == 0) {
                return;
            }

            int pa = pressedArea;

            synchronized (lock) {
                pressedArea = NONE;
            }

            switch (pa) {
            case ARROW1:
            case ARROW2:
            case SPACE1:
            case SPACE2:
                repaint();
                break;
            case HANDLE:
                // ϥɥŬ֤ذư
                rectChanged = true;
                repaint();
                break;
            default: // AVOID
            }

            return;
        }

        /** ޥä */
        public void mouseEntered(MouseEvent e) {
        }

        /** ޥ줿 */
        public void mouseExited(MouseEvent e) {
        }

        /** ޥɥå줿 */
        public void mouseDragged(MouseEvent e) {
            if (/*---*/!isVisible || !isEnabled || !e.getComponent().isEnabled()
                    || pressedArea == NONE) {
                return;
            }

            e.consume();

            boolean repaint = false;
            boolean change  = false;

            synchronized (lock) {
                int x = e.getX();
                int y = e.getY();

                // ϥɥʳΥɥåξϡȥͤĴ
                if (pressedArea != HANDLE) {
                    if (autoScroll != null) {
                        autoScroll.setPoint(x, y);
                    }
                    return;
                }

                int oldValue = value;
                if (isHorizontal) {
                    x -= pressedHandle.x;
                    x = Math.max(x, rectSlider.x);
                    x = Math.min(x, rectSlider.x + rectSlider.width  - rectHandle.width );
                    if (rectHandle.x != x) {
                        value = (maximum - visible - minimum)
                              * (x - rectSlider.x)
                              / (rectSlider.width  - rectHandle.width )
                              + minimum;
                        rectHandle.x = x;
                        repaint = true;
                    }
                } else {
                    y -= pressedHandle.y;
                    y = Math.max(y, rectSlider.y);
                    y = Math.min(y, rectSlider.y + rectSlider.height - rectHandle.height);
                    if (rectHandle.y != y) {
                        value = (maximum - visible - minimum)
                              * (y - rectSlider.y)
                              / (rectSlider.height - rectHandle.height)
                              + minimum;
                        rectHandle.y = y;
                        repaint = true;
                    }
                }
                change = (oldValue != value);
            }

            if (change) {
                changedValue();
            }

            // 
            if (repaint) {
                repaint();
            }

            return;
        }

        /** ޥư */
        public void mouseMoved(MouseEvent e) {
        }
    }

//### AutoScroll
    /** ư */
    private final class AutoScroll extends Thread {
        private static final int INIT_INTERVAL = 300;
        private static final int INIT_TIMER    = 300;
        private static final int NEXT_INTERVAL =  35;
        private int        interval  = INIT_INTERVAL;
        private boolean    pause;
        private long       startTime;
        private MouseEvent event;

        /** 󥹥󥹤 */
        private AutoScroll(MouseEvent e) {
            setName("AutoScroll-" + component.getName() + "-" + orientation);
            setDaemon(true);
            pause   = false;
            startTime = System.currentTimeMillis();
            event     = e;
            try {
                setPriority(Thread.MAX_PRIORITY);
            } catch (SecurityException ex) { }
        }

        /** Υ */
        private void setPoint(int x, int y) {
            boolean inner = false;

            synchronized (lock) {
                switch (pressedArea) {
                case ARROW1: inner = rectArrow1.contains(x, y); break;
                case ARROW2: inner = rectArrow2.contains(x, y); break;
                case SPACE1:
                case SPACE2:
                    if (rectSlider.contains(x, y) && !rectHandle.contains(x, y)) {
                        if (isHorizontal) {
                            inner = (x < rectHandle.x) ? (pressedArea == SPACE1)
                                                       : (pressedArea == SPACE2);
                        } else {
                            inner = (y < rectHandle.y) ? (pressedArea == SPACE1)
                                                       : (pressedArea == SPACE2);
                        }
                    }
                default: // AVOID
                }

                if (inner) {
                    event = new MouseEvent(event.getComponent(), event.getID(), event.getWhen(),
                                           event.getModifiers(), x, y, event.getClickCount(),
                                           event.isPopupTrigger());
                }
            }

            synchronized (this) {
                pause = !inner;
                if (inner) {
                    notify();
                }
            }
        }

        /** ¹ */
        public void run() {
            try {
                for (;;) {
                    if (isInterrupted()) {
                        break;
                    }
                    sleep(interval);

                    synchronized (this) {
                        if (pause) {
                            wait();
                            continue;
                        }
                    }

                    // 󥿡Х򶹤
                    if (/*---*/interval == INIT_INTERVAL
                            && startTime + INIT_TIMER < System.currentTimeMillis()) {
                        interval = NEXT_INTERVAL;
                    }

                    listener.mousePressed(event, false);
                }
            } catch (InterruptedException e) { }
        }

        /**  */
        public synchronized void pause() {
            pause = true;
        }

        /** Ƴ */
        public synchronized void resume(MouseEvent e) {
            if (!pause) {
                return;
            }

            this.event    = e;
            this.interval = INIT_INTERVAL;
            this.pause    = false;
            notify();
        }
    }
}
