/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.awt;

import net.hizlab.kagetaka.awt.event.StateEvent;
import net.hizlab.kagetaka.awt.event.StateListener;
import net.hizlab.kagetaka.awt.image.GrayFilter;
import net.hizlab.kagetaka.awt.image.ImageContainer;
import net.hizlab.kagetaka.awt.image.OffscreenImage;
import net.hizlab.kagetaka.awt.image.OffscreenObserver;
import net.hizlab.kagetaka.awt.image.WrappedObserver;

import java.awt.AWTEvent;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Polygon;
import java.awt.Rectangle;
import java.awt.event.InputEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionAdapter;
import java.awt.image.FilteredImageSource;

/**
 * ᡼դΥɽ饹Ǥ
 * ΥܥϡˤꣴĤΥ᡼̾С̵ˤѤޤ
 * ΤᡢĤξ֤٤ƤΥ᡼ꤹˡȡ
 * Ĥξ֡̾ˤΥ᡼뤤ϲлΥ᡼ꤷ
 * 줫ꤵƤʤ¾ξ֤ <code>ImageButton</code> 
 * Фˡޤ
 * <p>
 * ޤ<code>setHotspot</code> ᥽åɤǡ
 * ֥˥塼ѤΥ᡼ɲä뤳Ȥޤ
 * <p>
 * Ĥξ֤٤ƤΥ᡼ꤷƤ硢
 * <code>setHotspot</code> ᥽åɤǤϡĤΥѥϿ
 * ɬפޤΥ᥽åɤϿĤΰΰ
 * ޥưȡ֥˥塼ѤưԤޤ
 * Ĥξ֤Υ᡼Ƥ硢<code>setHotspot</code> ᥽åɤϡ
 * ĤΰΥѥϿɬפޤξ硢ۥåȥݥåѤ
 * ᡼ϡ̾Υ᡼α¦ɲäޤ
 * <p>
 * Ĥξ֤Υ᡼Ƥ硢᡼Ȱɽʸ
 * Ǥޤξ硢᡼α¦ǥۥåȥݥåȤ꺸¦
 * ʸɽޤ
 *
 * @kagetaka.bugs ľ󲽤ϡꥹʤ¸ʤޤ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.6 $
 */
public class ImageButton extends Component {
    private static final int IMAGE_FRAME = 3;
    private static final int LABEL_RIGHT = 3;

    /** ̾ξ */
    public static final int NORMAL       = 0;
    /** ޥ褿 */
    public static final int OVER         = 1;
    /** ޥ줿 */
    public static final int DOWN         = 2;
    /** ܥ̵ʾ */
    public static final int DISABLE      = 3;
    /** ޥۥåȥݥåȤξ褿 */
    public static final int HOTSPOT_OVER = 4;
    /** ۥåȥݥåȤ줿 */
    public static final int HOTSPOT_DOWN = 5;

    /** @serial å֥ */
    private Object lock  = new Object();

    /** @serial ե꡼ */
    private OffscreenImage    offscreenImage;
    /** @serial 侩 */
    private Dimension         preferredSize;
    /** @serial ե꡼󥤥᡼ */
    private WrappedObserver   backObserver;
    /** @serial ե꡼󥤥᡼ */
    private OffscreenObserver buttonObserver;
    /** @serial ե꡼󥹥ݥåȹ */
    private OffscreenObserver hotspotObserver;

    /** @serial طʥ᡼ */
    private ImageContainer backImageContainer;
    /** @serial ᡼꤫ */
    private boolean        isSeparate;
    /** @serial ̾᡼ */
    private ImageContainer buttonNormalImage;
    /** @serial С᡼ */
    private ImageContainer buttonOverImage;
    /** @serial ᡼ */
    private ImageContainer buttonDownImage;
    /** @serial ̵᡼ */
    private ImageContainer buttonDisableImage;
    /** @serial ۥåȥݥåȥС᡼ */
    private ImageContainer hotspotOverImage;
    /** @serial ۥåȥݥåȲ᡼ */
    private ImageContainer hotspotDownImage;
    /** @serial ۥåȥݥå̵᡼ */
    private ImageContainer hotspotDisableImage;
    /** @serial ᡼ˤۥåȥݥåȥꥢ */
    private Polygon        hotspotArea;
    /** @serial ٥ʸ */
    private String         label;

    /** @serial ºݤ褵Ƥܥΰ */
    private Rectangle buttonFixedArea;
    /** @serial ºݤ褵ƤۥåȥݥåȤΰ */
    private Polygon   hotspotFixedArea;

    /** @serial  */
    private int state;
    /** @serial Ρޥξ */
    private int stateEnable;
    /** @serial ξ */
    private int stateDown;

    /** @serial 줿 X  */
    private int pressX = -1;
    /** @serial 줿 Y  */
    private int pressY = -1;

    private transient StateListener stateListener;

    /**
     * ꤵ줿᡼ɽ롢᡼ܥޤ
     * <code>buttonBaseImage</code>
     * ǻꤵ줿᡼򸵤ˡƤξ֤ޤ
     *
     * @param  buttonBaseImage ̾ξ֤Υ᡼
     */
    public ImageButton(Image buttonBaseImage) {
        this(null, true,
             buttonBaseImage, null, null, null,
             null, null, null, null);
    }

    /**
     * ꤵ줿᡼ȥ٥ɽ롢᡼ܥޤ
     * <code>buttonBaseImage</code>
     * ǻꤵ줿᡼򸵤ˡƤξ֤ޤ
     *
     * @param  label           ܥΥ٥롢
     *                         ɽʤ <code>null</code>
     * @param  buttonBaseImage ̾ξ֤Υ᡼
     */
    public ImageButton(String label, Image buttonBaseImage) {
        this(label, true,
             buttonBaseImage, null, null, null,
             null, null, null, null);
    }

    /**
     * ꤵ줿᡼ɽ롢
     * ۥåȥݥåդΥ᡼ܥޤ
     * <code>buttonBaseImage, spot</code>
     * ǻꤵ줿᡼򸵤ˡƤξ֤ޤ
     *
     * @param  buttonBaseImage  ̾ξ֤Υ᡼
     * @param  hotspotBaseImage ۥåȥݥåȤΥ᡼
     *                          ۥåȥݥåȤ¸ߤʤ
     *                          <code>null</code>
     */
    public ImageButton(Image buttonBaseImage, Image hotspotBaseImage) {
        this(null, true,
             buttonBaseImage, null, null, null,
             hotspotBaseImage, null, null, null);
    }

    /**
     * ꤵ줿᡼ȥ٥ɽ롢
     * ۥåȥݥåդΥ᡼ܥޤ
     * <code>buttonBaseImage, hotspotBaseImage</code>
     * ǻꤵ줿᡼򸵤ˡƤξ֤ޤ
     *
     * @param  label            ܥΥ٥롢
     *                          ɽʤ <code>null</code>
     * @param  buttonBaseImage  ̾ξ֤Υ᡼
     * @param  hotspotBaseImage ۥåȥݥåȤΥ᡼
     *                          ۥåȥݥåȤ¸ߤʤ
     *                          code>null</code>
     */
    public ImageButton(String label, Image buttonBaseImage, Image hotspotBaseImage) {
        this(label, true,
             buttonBaseImage, null, null, null,
             hotspotBaseImage, null, null, null);
    }

    /**
     * ꤵ줿᡼ɽ롢᡼ܥޤ
     * <p>
     * <code>isSeparate</code>  <code>false</code> ξϡ
     * ꤵ줿᡼ܥΤɽ᡼Ȥǧ졢
     * ֤ˤäȤ䡢٥뤬ɽʤʤޤ
     * <code>true</code> ξ硢
     * ۥåȥݥåȤ <code>hotspotOverImage</code> Ʊΰˤʤޤ
     * <p>
     * <code>buttonOverImage, buttonDownImage, buttonDisableImage</code>
     * Τ줫 <code>null</code> ꤵ줿硢
     * <code>buttonNormalImage</code> Ŭڤʥ᡼ޤ
     * <code>hotspotDownImage, hotspotDisableImage</code>
     * Τ줫 <code>null</code> ꤵ줿硢
     * <code>hotspotOverImage</code> Ŭڤʥ᡼ޤ
     * <p>
     * ۥåȥݥåȤ¸ߤ<code>hotspotOverImage</code>
     * ꤵƤ˾ϡ
     * <code>isSeparate</code>  <code>true</code> ξΤ
     * <code>hotspotArea</code>  <code>null</code> ˻Ǥ
     * ξΥۥåȥݥåȥꥢ <code>hotspotOverImage</code>
     * ǻꤵ줿᡼ΤȤʤޤ
     * <code>isSeparate</code>  <code>false</code> ˤ⤫餺
     * <code>hotspotArea</code>  <code>null</code> ξ
     * ۥåȥݥåȥꥢꤵʤۥåȥݥåȤǽޤ
     *
     * @param  isSeparate          ᡼ܥΥ᡼ʬΤߤ
     *                             ɽ<code>true</code>
     *                             ܥΥ᡼Τɽ
     *                             <code>false</code>
     * @param  buttonNormalImage   ̾ξ
     * @param  buttonOverImage     ޥ˾ä֡
     *                             <code>buttonNormalImage</code>
     *                              <code>null</code>
     * @param  buttonDownImage     ܥ󤬲줿֡
     *                             <code>buttonNormalImage</code>
     *                              <code>null</code>
     * @param  buttonDisableImage  ܥ̵ξ֡
     *                             <code>buttonNormalImage</code>
     *                              <code>null</code>
     * @param  hotspotOverImage    ۥåȥݥåȾ褿֤Υ᡼
     *                             ۥåȥݥåȤ¸ߤʤ
     *                             <code>null</code>
     * @param  hotspotDownImage    ۥåȥݥåȤƤ֤Υ᡼
     *                             ۥåȥݥåȤ¸ߤʤ
     *                             <code>hotspotOverImage</code>
     *                              <code>null</code>
     * @param  hotspotDisableImage ۥåȥݥåȤ̵ξ֤Υ᡼
     *                             <code>isSeparate</code> 
     *                             <code>false</code> ξ䡢
     *                             ۥåȥݥåȤ¸ߤʤ硢
     *                             <code>hotspotOverImage</code>
     *                              <code>null</code>
     * @param  hotspotArea         <code>hotspotOverImage</code>
     *                             ФƤΥۥåȥݥåȤΰ֡
     *                             ۥåȥݥåȤ¸ߤʤ
     *                             <code>null</code>
     */
    public ImageButton(boolean isSeparate,
                       Image buttonNormalImage,
                       Image buttonOverImage,
                       Image buttonDownImage,
                       Image buttonDisableImage,
                       Image hotspotOverImage,
                       Image hotspotDownImage,
                       Image hotspotDisableImage,
                       Polygon hotspotArea) {
        this(null, isSeparate,
             buttonNormalImage, buttonDownImage, buttonOverImage, buttonDisableImage,
             hotspotOverImage, hotspotDownImage, hotspotDisableImage, hotspotArea);
    }

    /**
     * ꤵ줿᡼ȥ٥ɽ롢
     * ᡼ܥޤ
     *
     * @param  label               ܥΥ٥롢
     *                             ɽʤ <code>null</code>
     * @param  isSeparate          ᡼ܥΥ᡼ʬΤߤ
     *                             ɽ<code>true</code>
     *                             ܥΥ᡼Τɽ
     *                             <code>false</code>
     * @param  buttonNormalImage   ̾ξ
     * @param  buttonOverImage     ޥ˾ä֡
     *                             <code>buttonNormalImage</code>
     *                              <code>null</code>
     * @param  buttonDownImage     ܥ󤬲줿֡
     *                             <code>buttonNormalImage</code>
     *                              <code>null</code>
     * @param  buttonDisableImage  ܥ̵ξ֡
     *                             <code>buttonNormalImage</code>
     *                              <code>null</code>
     * @param  hotspotOverImage    ۥåȥݥåȾ褿֤Υ᡼
     *                             ۥåȥݥåȤ¸ߤʤ
     *                             <code>null</code>
     * @param  hotspotDownImage    ۥåȥݥåȤƤ֤Υ᡼
     *                             ۥåȥݥåȤ¸ߤʤ
     *                             <code>hotspotOverImage</code>
     *                              <code>null</code>
     * @param  hotspotDisableImage ۥåȥݥåȤ̵ξ֤Υ᡼
     *                             <code>isSeparate</code> 
     *                             <code>false</code> ξ䡢
     *                             ۥåȥݥåȤ¸ߤʤ硢
     *                             <code>hotspotOverImage</code>
     *                              <code>null</code>
     * @param  hotspotArea         <code>hotspotOverImage</code>
     *                             ФƤΥۥåȥݥåȤΰ֡
     *                             ۥåȥݥåȤ¸ߤʤ
     *                             <code>null</code>
     *
     * @see    #ImageButton(boolean, Image, Image, Image, Image, Image, Image, Image, Polygon)
     */
    public ImageButton(String label,
                       boolean isSeparate,
                       Image buttonNormalImage,
                       Image buttonOverImage,
                       Image buttonDownImage,
                       Image buttonDisableImage,
                       Image hotspotOverImage,
                       Image hotspotDownImage,
                       Image hotspotDisableImage,
                       Polygon hotspotArea) {
        setImageImpl(isSeparate,
                     buttonNormalImage, buttonDownImage, buttonOverImage, buttonDisableImage,
                     hotspotOverImage, hotspotDownImage, hotspotDisableImage, hotspotArea);
        setLabel(label);

        addMouseListener(
            new MouseAdapter() {
                /** ޥ줿 */
                public void mousePressed(MouseEvent e) {
                    if ((e.getModifiers() & InputEvent.BUTTON1_MASK) == 0) {
                        return;
                    }
                    int x = e.getX();
                    int y = e.getY();

                    if (buttonFixedArea == null || !buttonFixedArea.contains(x, y)) {
                        return;
                    }

                    if (hotspotFixedArea != null && hotspotFixedArea.contains(x, y)) {
                        changeState(HOTSPOT_DOWN);
                    } else {
                        changeState(DOWN);
                    }

                    pressX = e.getX();
                    pressY = e.getY();
                }

                /** ޥ줿 */
                public void mouseReleased(MouseEvent e) {
                    if ((e.getModifiers() & InputEvent.BUTTON1_MASK) == 0) {
                        return;
                    }
                    int x = e.getX();
                    int y = e.getY();

                    if (buttonFixedArea == null || !buttonFixedArea.contains(x, y)) {
                        changeState(NORMAL);
                    } else if (hotspotFixedArea != null && hotspotFixedArea.contains(e.getX(), e.getY())) {
                        changeState(HOTSPOT_OVER);
                    } else {
                        changeState(OVER);
                    }

                    if (e.getClickCount() > 0
                            && buttonFixedArea != null
                            && buttonFixedArea.contains(x, y)
                            && (x != pressX || y != pressY)) {
                        getToolkit().getSystemEventQueue().postEvent(
                            new MouseEvent(e.getComponent(),
                                           MouseEvent.MOUSE_CLICKED,
                                           e.getWhen(),
                                           e.getModifiers(),
                                           e.getX(),
                                           e.getY(),
                                           e.getClickCount(),
                                           e.isPopupTrigger()));
                    }
                }

                /** ޥä */
                public void mouseEntered(MouseEvent e) {
                    if (hotspotFixedArea != null && hotspotFixedArea.contains(e.getX(), e.getY())) {
                        changeState(HOTSPOT_OVER);
                    } else {
                        changeState(OVER);
                    }
                }

                /** ޥФ */
                public void mouseExited(MouseEvent e) {
                    changeState(NORMAL);
                }
            }
        );

        addMouseMotionListener(
            new MouseMotionAdapter() {
                /** ޥɥå줿 */
                public void mouseDragged(MouseEvent e) {
                    if (state == NORMAL) {
                        return;
                    }

                    int x = e.getX();
                    int y = e.getY();

                    if (buttonFixedArea == null || !buttonFixedArea.contains(x, y)) {
                        return;
                    }

                    changeState(stateDown);
                }

                /** ޥư줿 */
                public void mouseMoved(MouseEvent e) {
                    if (hotspotFixedArea == null) {
                        return;
                    }

                    if (hotspotFixedArea.contains(e.getX(), e.getY())) {
                        changeState(HOTSPOT_OVER);
                    } else {
                        changeState(OVER);
                    }
                }
            }
        );
    }

//### Original
    /**
     * ꤵ줿᡼ܥΥ᡼ʬɤ֤ޤ
     *
     * @return ܥΥ᡼ʬξ <code>true</code>
     *         ܥΤξ <code>false</code>
     */
    public boolean isSeparate() {
        return isSeparate;
    }

    /**
     * ۥåȥݥåȤꤵƤ뤫֤ޤ
     *
     * @return ۥåȥݥåȤꤵƤ <code>true</code>
     *         ʳξ <code>false</code>
     */
    public boolean hasHotspot() {
        return (hotspotFixedArea != null);
    }

    /**
     * ܥɽ٥֤ޤ
     *
     * @return ܥΥ٥
     */
    public String getLabel() {
        return label;
    }

    /**
     * ܥɽ٥ꤷޤ
     *
     * @param  label ܥΥ٥
     */
    public void setLabel(String label) {
        synchronized (lock) {
            if (!isSeparate) {
                return;
            }

            if ((this.label == label)
                    || (this.label != null && label != null && this.label.compareTo(label) == 0)) {
                return;
            }

            this.label = label;
        }

        invalidate();
    }

    /**
     * ᡼ꤷޤ
     *
     * @param  buttonNormalImage ̾ξ
     *
     * @see    #ImageButton(Image)
     */
    public void setImage(Image buttonNormalImage) {
        setImageImpl(true, buttonNormalImage, null, null, null, null, null, null, null);
    }

    /**
     * ᡼ꤷޤ
     *
     * @param  buttonNormalImage ̾ξ
     * @param  hotspotBaseImage ۥåȥݥåȤΥ᡼
     *              ۥåȥݥåȤ¸ߤʤ <code>null</code>
     *
     * @see    #ImageButton(Image, Image)
     */
    public void setImage(Image buttonNormalImage, Image hotspotBaseImage) {
        setImageImpl(true, buttonNormalImage, null, null, null, hotspotBaseImage, null, null, null);
    }

    /**
     * ᡼ꤷޤ
     *
     * @param  isSeparate          ᡼ܥΥ᡼ʬΤߤ
     *                             ɽ<code>true</code>
     *                             ܥΥ᡼Τɽ
     *                             <code>false</code>
     * @param  buttonNormalImage   ̾ξ
     * @param  buttonOverImage     ޥ˾ä֡
     *                             <code>buttonNormalImage</code>
     *                              <code>null</code>
     * @param  buttonDownImage     ܥ󤬲줿֡
     *                             <code>buttonNormalImage</code>
     *                              <code>null</code>
     * @param  buttonDisableImage  ܥ̵ξ֡
     *                             <code>buttonNormalImage</code>
     *                              <code>null</code>
     * @param  hotspotOverImage    ۥåȥݥåȾ褿֤Υ᡼
     *                             ۥåȥݥåȤ¸ߤʤ
     *                             <code>null</code>
     * @param  hotspotDownImage    ۥåȥݥåȤƤ֤Υ᡼
     *                             ۥåȥݥåȤ¸ߤʤ
     *                             <code>hotspotOverImage</code>
     *                              <code>null</code>
     * @param  hotspotDisableImage ۥåȥݥåȤ̵ξ֤Υ᡼
     *                             <code>isSeparate</code> 
     *                             <code>false</code> ξ䡢
     *                             ۥåȥݥåȤ¸ߤʤ硢
     *                             <code>hotspotOverImage</code>
     *                              <code>null</code>
     * @param  hotspotArea         <code>hotspotOverImage</code>
     *                             ФƤΥۥåȥݥåȤΰ֡
     *                             ۥåȥݥåȤ¸ߤʤ
     *                             <code>null</code>
     *
     * @see    #ImageButton(boolean, Image, Image, Image, Image, Image, Image, Image, Polygon)
     */
    public void setImage(boolean isSeparate,
                         Image buttonNormalImage,
                         Image buttonOverImage,
                         Image buttonDownImage,
                         Image buttonDisableImage,
                         Image hotspotOverImage,
                         Image hotspotDownImage,
                         Image hotspotDisableImage,
                         Polygon hotspotArea) {
        setImageImpl(isSeparate,
                     buttonNormalImage, buttonOverImage, buttonDownImage, buttonDisableImage,
                     hotspotOverImage, hotspotDownImage, hotspotDisableImage, hotspotArea);
    }

    /**
     * ֤֤ޤ
     *
     * @return 
     */
    public int getState() {
        return state;
    }

    /**
     * ۥåȥݥåȤȤƤꤵƤ֤֤ޤ
     *
     * @return   ۥåȥݥåȤΰ
     */
    public Polygon getHotspotArea() {
        return hotspotFixedArea;
    }

    /**
     * طʲꤷޤ
     *
     * @param  image 
     */
    public void setBackImage(Image image) {
        ImageContainer oldIc = backImageContainer;
        ImageContainer newIc = ImageContainer.setupImageContainer(oldIc, image);
        if (oldIc != newIc) {
            backImageContainer = newIc;
            repaintForce();
        }
    }

//### Override
    /**
     * ݡͥȤѲǽˤޤ
     *
     * @param  b Ѳǽˤ <code>true</code>
     *           ԲĤˤ <code>false</code>
     */
    public void setEnabled(boolean b) {
        super.setEnabled(b);

        changeState((b ? stateEnable : DISABLE));
    }

    /**
     * ơСκǾΥ֤ޤ
     * ⤵Τ¤Ƥޤ
     *
     * @return Ǿ
     */
    public Dimension getMinimumSize() {
        return getPreferredSize();
    }

    /**
     * ơСκ祵֤ޤ
     * ⤵Τ¤Ƥޤ
     *
     * @return 祵
     */
    public Dimension getMaximumSize() {
        return getPreferredSize();
    }

    /**
     * ơСο侩֤ޤ
     * ⤵Τ߻ꤵƤޤ
     *
     * @return 侩
     */
    public Dimension getPreferredSize() {
        Dimension preferredSize = this.preferredSize;
        if (preferredSize != null && isValid()) {
            return preferredSize;
        }

        String label = this.label;
        if (label != null) {
            FontMetrics fm = getFontMetrics(getFont());
            return this.preferredSize = getTotalSize(fm.stringWidth(label) + LABEL_RIGHT,
                                                     fm.getHeight());
        } else {
            return this.preferredSize = getTotalSize(0, 0);
        }
    }

    /**
     * ݡͥȤ̵ˤޤ
     */
    public void invalidate() {
        preferredSize = null;
        super.invalidate();
    }

    /**
     * ̤򥢥åץǡȤޤ
     *
     * @param  g եå
     */
    public void update(Graphics g) {
        paint(g);
    }

    /**
     * ᡼ºݤ褷ޤ
     *
     * @param  g եå
     */
    public void paint(Graphics g) {
        offscreenImage.paint(g);
    }

    /**
     *  ݡͥȤƥʤɲä줿Ȥ򡢤ΥݡͥȤΤ
     *  ԥɬפǤСʤФʤޤ
     */
    public void addNotify() {
        synchronized (getTreeLock()) {
            super.addNotify();

            // ե꡼
            offscreenImage = new OffscreenImage(this) {
                public void update(Image offscreen, Graphics g, Dimension size) {
                    refresh(offscreen, g, size);
                }
            };
        }
    }

    /**
     * ݡͥȤƥʤ줿Ȥ򤽤ΥݡͥȤΤ
     * ԥ¸ߤ˲ޤ
     */
    public void removeNotify() {
        synchronized (getTreeLock()) {
            super.removeNotify();

            // ե꡼
            if (offscreenImage != null) {
                releaseObserver();
                offscreenImage.dispose();
                offscreenImage = null;
            }
        }
    }

    /**
     * Υ֤Υѥ᡼ʸ֤ޤ
     *
     * @return ѥ᡼ʸ
     */
    protected String paramString() {
        String str = super.paramString();
        return str + ",state=" + state;
    }

//### Listener
    /**
     * ֥ꥹʤϿޤ
     *
     * @param  l Ͽ֥ꥹ
     */
    public synchronized void addStateListener(StateListener l) {
        enableEvents(0);
        stateListener = AWTEventMulticaster.add(stateListener, l);
    }

    /**
     * ֥ꥹʤޤ
     *
     * @param  l ֥ꥹ
     */
    public synchronized void removeStateListener(StateListener l) {
        stateListener = AWTEventMulticaster.remove(stateListener, l);
    }

    /**
     * ΥݡͥȤȯ륳ݡͥȥ٥Ȥޤ
     *
     * @param  e ٥
     */
    protected void processEvent(AWTEvent e) {
        if (e instanceof StateEvent) {
            processStateEvent((StateEvent) e);
            return;
        }
        super.processEvent(e);
    }

    /**
     * ΥݡͥȤȯѹ٥Ȥ
     * ϿƤ뤹٤Ƥ {@link StateListener} 뤳Ȥˤꡢ
     * ѹ٥Ȥޤ
     *
     * @param  e ٥
     */
    protected void processStateEvent(StateEvent e) {
        if (stateListener == null) {
            return;
        }

        switch (e.getID()) {
        case StateEvent.STATE_CHANGED: stateListener.stateChanged(e); break;
        default: // AVOID
        }
    }

//### private
    /** ߤΥơˤä֤˺ */
    private void refresh(Image offscreen, Graphics g, Dimension size) {
        synchronized (lock) {
            releaseObserver();

            Font font;
            int  labelWidth;
            int  labelHeight;
            // ٥׻
            if (label != null) {
                font = getFont();
                FontMetrics fm = getFontMetrics(font);
                labelWidth  = fm.stringWidth(label) + LABEL_RIGHT;
                labelHeight = fm.getHeight();
            } else {
                font        = null;
                labelWidth  = 0;
                labelHeight = 0;
            }

            // Τ׻
            Dimension totalSize = getTotalSize(labelWidth, labelHeight);

            // طʿ
            Color bg = getBackground();
            g.setColor(bg);
            g.fillRect(0, 0, size.width, size.height);

            // طʲ
            if (backImageContainer != null) {
                backObserver = new WrappedObserver(this);
                Image backImage  = backImageContainer.image;
                int   backWidth  = backImageContainer.width;
                int   backHeight = backImageContainer.height;
                int x, y;
                for (x = 0; x < size.width; x += backWidth) {
                    for (y = 0; y < size.height; y += backHeight) {
                        g.drawImage(backImage, x, y, backObserver);
                    }
                }
            }

            int x = (size.width - totalSize.width) / 2;
            int y = 0;

            // ơȤɽ
            ImageContainer buttonIc = null, hotspotIc = null;
            if (isSeparate) {
                int sw = (buttonNormalImage != null ? buttonNormalImage.width + IMAGE_FRAME : 0)
                       + labelWidth;
                switch (state) {
                case NORMAL:
                    x += 1;
                    y += 1;
                    buttonIc  = buttonNormalImage;
                    hotspotIc = hotspotOverImage;
                    break;
                case OVER:
                case HOTSPOT_OVER:
                    g.setColor(ColorConverter.getBrighter(bg));
                    g.drawLine(x, y, x + size.width  - 2, y);
                    g.drawLine(x, y, x, y + size.height - 2);
                    if (hotspotOverImage != null) {
                        g.drawLine(x + sw, y, x + sw, y + size.height - 2);
                    }
                    g.setColor(ColorConverter.getDarker(bg));
                    g.drawLine(x, y + size.height - 1, x + size.width - 1, y + size.height - 1);
                    g.drawLine(x + size.width  - 1, y, x + size.width - 1, y + size.height - 1);
                    if (hotspotOverImage != null) {
                        g.drawLine(x + sw - 1, y, x + sw - 1, y + size.height - 1);
                    }
                    x += 1;
                    y += 1;
                    buttonIc  = (buttonOverImage != null ? buttonOverImage : buttonNormalImage);
                    hotspotIc = hotspotOverImage;
                    break;
                case DOWN:
                    g.setColor(ColorConverter.getDarker(bg));
                    g.drawLine(x, y, x + size.width  - 1, y);
                    g.drawLine(x, y, x, y + size.height - 1);
                    if (hotspotOverImage != null) {
                        g.drawLine(x + sw, y, x + sw, y + size.height - 2);
                    }
                    g.setColor(ColorConverter.getBrighter(bg));
                    g.drawLine(x + 1, y + size.height - 1, x + size.width - 1, y + size.height - 1);
                    g.drawLine(x + size.width  - 1, y + 1, x + size.width - 1, y + size.height - 1);
                    if (hotspotOverImage != null) {
                        g.drawLine(x + sw - 1, y, x + sw - 1, y + size.height - 1);
                    }
                    x += 2;
                    y += 2;
                    buttonIc  = (buttonDownImage  != null ? buttonDownImage  : buttonNormalImage);
                    hotspotIc = (hotspotDownImage != null ? hotspotDownImage : hotspotOverImage );
                    break;
                case DISABLE:
                    if (buttonDisableImage == null) {
                        buttonDisableImage = getDisableImage(buttonNormalImage);
                    }
                    if (hotspotOverImage != null && hotspotDisableImage == null) {
                        hotspotDisableImage = getDisableImage(hotspotOverImage);
                    }
                    x += 1;
                    y += 1;
                    buttonIc  = buttonDisableImage;
                    hotspotIc = hotspotDisableImage;
                    break;
                case HOTSPOT_DOWN:
                    g.setColor(ColorConverter.getBrighter(bg));
                    g.drawLine(x, y, x + sw - 2, y);
                    g.drawLine(x, y, x, y + size.height - 2);
                    g.drawLine(x + size.width  - 1, y, x + size.width - 1, y + size.height - 1);
                    g.drawLine(x + sw, y + size.height - 1, x + size.width - 2, y + size.height - 1);
                    g.setColor(ColorConverter.getDarker(bg));
                    g.drawLine(x, y + size.height - 1, x + sw - 1, y + size.height - 1);
                    g.drawLine(x + sw - 1, y, x + sw - 1, y + size.height - 2);
                    g.drawLine(x + sw, y, x + size.width - 2, y);
                    g.drawLine(x + sw, y, x + sw, y + size.height - 2);
                    x += 1;
                    y += 1;
                    buttonIc  = buttonNormalImage;
                    hotspotIc = (hotspotDownImage != null ? hotspotDownImage : hotspotOverImage);
                    break;
                default: // AVOID
                }
            } else {
                switch (state) {
                case NORMAL      : buttonIc = buttonNormalImage ; break;
                case OVER        : buttonIc = buttonOverImage   ; break;
                case DOWN        : buttonIc = buttonDownImage   ; break;
                case DISABLE     :
                    if (buttonDisableImage == null) {
                        buttonDisableImage = getDisableImage(buttonNormalImage);
                    }
                    buttonIc = buttonDisableImage;
                    break;
                case HOTSPOT_OVER: buttonIc = hotspotOverImage  ; break;
                case HOTSPOT_DOWN: buttonIc = hotspotDownImage  ; break;
                default: // AVOID
                }
            }

            int offset;

            // 
            if (buttonIc != null) {
                offset = (size.height - (buttonIc.height + (isSeparate ? IMAGE_FRAME : 0))) / 2;
                buttonObserver = new OffscreenObserver(this, offscreen, x, y + offset);
                g.drawImage(buttonIc.image, x, y + offset, buttonObserver);
                x += buttonNormalImage.width;
                if (isSeparate) {
                    x += IMAGE_FRAME;
                }
            }

            // ٥
            if (label != null) {
                g.setFont (font);
                Color color = getForeground();
                if (state == DISABLE) {
                    color = new Color((new GrayFilter(getBackground())).filterRGB(0, 0, color.getRGB()));
                }
                g.setColor(color);
                FontData    fd = FontData.getInstance(this, font);
                FontMetrics fm = fd.getFontMetrics();
                offset = (int) ((size.height - fd.getFullSize().height - 3) / 2.0 + 0.5);
                int y1 = fd.getHalfBase() + offset + y;
                int y2 = fd.getFullBase() + offset + y;

                char[] cs = new char[1];
                char   c;
                int    w;
                for (int i = 0; i < label.length(); i++) {
                    cs[0] = c = label.charAt(i);
                    w = fm.charWidth(c);
                    g.drawChars(cs, 0, 1, x, ((c <= 0xFF) || (0xFF61 <= c && c <= 0xFF9F) ? y1 : y2));
                    x += w;
                }
                x += LABEL_RIGHT;
            }

            // ݥåȥ᡼
            if (hotspotIc != null) {
                if (state == HOTSPOT_DOWN) {
                    x += 1;
                    y += 1;
                }

                offset = (size.height - (hotspotIc.height + (isSeparate ? IMAGE_FRAME : 0))) / 2;
                hotspotObserver = new OffscreenObserver(this, offscreen, x, y + offset);
                g.drawImage(hotspotIc.image, x, y + offset, hotspotObserver);
            }

            calculateArea(size, totalSize, labelWidth);
        }
    }

    /** Ū˺ɽ */
    private void repaintForce() {
        OffscreenImage oi = offscreenImage;
        if (oi != null) {
            oi.repaint();
        }
    }

    /** ֥Фγ */
    private void releaseObserver() {
        if (backObserver != null) {
            backObserver.dispose();
            backObserver = null;
        }
        if (buttonObserver != null) {
            buttonObserver.dispose();
            buttonObserver = null;
        }
        if (hotspotObserver != null) {
            hotspotObserver.dispose();
            hotspotObserver = null;
        }
    }

    /** ºݤ˥᡼ */
    private void setImageImpl(boolean isSeparate,
                              Image buttonNormalImage,
                              Image buttonOverImage,
                              Image buttonDownImage,
                              Image buttonDisableImage,
                              Image hotspotOverImage,
                              Image hotspotDownImage,
                              Image hotspotDisableImage,
                              Polygon hotspotArea) {
        synchronized (lock) {
            this.isSeparate          = isSeparate;
            this.buttonNormalImage   = ImageContainer.setupImageContainer(this.buttonNormalImage  , buttonNormalImage  );
            this.buttonOverImage     = ImageContainer.setupImageContainer(this.buttonOverImage    , buttonOverImage    );
            this.buttonDownImage     = ImageContainer.setupImageContainer(this.buttonDownImage    , buttonDownImage    );
            this.buttonDisableImage  = ImageContainer.setupImageContainer(this.buttonDisableImage , buttonDisableImage );
            this.hotspotOverImage    = ImageContainer.setupImageContainer(this.hotspotOverImage   , hotspotOverImage   );
            this.hotspotDownImage    = ImageContainer.setupImageContainer(this.hotspotDownImage   , hotspotDownImage   );
            this.hotspotDisableImage = ImageContainer.setupImageContainer(this.hotspotDisableImage, hotspotDisableImage);
            this.hotspotArea         = hotspotArea;

            // ʬΥ᡼Ǥ̵ϡ٥뤬ѤǤʤ
            if (!isSeparate) {
                this.label = null;
            }
        }

        invalidate();
    }

    /**
     * ݡͥȤɬפʥ֤ޤ
     *
     * @param  labelWidth  ٥
     * @param  labelHeight ٥ι⤵
     *
     * @return ݡͥȤɬץ
     */
    private Dimension getTotalSize(int labelWidth, int labelHeight) {
        synchronized (lock) {
            Dimension size = new Dimension(buttonNormalImage.width,
                                           buttonNormalImage.height);
            if (isSeparate) {
                size.width  += IMAGE_FRAME;
                size.height += IMAGE_FRAME;

                // ۥåȥݥåȤû
                if (hotspotOverImage != null) {
                    size.width  += hotspotOverImage.width + IMAGE_FRAME;
                    size.height =  Math.max(size.height, hotspotOverImage.height + IMAGE_FRAME);
                }
            }

            // ٥û
            if (label != null) {
                size.width  += labelWidth;
                size.height =  Math.max(size.height, labelHeight);
            }

            return size;
        }
    }

    /** ꥢΥ */
    private void calculateArea(Dimension size, Dimension totalSize, int labelWidth) {
        // ١ΥܥϡºݤΥ᡼ǤϤʤݡͥȥ
        buttonFixedArea = new Rectangle(0, 0, size.width, size.height);

        if (hotspotOverImage == null) {
            hotspotFixedArea = null;
            return;
        }

        if (hotspotArea != null) {
            // ºݤΥ礭ϥ󥿥󥰤
            int w = (size.width  - totalSize.width ) / 2;
            int h;
            if (isSeparate) {
                h = (size.height - (hotspotOverImage.height + IMAGE_FRAME)) / 2;
                w += buttonNormalImage.width + IMAGE_FRAME + labelWidth;
            } else {
                h = (size.height - totalSize.height) / 2;
            }

            if (w == 0 && h == 0) {
                hotspotFixedArea = hotspotArea;
            } else {
                int   npoints = hotspotArea.npoints;
                int[] xpoints = new int[npoints];
                int[] ypoints = new int[npoints];
                System.arraycopy(hotspotArea.xpoints, 0, xpoints, 0, npoints);
                System.arraycopy(hotspotArea.ypoints, 0, ypoints, 0, npoints);
                for (int i = 0; i < npoints; i++) {
                    xpoints[i] += w;
                    ypoints[i] += h;
                }
                hotspotFixedArea = new Polygon(xpoints, ypoints, npoints);
            }
        } else {
            // ºݤΥ礭ϡϥ󥿥󥰤岼Ϲ
            int w = buttonNormalImage.width + (size.width - totalSize.width) / 2
                  + IMAGE_FRAME + labelWidth;
            hotspotFixedArea = new Polygon(new int[]{w,
                                                     w + hotspotOverImage.width + IMAGE_FRAME,
                                                     w + hotspotOverImage.width + IMAGE_FRAME,
                                                     w},
                                           new int[]{0,
                                                     0,
                                                     size.height,
                                                     size.height},
                                           4);
        }
    }

    /** ̵Υܥ */
    private ImageContainer getDisableImage(ImageContainer ic) {
        if (!ImageUtils.load(ic.image, this)) {
            return new ImageContainer(null);
        }

        Image image = createImage(new FilteredImageSource(ic.image.getSource(),
                                                          new GrayFilter(getBackground())));
        return new ImageContainer(image, ic.width, ic.height);
    }

    /** ֤ѹ */
    private void changeState(int s) {
        synchronized (lock) {
            if (s != DISABLE) {
                stateEnable = s;
            }

            if (!isEnabled()) {
                s = DISABLE;
            }

            if (state != s) {
                state = s;

                switch (state) {
                case DOWN        :
                case HOTSPOT_DOWN:
                    stateDown = state; break;
                default: // AVOID
                }

                repaintForce();
                postStateEvent(s);
            }
        }
    }

    /** ѹ٥Ȥȯޤ */
    private void postStateEvent(int state) {
        StateEvent e = new StateEvent(this, StateEvent.STATE_CHANGED, state);
        try {
            getToolkit().getSystemEventQueue().postEvent(e);
        } catch (SecurityException ex) {
            processStateEvent(e);
        }
    }
}
