/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.addin.table;

import net.hizlab.kagetaka.rendering.Canvas;
import net.hizlab.kagetaka.rendering.Drawkit;
import net.hizlab.kagetaka.rendering.Status;
import net.hizlab.kagetaka.rendering.block.Block;
import net.hizlab.kagetaka.rendering.block.ContainerBlock;
import net.hizlab.kagetaka.rendering.block.FloatManager;
import net.hizlab.kagetaka.token.TokenTypes;
import net.hizlab.kagetaka.token.Value;

import java.awt.Dimension;
import java.awt.Insets;

/**
 * ơ֥Τޤ֥åǤɽޤ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.7 $
 */
class TableBlock extends Block {
    private ErrorBlock   errorBlock;        // 顼֥å
    private CaptionBlock captionBlock;      // ץ֥å
    private GridBlock    gridBlock;         // åɥ֥å

    private int    captionSide;             // ץΰ
    private int    captionHeight;           // ץι⤵RIGHT or LEFT

    /**
     * ơ֥Τޤ֥åޤ
     *
     * @param  drawkit   ɥå
     * @param  status    ơ
     * @param  container ƥʥ֥å
     *                   եȤαƶʤ <code>null</code>
     * @param  parent    ƥ֥å
     */
    TableBlock(Drawkit drawkit, Status status,
               ContainerBlock container, Block parent) {
        super(drawkit, status.createChild(), container, parent);
        this.status.type = TokenTypes.X_BLOCK_START;

        // åɥ֥å
        gridBlock = new GridBlock(drawkit, status, container, this);
        contents.addElement(gridBlock);
    }

    /** {@inheritDoc} */
    protected void calculatePreferred() {
        // ̵analyze 
    }

    /** {@inheritDoc} */
    public Block createBlock(Status status, Status markerStatus) {
        switch (status.display) {
        case Value.DISPLAY_TABLE_CELL:
            return gridBlock.createBlock(status, markerStatus);
        case Value.DISPLAY_TABLE_CAPTION:
            if (captionBlock != null) {
                // ߡ
                return new CaptionBlock(drawkit, status, this);
            }
            captionBlock = new CaptionBlock(drawkit, status, this);
            captionSide  = status.captionSide;
            // ץΰ֤
            switch (captionSide) {
            case Value.ALIGN_BOTTOM:
            case Value.ALIGN_LEFT:
            case Value.ALIGN_RIGHT:
                break;
            default:
                captionSide = Value.ALIGN_TOP;
            }
            contents.addElement(captionBlock);
            return captionBlock;
        default:
            ensureDefaultBlock();
            return errorBlock.createBlock(status, markerStatus);
        }
    }

    /** {@inheritDoc} */
    public Block commitBlock() {
        // 顼֥å򥳥ߥå
        if (errorBlock != null) {
            errorBlock.commitBlock();
        }

        gridBlock.commitBlock();

        return super.commitBlock();
    }

    /** {@inheritDoc} */
    public void analyze(int minParentHeight, int maxParentHeight) {
//Debug.out.println(this + ":TableBlock.analyze: " + minParentHeight + "," + maxParentHeight);
        int min = 0;
        int max = 0;

        // åɤ analyze
        gridBlock.analyze(minParentHeight, maxParentHeight);
        min = gridBlock.minHeight;
        max = gridBlock.maxHeight;

        // ץ analyze
        if (captionBlock != null) {
            captionBlock.analyze(minParentHeight, maxParentHeight);

            switch (captionSide) {
            case Value.ALIGN_LEFT: // ץ󤬾¦
            {
                // ޡǤä
                int margin = captionBlock.margin.bottom;
                this.captionHeight = captionBlock.minHeight;

                min += captionHeight
                    + getMarginOffset(margin,
                                      gridBlock.getMargin(minParentHeight, MARGIN_TOP));
                max += captionHeight
                    + getMarginOffset(margin,
                                      gridBlock.getMargin(maxParentHeight, MARGIN_TOP));

                break;
            }
            case Value.ALIGN_RIGHT: // ץ󤬲¦ˤ
            {
                // ޡǤä
                int margin = captionBlock.margin.top;
                this.captionHeight = captionBlock.minHeight;

                min += captionHeight
                    +  getMarginOffset(gridBlock.getMargin(minParentHeight, MARGIN_BOTTOM),
                                       margin);
                max += captionHeight
                    +  getMarginOffset(gridBlock.getMargin(maxParentHeight, MARGIN_BOTTOM),
                                       margin);

                break;
            }
            default:
                min = Math.max(min, captionHeight);
                max = Math.max(max, captionHeight);
                break;
            }
        }

        // 顼֥å analyze
        if (errorBlock != null) {
            errorBlock.analyze(minParentHeight, maxParentHeight);
            int errorHeight = errorBlock.minHeight;
            min = Math.max(min, errorHeight);
            max = Math.max(max, errorHeight);
        }

        this.minHeight = min;
        this.maxHeight = max;

        // 侩׻
        Dimension parentPreferred = parent.getPreferredSize();
        this.preferredWidth  = parentPreferred.width;
        this.preferredHeight = parentPreferred.height;
        if (this.preferredWidth  == SIZE_PERCENT) {
            status.width  = parent.status.width;
        }
        if (this.preferredHeight == SIZE_PERCENT) {
            status.height = parent.status.height;
        }
    }

    /** {@inheritDoc} */
    protected void calculateFrame(int parentHeight) {
        parentHeight -= captionHeight;

        margin.right = getMarginRight(parentHeight);

        // ե졼̵
        if (frame == null) {
            frame = new Insets(0, 0, 0, 0);
        }
        frameWidth = frame.right = margin.right;
    }

    /** {@inheritDoc} */
    protected void layoutBlock(int newHeight, int containerX, int containerY) {
//Debug.out.println(this + ":TableBlock.layoutBlock: " + newHeight + "," + minHeight + "," + maxHeight);

        // եȤ򤱤
        if (container != null) {
            FloatManager.Trench t = container.getFloatTrench(containerX);
            if (t != null) {
                int diff = Math.max(t.top - containerY, 0);
                // ơ֥եȤβˤ⤰ꤳޤ
                this.top  += diff;
                // եȤη֤褦ˤ
                newHeight -= Math.max((containerY + newHeight) - t.bottom, 0) + diff;
            }
        }

        // åɤ˹碌
        int gridHeight = gridBlock.getHeight(newHeight - captionHeight);

        // 
        int     y          = 0;
        int     marginLeft = 0;
        int     width      = -margin.right; // ʪ
        int     height     = 0;
        int     drawTop    = 0;             // ʪɬפʾ
        int     drawRight  = 0;             // ʪɬפʱ
        int     drawBottom = 0;             // ʪɬפʲ
        int     drawLeft   = 0;             // ʪɬפʺ
        Block[] blocks = new Block[3];
        int     index  = 0;
        Block   block;
        int     captionMargin = 0;

        // 顼֥å
        if (errorBlock != null) {
            block = errorBlock;
            blocks[index++] = block;
            block.validate(gridHeight + captionHeight, marginLeft, width, 0, containerX, containerY);

            marginLeft = block.margin.left;
            width      = block.left;
        }

        switch (captionSide) {
        case Value.ALIGN_TOP:
            // ץ֤ TOP ξ
            block = captionBlock;
            blocks[index++] = block;
            block.validate(gridHeight, marginLeft, width, 0, containerX, containerY);

            gridHeight = Math.max(gridHeight, captionBlock.minHeight);
            marginLeft = block.margin.left;
            width      = block.left;
            break;
        case Value.ALIGN_LEFT:
            // ץ֤ LEFT ξ
            block = captionBlock;
            blocks[index++] = block;
            block.validate(block.minHeight, marginLeft, width, 0, containerX, containerY);

            // ץȥåɴ֤ΥޡǤä
            captionMargin =  getMarginOffset(block.margin.bottom,
                                             gridBlock.getMargin(gridHeight, MARGIN_TOP));
            gridHeight    -= captionMargin;
            y = block.bottom + captionMargin;
            break;
        case Value.ALIGN_RIGHT:
            // ץȥåɴ֤ΥޡǤä
            captionMargin =  getMarginOffset(gridBlock.getMargin(gridHeight, MARGIN_BOTTOM),
                                             captionBlock.margin.top);
            gridHeight    -= captionMargin;
            break;
        case Value.ALIGN_BOTTOM:
            gridHeight = Math.max(gridHeight, captionBlock.minHeight);
            break;
        default: // AVOID
        }

        // åɥ֥å
        block = gridBlock;
        blocks[index++] = block;
        block.validate(gridHeight, marginLeft, width, y, containerX, containerY);

        switch (captionSide) {
        case Value.ALIGN_BOTTOM:
            marginLeft = block.margin.left;
            width      = block.left;

            block = captionBlock;
            blocks[index++] = block;
            block.validate(gridHeight, marginLeft, width, 0, containerX, containerY);

            width      = block.left;
            marginLeft = block.margin.left;
            break;
        case Value.ALIGN_RIGHT:
            // ץȥåɴ֤ΥޡǤäblock = gridBlock
            y = block.bottom + captionMargin;

            block = captionBlock;
            blocks[index++] = block;
            block.validate(block.minHeight, marginLeft, width, y, containerX, containerY);

            width      = Math.max(gridBlock.left, block.left);
            marginLeft = Math.min(gridBlock.margin.left + width - gridBlock.left,
                                  block    .margin.left + width - block    .left);
            break;
        case Value.ALIGN_LEFT:
            width      = Math.max(captionBlock.left, block.left);
            marginLeft = Math.min(block       .margin.left + width - block       .left,
                                  captionBlock.margin.left + width - captionBlock.left);
            break;
        default: // align=right, caption ̵
            width      = block.left;
            marginLeft = block.margin.left;
        }
        // ץ󡦥åɥ֥å麸ޡ
        width       -= marginLeft;
        margin.left =  marginLeft;
        frame .left += marginLeft;
        frameWidth  += marginLeft;

        // 
        for (int i = 0; i < index; i++) {
            block = blocks[i];
            height     = Math.max(height    , block.bottom    );
            drawTop    = Math.min(drawTop   , block.drawTop   );
            drawRight  = Math.min(drawRight , block.drawRight );
            drawBottom = Math.max(drawBottom, block.drawBottom);
            drawLeft   = Math.max(drawLeft  , block.drawLeft  );
        }

        // ·
        int oddHeight = newHeight - height;
        if (oddHeight > 0) {
            switch (parent.status.align) {
            case Value.ALIGN_CENTER: top += (oddHeight) / 2; break;
            case Value.ALIGN_RIGHT : top += (oddHeight)    ; break;
            default: // AVOID
            }
        }

        // ֥åΤ礭
        this.width      = width;
        this.height     = height;
        this.bottom     = top   + height;
        this.left       = right + width + frameWidth;
        this.drawTop    = top   + Math.min(drawTop, 0);
        this.drawRight  = right + Math.min(frame.right + drawRight, 0);
        this.drawBottom = top   + height + Math.max(drawBottom - height, 0);
        this.drawLeft   = right + frame.right + width
                        + Math.max(drawLeft   - width    , frame.left  );
    }

    /** {@inheritDoc} */
    protected void drawBackground(Canvas canvas, int x, int y) {
        // ̵
    }

    /** {@inheritDoc} */
    protected void drawBorder(Canvas canvas, int x, int y) {
        // ̵
    }

//### 
    /** {@inheritDoc} */
    protected void ensureDefaultBlock() {
        if (defaultBlock != null) {
            return;
        }
        defaultBlock = errorBlock = new ErrorBlock(drawkit, status, this);
        contents.addElement(defaultBlock);
    }

    /** {@inheritDoc} */
    public void appendString(String text) {
        if (defaultBlock == null && text.trim().length() == 0) {
            return;
        }
        ensureDefaultBlock();
        defaultBlock.appendString(text);
    }

    /** {@inheritDoc} */
    public void appendNewLine() {
        if (defaultBlock == null) {
            return;
        }
        ensureDefaultBlock();
        defaultBlock.appendNewLine();
    }

//### ơ֥
    /** {@inheritDoc} */
    public void commitRow() {
        gridBlock.commitRow();
    }

//### Utils
    /** {@inheritDoc} */
    protected int getFrameHeight(int parentHeight) {
        return 0;
    }

    /** {@inheritDoc} */
    public int getMargin(int parentHeight, int sense) {
        //// RIGHT ʳ validate ޤʤ
        switch (sense) {
//        case MARGIN_TOP   :
        case MARGIN_RIGHT : return getMarginRight(parentHeight);
//        case MARGIN_BOTTOM:
//        case MARGIN_LEFT  :
        default: // AVOID
            throw new IllegalArgumentException("TableBlock.getMargin(" + sense + ")");
        }
//        return 0;
    }

//### private
    /** ޡ֤ */
    private int getMarginRight(int parentHeight) {
        // Υ֥åαޡ򡢥ץ󡦥åɥ֥å
        if (errorBlock == null) {
            switch (captionSide) {
            case Value.ALIGN_TOP:
                return captionBlock.getMargin(parentHeight, MARGIN_RIGHT);
            case Value.ALIGN_BOTTOM:
                return gridBlock.getMargin(parentHeight, MARGIN_RIGHT);
            case Value.ALIGN_LEFT:
            case Value.ALIGN_RIGHT:
                return getAppendMargin(gridBlock.getMargin(parentHeight, MARGIN_RIGHT),
                                       captionBlock.margin.right);
            default:
                return gridBlock.getMargin(parentHeight, MARGIN_RIGHT);
            }
        }
        return errorBlock.getMargin(parentHeight, MARGIN_RIGHT);
    }

    /** ץȥåɤκޡ֤ */
    private int getAppendMargin(int a, int b) {
        return (a >= 0
                ? (b >= 0 ? Math.min(a, b) : a)
                : (b >= 0 ? b : Math.max(a, b)));
    }
}
