/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
package net.hizlab.kagetaka.addin.style;

import net.hizlab.kagetaka.Debug;
import net.hizlab.kagetaka.Resource;
import net.hizlab.kagetaka.rendering.Content;
import net.hizlab.kagetaka.rendering.Document;
import net.hizlab.kagetaka.rendering.HawkContext;
import net.hizlab.kagetaka.rendering.Reporter;
import net.hizlab.kagetaka.rendering.Request;
import net.hizlab.kagetaka.util.Charset;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.StreamTokenizer;
import java.io.StringReader;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.net.MalformedURLException;
import java.util.Vector;

/**
 * CSS αǡեϤ뤿Υ饹Ǥ
 *
 * @author  <A HREF="mailto:hizuya@hizlab.net">Hizuya Atsuzaki</A>
 * @version $Revision: 1.11 $
 */
class CSSParser {
    private static final String RESOURCE = "net.hizlab.kagetaka.addin.style.Resources";

    private static final int INPUT_BUFFER = 4096;
    private static final int TT_NOTHING   = 0;
    private static final int TT_EOF       = 1;
    private static final int TT_STRINGS   = 2;
    private static final int TT_KEYWORD   = 3;
    private static final int TT_SYNTAX    = 4;
    private static final String[] START_BLOCK  = {"{"};
    private static final String[] END_PROPERTY = {":"};
    private static final String[] END_LINE     = {";"};
    private static final String[] END_STYLE    = {"}", ";"};

    private HawkContext context;            // 륳ƥ
    private Document    document;           // ɥ
    private Content     content;            // ƥ
    private Request     request;            // ꥯ
    private Reporter    reporter;           // 顼ݡ

    private CSSReader           reader;     // CSS ꡼
    private InputStream         is;         // ץåȥȥ꡼
    private BufferedInputStream bis;        // Хåեץåȥȥ꡼
    private String              encoding;   // 󥳡ǥ

    private boolean   commitEncoding;       // 󥳡ǥ󥰤ꤷ
    private boolean   existMedia;           // ǥи
    private boolean   existStyle;           // и
    private int       media;                // ǥ
    private CSSParser nestedParser;         // ͥȤѡ

    // 
    private StringBuffer sb = new StringBuffer();

    /**
     * CSS ΥǡϤѡޤ
     *
     * @param  context  륳ƥ
     * @param  document ɥ
     * @param  content  ¸ߤ륳ƥ
     * @param  request  ꥯ
     * @param  data     Ϥǡ
     */
    CSSParser(HawkContext context, Document document,
              Content content, Request request, String data) {
        this.context  = context;
        this.document = document;
        this.content  = content;
        this.request  = request;
        this.reporter = context.getReporter();
        this.reader   = new CSSReader(content, reporter, new BufferedReader(new StringReader(data)));
        this.commitEncoding = true;
    }

    /**
     * CSS ΥǡեϤѡޤ
     *
     * @param  context  륳ƥ
     * @param  document ɥ
     * @param  content  륳ƥ
     * @param  request  ꥯ
     *
     * @throws IOException IO 顼ȯ
     */
    CSSParser(HawkContext context, Document document,
              Content content, Request request)
            throws IOException {
        this.context  = context;
        this.document = document;
        this.content  = content;
        this.request  = request;
        this.reporter = context.getReporter();
        this.is       = content.getInputStream(null);
        this.encoding = content.getEncoding   ();

        BufferedReader br = null;

        if (encoding != null && (br = createBufferedReader(is, encoding)) != null) {
            this.commitEncoding = true;
        } else {
            // 󥳡ǥ󥰤ǤƤʤ
            this.bis = new BufferedInputStream(is, INPUT_BUFFER);
            this.bis.mark(INPUT_BUFFER * 4);
            br = createBufferedReader(bis, Charset.getDefaultEncoding());
        }

        this.reader = new CSSReader(content, reporter, br);
    }

    /** Хåե꡼ */
    private BufferedReader createBufferedReader(InputStream is, String encoding) {
        if (encoding != null) {
            try {
                return new BufferedReader(new InputStreamReader(is, encoding), INPUT_BUFFER);
            } catch (UnsupportedEncodingException e) {
                reportMessage(null, Reporter.WARNING, 0,
                              "charset.warning.invalid", new String[]{encoding});
            }
            return null;
        }
        return new BufferedReader(new InputStreamReader(is), INPUT_BUFFER);
    }

    /**
     * style °ͤϤޤ
     *
     * @param  context 륳ƥ
     * @param  content ¸ߤ륳ƥ
     * @param  s       Ϥ륹ʸ
     *
     * @return 롢ϤǤʤ <code>null</code>
     */
    static Style valueOf(HawkContext context, Content content, String s) {
        s = "{" + s + "}";
        CSSReader reader = new CSSReader(content,
                                         context.getReporter(),
                                         new BufferedReader(new StringReader(s)));
        try {
            return reader.readStyle(0, null, false);
        } catch (EOFException e) {
Debug.out.println("### WARNING ### CSSParser.valueOf : [" + s + "]");
        } catch (IOException e) {
            //### ERROR
Debug.err.println("### ERROR ### CSSParser.valueOf : [" + s + "]");
        }
        return null;
    }

    /**
     * ǡեϤϰ֤ƬΥ֤ޤ
     *
     * @return Ϥ̤ƬΥ롢
     *         뤬¸ߤʤ <code>null</code>
     *
     * @throws IOException IO 顼ȯ
     */
    synchronized Style next()
            throws IOException {
        try {
            Style    style;
            String   token;
            String   tokenLC;
            Selector selector;
            for (;;) {
                // @import ˤͥȤ
                if (nestedParser != null) {
                    if ((style = nestedParser.next()) != null) {
                        return style;
                    }
                    nestedParser = null;
                }

                // ƬФԤ
                while ((token = reader.readToken(true, true)) == null) {
                    // AVOID
                }
                tokenLC = token.toLowerCase();

                // @charset §
                if (tokenLC.compareTo("@charset") == 0) {
                    if ((token = reader.readToken(false, true)) != null && reader.isSyntax(END_LINE)) {
                        if (!commitEncoding) {
                            resetEncoding(token);
                        }
                    }
                    continue;
                }
                // @charset Ƭʳ̵
                if (!commitEncoding) {
                    commitEncoding = true;
                }

                // @media §
                if (tokenLC.compareTo("@media") == 0) {
                    existStyle = true;
                    if (existMedia) {
                        reader.skipNextBlock("cssparser.warning.nestedmedia", null);
                    } else {
                        existMedia = true;
                        if ((media = Style.convertMedia(reader.getList(true, true))) == 0) {
                            reader.skipNextBlock("cssparser.warning.invalidmedia", null);
                        } else if ((token = reader.readToken(true, true)) == null) {
                            media = 0;
                        } else if (token.compareTo("{") != 0) {
                            reader.skipNextBlock("cssparser.warning.invalidposition", reader.lastToken);
                            media = 0;
                        }
                    }
                    continue;
                }

                // @import §
                if (tokenLC.compareTo("@import") == 0) {
                    String[] values;
                    if ((values = reader.getList(false, false)) != null
                            && reader.isSyntax(END_LINE)) {
                        if (!existStyle) {
                            importStyle(values);
                        } else {
                            reportMessage(RESOURCE, Reporter.WARNING, reader.lineno(),
                                          "cssparser.warning.invalidposition", new String[]{token});
                        }
                    }
                    continue;
                }

                // @page §
                if (tokenLC.compareTo("@page") == 0) {
                    existStyle = true;
                    if ((style = reader.readStyle(media, null, true)) != null) {
                        return style;
                    }
                    continue;
                }

                // ¾ @ §
                if (tokenLC.startsWith("@")) {
                    reader.skipNextBlock("cssparser.warning.unknown", token);
                    continue;
                }

                // Ĥ "}" (@media Ĥ̤ʤ)
                if (tokenLC.compareTo("}") == 0) {
                    if (media == 0) {
                        reader.moveNextLine("cssparser.warning.invalidposition", token);
                    } else {
                        media = 0;
                    }
                    continue;
                }

                existStyle = true;

                // 󥿥å
                if (reader.lastTokenType == TT_SYNTAX) {
                    reader.pushBack();
                    reader.skipNextBlock("cssparser.warning.invalidposition", reader.lastToken);
                    continue;
                }

                // 쥯줿
                reader.pushBack();
                boolean beforeSyntax = true;
                boolean nowSyntax;
                for (;;) {
                    token = reader.readToken(true, false);
                    if (token != null && token.compareTo("{") == 0) {
                        reader.pushBack();
                        break;
                    }
                    nowSyntax = (reader.lastTokenType == TT_SYNTAX);
                    if (!beforeSyntax && !nowSyntax) {
                        sb.append(' ');
                    }
                    sb.append(reader.lastToken);
                    beforeSyntax = nowSyntax;
                }
                if (sb.length() > 0) {
                    token = sb.toString();
                    sb.setLength(0);
                    try {
                        selector = new Selector(token);
                    } catch (java.text.ParseException e) {
                        if (reporter != null) {
                            reporter.report(Reporter.STYLE, Reporter.WARNING, Reporter.NONE,
                                            content, reader.lineNumber, 0,
                                            "CSSParser", e.getMessage());
                        }
                        reader.skipNextBlock(null, null);
                        continue;
                    }
                } else {
                    selector = null;
                }
                if ((style = reader.readStyle(media, selector, false)) != null) {
                    return style;
                }
            }
        } catch (EOFException e) { }

        return null;
    }

    /** 󥳡ǥ󥰤ѹ */
    private void resetEncoding(String charset) {
        if (charset == null) {
            return;
        }

        int lineno = reader.lineno();

        String encoding = Charset.toEncoding(charset);
        if (encoding == null) {
            reportMessage(null, Reporter.WARNING, lineno,
                          "charset.warning.invalid", new String[]{charset});
            return;
        }

        try {
            bis.reset();
            bis.mark(-1);
            reader = new CSSReader(content, reporter, createBufferedReader(bis, encoding));
        } catch (IOException e) {
            // ꥻåȤ˼ԤǤ⡢󥳡ǥ󥰤ꤷ³
            reportMessage(null, Reporter.WARNING, lineno,
                          "charset.warning.reset", new String[]{charset});
        }

        this.encoding       = encoding;
        this.commitEncoding = true;
        content.setEncoding(encoding);

        reportMessage(null, Reporter.INFO, lineno,
                      "charset.info.encoding", new String[]{encoding});
    }

    /** ¾Υ륷Ȥɤ߹ */
    private void importStyle(String[] values) {
        // CSS @import § - 6.3
        String urlString;
        if (values.length > 0) {
            if (values[0].toLowerCase().startsWith("url(")
                    && values[0].endsWith(")")) {
                urlString = values[0].substring(4, values[0].length() - 1);
            } else {
                urlString = values[0];
            }
        } else {
            reportMessage(RESOURCE, Reporter.WARNING, reader.lineno(),
                          "cssparser.warning.invalidimport", null);
            return;
        }

        // ǥפꤵƤ
        int media = 0;
        if (values.length > 1) {
            Vector list = new Vector();
            PARSE:
            for (int i = 1; i < values.length;) {
                list.addElement(values[i]);
                if (++i >= values.length) {
                    break;
                }
                if (values[i].compareTo(",") != 0) {
                    reportMessage(RESOURCE, Reporter.WARNING, reader.lineno(),
                                  "cssparser.warning.invalidimport", null);
                    return;
                }
                i++;
                for (;;) {
                    if (i >= values.length) {
                        break PARSE;
                    }
                    if (values[i].compareTo(",") != 0) {
                        break;
                    }
                    i++;
                }
            }
            if (list.size() > 0) {
                String[] medias = new String[list.size()];
                list.copyInto(medias);
                if ((media = Style.convertMedia(medias)) == 0) {
                    reportMessage(RESOURCE, Reporter.WARNING, reader.lineno(),
                                  "cssparser.warning.invalidmedia", null);
                    return;
                }
                if ((media & (Value.MEDIA_SCREEN | Value.MEDIA_PRINT)) == 0) {
                    return;
                }
            }
        }

        // URL 
        URL url;
        try {
            url = new URL(content.url, urlString);
        } catch (MalformedURLException e) {
            reportMessage(null, Reporter.WARNING, reader.lineno(),
                          "render.warning.href", new String[]{urlString, e.toString()});
            return;
        }

        // 
        try {
            Request request = this.request.createRequest(url);
            Content content = request.getContent(context, true);
            if (content == null) {
                return;
            }

            nestedParser = new CSSParser(context, document, content, request);
            if (media != 0) {
                nestedParser.media = media;
            }
        } catch (IOException e) {
            String s = e.getMessage();
            reportMessage(null, Reporter.WARNING, reader.lineno(),
                          "engine.status.connect.error",
                          new Object[]{urlString,
                                       new Integer(s != null ? 1 : 0),
                                       s,
                                       e.getClass().getName()});
        }
    }

    /**
     * ߤɤ߹ΥǡΥ󥳡ǥ֤̾ޤ
     *
     * @return 󥳡ǥ̤̾ξ <code>null</code>
     */
    String getEncoding() {
        return encoding;
    }

    /**
     * ѡλ꥽ޤ
     */
    void close() {
        try {
            if (bis != null) {
                bis.close();
            }
        } catch (IOException e) {
            reportMessage(RESOURCE, Reporter.WARNING, 0,
                          "cssparser.warning.stream.close", new String[]{e.toString()});
        }
        try {
            if (is != null) {
                is.close();
            }
        } catch (IOException e) {
            reportMessage(RESOURCE, Reporter.WARNING, 0,
                          "cssparser.warning.stream.close", new String[]{e.toString()});
        }
    }

    /** ٹɲä */
    private void reportMessage(String path, int level, int lineNumber, String key, Object[] args) {
        String value = (path != null
                        ? Resource.getMessage(path, key, args)
                        : Resource.getMessage(key, args));
        reporter.report(Reporter.STYLE, level, Reporter.NONE,
                        content, lineNumber, 0,
                        "CSSParser", value);
    }

    /**
     * CSS ѤΥפŸޤ
     *
     * @param  value    ʸ
     * @param  start    ϰ
     * @param  end      λ
     * @param  isString ʸξ <code>true</code>
     *                  ʳξ <code>false</code>
     *
     * @return פŸʸ
     */
    static String unescape(String value, int start, int end, boolean isString) {
        char c = value.charAt(start);
        if (isString
                && (end - start >= 2)
                && (c == '\'' || c == '"')
                && value.charAt(end - 1) == c) {
            start++;
            end--;
        }

        int p = value.indexOf('\\', start);
        if (p == -1 || p >= end) {
            return value.substring(start, end);
        }

        StringBuffer sb = new StringBuffer(end - start);
        for (int i = start; i < end; i++) {
            if ((c = value.charAt(i)) == '\\') {
                int save = i;
                int n = 0;
                while (i + 1 < end) {
                    i++;
                    c = value.charAt(i);
                    n = n << 4;
                    if ('0' <= c && c <= '9') {
                        n += (c - '0');
                    } else if ('A' <= c && c <= 'F') {
                        n += (c - 0x37);               // c - 'A' + 10
                    } else if ('a' <= c && c <= 'f') {
                        n += (c - 0x57);               // c - 'a' + 10
                    } else {
                        i--;
                        break;
                    }
                }
                if (save == i) {
                    continue;
                }
                c = (char) n;
            }
            sb.append(c);
        }

//Debug.out.println("###unescape=["+sb.toString()+"]");
        return sb.toString();
    }

//### CSSReader
    /** CSS ꡼ */
    private static final class CSSReader {
        private Content         content       = null;
        private Reporter        reporter      = null;
        private StreamTokenizer tokenizer     = null;
        private boolean         pushBack      = false;
        private String          lastToken     = null;
        private int             lastTokenType = TT_NOTHING;
        private int             lineNumber    = 0;

        private String token = null;

        /** 󥹥󥹤 */
        private CSSReader(Content content, Reporter reporter, BufferedReader br) {
            this.content   = content;
            this.reporter  = reporter;
            this.tokenizer = new StreamTokenizer(br);

            // ȡʥ
            tokenizer.resetSyntax();

            tokenizer.wordChars   (0x00, 0xFFFF);
            tokenizer.ordinaryChar('"');
            tokenizer.ordinaryChar(':');
            initTokenizerEndText  ();

            tokenizer.eolIsSignificant  (false);
            tokenizer.slashSlashComments(false);
        }

        /** ʸ¦Ѥ˥ȡʥ */
        private void initTokenizerStartText() {
            tokenizer.wordChars(';', ';');
            tokenizer.wordChars('{', '{');
            tokenizer.wordChars('}', '}');
            tokenizer.wordChars(',', ',');
            tokenizer.wordChars('/', '/');          // 
            tokenizer.wordChars('(', '(');          // url "("
            tokenizer.wordChars('>', '>');          // 쥯
            tokenizer.wordChars('+', '+');          // 쥯

            tokenizer.wordChars(0x09, 0x10);
            tokenizer.wordChars(0x12, 0x13);
            tokenizer.wordChars(0x20, 0x20);

            tokenizer.slashStarComments(false);
        }

        /** ʸγ¦Ѥ˥ȡʥ */
        private void initTokenizerEndText() {
            tokenizer.ordinaryChar(';');
            tokenizer.ordinaryChar('{');
            tokenizer.ordinaryChar('}');
            tokenizer.ordinaryChar(',');
            tokenizer.ordinaryChar('/');
            tokenizer.ordinaryChar('(');
            tokenizer.ordinaryChar('>');
            tokenizer.ordinaryChar('+');

            tokenizer.whitespaceChars(0x09, 0x10);
            tokenizer.whitespaceChars(0x12, 0x13);
            tokenizer.whitespaceChars(0x20, 0x20);

            tokenizer.slashStarComments(true);
        }

        /** Υ֥å򥹥ȤƲϤ֤ޤ { key : value; ... }  */
        private Style readStyle(int media, Selector selector, boolean page)
                throws IOException, EOFException {
            int toplineno = tokenizer.lineno();

            tokenizer.ordinaryChar(':');

            // Ƭ "{" ǻϤޤ뤫å
            if (!isSyntax(START_BLOCK)) {
                return null;
            }

            String   name   = null;
            String[] values = null;
            Style    style = new Style(reporter, content, toplineno, 0,
                                       (media != 0 ? media : Value.MEDIA_ALL), selector, page);

            for (;;) {
                // ƬФԤ
                while ((token = readToken(true, true)) == null) {
                    // AVOID
                }

                if (token.compareTo("}") == 0) {
                    break;
                }
                if (token.compareTo(";") == 0) {
                    continue;
                }
                if (lastTokenType == TT_SYNTAX) {
                    moveNextLine("cssparser.warning.invalidposition", token);
                    continue;
                }

                // ץѥƥ
                name = token;
                if (!isSyntax(END_PROPERTY)) {
                    continue;
                }

                // 
                values = getList(false, false);
                if (!isSyntax(END_STYLE)) {
                    continue;
                }

                if (values != null) {
                    style.initProperties(name, values);
                }

                // ";" ǤϤʤ "}" ǽäƤ
                if (token.compareTo("}") == 0) {
                    break;
                }
            }

            tokenizer.wordChars(':', ':');

//Debug.out.println("S[" + style + "]");
            return style;
        }

        /**
         * Υȡ֤
         * (keyword ? ʸλ null : ɤλ null)
         * (error ? null ξϡιԤ˰ư)
         */
        private String readToken(boolean keyword, boolean error)
                throws IOException, EOFException {
            if (pushBack) {
                pushBack = false;
//Debug.out.println("@[" + lastToken + "]");
                return lastToken;
            }

            String returnToken = null;

            READ:
            switch (tokenizer.nextToken()) {
            case StreamTokenizer.TT_EOF:
                lastToken     = null;
                lastTokenType = TT_EOF;
                lineNumber    = tokenizer.lineno();
                throw new EOFException();
            case StreamTokenizer.TT_WORD:
                returnToken   = lastToken = tokenizer.sval;
                lastTokenType = TT_KEYWORD;
                lineNumber    = tokenizer.lineno();
                break READ;
            case '"':
            case '(':
            {
                lineNumber = tokenizer.lineno();

                // ʸνõ
                StringBuffer sb = new StringBuffer();
                int     start  = tokenizer.ttype;
                boolean escape = false;
                int     length = 0, i;
                String  value;

                initTokenizerStartText();
                if (start == '(') {
                    sb.append('(');
                    tokenizer.wordChars   ('"', '"');
                    tokenizer.ordinaryChar(')');
                }
                for (;;) {
                    switch (tokenizer.nextToken()) {
                    case StreamTokenizer.TT_EOF:
                        lastToken     = null;
                        lastTokenType = TT_EOF;
                        lineNumber    = tokenizer.lineno();
                        throw new EOFException();
                    case '"':
                    case ')':
                        if (escape) {
                            sb.append((char) tokenizer.ttype);
                            escape = true;
                            break;
                        }
                        if (start == '(') {
                            sb.append(')');
                            tokenizer.ordinaryChar('"');
                            tokenizer.wordChars   (')', ')');
                        }
                        initTokenizerEndText();
                        lastToken     = sb.toString();
                        lastTokenType = TT_STRINGS;
                        if (!keyword) {
                            returnToken = lastToken;
                        } else if (error) {
                            moveNextLine("cssparser.warning.invalidstrings", lastToken);
                        }
                        break READ;
                    default:
                        {
                            value  = tokenizer.sval;
                            if (value == null) {
                                value = String.valueOf(tokenizer.ttype);
                            }
                            length = value.length();
                            for (i = 0; i < length; i++) {
                                if (escape) {
                                    escape = false;
                                } else if (value.charAt(i) == '\\') {
                                    escape = true;
                                }
                            }
                            sb.append(value);
                        }
                    }
                }
                // Ϥʤ
            }
            default:
                lastToken     = String.valueOf((char) tokenizer.ttype);
                lastTokenType = TT_SYNTAX;
                lineNumber    = tokenizer.lineno();
                if (keyword) {
                    returnToken = lastToken;
                } else if (error) {
                    moveNextLine("cssparser.warning.invalidposition", lastToken);
                }
            }

//Debug.out.println("@[" + lastToken + "],[" + returnToken + "]");
            return returnToken;
        }

        /** ǸΤ⤦֤褦ˤ */
        private void pushBack() {
//Debug.out.println("@pushBack");
            pushBack = true;
        }

        /** ֹ֤ */
        private int lineno() {
            return lineNumber;
        }

        /** Υȡ󤬻ꤵ줿󥿥åΤɤ줫 */
        private boolean isSyntax(String[] syntax)
                throws IOException, EOFException {
            if ((token = readToken(true, true)) == null) {
                return false;
            }

            if (lastTokenType == TT_SYNTAX) {
                for (int i = 0; i < syntax.length; i++) {
                    if (syntax[i].compareTo(token) == 0) {
                        return true;
                    }
                }
            }

            moveNextLine("cssparser.warning.invalidposition", token);
            return false;
        }

        /** ˸ ";" ľ夫 "}" ľ"{" б "}" ľޤǰư */
        private void moveNextLine(String key, String args)
                throws IOException, EOFException {
            if (key != null) {
                reportMessage(Reporter.WARNING, tokenizer.lineno(),
                              key, (args != null ? new String[]{args} : null));
            }

            for (;;) {
                // ƬФԤ
                while ((token = readToken(true, false)) == null) {
                    // AVOID
                }

                // ";" ľذư
                if (token.compareTo(";") == 0) {
                    return;
                }

                // "}" ľذư
                if (token.compareTo("}") == 0) {
                    pushBack();
                    return;
                }

                // "{" б "}" ľذư
                if (token.compareTo("{") == 0) {
                    int level = 1;
                    for (;;) {
                        while ((token = readToken(true, false)) == null) {
                            // AVOID
                        }
                        if (token.compareTo("{") == 0) {
                            level++;
                        } else if (token.compareTo("}") == 0) {
                            if (--level == 0) {
                                return;
                            }
                        }
                    }
                }
            }
        }

        /** Υ֥åΤ̵ ... { ... }  */
        private void skipNextBlock(String key, String args)
                throws IOException, EOFException {
            if (key != null) {
                reportMessage(Reporter.WARNING, tokenizer.lineno(),
                              key, (args != null ? new String[]{args} : null));
            }

            //  "{" ޤɤФ
            while (/**/(token = readToken(true, false)) == null
                    || token.compareTo("{") != 0) {
                // AVOID
            }

            pushBack();

            // Ƭ "{" ʤΤǡ"}" ޤǳμ¤ɤФƤ
            moveNextLine(null, null);
        }

        /** ꥹȤ(ꥹʬʤޤʤ) */
        private String[] getList(boolean comma, boolean keyword)
                throws IOException, EOFException {
            Vector list = new Vector();
            boolean nowcomma = false;

            for (;;) {
                token = readToken((keyword ? true : comma && nowcomma), false);

                // ȡ null ξǤ⡢̵ʸꥹȤϥޤоݤˤ
                if (token == null) {
                    if (!comma && !keyword) {
                        token = lastToken;
                    } else {
                        break;
                    }
                }

                // url( Ҥ
                if (!comma && token.equalsIgnoreCase("url")) {
                    String nextToken = readToken((keyword ? true : comma && nowcomma), false);
                    if (nextToken != null) {
                        token += nextToken;
                    }
                }
                if (comma) {
                    if (nowcomma) {
                        // ް֤ξ
                        if (token.compareTo(",") == 0) {
                            nowcomma = false;
                            continue;
                        }
                        // ްʳƤ齪λ
                        break;
                    }
                    nowcomma = true;
                } else {
                    // ޤʤξϡްʳΥ󥿥åǥꥹȤνȤ
                    if (lastTokenType == TT_SYNTAX && token.compareTo(",") != 0) {
                        break;
                    }
                }

                list.addElement(token);
            }

            pushBack();

            if (list.size() == 0) {
                return null;
            }

            String[] values = new String[list.size()];
            list.copyInto(values);
            return values;
        }

        /** ٹɲä */
        private void reportMessage(int level, int lineNumber, String key, String[] args) {
//(new Exception()).printStackTrace(Debug.out);
            if (reporter != null) {
                reporter.report(Reporter.STYLE, level, Reporter.NONE,
                                content, lineNumber, 0,
                                "CSSReader", Resource.getMessage(RESOURCE, key, args));
            }
        }
    }

//### EOFException
    /** EOF 㳰 */
    private static final class EOFException extends Exception {
        /** EOF 㳰 */
        private EOFException() {
        }
    }
}
