/* ----- BEGIN LICENSE BLOCK -----
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Kagetaka Libraries.
 *
 * The Initial Developer of the Original Code is Hizuya Atsuzaki
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): Hizuya Atsuzaki <hizuya@hizlab.net>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ----- END LICENSE BLOCK ----- */
#include "global.h"
#include "resource.h"
#include "util.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <windows.h>
#include <direct.h>
#include <process.h>

#define getcwd _getcwd
#define strcasecmp _stricmp

#define MAX_PATH_LENGTH   2000

#define   DIR_SEPARATOR  '\\'
#define   PATH_SEPARATOR ';'
#define   JAVA_COMMAND   "javaw.exe"

char *g_Title  = NULL;
HWND  g_Window = 0;

/* Global Data */
static char*  program     = NULL;       /* full pathname of the program */
static char*  homeDir     = NULL;       /* directory where program resides */
static char*  javaPath    = NULL;       /* full pathname of the Java VM to run */

/* Define the special exit codes returned from Kagetaka. */
static int   restartExitCode        = 98;

/* Define the required VM arguments (all platforms). */
#define       JAR_PATH "..\\lib\\kagetaka-browser.jar"
static char*  reqVMarg[] = { "-cp", JAR_PATH, "net.hizlab.kagetaka.viewer.Loader", NULL };

static WNDPROC oldProc;

static char*   jvmCommand     = NULL;
static int     jvmProcess     = 0;
static int     jvmExitCode    = 0;
static int     jvmExitTimeout = 100;
static int     jvmExitTimerId = 99;

static char  *find_command(char* command);
static char **get_vm_command(int argc, char* argv[]);
static char  *format_vm_command(char* args[]);
static char  *get_install_dir();
static void   init_system(int* argc, char* argv[]);
static int    start_javavm(char* args[]);

static void    CALLBACK  detectJvmExit(HWND hwnd, UINT uMsg, UINT id, DWORD dwTime);
static LRESULT WINAPI    WndProc      (HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam);


int main(int argc, char* argv[])
{
	char **vmCommand;
	char  *vmCommandMsg;
	int    exitCode;

	g_Title = get_string_resource(IDS_TITLE);

	if (!(program = find_command(argv[0]))) {
		program = malloc(MAX_PATH_LENGTH + 1);
		GetModuleFileName(NULL, program, MAX_PATH_LENGTH);
	}

	init_system(&argc, argv); 

	if (!(homeDir = get_install_dir())) {
		show_message(IDS_ERROR_NOJAR, JAR_PATH);
		exit(1);
	}
	
	javaPath = find_command(JAVA_COMMAND);
	
	if (javaPath == NULL)
	{
		show_message(IDS_ERROR_NOVM, JAVA_COMMAND);
		exit(1);
	}

	vmCommand    = get_vm_command(argc, argv);
	vmCommandMsg = format_vm_command(vmCommand);
	
	exitCode = restartExitCode;
	while (exitCode == restartExitCode) {
		exitCode = start_javavm(vmCommand);
		
		if (exitCode != 0 && exitCode != restartExitCode) {
			show_message(IDS_ERROR_EXIT, exitCode, vmCommandMsg);
		}
	}
	
	free(homeDir);
	free(program);
	free(vmCommandMsg);
	
	return 0;
}

#define EXTRA 20
char* find_command(char* command)
{ 
	char *cmdPath;
	int   length;
	char *ch;
	char *dir;
	char *path;
	struct _stat stats;

	if (command[0] == DIR_SEPARATOR || 
	   (strlen(command) > 2 && command[1] == ':')) {
		length = strlen(command);
		cmdPath = malloc(length + EXTRA); /* add extra space for a possible ".exe" extension */
		strcpy(cmdPath, command);
	} else {
		if (strchr(command, (int)DIR_SEPARATOR) != NULL) {
			length = MAX_PATH_LENGTH + EXTRA + strlen(command);
			cmdPath = malloc(length);
			getcwd(cmdPath, length);
			if (cmdPath[strlen(cmdPath) - 1] != DIR_SEPARATOR) {
				length = strlen(cmdPath);
				cmdPath[length] = DIR_SEPARATOR;
				cmdPath[length+1] = '\0';
			}
			strcat(cmdPath, command);
		} else {
			path    = getenv("PATH");
			length  = strlen(path) + strlen(command) + MAX_PATH_LENGTH;
			cmdPath = malloc(length);
			
			dir = path;
			while (dir != NULL && *dir != '\0') {
				ch = strchr(dir, (int)PATH_SEPARATOR);
				if (ch == NULL) {
					strcpy(cmdPath, dir);
				} else {
					length = ch - dir;
					strncpy(cmdPath, dir, length);
					cmdPath[length] = '\0';
					ch++;
				}
				dir = ch;

				if (cmdPath[0] == '.' &&
				   (strlen(cmdPath) == 1 || (strlen(cmdPath) == 2 && cmdPath[1] == DIR_SEPARATOR))) {
					getcwd(cmdPath, MAX_PATH_LENGTH);
				}
				if (cmdPath[strlen(cmdPath) - 1] != DIR_SEPARATOR) {
					length = strlen(cmdPath);
					cmdPath[length] = DIR_SEPARATOR;
					cmdPath[length+1] = '\0';
				}
				strcat(cmdPath, command);
				
				if (_stat(cmdPath, &stats) == 0 && (stats.st_mode & S_IFREG) != 0)
					dir = NULL;
			}
		}
	}

	/* If the command does not exist */
	if (_stat(cmdPath, &stats) != 0 || (stats.st_mode & S_IFREG) == 0) {
		/* If the command does not end with .exe, append it an try again. */
		length = strlen(cmdPath);
		if (length > 4 && strcasecmp(&cmdPath[length - 4], ".exe") != 0)
		    strcat(cmdPath, ".exe");
	}
	
	/* Verify the resulting command actually exists. */
	if (_stat(cmdPath, &stats) != 0 || (stats.st_mode & S_IFREG) == 0) {
		free(cmdPath);
		cmdPath = NULL;
	}

	/* Return the absolute command pathname. */
	return cmdPath;
}


static char **get_vm_command(int argc, char* argv[])
{ 
	int    nReqVMarg = 0;
	int    totalArgs;
	char **execArg;
	int    src;
	int    dst;
	
	while (reqVMarg[nReqVMarg] != NULL)
		nReqVMarg++;
	
	totalArgs = 1 + nReqVMarg + argc + 1;
	execArg = (char **)malloc(totalArgs * sizeof(char*));
	
	dst = 0;
	execArg[dst++] = javaPath;
	
	for (src = 1; src < argc; src++) {
		if (strlen(argv[src]) >= 2 &&
		    argv[src][0] == '-' &&
		    argv[src][1] == 'J') {
			execArg[dst++] = argv[src] + 2;
		}
	}
	
	for (src = 0; src < nReqVMarg; src++) {
		if (strcmp(reqVMarg[src], JAR_PATH) != 0) {
			execArg[dst++] = reqVMarg[src];
		} else {
			char *jarFile = malloc(strlen(homeDir) + strlen(JAR_PATH) + 1);
			
			strcpy(jarFile, homeDir);
			strcat(jarFile, JAR_PATH);
			execArg[dst++] = jarFile;
		}
	}
	
	for (src = 1; src < argc; src++) {
		if (strlen(argv[src]) < 2 ||
		    (argv[src][0] != '-' &&
		     argv[src][1] != 'J')) {
			execArg[dst++] = argv[src];
		}
	}
	execArg[dst++] = NULL;
	
	return execArg;
}

static char* format_vm_command(char* args[]) 
{
	int   index;
	int   length;
	char* ch;
	char* message;

	length = 0;
	for (index = 0; args[index] != NULL; index++)
		length += strlen(args[index]) + 1;
	message = (char *)malloc(length);
	
	ch = message;
	for (index = 0; args[index] != NULL; index++) 
	{
		strcpy(ch, args[index]);
		ch += strlen(args[index]);
		*ch++ = ' ';
	}
	*ch = '\0';
	
	return message;
}

static char* get_install_dir() 
{
	char*  ch;
	char*  installDir;
	char   path[MAX_PATH_LENGTH];
	struct _stat stats;
	
	installDir = (char *)malloc(strlen(program) + 1);
	strcpy(installDir, program);
	ch = strrchr(installDir, (int)DIR_SEPARATOR);
	
	while (ch != NULL) {
		*(ch+1) = '\0';
		sprintf(path, "%s%s", installDir, JAR_PATH);
		
		if (_stat(path, &stats) == 0 && (stats.st_mode & S_IFREG) != 0) {
			return installDir;
		} else {
			*ch = '\0';	
			ch = strrchr(installDir, (int)DIR_SEPARATOR);
		}
	}
	
	free(installDir);
	return NULL;
}

void init_system(int* pArgc, char* argv[])
{
	g_Window = CreateWindowEx(0, "STATIC", g_Title, SS_BITMAP | WS_POPUP,
	                          0, 0, 0, 0, NULL, NULL,
	                          GetModuleHandle (NULL), NULL);
	oldProc = (WNDPROC)GetWindowLong(g_Window, GWL_WNDPROC);
	SetWindowLong(g_Window, GWL_WNDPROC, (LONG)WndProc);
}

int start_javavm(char* args[]) 
{
	MSG   msg;
	int   index;
	char* newArg;

	if (jvmCommand == NULL) {
		jvmCommand = malloc(strlen(args[0]) + 1);
		strcpy(jvmCommand, args[0]);
		
		for (index = 0; args[index] != NULL; index++) {
			if (strchr(args[index], ' ') != NULL) {
				newArg = malloc(strlen(args[index]) + 3);
				sprintf(newArg, "\"%s\"", args[index]);
				args[index] = newArg;
			}
		}
	}

	if ((jvmProcess = _spawnv(_P_NOWAIT, jvmCommand, args)) == -1) {
		jvmExitCode = errno;
		jvmProcess  = 0;
	} else {
		SetTimer(g_Window, jvmExitTimerId, jvmExitTimeout, detectJvmExit);
		
		while (jvmProcess != 0) {
			GetMessage(&msg, NULL, 0, 0);
			TranslateMessage(&msg);
			DispatchMessage (&msg);
		}
		
		KillTimer(g_Window, jvmExitTimerId);
	}
	
	return jvmExitCode;
}

/* Local functions */

static void CALLBACK detectJvmExit(HWND hwnd, UINT uMsg, UINT id, DWORD dwTime)
{
	DWORD   exitCode;
	
	if (!GetExitCodeProcess((HANDLE)jvmProcess, &exitCode) ||
	    exitCode != STILL_ACTIVE) {
		jvmExitCode = exitCode;
		jvmProcess = 0;
	}
}

static LRESULT WINAPI WndProc (HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	switch (uMsg) 
	{
	case WM_NCHITTEST:
		return HTCLIENT;
	case WM_CLOSE: 
		PostQuitMessage(0);
		break;
	}
	return CallWindowProc(oldProc, hwnd, uMsg, wParam, lParam);
}
