// $Id: ACELogCmd.cpp,v 1.1.1.1 2002/08/31 04:47:24 fukasawa Exp $

//=============================================================================
/**
 *  @file    ACELogCmd.cpp
 *
 *  @author Fukasawa Mitsuo
 *
 *
 *    Copyright (C) 1998-2001 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#include "beesecs.h"
#include "BS2Version.h"
#include "tcl.h"

static char _versionNum[] = {BS2_VERSION};
ACE_Object_Manager ace_object_manager;

/*---------------------------------------------------------------------------*/
/* Mask bit name table                                                       */
/*---------------------------------------------------------------------------*/
struct LogMaskName
{
    TCHAR * m_name;
    ULONG m_bit;
};

static LogMaskName _maskNameTable[] =
{
    {_T("SHUTDOWN"),  LM_SHUTDOWN},      //  0: 1
    {_T("TRACE"),     LM_TRACE},         //  1: 2
    {_T("DEBUG"),     LM_DEBUG},         //  2: 4
    {_T("INFO"),      LM_INFO},          //  3: 8
    {_T("NOTICE"),    LM_NOTICE},        //  4: 16
    {_T("WARNING"),   LM_WARNING},       //  5: 32
    {_T("STARTUP"),   LM_STARTUP},       //  6: 64
    {_T("ERROR"),     LM_ERROR},         //  7: 128
    {_T("CRITICAL"),  LM_CRITICAL},      //  8: 256
    {_T("ALERT"),     LM_ALERT},         //  9: 512
    {_T("EMERGENCY"), LM_EMERGENCY},     // 10: 1024
    {NULL, 0},                           // 11: tail
};

ULONG LogMaskTokenize(char *flag_string)
{
    int maskbits = 0;
    for (char *flag = ACE_OS::strtok (flag_string, " "); flag != NULL;
        flag = ACE_OS::strtok (0, " ")) {
        for (int i = 0; (i < 11) && (_maskNameTable[i].m_name != NULL); i++) {
            if (ACE_OS::strcmp(flag, _maskNameTable[i].m_name) == 0)
                maskbits |= _maskNameTable[i].m_bit;
        }
    }
    return maskbits;
}

void LogMaskDumper(Tcl_Interp * interp, ULONG maskPattern)
{
    char names[256];
    int bitnum = 1;
    names[0] = '\0';
    for (int i = 0; (i < 11) && (_maskNameTable[i].m_name != NULL); i++) {
        if (maskPattern & bitnum) {
            if (names[0] != '\0')
                ACE_OS::strcat(names, _T(" "));
            ACE_OS::strcat(names, _maskNameTable[i].m_name);
        }
        bitnum <<= 1;
    }
    Tcl_AppendResult(interp, names, NULL);
}

void LogMaskHelp(Tcl_Interp * interp)
{
    char names[256];
    names[0] = '\0';
    for (int i = 0; (i < 11) && (_maskNameTable[i].m_name != NULL); i++) {
        if (names[0] != '\0')
            ACE_OS::strcat(names, _T(" "));
        ACE_OS::strcat(names, _maskNameTable[i].m_name);
    }
    Tcl_AppendResult(interp, names, NULL);
}

/*
 *----------------------------------------------------------------------
 *
 * Ace_logCmd --
 *
 *    This procedure is invoked to process "secs::log" Tcl command.
 *    See the user documentation for details on what it does.
 *
 * Results:
 *    A standard Tcl result.
 *
 * Side effects:
 *    See the user documentation.
 *
 *----------------------------------------------------------------------
 */
int Ace_logCmd (
    ClientData ,                /* not used */
    Tcl_Interp *interp,         /* Current interpreter. */
    int objc,                   /* Number of arguments. */
    Tcl_Obj *CONST objv[])      /* The argument objects. */
{
    if (objc < 2) {
        Tcl_WrongNumArgs(interp, 1, objv, "log sub-command ?bit-name ...?");
        return TCL_ERROR;
    }

    static char *options[] = {
        "start", "flags", "set", "reset", "help", NULL
    };
    enum SubCommandEnum {
        LOG_START, LOG_FLAGS, LOG_SET, LOG_RESET, LOG_HELP
    };

    int index;
    if (Tcl_GetIndexFromObj(interp, objv[1], options, "sub-command", 0,
                            &index) != TCL_OK) {
        return TCL_ERROR;
    }

    switch (index) {
    case LOG_START:
        if (ACE_LOG_MSG->open("SecsTcl", ACE_Log_Msg::LOGGER,
                              ACE_DEFAULT_LOGGER_KEY) == -1)
        {
            ACE_DEBUG((LM_DEBUG, "*** open logger ***\n"));
            ACE_DEBUG((LM_DEBUG, "version: %s\n", _versionNum));
            return false;
        }
        else
        {
            ACE_DEBUG((LM_DEBUG, "******************************************\n"));
            ACE_DEBUG((LM_DEBUG, "*   SECS2 communication task log start   *\n"));
            ACE_DEBUG((LM_DEBUG, "******************************************\n"));
            ACE_DEBUG((LM_DEBUG, "version: %s\n", _versionNum));

            ACE_Trace::stop_tracing();   // disable trace
        }
        return TCL_OK;
    case LOG_FLAGS:
        {
            ULONG mask_bits = ACE_Log_Msg::instance()->priority_mask(ACE_Log_Msg::PROCESS);
            LogMaskDumper(interp, mask_bits);
        }
        return TCL_OK;
    case LOG_SET:
        {
            ULONG priority_mask = ACE_Log_Msg::instance()->priority_mask(ACE_Log_Msg::PROCESS);
            ULONG update_mask = LogMaskTokenize(Tcl_GetStringFromObj(objv[2], (int *)0));
            ACE_SET_BITS(priority_mask, update_mask);
            ACE_Log_Msg::instance()->priority_mask(priority_mask, ACE_Log_Msg::PROCESS);
            if (ACE_BIT_ENABLED(priority_mask, LM_TRACE))
                ACE_Trace::start_tracing();
        }
        return TCL_OK;
    case LOG_RESET:
        {
            ULONG priority_mask = ACE_Log_Msg::instance()->priority_mask(ACE_Log_Msg::PROCESS);
            ULONG update_mask = LogMaskTokenize(Tcl_GetStringFromObj(objv[2], NULL));
            ACE_CLR_BITS(priority_mask, update_mask);
            ACE_Log_Msg::instance()->priority_mask(priority_mask, ACE_Log_Msg::PROCESS);
            if (ACE_BIT_DISABLED(priority_mask, LM_TRACE))
                ACE_Trace::stop_tracing();
        }
        return TCL_OK;
    case LOG_HELP:
        {
            LogMaskHelp(interp);
        }
        return TCL_OK;
    }
    return TCL_ERROR;
}

