// $Id: BS2Device.cpp,v 1.1.1.1 2002/08/31 04:47:23 fukasawa Exp $

//=============================================================================
/**
 *  @file    BS2Device.cpp
 *
 *  @author Fukasawa Mitsuo
 *
 *
 *    Copyright (C) 1998-2002 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#define BEE_BUILD_DLL

#include "BS2Device.h"
#include "BS2SECSReceiver.h"
#include "BS2HSMSReceiver.h"
#include "BS2Sender.h"
#include "BS2Serial.h"
#include "BS2Socket.h"
#include "BS2Driver.h"
#include "BS2TransactionManager.h"
#include "BS2MessageInfo.h"
#include "SECSXmlParser.h"

//-----------------------------------------------------------------------------
// Constructor/Destructor
//-----------------------------------------------------------------------------
BS2Device::BS2Device()
        : m_devid(0), m_action(EQUIPMENT), m_driver(NULL), m_sender(NULL),
          m_receiver(NULL)
{
    TRACE_FUNCTION(TRL_CONSTRUCT, "BS2Device::BS2Device");
    m_trmgr = new BS2TransactionManager;
    m_logmask = ACE_Log_Msg::instance()->priority_mask();
}

//-----------------------------------------------------------------------------
BS2Device::~BS2Device()
{
    TRACE_FUNCTION(TRL_CONSTRUCT, "BS2Device::~BS2Device");
    if (m_trmgr)
        delete m_trmgr;
    if (m_driver)
        delete m_driver;
    if (m_sender)
        delete m_sender;
    if (m_receiver)
        delete m_receiver;
}

//-----------------------------------------------------------------------------
// Create new receiver and sender.
//-----------------------------------------------------------------------------
void BS2Device::hexDump(bool tf)
{
    TRACE_FUNCTION(TRL_LOW, "BS2Device::doDump");
	if (m_driver != NULL)
        m_driver->hexDump(tf);
}

//-----------------------------------------------------------------------------
// Create new receiver and sender.
//-----------------------------------------------------------------------------
int BS2Device::initialize(DeviceParameter * parm)
{
    TRACE_FUNCTION(TRL_LOW, "BS2Device::initialize");

    int result;
    ACE_TCHAR name[64];

    result = SECSXmlParser::instance()->init(parm->m_xmlname);
    if (result < 0)
    {
        return result;
    }
    result = SECSXmlParser::instance()->parse();
    if (result < 0)
    {
        return result;
    }

    m_sender = new BS2Sender(this);
    m_devid = parm->m_deviceId;
    m_action = (parm->m_slave == 0) ? EQUIPMENT : HOST;
    m_trmgr->sourceId(parm->m_sourceId);
    if (parm->m_dtype == DRIVER_SOCKET)
    {
        BS2HSMSReceiver * hsms = new BS2HSMSReceiver(this);
        m_receiver = hsms;
        BS2Socket * socket = new BS2Socket();
        m_driver = socket;
        ACE_OS::strcpy(name, "HSMS");
        m_sender->T6(((SocketParameter *)parm)->m_t6timeout);
    }
    else
    {
        BS2SECSReceiver * secs1 = new BS2SECSReceiver(this);
        m_receiver = secs1;
        BS2Serial * serial = new BS2Serial();
        m_driver = serial;
        ACE_OS::strcpy(name, "SECS1");
    }

    // initial control tables
    if (m_driver->initialize(parm, this) >= 0)
    {
        result = m_driver->open();
        if (result >= 0)
        {
            if (m_sender->open() >= 0)
            {
                m_sender->T3(parm->m_t3timeout);
                m_sender->setLinktestTime(parm->m_hbtimeout);
                if (m_receiver->open() >= 0)
                {
                    return BEE_SUCCESS;
                }
                else
                {
                    ACE_ERROR((LM_ERROR, "(%t) receiver is not opend %s %d\n",
                                         name, parm->m_deviceId));
                }
            }
            else
            {
                ACE_ERROR((LM_ERROR, "(%t) sender is not opend %s %d\n",
                                     name, parm->m_deviceId));
            }
        }
        else
        {
            ACE_ERROR((LM_ERROR, "(%t) device module is not opened %s %d\n",
                                 name, parm->m_deviceId));
        }
    }
    else
    {
        ACE_ERROR((LM_ERROR, "(%t) device module is not initalized %s %d\n",
                             name, parm->m_deviceId));
    }

    // fail open module.
    delete m_driver;
    m_driver = NULL;
    delete m_sender;
    m_sender = NULL;
    delete m_receiver;
    m_receiver = NULL;
    return BEE_ERROR;
}

//-----------------------------------------------------------------------------
// Open driver.
//-----------------------------------------------------------------------------
int BS2Device::open(void *)
{
    TRACE_FUNCTION(TRL_LOW, "BS2Device::open");

    if (this->activate(THR_NEW_LWP | THR_DETACHED) == -1)
        ACE_ERROR_RETURN((LM_ERROR, "%p\n", "spawn"), -1);

    return BEE_SUCCESS;
}

//-----------------------------------------------------------------------------
// Change device id.
//-----------------------------------------------------------------------------
int BS2Device::setDeviceId(int id)
{
    TRACE_FUNCTION(TRL_LOW, "BS2Device::setDeviceId");
    // Caution! Not mutex lock.
    int retval = m_devid;
    m_devid = id;
    m_driver->deviceId(id);
    return retval;
}

//-----------------------------------------------------------------------------
// Restart receive interpreter.
//-----------------------------------------------------------------------------
int BS2Device::restart(void *)
{
    TRACE_FUNCTION(TRL_LOW, "BS2Device::restart");

    if (this->activate(THR_NEW_LWP | THR_DETACHED) == -1)
        ACE_ERROR_RETURN((LM_ERROR, "%p\n", "spawn"), -1);

    return BEE_SUCCESS;
}

//-----------------------------------------------------------------------------
// Close
//-----------------------------------------------------------------------------
int BS2Device::close(int exit_status)
{
    TRACE_FUNCTION(TRL_LOW, "BS2Receiver::close");

    if (exit_status < 0)
    {   // return by error
        if (m_receiver)
            m_receiver->close();
        if (m_sender)
            m_sender->open();
        if (m_driver)
            m_driver->close();
    }
    TRACE_DEBUG((_TX("(%t) thread is exiting with status %d in module %s\n"),
                 exit_status, this->getName()));

    return BEE_SUCCESS;
}

//-----------------------------------------------------------------------------
// Send message.
//-----------------------------------------------------------------------------
int BS2Device::send(BS2Message * msg)
{
    TRACE_FUNCTION(TRL_LOW, "BS2Device::send");

    int result;
    result = m_sender->send(msg);
    return result;
}

//-----------------------------------------------------------------------------
// receive message.
//-----------------------------------------------------------------------------
int BS2Device::receive(BS2MessageInfo& ret)
{
    TRACE_FUNCTION(TRL_LOW, "BS2Device::receive");

    int result;
    result = m_receiver->receive(ret);
    return result;
}

//-----------------------------------------------------------------------------
// receive message with limit time.
//-----------------------------------------------------------------------------
int BS2Device::receiveWithLimit(BS2MessageInfo& ret, ACE_Time_Value *tv)
{
    TRACE_FUNCTION(TRL_LOW, "BS2Device::receiveWithLimit");

    int result;
    result = m_receiver->receiveWithLimit(ret, tv);
    return result;
}

//-----------------------------------------------------------------------------
// receive message.
//-----------------------------------------------------------------------------
int BS2Device::stop_receive()
{
    TRACE_FUNCTION(TRL_LOW, "BS2Device::receive");

    int result;
    putCommand(NULL);
    result = m_receiver->stopReceive();
    return result;
}

//-----------------------------------------------------------------------------
// Cancel receive message.
//-----------------------------------------------------------------------------
int BS2Device::cancelReceive()
{
    TRACE_FUNCTION(TRL_LOW, "BS2Device::cancelReceive");

    int result;

    result = m_receiver->stopReceive();
    return result;
}

//-----------------------------------------------------------------------------
// sense received message.
//-----------------------------------------------------------------------------
int BS2Device::sense(BS2MessageInfo& ret)
{
    TRACE_FUNCTION(TRL_LOW, "BS2Device::sense");

    int result;
    result = m_receiver->sense(ret);
    return result;
}

//-----------------------------------------------------------------------------
// Simply enqueue the Message_Block into the end of the queue.
//-----------------------------------------------------------------------------
int BS2Device::put(ACE_Message_Block *mb, ACE_Time_Value *tv)
{
    return this->putq(mb, tv);
}

//-----------------------------------------------------------------------------
// Receive message thread.
//-----------------------------------------------------------------------------
int BS2Device::svc(void)
{
    int result = 0;
    ACE_Time_Value delaytm;

    for (;;)
    {
        ACE_Message_Block *mb;
        ACE_Log_Msg::instance()->priority_mask(m_logmask);

            // delay(1 sec) for task switching
        delaytm.set(ACE_OS::time(0), 1000000);
        result = this->getq(mb, &delaytm);
        if (result == -1)
        {    // time out

        }
        else
        {
            int length = mb->length();
            if (length > 0)
            {
                BCHAR * top = mb->rd_ptr();

            }
            mb->release ();

            if (length == 0)             // shutdown
                break;
        }
    }

    ACE_DEBUG((LM_DEBUG, "(%t) %s \"device %d\" dispatcher is deleted.\n",
                         this->getName(), m_devid));
    return 0;
}

//-----------------------------------------------------------------------------
// Put tcl command in thread interpreter
//-----------------------------------------------------------------------------
int BS2Device::putCommand(BCHAR * cmd)
{
    ACE_Message_Block *mb;

    if (cmd == NULL)
    {
        ACE_NEW_RETURN(mb, ACE_Message_Block(BUFSIZ + 1), -1);
        // Send a shutdown message to the other thread and exit.
        mb->length(0);
        if (this->put(mb) == -1)
            ACE_ERROR((LM_ERROR, "(%t) %p\n", "put"));
        return 0;
    }

    int len = ACE_OS::strlen(cmd) + 1;
    ACE_NEW_RETURN(mb, ACE_Message_Block(len + 128), -1);
    ACE_OS::memcpy(mb->rd_ptr(), cmd, len);

    mb->length(len);
    // NUL-terminate the string (since we use strlen() on it later).

    if (this->put(mb) == -1)
        ACE_ERROR ((LM_ERROR, "(%t) %p\n", "put_next"));

    return 0;
}

