%{
/* $Id*/
/*---------------------------------------------------------------------------*/
/**
 *  @file    secs2l.l
 *
 *  @author Fukasawa Mitsuo
 *
 *
 *    Copyright (C) 1998-2001 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
/*---------------------------------------------------------------------------*/

/* #define yylex(p)        yyyylex(p) */

#include "beelib.h"
#include "tcl.h"
#include "y.tab.h"

extern int LineNum;

#ifdef FLEX_SCANNER
extern char * myinputptr;
extern char * myinputlim;
extern Tcl_Interp * myinterp;
static int my_yyinput(char * buf, int max_size);
#endif  /* FLEX_SCANNER */

extern void warning(char * s, char * t);
int findname(char * yytext, YYSTYPE * valptr);

#ifndef FLEX_SCANNER
#undef  input
#undef  unput
#define input()   (*myinputptr++)
#define unput(c)  (*--myinputptr = (c))
#else
#undef  YY_INPUT
#define YY_INPUT(b, r, ms)   (r = my_yyinput(b, ms))
#endif

%}

ws            [ \t]+
string        \"
digit         [0-9]+
sign          [-+]
integer       ({sign}{digit})
uinteger      ({digit}|[+]{digit})
u1            ({uinteger}[uU]1)
u2            ({uinteger}[uU]2)
u4            ({uinteger}[uU]4)
u8            ({uinteger}[uU]8)
i1            (({integer}[iI]1)|({uinteger}[iI]1))
i2            (({integer}[iI]2)|({uinteger}[iI]2))
i4            (({integer}[iI]4)|({uinteger}[iI]4))
i8            (({integer}[iI]8)|({uinteger}[iI]8))
sfnum         [0-9]|[0-9][0-9]|[0-2][0-9][0-9]
number        {digit}
hex           0x[0-9a-fA-F]+
name          $[a-zA-Z_][0-9a-zA-Z_(),\-:]*
nl            \n
float         (({digit}\.{digit})|({sign}{digit}\.{digit}))
real          ({float}|({float}[eE]{digit})|({float}[eE]{sign}{digit}))
f4            ({float}[fF]|{real}[fF])
f8            ({float}[dD]|{real}[dD])
sxxfxx        S{sfnum}F{sfnum}

%s NORMAL LABEL
%e 3000
%p 7000
%n 1024

/*---------------------------------------------------------------------------*/
/*  START                                                                    */
/*---------------------------------------------------------------------------*/

%%

%{

    BEGIN NORMAL;

%}

<NORMAL>{ws}
<NORMAL>"MESSAGE"   { return MESSAGE; }
<NORMAL>"message"   { return MESSAGE; }
<NORMAL>"{"         { return LBRACE; }
<NORMAL>"}"         { return RBRACE; }
<NORMAL>"("         { return LPAREN; }
<NORMAL>")"         { return RPAREN; }
<NORMAL>";"         { return SEMICOLON; }
<NORMAL>"WAIT"      { return WAIT; }
<NORMAL>"NOWAIT"    { return NOWAIT; }
<NORMAL>"BIN"       { return BIN; }
<NORMAL>"true"      { yylval.ival = 1; return BOOLEAN; }
<NORMAL>"TRUE"      { yylval.ival = 1; return BOOLEAN; }
<NORMAL>"false"     { yylval.ival = 0; return BOOLEAN; }
<NORMAL>"FALSE"     { yylval.ival = 0; return BOOLEAN; }
<NORMAL>"carriers"  { return CARRIERS; }
<NORMAL>"S1F0"      { return S1F0; }    /* Abort Transaction */
<NORMAL>"S1F1"      { return S1F1; }    /* Are You There Request */
<NORMAL>"S1F2"      { return S1F2; }    /* On Line Data */
<NORMAL>"S1F3"      { return S1F3; }    /* Selected Equipment Status Request */
<NORMAL>"S1F4"      { return S1F4; }    /* Selected Equipment Status Data */
<NORMAL>"S1F5"      { return S1F5; }    /* Formatted Status Request */
<NORMAL>"S1F6"      { return S1F6; }    /* Formatted Status Data */
<NORMAL>"S1F7"      { return S1F7; }    /* Fixed Form Request */
<NORMAL>"S1F8"      { return S1F8; }    /* Fixed Form Data */
<NORMAL>"S1F9"      { return S1F9; }    /* Material Transfer Status Request */
<NORMAL>"S1F10"     { return S1F10; }   /* Material Transfer Status Data */
<NORMAL>"S1F11"     { return S1F11; }   /* Status Variable Namelist Request */
<NORMAL>"S1F12"     { return S1F12; }   /* Status Variable Namelist Reply */
<NORMAL>"S1F13"     { return S1F13; }   /* Establish Communications Requet */
<NORMAL>"S1F14"     { return S1F14; }   /* Establish Communications Requet Acknowledge */
<NORMAL>"S1F15"     { return S1F15; }   /* Request OFF-LINE */
<NORMAL>"S1F16"     { return S1F16; }   /* OFF-LINE Acknowledge */
<NORMAL>"S1F17"     { return S1F17; }   /* Request ON-LINE */
<NORMAL>"S1F18"     { return S1F18; }   /* ON-LINE Acknowledge */
<NORMAL>"S1F19"     { return S1F19; }   /* Get Attribute */
<NORMAL>"S1F20"     { return S1F20; }   /* Attribute Data */
<NORMAL>"S2F0"      { return S2F0; }    /* Abort Transaction */
<NORMAL>"S2F1"      { return S2F1; }    /* Service Program Load Inquire */
<NORMAL>"S2F2"      { return S2F2; }    /* Service Program Load Grant */
<NORMAL>"S2F3"      { return S2F3; }    /* Service Program Send */
<NORMAL>"S2F4"      { return S2F4; }    /* Service Program Send Acknowledge */
<NORMAL>"S2F5"      { return S2F5; }    /* Service Program Load Request */
<NORMAL>"S2F6"      { return S2F6; }    /* Service Program Load Data */
<NORMAL>"S2F7"      { return S2F7; }    /* Service Program Run Send */
<NORMAL>"S2F8"      { return S2F8; }    /* Service Program Run Acknowledge */
<NORMAL>"S2F9"      { return S2F9; }    /* Service Program Result Request */
<NORMAL>"S2F10"     { return S2F10; }   /* Service Program Result Data */
<NORMAL>"S2F11"     { return S2F11; }   /* Service Program Directory Request */
<NORMAL>"S2F12"     { return S2F12; }   /* Service Program Directory Data */
<NORMAL>"S2F13"     { return S2F13; }   /* Equipment Constant Request */
<NORMAL>"S2F14"     { return S2F14; }   /* Equipment Constant Data */
<NORMAL>"S2F15"     { return S2F15; }   /* New Equipment Constant Send */
<NORMAL>"S2F16"     { return S2F16; }   /* New Equipment Constant Acknowledge */
<NORMAL>"S2F17"     { return S2F17; }   /* Date and Time Request */
<NORMAL>"S2F18"     { return S2F18; }   /* Date and Time Data */
<NORMAL>"S2F19"     { return S2F19; }   /* Reset/Initialize Send */
<NORMAL>"S2F20"     { return S2F20; }   /* Reset Acknowledge */
<NORMAL>"S2F21"     { return S2F21; }   /* Remote Command Send */
<NORMAL>"S2F22"     { return S2F22; }   /* Remote Command Acknowledge */
<NORMAL>"S2F23"     { return S2F23; }   /* Trace Initialize Send */
<NORMAL>"S2F24"     { return S2F24; }   /* Trace Initialize Acknowledge */
<NORMAL>"S2F25"     { return S2F25; }   /* Loopback Diagnostic Request */
<NORMAL>"S2F26"     { return S2F26; }   /* Loopback Diagnostic Data */
<NORMAL>"S2F27"     { return S2F27; }   /* Initiate Processing Request */
<NORMAL>"S2F28"     { return S2F28; }   /* Initiate Processing Acknowledge */
<NORMAL>"S2F29"     { return S2F29; }   /* Equipment Constant Namelist Request */
<NORMAL>"S2F30"     { return S2F30; }   /* Equipment Constant Namelist */
<NORMAL>"S2F31"     { return S2F31; }   /* Date and Time Set Request */
<NORMAL>"S2F32"     { return S2F32; }   /* Date and Time Set Acknowledge */
<NORMAL>"S2F33"     { return S2F33; }   /* Define Report */
<NORMAL>"S2F34"     { return S2F34; }   /* Define Report Acknowledge */
<NORMAL>"S2F35"     { return S2F35; }   /* Link Report */
<NORMAL>"S2F36"     { return S2F36; }   /* Link Report Acknowledge */
<NORMAL>"S2F37"     { return S2F37; }   /* Enable/Disable Event Report */
<NORMAL>"S2F38"     { return S2F38; }   /* Enable/Disable Event Report Acknowledge */
<NORMAL>"S2F39"     { return S2F39; }   /* Multi-block Inquire */
<NORMAL>"S2F40"     { return S2F40; }   /* Multi-block Grant */
<NORMAL>"S2F41"     { return S2F41; }   /* Host Command Send */
<NORMAL>"S2F42"     { return S2F42; }   /* Host Command Acknowledge */
<NORMAL>"S2F43"     { return S2F43; }   /* Reset Spooling Streams and Functions */
<NORMAL>"S2F44"     { return S2F44; }   /* Reset Spooling Acknowledge */
<NORMAL>"S2F45"     { return S2F45; }   /* Define Variable Limit Attributes */
<NORMAL>"S2F46"     { return S2F46; }   /* Variable Limit Attribute Acknowledge */
<NORMAL>"S2F47"     { return S2F47; }   /* Variable Limit Attribute Request */
<NORMAL>"S2F48"     { return S2F48; }   /* Variable Limit Attributes Send */
<NORMAL>"S2F49"     { return S2F49; }   /* Enhanced Remote Command */
<NORMAL>"S2F50"     { return S2F50; }   /* Enhanced Remote Command Acknowledge */
<NORMAL>"S3F0"      { return S3F0; }    /* Abort Transaction */
<NORMAL>"S3F1"      { return S3F1; }    /* Material Status Request */
<NORMAL>"S3F2"      { return S3F2; }    /* Material Status Data */
<NORMAL>"S3F3"      { return S3F3; }    /* Time to Completion Request */
<NORMAL>"S3F4"      { return S3F4; }    /* Time to Completion Data */
<NORMAL>"S3F5"      { return S3F5; }    /* Material Found Send */
<NORMAL>"S3F6"      { return S3F6; }    /* Material Found Acknowledge */
<NORMAL>"S3F7"      { return S3F7; }    /* Material Lost Send */
<NORMAL>"S3F8"      { return S3F8; }    /* Material Lost Acknowledge */
<NORMAL>"S3F9"      { return S3F9; }    /* Material ID Equate Send */
<NORMAL>"S3F10"     { return S3F10; }   /* Material ID Equate Acknowledge */
<NORMAL>"S3F11"     { return S3F11; }   /* Material ID Request */
<NORMAL>"S3F12"     { return S3F12; }   /* Material ID Request Acknowledge */
<NORMAL>"S3F13"     { return S3F13; }   /* Material ID Send */
<NORMAL>"S3F14"     { return S3F14; }   /* Material ID Acknowledge */
<NORMAL>"S3F15"     { return S3F15; }   /* Materials Multi-block Inquire */
<NORMAL>"S3F16"     { return S3F16; }   /* Materials Multi-block Grant */
<NORMAL>"S3F17"     { return S3F17; }   /* Carrier Action Request */
<NORMAL>"S3F18"     { return S3F18; }   /* Carrier Action Acknowledge */
<NORMAL>"S3F19"     { return S3F19; }   /* Cancel All Carrier Out Request */
<NORMAL>"S3F20"     { return S3F20; }   /* Cancel All Carrier Out Acknowledge */
<NORMAL>"S3F21"     { return S3F21; }   /* Port Group Definition */
<NORMAL>"S3F22"     { return S3F22; }   /* Port Group Definition Acknowledge */
<NORMAL>"S3F23"     { return S3F23; }   /* Port Group Action Request */
<NORMAL>"S3F24"     { return S3F24; }   /* Port Group Action Acknowledge */
<NORMAL>"S3F25"     { return S3F25; }   /* Port Action Request */
<NORMAL>"S3F26"     { return S3F26; }   /* Port Action Acknowledge */
<NORMAL>"S3F27"     { return S3F27; }   /* Change Access */
<NORMAL>"S3F28"     { return S3F28; }   /* Change Access Acknowledge */
<NORMAL>"S3F29"     { return S3F29; }   /* Carrier Tag Read Request */
<NORMAL>"S3F30"     { return S3F30; }   /* Carrier Tag Read Data */
<NORMAL>"S3F31"     { return S3F31; }   /* Carrier Tag Write Data Request */
<NORMAL>"S3F32"     { return S3F32; }   /* Carrier Tag Write Data Acknowledge */
<NORMAL>"S4F0"      { return S4F0; }    /* Abort Transaction */
<NORMAL>"S4F1"      { return S4F1; }    /* Ready to Send Material */
<NORMAL>"S4F2"      { return S4F2; }    /* Ready to Send Acknowledge */
<NORMAL>"S4F3"      { return S4F3; }    /* Send Material */
<NORMAL>"S4F5"      { return S4F5; }    /* Handshake Complete */
<NORMAL>"S4F7"      { return S4F7; }    /* Not Ready to Send */
<NORMAL>"S4F9"      { return S4F9; }    /* Stuck in Sender */
<NORMAL>"S4F11"     { return S4F11; }   /* Stuck in Receiver */
<NORMAL>"S4F13"     { return S4F13; }   /* Send Incomplete Timeout */
<NORMAL>"S4F15"     { return S4F15; }   /* Material Received */
<NORMAL>"S4F17"     { return S4F17; }   /* Request to Receive */
<NORMAL>"S4F18"     { return S4F18; }   /* Request to Receive Acknowledge */
<NORMAL>"S4F19"     { return S4F19; }   /* Transfer Job Create */
<NORMAL>"S4F20"     { return S4F20; }   /* Transfer Job Acknowledge */
<NORMAL>"S4F21"     { return S4F21; }   /* Transfer Job Command */
<NORMAL>"S4F22"     { return S4F22; }   /* Transfer Command Acknowledge */
<NORMAL>"S4F23"     { return S4F23; }   /* Transfer Job Alert */
<NORMAL>"S4F24"     { return S4F24; }   /* Transfer Alert Acknowledge */
<NORMAL>"S4F25"     { return S4F25; }   /* Multi-Block Inquire */
<NORMAL>"S4F26"     { return S4F26; }   /* Multi-Block Grant */
<NORMAL>"S4F27"     { return S4F27; }   /* Handoff Ready */
<NORMAL>"S4F29"     { return S4F29; }   /* Handoff Command */
<NORMAL>"S4F31"     { return S4F31; }   /* Handoff Command Complete */
<NORMAL>"S4F33"     { return S4F33; }   /* Handoff Verify */
<NORMAL>"S4F35"     { return S4F35; }   /* Handoff Cancel Ready */
<NORMAL>"S4F37"     { return S4F37; }   /* Handoff Cancel Ready Acknowledge */
<NORMAL>"S4F39"     { return S4F39; }   /* Handoff Halt */
<NORMAL>"S4F41"     { return S4F41; }   /* Handoff Halt Acknowledge */
<NORMAL>"S5F0"      { return S5F0; }    /* Abort Transaction */
<NORMAL>"S5F1"      { return S5F1; }    /* Alarm Report Send */
<NORMAL>"S5F2"      { return S5F2; }    /* Alarm Report Acknowledge */
<NORMAL>"S5F3"      { return S5F3; }    /* Enable/Disable Alarm Send */
<NORMAL>"S5F4"      { return S5F4; }    /* Enable/Disable Alarm Acknowledge */
<NORMAL>"S5F5"      { return S5F5; }    /* List Alarms Request */
<NORMAL>"S5F6"      { return S5F6; }    /* List Alarm Data */
<NORMAL>"S5F7"      { return S5F7; }    /* List Enabled Alarm Request */
<NORMAL>"S5F8"      { return S5F8; }    /* List Enabled Alarm Data */
<NORMAL>"S5F9"      { return S5F9; }    /* Exception Post Notify */
<NORMAL>"S5F10"     { return S5F10; }   /* Exception Post Confirm */
<NORMAL>"S5F11"     { return S5F11; }   /* Exception Clear Notify */
<NORMAL>"S5F12"     { return S5F12; }   /* Exception Clear Confirm */
<NORMAL>"S5F13"     { return S5F13; }   /* Exception Recovery Request */
<NORMAL>"S5F14"     { return S5F14; }   /* Exception Recovery Acknowledge */
<NORMAL>"S5F15"     { return S5F15; }   /* Exception Recovery Complete Notify */
<NORMAL>"S5F16"     { return S5F16; }   /* Exception Recovery Complete Confirm */
<NORMAL>"S5F17"     { return S5F17; }   /* Exception Recovery Abort Request */
<NORMAL>"S5F18"     { return S5F18; }   /* Exception Recovery Abort Acknowledge */
<NORMAL>"S6F0"      { return S6F0; }    /* Abort Transaction */
<NORMAL>"S6F1"      { return S6F1; }    /* Trace Data Send */
<NORMAL>"S6F2"      { return S6F2; }    /* Trace Data Acknowledge */
<NORMAL>"S6F3"      { return S6F3; }    /* Discrete Variable Data Send */
<NORMAL>"S6F4"      { return S6F4; }    /* Discrete Variable Data Acknowledge */
<NORMAL>"S6F5"      { return S6F5; }    /* Multi-Block Data Send Inquire */
<NORMAL>"S6F6"      { return S6F6; }    /* Multi-Block Grant */
<NORMAL>"S6F7"      { return S6F7; }    /* Data Transfer Request */
<NORMAL>"S6F8"      { return S6F8; }    /* Data Transfer data */
<NORMAL>"S6F9"      { return S6F9; }    /* Formatted Variable Send */
<NORMAL>"S6F10"     { return S6F10; }   /* Formatted Variable Acknowledge */
<NORMAL>"S6F11"     { return S6F11; }   /* Event Report Send */
<NORMAL>"S6F12"     { return S6F12; }   /* Event Report Acknowledge */
<NORMAL>"S6F13"     { return S6F13; }   /* Annotated Event Report Send */
<NORMAL>"S6F14"     { return S6F14; }   /* Annotated Event Report Acknowledge */
<NORMAL>"S6F15"     { return S6F15; }   /* Event Report Request */
<NORMAL>"S6F16"     { return S6F16; }   /* Event Report Data */
<NORMAL>"S6F17"     { return S6F17; }   /* Annotated Event Report Request */
<NORMAL>"S6F18"     { return S6F18; }   /* Annotated Event Report Data */
<NORMAL>"S6F19"     { return S6F19; }   /* Indivisual Report Request */
<NORMAL>"S6F20"     { return S6F20; }   /* Indivisual Report Data */
<NORMAL>"S6F21"     { return S6F21; }   /* Annotated Indivisual Report Request */
<NORMAL>"S6F22"     { return S6F22; }   /* Annotated Indivisual Report Data */
<NORMAL>"S6F23"     { return S6F23; }   /* Request Spooled Data */
<NORMAL>"S6F24"     { return S6F24; }   /* Request Spooled Data Acknowledge Send */
<NORMAL>"S6F25"     { return S6F25; }   /* Notification Report Send */
<NORMAL>"S6F26"     { return S6F26; }   /* Notification Report Send Acknowledge */
<NORMAL>"S6F27"     { return S6F27; }   /* Trace Report Send */
<NORMAL>"S6F28"     { return S6F28; }   /* Trace Report Send Acknowledge */
<NORMAL>"S6F29"     { return S6F29; }   /* Trace Report Request */
<NORMAL>"S6F30"     { return S6F30; }   /* Trace Report Data */
<NORMAL>"S7F0"      { return S7F0; }    /* Abort Transaction */
<NORMAL>"S7F1"      { return S7F1; }    /* Process Program Load Inquire */
<NORMAL>"S7F2"      { return S7F2; }    /* Process Program Load Grant */
<NORMAL>"S7F3"      { return S7F3; }    /* Process Program Send */
<NORMAL>"S7F4"      { return S7F4; }    /* Process Program Acknowledge */
<NORMAL>"S7F5"      { return S7F5; }    /* Process Program Request */
<NORMAL>"S7F6"      { return S7F6; }    /* Process Program Data */
<NORMAL>"S7F7"      { return S7F7; }    /* Process Program ID Request */
<NORMAL>"S7F8"      { return S7F8; }    /* Process Program ID Data */
<NORMAL>"S7F9"      { return S7F9; }    /* M/P M Request */
<NORMAL>"S7F10"     { return S7F10; }   /* M/P M Data */
<NORMAL>"S7F11"     { return S7F11; }   /* M/P M Update Send */
<NORMAL>"S7F12"     { return S7F12; }   /* M/P M Update Acknowledge */
<NORMAL>"S7F13"     { return S7F13; }   /* Delete M/P M Entry Send */
<NORMAL>"S7F14"     { return S7F14; }   /* Delete M/P M Entry Acknowledge */
<NORMAL>"S7F15"     { return S7F15; }   /* Matrix Mode Select Send */
<NORMAL>"S7F16"     { return S7F16; }   /* Matrix Mode Select Acknowledge */
<NORMAL>"S7F17"     { return S7F17; }   /* Delete Process Program Send */
<NORMAL>"S7F18"     { return S7F18; }   /* Delete Process Program Acknowledge */
<NORMAL>"S7F19"     { return S7F19; }   /* Current EPPD Request */
<NORMAL>"S7F20"     { return S7F20; }   /* Current EPPD Data */
<NORMAL>"S7F21"     { return S7F21; }   /* Equipment Process Capabilities Request */
<NORMAL>"S7F22"     { return S7F22; }   /* Equipment Process Capabilities Data */
<NORMAL>"S7F23"     { return S7F23; }   /* Formatted Process Program Send */
<NORMAL>"S7F24"     { return S7F24; }   /* Formatted Process Program Acknowledge */
<NORMAL>"S7F25"     { return S7F25; }   /* Formatted Process Program Request */
<NORMAL>"S7F26"     { return S7F26; }   /* Formatted Process Program Data */
<NORMAL>"S7F27"     { return S7F27; }   /* Process Program Verification Send */
<NORMAL>"S7F28"     { return S7F28; }   /* Process Program Verification Acknowledge */
<NORMAL>"S7F29"     { return S7F29; }   /* Process Program Verification Inquire */
<NORMAL>"S7F30"     { return S7F30; }   /* Process Program Verification Grant */
<NORMAL>"S7F31"     { return S7F31; }   /* Verification Request Send */
<NORMAL>"S7F32"     { return S7F32; }   /* Verification Request Acknowledge */
<NORMAL>"S7F33"     { return S7F33; }   /* Process Program Available Request */
<NORMAL>"S7F34"     { return S7F34; }   /* Process Program Availability Data */
<NORMAL>"S7F35"     { return S7F35; }   /* Process Program for MID Request */
<NORMAL>"S7F36"     { return S7F36; }   /* Process Program for MID Data */
<NORMAL>"S8F0"      { return S8F0; }    /* Abort Transaction */
<NORMAL>"S8F1"      { return S8F1; }    /* Boot Program Request */
<NORMAL>"S8F2"      { return S8F2; }    /* Boot Program Data */
<NORMAL>"S8F3"      { return S8F3; }    /* Executive Program Request */
<NORMAL>"S8F4"      { return S8F4; }    /* Executive Program Data */
<NORMAL>"S9F0"      { return S9F0; }    /* Abort Transaction */
<NORMAL>"S9F1"      { return S9F1; }    /* Unrecognized Device ID */
<NORMAL>"S9F3"      { return S9F3; }    /* Unrecognized Stream Type */
<NORMAL>"S9F5"      { return S9F5; }    /* Unrecognized Function Type */
<NORMAL>"S9F7"      { return S9F7; }    /* Illegal Data */
<NORMAL>"S9F9"      { return S9F9; }    /* Transaction Timer Timeout */
<NORMAL>"S9F11"     { return S9F11; }   /* Data Too Long */
<NORMAL>"S9F13"     { return S9F13; }   /* Conversation Timeout */
<NORMAL>"S10F0"     { return S10F0; }   /* Abort Transaction */
<NORMAL>"S10F1"     { return S10F1; }   /* Terminal Request */
<NORMAL>"S10F2"     { return S10F2; }   /* Terminal Request Acknowledge */
<NORMAL>"S10F3"     { return S10F3; }   /* Terminal Display, Single */
<NORMAL>"S10F4"     { return S10F4; }   /* Terminal Display, Single Acknowledge */
<NORMAL>"S10F5"     { return S10F5; }   /* Terminal Display, Multi-Block */
<NORMAL>"S10F6"     { return S10F6; }   /* Terminal Display, Multi-Block Acknowledge */
<NORMAL>"S10F7"     { return S10F7; }   /* Multi-Block Not Allowed */
<NORMAL>"S10F9"     { return S10F9; }   /* Broadcast */
<NORMAL>"S10F10"    { return S10F10; }  /* Broadcast Acknowledge */
<NORMAL>"S11F0"     { return S11F0; }   /* Abort Transaction */
<NORMAL>"S12F0"     { return S12F0; }   /* Abort Transaction */
<NORMAL>"S12F1"     { return S12F1; }   /* Map Set-up Data Send */
<NORMAL>"S12F2"     { return S12F2; }   /* Map Set-up Data Acknowledge */
<NORMAL>"S12F3"     { return S12F3; }   /* Map Set-up Data Request */
<NORMAL>"S12F4"     { return S12F4; }   /* Map Set-up Data */
<NORMAL>"S12F5"     { return S12F5; }   /* Map Transmit Inquire */
<NORMAL>"S12F6"     { return S12F6; }   /* Map Transmit Grant */
<NORMAL>"S12F7"     { return S12F7; }   /* Map Data Send Type 1 */
<NORMAL>"S12F8"     { return S12F8; }   /* Map Data Acknowledge Type 1 */
<NORMAL>"S12F9"     { return S12F9; }   /* Map Data Send Type 2 */
<NORMAL>"S12F10"    { return S12F10; }  /* Map Data Acknowledge Type 2 */
<NORMAL>"S12F11"    { return S12F11; }  /* Map Data Send Type 3 */
<NORMAL>"S12F12"    { return S12F12; }  /* Map Data Acknowledge Type 3 */
<NORMAL>"S12F13"    { return S12F13; }  /* Map Data Request Type 1 */
<NORMAL>"S12F14"    { return S12F14; }  /* Map Data Type 1 */
<NORMAL>"S12F15"    { return S12F15; }  /* Map Data Request Type 2 */
<NORMAL>"S12F16"    { return S12F16; }  /* Map Data Type 2 */
<NORMAL>"S12F17"    { return S12F17; }  /* Map Data Request Type 3 */
<NORMAL>"S12F18"    { return S12F18; }  /* Map Data Type 3 */
<NORMAL>"S12F19"    { return S12F19; }  /* Map Error Report Send */
<NORMAL>"S13F0"     { return S13F0; }   /* Abort Transaction */
<NORMAL>"S13F1"     { return S13F1; }   /* Send Data Set Send */
<NORMAL>"S13F2"     { return S13F2; }   /* Send Data Set Acknowledge */
<NORMAL>"S13F3"     { return S13F3; }   /* Open Data Set Request */
<NORMAL>"S13F4"     { return S13F4; }   /* Open Data Set Data */
<NORMAL>"S13F5"     { return S13F5; }   /* Read Data Set Request */
<NORMAL>"S13F6"     { return S13F6; }   /* Read Data Set Data */
<NORMAL>"S13F7"     { return S13F7; }   /* Close Data Set Send */
<NORMAL>"S13F8"     { return S13F8; }   /* Close Data Set Acknowledge */
<NORMAL>"S13F9"     { return S13F9; }   /* Reset Data Set Send */
<NORMAL>"S13F10"    { return S13F10; }  /* Reset Data Set Acknowledge */
<NORMAL>"S13F11"    { return S13F11; }  /* Data Set Object Multi-Block Inquire */
<NORMAL>"S13F12"    { return S13F12; }  /* Data Set Object Multi-Block Grant */
<NORMAL>"S13F13"    { return S13F13; }  /* Table Data Send */
<NORMAL>"S13F14"    { return S13F14; }  /* Table Data Acknowledge */
<NORMAL>"S13F15"    { return S13F15; }  /* Table Data Request */
<NORMAL>"S13F16"    { return S13F16; }  /* Table Data */
<NORMAL>"S14F0"     { return S14F0; }   /* Abort Transaction */
<NORMAL>"S14F1"     { return S14F1; }   /* GetAttr Request */
<NORMAL>"S14F2"     { return S14F2; }   /* GetAttr Data */
<NORMAL>"S14F3"     { return S14F3; }   /* SetAttr Request */
<NORMAL>"S14F4"     { return S14F4; }   /* SetAttr Data */
<NORMAL>"S14F5"     { return S14F5; }   /* GetType Request */
<NORMAL>"S14F6"     { return S14F6; }   /* GetType Data */
<NORMAL>"S14F7"     { return S14F7; }   /* GetAttrName Request */
<NORMAL>"S14F8"     { return S14F8; }   /* GetAttrName Data */
<NORMAL>"S14F9"     { return S14F9; }   /* Create Object Request */
<NORMAL>"S14F10"    { return S14F10; }  /* Create Object Acknowledge */
<NORMAL>"S14F11"    { return S14F11; }  /* Delete Object Request */
<NORMAL>"S14F12"    { return S14F12; }  /* Delete Object Request Acknowledge */
<NORMAL>"S14F13"    { return S14F13; }  /* Object Attach Request */
<NORMAL>"S14F14"    { return S14F14; }  /* Object Attach Acknowledge */
<NORMAL>"S14F15"    { return S14F15; }  /* Attached Object Action Request */
<NORMAL>"S14F16"    { return S14F16; }  /* Attached Object Action Acknowledge */
<NORMAL>"S14F17"    { return S14F17; }  /* Supervised Object Action Request */
<NORMAL>"S14F18"    { return S14F18; }  /* Supervised Object Action Acknowledge */
<NORMAL>"S14F19"    { return S14F19; }  /* Generic Service Request */
<NORMAL>"S14F20"    { return S14F20; }  /* Generic Service Acknowledge */
<NORMAL>"S14F21"    { return S14F21; }  /* Generic Service Completion Onformation */
<NORMAL>"S14F22"    { return S14F22; }  /* Generic Service Completion Acknowledge */
<NORMAL>"S14F23"    { return S14F23; }  /* Multi-Block Generic Service Data Inquire */
<NORMAL>"S14F24"    { return S14F24; }  /* Multi-Block Generic Service Data Grant */
<NORMAL>"S14F25"    { return S14F25; }  /* Get Service Name Request */
<NORMAL>"S14F26"    { return S14F26; }  /* Get Service Name Data */
<NORMAL>"S14F27"    { return S14F27; }  /* Get Service Parameter Name Request */
<NORMAL>"S14F28"    { return S14F28; }  /* Get Service Parameter Name Request */
<NORMAL>"S15F0"     { return S15F0; }   /* Abort Transaction */
<NORMAL>"S15F1"     { return S15F1; }   /* Recipe Management Multi-block Inquire */
<NORMAL>"S15F2"     { return S15F2; }   /* Recipe Management Multi-block Grant */
<NORMAL>"S15F3"     { return S15F3; }   /* Recipe Namespace Action Request */
<NORMAL>"S15F4"     { return S15F4; }   /* Recipe Namespace Action Acknowledge */
<NORMAL>"S15F5"     { return S15F5; }   /* Recipe Namespace Rename Request */
<NORMAL>"S15F6"     { return S15F6; }   /* Recipe Namespace Rename Acknowledge */
<NORMAL>"S15F7"     { return S15F7; }   /* Recipe Space Request */
<NORMAL>"S15F8"     { return S15F8; }   /* Recipe Space Data */
<NORMAL>"S15F9"     { return S15F9; }   /* Recipe Status Request */
<NORMAL>"S15F10"    { return S15F10; }  /* Recipe Status Data */
<NORMAL>"S15F11"    { return S15F11; }  /* Recipe Version Request */
<NORMAL>"S15F12"    { return S15F12; }  /* Recipe Version Data */
<NORMAL>"S15F13"    { return S15F13; }  /* Recipe Create Request */
<NORMAL>"S15F14"    { return S15F14; }  /* Recipe Create Acknowledge */
<NORMAL>"S15F15"    { return S15F15; }  /* Recipe Store Request */
<NORMAL>"S15F16"    { return S15F16; }  /* Recipe Store Acknowledge */
<NORMAL>"S15F17"    { return S15F17; }  /* Recipe Retrieve Request */
<NORMAL>"S15F18"    { return S15F18; }  /* Recipe Retrieve Data */
<NORMAL>"S15F19"    { return S15F19; }  /* Recipe Rename Request */
<NORMAL>"S15F20"    { return S15F20; }  /* Recipe Rename Acknowledge */
<NORMAL>"S15F21"    { return S15F21; }  /* Recipe Action Request */
<NORMAL>"S15F22"    { return S15F22; }  /* Recipe Action Acknowledge */
<NORMAL>"S15F23"    { return S15F23; }  /* Recipe Descriptor Request */
<NORMAL>"S15F24"    { return S15F24; }  /* Recipe Descriptor Data */
<NORMAL>"S15F25"    { return S15F25; }  /* Recipe Parameter Update Request */
<NORMAL>"S15F26"    { return S15F26; }  /* Recipe Parameter Update Acknowledge */
<NORMAL>"S15F27"    { return S15F27; }  /* Recipe Download Request */
<NORMAL>"S15F28"    { return S15F28; }  /* Recipe Download Acknowledge */
<NORMAL>"S15F29"    { return S15F29; }  /* Recipe Verify Request */
<NORMAL>"S15F30"    { return S15F30; }  /* Recipe Verify Acknowledge */
<NORMAL>"S15F31"    { return S15F31; }  /* Recipe Upload Request */
<NORMAL>"S15F32"    { return S15F32; }  /* Recipe Upload Data */
<NORMAL>"S15F33"    { return S15F33; }  /* Recipe Select Request */
<NORMAL>"S15F34"    { return S15F34; }  /* Recipe Select Acknowledge */
<NORMAL>"S15F35"    { return S15F35; }  /* Recipe Delete Request */
<NORMAL>"S15F36"    { return S15F36; }  /* Recipe Delete Acknowledge */
<NORMAL>"S15F37"    { return S15F37; }  /* DRNS Segment Approve Action Request */
<NORMAL>"S15F38"    { return S15F38; }  /* DRNS Segment Approve Action Acknowledge */
<NORMAL>"S15F39"    { return S15F39; }  /* DRNS Recorder Segment Request */
<NORMAL>"S15F40"    { return S15F40; }  /* DRNS Recorder Segment Acknowledge */
<NORMAL>"S15F41"    { return S15F41; }  /* DRNS Recorder Modify Request */
<NORMAL>"S15F42"    { return S15F42; }  /* DRNS Recorder Modify Acknowledge */
<NORMAL>"S15F43"    { return S15F43; }  /* DRNS Get Change Request */
<NORMAL>"S15F44"    { return S15F44; }  /* DRNS Get Change Request Data */
<NORMAL>"S15F45"    { return S15F45; }  /* DRNS Manager Segment Change Approval Request */
<NORMAL>"S15F46"    { return S15F46; }  /* DRNS Manager Segment Change Approval Acknowledge */
<NORMAL>"S15F47"    { return S15F47; }  /* DRNS Manager Rebuild Request */
<NORMAL>"S15F48"    { return S15F48; }  /* DRNS Manager Rebuild Acknowledge */
<NORMAL>"S16F0"     { return S16F0; }   /* Abort Transaction */
<NORMAL>"S16F1"     { return S16F1; }   /* Multi-block Process Job Data Inquire */
<NORMAL>"S16F2"     { return S16F2; }   /* Multi-block Process Job Data Grant */
<NORMAL>"S16F3"     { return S16F3; }   /* Process Job Create Request */
<NORMAL>"S16F4"     { return S16F4; }   /* Process Job Create Acknowledge */
<NORMAL>"S16F5"     { return S16F5; }   /* Process Job Command Request */
<NORMAL>"S16F6"     { return S16F6; }   /* Process Job Command Acknowledge */
<NORMAL>"S16F7"     { return S16F7; }   /* Process Job Alert Notify */
<NORMAL>"S16F8"     { return S16F8; }   /* Process Job Alert Confirm */
<NORMAL>"S16F9"     { return S16F9; }   /* Process Job Event Notify */
<NORMAL>"S16F10"    { return S16F10; }  /* Process Job Event Confirm */
<NORMAL>"S16F11"    { return S16F11; }  /* PRJobCreateEnh */
<NORMAL>"S16F12"    { return S16F12; }  /* PRJobCreateEnh Acknowledge */
<NORMAL>"S16F13"    { return S16F13; }  /* PRJobDuplicateCreate */
<NORMAL>"S16F14"    { return S16F14; }  /* PRJobDuplicateCreate Acknowledge */
<NORMAL>"S16F15"    { return S16F15; }  /* PRJobMultiCreate */
<NORMAL>"S16F16"    { return S16F16; }  /* PRJobMultiCreate Acknowledge */
<NORMAL>"S16F17"    { return S16F17; }  /* PRJobDequeue */
<NORMAL>"S16F18"    { return S16F18; }  /* PRJobDequeue Acknowledge */
<NORMAL>"S16F19"    { return S16F19; }  /* PRGetAllJobs */
<NORMAL>"S16F20"    { return S16F20; }  /* PRGetAllJobs Send */
<NORMAL>"S16F21"    { return S16F21; }  /* PRGetSpace */
<NORMAL>"S16F22"    { return S16F22; }  /* PRGetSpace Send */
<NORMAL>"S16F23"    { return S16F23; }  /* PRJobSetRecipeVariable */
<NORMAL>"S16F24"    { return S16F24; }  /* PRJobSetRecipeVariable Acknowledge */
<NORMAL>"S16F25"    { return S16F25; }  /* PRJobSetStartMethod */
<NORMAL>"S16F26"    { return S16F26; }  /* PRJobSetStartMethod Acknowledge */
<NORMAL>"S16F27"    { return S16F27; }  /* Control Job Command Request */
<NORMAL>"S16F28"    { return S16F28; }  /* Control Job Command Acknowledge */
<NORMAL>"S16F29"    { return S16F29; }  /* PRSetMtrlOrder */
<NORMAL>"S16F30"    { return S16F30; }  /* PRSetMtrlOrder Acknowledge */
<NORMAL>"S17F0"     { return S17F0; }   /* Abort Transaction */
<NORMAL>"S17F1"     { return S17F1; }   /* Data Report Create Request */
<NORMAL>"S17F2"     { return S17F2; }   /* Data Report Create Acknowledge */
<NORMAL>"S17F3"     { return S17F3; }   /* Data Report Delete Request */
<NORMAL>"S17F4"     { return S17F4; }   /* Data Report Delete Acknowledge */
<NORMAL>"S17F5"     { return S17F5; }   /* Trace Create Request */
<NORMAL>"S17F6"     { return S17F6; }   /* Trace Create Acknowledge */
<NORMAL>"S17F7"     { return S17F7; }   /* Trace Delete Request */
<NORMAL>"S17F8"     { return S17F8; }   /* Trace Delete Acknowledge */
<NORMAL>"S17F9"     { return S17F9; }   /* Collection Event Link Request */
<NORMAL>"S17F10"    { return S17F10; }  /* Collection Event Link Acknowledge */
<NORMAL>"S17F11"    { return S17F11; }  /* Collection Event Unlink Request */
<NORMAL>"S17F12"    { return S17F12; }  /* Collection Event Unlink Acknowledge */
<NORMAL>"S17F13"    { return S17F13; }  /* Trace Reset Request */
<NORMAL>"S17F14"    { return S17F14; }  /* Trace Reset Acknowledge */
<NORMAL>"S18F0"     { return S18F0; }   /* Abort Transaction */
<NORMAL>"S18F1"     { return S18F1; }   /* Read Attribute Request */
<NORMAL>"S18F2"     { return S18F2; }   /* Read Attribute Data */
<NORMAL>"S18F3"     { return S18F3; }   /* Write Attribute Request */
<NORMAL>"S18F4"     { return S18F4; }   /* Write Attribute Acknowledge */
<NORMAL>"S18F5"     { return S18F5; }   /* Read Request */
<NORMAL>"S18F6"     { return S18F6; }   /* Read Data */
<NORMAL>"S18F7"     { return S18F7; }   /* Write Data Request */
<NORMAL>"S18F8"     { return S18F8; }   /* Write Data Acknowledge */
<NORMAL>"S18F9"     { return S18F9; }   /* Read ID Request */
<NORMAL>"S18F10"    { return S18F10; }  /* Read ID Data */
<NORMAL>"S18F11"    { return S18F11; }  /* Write ID Request */
<NORMAL>"S18F12"    { return S18F12; }  /* Write ID Acknowledge */
<NORMAL>"S18F13"    { return S18F13; }  /* Subsystem Command Request */
<NORMAL>"S18F14"    { return S18F14; }  /* Subsystem Command Acknowledge */
<NORMAL>{sxxfxx}    {
                        yylval.string = malloc(strlen(yytext)+1);
                        strcpy(yylval.string, yytext);
                        return SXXFXX;
                    }
<NORMAL>{name}      {
                        int token;
                        token = findname(yytext, &yylval);
                        return token;
                    }
<NORMAL>{hex}       {
                        yylval.ival = strtoul(yytext+2, NULL, 16);
                        if (yylval.ival < 256)
                            return HEX;
                        else
                            return UINTEGER;
                    }

<NORMAL>{number}    {
                        if (strlen(yytext) < strlen(MAX_INTSTR))
                        {
                            yylval.uival = strtoul(yytext, NULL, 10);
                            return UINTEGER;
                        }
                        else if ((strlen(yytext) == strlen(MAX_INTSTR)) &&
                                 (strcmp(yytext, MAX_INTSTR) <= 0))
                        {
                            yylval.uival = strtoul(yytext, NULL, 10);
                            return UINTEGER;
                        }
                        else if ((strlen(yytext) == strlen(MAX_UINTSTR)) &&
                                 (strcmp(yytext, MAX_UINTSTR) <= 0))
                        {
                            yylval.uival= strtoul(yytext, NULL, 10);
                            return UINTEGER;
                        }
                        else
                        {
#ifdef _MSC_VER
                            yylval.llval = _atoi64(yytext);
#else
                            yylval.llval = strtoll(yytext, NULL, 10);
#endif
                            return LONGLONG;
                        }
                    }

<NORMAL>{u1}        {
                        yylval.uival = strtoul(yytext, NULL, 10);
                        if (yylval.uival <= UCHAR_MAX)
                            return UINT1;
                        warning("unsigned char range over", NULL);
                    }
<NORMAL>{u2}        {
                        yylval.uival = strtoul(yytext, NULL, 10);
                        if (yylval.uival <= USHRT_MAX)
                            return UINT2;
                        warning("unsigned short integer range over", NULL);
                    }
<NORMAL>{u4}        {
                        yylval.uival = strtoul(yytext, NULL, 10);
                        return UINTEGER;
                    }
<NORMAL>{u8}        {
#ifdef _MSC_VER
                        yylval.ullval = _atoi64(yytext);
#else
                        yylval.ullval = strtoull(yytext, NULL, 10);
#endif
                        return ULONGLONG;
                    }
<NORMAL>{i1}        {
                        yylval.ival = strtol(yytext, NULL, 10);
                        if (yylval.ival <= SCHAR_MAX &&
                            yylval.ival >= SCHAR_MIN)
                            return INT1;
                        warning("signed char range over", NULL);
                    }
<NORMAL>{i2}        {
                        yylval.ival = strtol(yytext, NULL, 10);
                        if (yylval.ival <= SHRT_MAX && yylval.ival >= SHRT_MIN)
                            return INT2;
                        warning("short integer range over", NULL);
                    }
<NORMAL>{i4}        {
                        yylval.ival = strtol(yytext, NULL, 10);
                        if (yylval.ival <= INT_MAX && yylval.ival >= INT_MIN)
                            return INTEGER;
                        warning("integer range over", NULL);
                    }
<NORMAL>{i8}        {
#ifdef _MSC_VER
                        yylval.llval = _atoi64(yytext);
#else
                        yylval.llval = strtoll(yytext, NULL, 10);
#endif
                        return LONGLONG;
                    }

<NORMAL>{integer}   {
                        if (strlen(yytext) < strlen(MAX_INTSTR))
                        {
                            yylval.ival = strtol(yytext, NULL, 10);
                            return INTEGER;
                        }
                        else if ((strlen(yytext) == strlen(MAX_INTSTR)) &&
                                 (strcmp(yytext, MAX_INTSTR) <= 0))
                        {
                            yylval.ival= strtol(yytext, NULL, 10);
                            return INTEGER;
                        }
                        else
                        {
#ifdef _MSC_VER
                            yylval.llval = _atoi64(yytext);
#else
                            yylval.llval = strtoll(yytext, NULL, 10);
#endif
                            return LONGLONG;
                        }
                    }

<NORMAL>{f4}        {
                        sscanf(yytext, "%f", &yylval.fval);
                        return FLOAT;
                    }
<NORMAL>{real}      {
                        yylval.dval = atof(yytext);
                        return REAL;
                    }
<NORMAL>{string}    {
                        int c;
                        int incsize = 128;
                        int total = incsize;
                        int current = 0;
                        yylval.string = malloc(total);
                        c = input();
                        while (c != '"' && c != '\0' && c != '\n')
                        {
                            yylval.string[current++] = c;
                            if (total <= (current + 1))
                            {
                                yylval.string = realloc(yylval.string,
                                                        total + incsize);
                                total += incsize;
                            }
                            c = input();
                        }
                        if (c != '"')
                             warning("Unterminated character string",
                                     (char *)0);
                        yylval.string[current] = '\0';
                        if (Tcl_NumUtfChars(yylval.string, current) == current)
                            return STRING;
                        else
                            return JSTRING;
                    }

<NORMAL>":"         BEGIN LABEL;
<NORMAL>{nl}        { LineNum++; }
<NORMAL>.           { return yytext[0]; }

<LABEL>":"          { BEGIN NORMAL; }
<LABEL>.            ;

%%

#ifdef FLEX_SCANNER
int input_position = -1;
struct {
    char * m_myinputptr;
    char * m_myinputlim;
} save_input[8];

int my_yyinput(char * buf, int max_size)
{
    int n = min(max_size, myinputlim - myinputptr);
    if (n > 0)
    {
        memcpy(buf, myinputptr, n);
        myinputptr += n;
    }
    else
    {
        *buf = '\0';
        if (input_position >= 0)
        {
            myinputptr = save_input[input_position].m_myinputptr;
            myinputlim = save_input[input_position].m_myinputlim;
            input_position--;
            n = min(max_size, myinputlim - myinputptr);
        }
    }
    return n;
}
#endif

int findname(char * yytext, YYSTYPE * valptr)
{
    int len;
    char * valuePtr = Tcl_GetVar(myinterp, yytext+1, 0);
    if (valuePtr == NULL)
        return 0;
    len = strlen(valuePtr);
    if (isdigit(*valuePtr))
    {
        char * dot = strchr(valuePtr, '.');
        if (dot != NULL)
        {
            valptr->dval= atof(valuePtr);
            return REAL;
        }
        else if (*valuePtr == '0' && *(valuePtr+1) == 'x')
        {
            valptr->ival = strtoul(valuePtr, NULL, 16);
            return HEX;
        }
        else
        {
            valptr->ival = strtoul(valuePtr, NULL, 0);
            return INTEGER;
        }
    }
    else if ((*valuePtr == '-' || *valuePtr == '+') && isdigit(*(valuePtr+1)))
    {
        char * dot = strchr(valuePtr, '.');
        if (dot != NULL)
        {
            valptr->dval= atof(valuePtr);
            return REAL;
        }
        else
        {
            valptr->ival = strtol(valuePtr, NULL, 10);
            return INTEGER;
        }
    }
    else if (*valuePtr == '"')
    {
        valptr->string = malloc(len + 1);
        strcpy(valptr->string, valuePtr+1);
        if (valptr->string[len - 2] != '"')
        {
            warning("Unterminated character string", (char *)0);
            return 0;
        }
        else
            valptr->string[len - 2] = '\0';
        return STRING;
    }
    else
    {
        valptr->string = malloc(len + 1);
        strcpy(valptr->string, valuePtr);
        return STRING;
    }
}

/*
 * Reset scanner variables
 */
void un_lex()
{
    BEGIN INITIAL;
    yy_init = 1;        /* whether we need to initialize */
    yy_start = 0;       /* start state number */
    input_position = -1;
}

