/* $Id: SECS.java,v 1.5 2004/06/27 08:01:56 fukasawa Exp $ */

/**
 *  @file    SECS.java
 *
 *  @author  Fukasawa Mitsuo
 *
 *
 *    Copyright (C) 2001-2004 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package com.jyugem.secs;

import java.util.ArrayList;
import java.util.StringTokenizer;
import java.io.UnsupportedEncodingException;

public final class SECS {
    public static final int LIST = 0;
    public static final int BINARY = 010;
    public static final int BOOLEAN = 011;
    public static final int ASCII = 020;
    public static final int JIS = 021;
    public static final int UNICODE = 022;
    public static final int INT1 = 031;
    public static final int INT2 = 032;
    public static final int INT4 = 034;
    public static final int INT8 = 030;
    public static final int UINT1 = 051;
    public static final int UINT2 = 052;
    public static final int UINT4 = 054;
    public static final int UINT8 = 050;
    public static final int FLOAT8 = 040;
    public static final int FLOAT4 = 044;
    public static final int ARRAY = 0100;

    public static String toString(int format) {
        String formatName;
        switch (format) {
        case LIST:    formatName = "list";    break;
        case ASCII:   formatName = "ascii";   break;
        case JIS:     formatName = "jis";     break;
        case UNICODE: formatName = "utf-8";   break;
        case BINARY:  formatName = "binary";  break;
        case BOOLEAN: formatName = "bool";    break;
        case INT1:    formatName = "int1";    break;
        case INT2:    formatName = "int2";    break;
        case INT4:    formatName = "int4";    break;
        case INT8:    formatName = "int8";    break;
        case FLOAT4:  formatName = "float4";  break;
        case FLOAT8:  formatName = "float8";  break;
        case UINT1:   formatName = "uint1";   break;
        case UINT2:   formatName = "uint2";   break;
        case UINT4:   formatName = "uint4";   break;
        case UINT8:   formatName = "uint8";   break;
        case (INT1 | ARRAY):    formatName = "int1[]";    break;
        case (INT2 | ARRAY):    formatName = "int2[]";    break;
        case (INT4 | ARRAY):    formatName = "int4[]";    break;
        case (INT8 | ARRAY):    formatName = "int8[]";    break;
        case (FLOAT4 | ARRAY):  formatName = "float4[]";  break;
        case (FLOAT8 | ARRAY):  formatName = "float8[]";  break;
        case (UINT1 | ARRAY):   formatName = "uint1[]";   break;
        case (UINT2 | ARRAY):   formatName = "uint2[]";   break;
        case (UINT4 | ARRAY):   formatName = "uint4[]";   break;
        case (UINT8 | ARRAY):   formatName = "uint8[]";   break;

        default:
            formatName = null;
            break;
        }
        return formatName;
    }

    public static int toFormat(Object value) throws SECSException {
        int format = (value.getClass().isArray()) ? ARRAY : 0;
        if (value instanceof SECSList) {
            format |= LIST;
        } else if (value instanceof SECSItem) {
            SECSItem item = (SECSItem)value;
            format |= item.type();
        } else if (value instanceof String) {
            format |= ASCII;
        } else if (value instanceof Byte) {
            format |= BINARY;
        } else if (value instanceof Boolean) {
            format |= BOOLEAN;
        } else if (value instanceof Int1) {
            format |= INT1;
        } else if (value instanceof Short) {
            format |= INT2;
        } else if (value instanceof Integer) {
            format |= INT4;
        } else if (value instanceof Long) {
            format |= INT8;
        } else if (value instanceof Float) {
            format |= FLOAT4;
        } else if (value instanceof Double) {
            format |= FLOAT8;
        } else if (value instanceof UInt1) {
            format |= UINT1;
        } else if (value instanceof UShort) {
            format |= UINT2;
        } else if (value instanceof UInteger) {
            format |= UINT4;
        } else if (value instanceof ULong) {
            format |= UINT8;
        } else if (value instanceof Jis) {
            format |= JIS;
        } else if (value instanceof Unicode) {
            format |= UNICODE;
        } else {
            String errmsg = "Unknown value type " + value.getClass().toString();
            throw new SECSException(errmsg);
        }
        return format;
    }

    public static int toFormat(String typeName) throws SECSException {
        String token = typeName.toLowerCase();
        if (token.equals("list")) {
            return LIST;
        } else if (token.equals("ascii")) {
            return ASCII;
        } else if (token.equals("int4")) {
            return INT4;
        } else if (token.equals("uint4")) {
            return UINT4;
        } else if (token.equals("int2")) {
            return INT2;
        } else if (token.equals("uint2")) {
            return UINT2;
        } else if (token.equals("binary")) {
            return BINARY;
        } else if (token.equals("boolean")) {
            return BOOLEAN;
        } else if (token.equals("int8")) {
            return INT8;
        } else if (token.equals("int1")) {
            return INT1;
        } else if (token.equals("float8")) {
            return FLOAT8;
        } else if (token.equals("float4")) {
            return FLOAT4;
        } else if (token.equals("uint8")) {
            return UINT8;
        } else if (token.equals("uint1")) {
            return UINT1;
        } else if (token.equals("jis")) {
            return JIS;
        } else if (token.equals("utf-8")) {
            return UNICODE;
        } else if (token.equals("int1[]")) {
            return (INT1 | ARRAY);
        } else if (token.equals("int2[]")) {
            return (INT2 | ARRAY);
        } else if (token.equals("int4[]")) {
            return (INT4 | ARRAY);
        } else if (token.equals("int8[]")) {
            return (INT8 | ARRAY);
        } else if (token.equals("uint1[]")) {
            return (UINT1 | ARRAY);
        } else if (token.equals("uint2[]")) {
            return (UINT2 | ARRAY);
        } else if (token.equals("uint4[]")) {
            return (UINT4 | ARRAY);
        } else if (token.equals("uint8[]")) {
            return (UINT8 | ARRAY);
        } else if (token.equals("float4[]")) {
            return (FLOAT4 | ARRAY);
        } else if (token.equals("float8[]")) {
            return (FLOAT8 | ARRAY);
        } else {
            String errmsg = "Unknown value type (" + typeName + ")";
            throw new SECSException(errmsg);
        }
    }

    //
    // Parse string to value
    //
    public static Object parseValueOf(int typeOrg, String varStr) throws SECSException {
        int type = typeOrg & 077;   // Reset array bit
        Object valObj = null;
        ArrayList objs = new ArrayList();
        Object aObj;
        try {
            if (typeOrg == BINARY || typeOrg > ARRAY) {
                StringTokenizer tokenizer = new StringTokenizer(varStr);
                while (tokenizer.hasMoreTokens()) {
                    String token = tokenizer.nextToken();
                    aObj = convertValueOf(type, token);
                    objs.add(aObj);
                }
            } else {
                aObj = convertValueOf(type, varStr);
                objs.add(aObj);
            }
        } catch (SECSException e) {
            return null;
        }

        if (typeOrg == BINARY || typeOrg > ARRAY) {
            int dsize = objs.size();
            switch (typeOrg) {
            case SECS.BINARY:
                Byte[] bin = new Byte[dsize];
                for (int i = 0; i < dsize; i++) {
                    bin[i] = (Byte)objs.get(i);
                }
                valObj = bin;
                break;
            case (INT2 | ARRAY):
                Short[] i2v = new Short[dsize];
                for (int i = 0; i < dsize; i++) {
                    i2v[i] = (Short)objs.get(i);
                }
                valObj = i2v;
                break;
            case (INT4 | ARRAY):
                Integer[] i4v = new Integer[dsize];
                for (int i = 0; i < dsize; i++) {
                    i4v[i] = (Integer)objs.get(i);
                }
                valObj = i4v;
                break;
            case (INT8 | ARRAY):
                Long[] i8v = new Long[dsize];
                for (int i = 0; i < dsize; i++) {
                    i8v[i] = (Long)objs.get(i);
                }
                valObj = i8v;
                break;
            case (FLOAT4 | ARRAY):
                Float[] f4v = new Float[dsize];
                for (int i = 0; i < dsize; i++) {
                    f4v[i] = (Float)objs.get(i);
                }
                valObj = f4v;
                break;
            case (FLOAT8 | ARRAY):
                Double[] f8v = new Double[dsize];
                for (int i = 0; i < dsize; i++) {
                    f8v[i] = (Double)objs.get(i);
                }
                valObj = f8v;
                break;
            case (INT1 | ARRAY):
                Int1[] i1v = new Int1[dsize];
                for (int i = 0; i < dsize; i++) {
                    i1v[i] = (Int1)objs.get(i);
                }
                valObj = i1v;
                break;
            case (UINT1 | ARRAY):
                UInt1[] u1v = new UInt1[dsize];
                for (int i = 0; i < dsize; i++) {
                    u1v[i] = (UInt1)objs.get(i);
                }
                valObj = u1v;
                break;
            case (UINT2 | ARRAY):
                UShort[] u2v = new UShort[dsize];
                for (int i = 0; i < dsize; i++) {
                    u2v[i] = (UShort)objs.get(i);
                }
                valObj = u2v;
                break;
            case (UINT4 | ARRAY):
                UInteger[] u4v = new UInteger[dsize];
                for (int i = 0; i < dsize; i++) {
                    u4v[i] = (UInteger)objs.get(i);
                }
                valObj = u4v;
                break;
            case (UINT8 | ARRAY):
                ULong[] u8v = new ULong[dsize];
                for (int i = 0; i < dsize; i++) {
                    u8v[i] = (ULong)objs.get(i);
                }
                valObj = u8v;
                break;
            default:
                throw new SECSException("Not support format." + type);
            }
        } else {
            valObj = objs.get(0);
        }
        return valObj;
    }


    public static Object convertValueOf(int typeOrg, String token)  throws SECSException {
        int type = typeOrg & ~ARRAY;   // Reset array bit
        Object valObj = null;

        try {
            switch (type) {
            case BINARY:
                byte b = Integer.decode(token).byteValue();
                valObj = new Byte(b);
                // valObj = Byte.decode(token);  // min/max error
                break;
            case BOOLEAN:  valObj = Boolean.valueOf(token);  break;
            case ASCII:    valObj = token;                   break;
            case JIS:      valObj = Jis.decode(token);       break;
            case UNICODE:  valObj = Unicode.decode(token);   break;
            case INT1:     valObj = Int1.decode(token);      break;
            case INT2:     valObj = Short.decode(token);     break;
            case INT4:     valObj = Integer.decode(token);   break;
            case INT8:     valObj = Long.decode(token);      break;
            case UINT1:    valObj = UInt1.decode(token);     break;
            case UINT2:    valObj = UShort.decode(token);    break;
            case UINT4:    valObj = UInteger.decode(token);  break;
            case UINT8:    valObj = ULong.decode(token);     break;
            case FLOAT4:   valObj = Float.valueOf(token);    break;
            case FLOAT8:   valObj = Double.valueOf(token);   break;
            default:
                throw new SECSException("Not support format(" + type + ") and token:" + token);
            }
        } catch (UnsupportedEncodingException e) {
            throw new SECSException("Can't Encoding");
        }
        return valObj;
    }

    //
    // Dumper for debug
    //
    static String hexTable[] = {
            "0x00", "0x01", "0x02", "0x03", "0x04", "0x05", "0x06", "0x07",
            "0x08", "0x09", "0x0A", "0x0B", "0x0C", "0x0D", "0x0E", "0x0F",
            "0x10", "0x11", "0x12", "0x13", "0x14", "0x15", "0x16", "0x17",
            "0x18", "0x19", "0x1A", "0x1B", "0x1C", "0x1D", "0x1E", "0x1F",
            "0x20", "0x21", "0x22", "0x23", "0x24", "0x25", "0x26", "0x27",
            "0x28", "0x29", "0x2A", "0x2B", "0x2C", "0x2D", "0x2E", "0x2F",
            "0x30", "0x31", "0x32", "0x33", "0x34", "0x35", "0x36", "0x37",
            "0x38", "0x39", "0x3A", "0x3B", "0x3C", "0x3D", "0x3E", "0x3F",
            "0x40", "0x41", "0x42", "0x43", "0x44", "0x45", "0x46", "0x47",
            "0x48", "0x49", "0x4A", "0x4B", "0x4C", "0x4D", "0x4E", "0x4F",
            "0x50", "0x51", "0x52", "0x53", "0x54", "0x55", "0x56", "0x57",
            "0x58", "0x59", "0x5A", "0x5B", "0x5C", "0x5D", "0x5E", "0x5F",
            "0x60", "0x61", "0x62", "0x63", "0x64", "0x65", "0x66", "0x67",
            "0x68", "0x69", "0x6A", "0x6B", "0x6C", "0x6D", "0x6E", "0x6F",
            "0x70", "0x71", "0x72", "0x73", "0x74", "0x75", "0x76", "0x77",
            "0x78", "0x79", "0x7A", "0x7B", "0x7C", "0x7D", "0x7E", "0x7F",
            "0x80", "0x81", "0x82", "0x83", "0x84", "0x85", "0x86", "0x87",
            "0x88", "0x89", "0x8A", "0x8B", "0x8C", "0x8D", "0x8E", "0x8F",
            "0x90", "0x91", "0x92", "0x93", "0x94", "0x95", "0x96", "0x97",
            "0x98", "0x99", "0x9A", "0x9B", "0x9C", "0x9D", "0x9E", "0x9F",
            "0xA0", "0xA1", "0xA2", "0xA3", "0xA4", "0xA5", "0xA6", "0xA7",
            "0xA8", "0xA9", "0xAA", "0xAB", "0xAC", "0xAD", "0xAE", "0xAF",
            "0xB0", "0xB1", "0xB2", "0xB3", "0xB4", "0xB5", "0xB6", "0xB7",
            "0xB8", "0xB9", "0xBA", "0xBB", "0xBC", "0xBD", "0xBE", "0xBF",
            "0xC0", "0xC1", "0xC2", "0xC3", "0xC4", "0xC5", "0xC6", "0xC7",
            "0xC8", "0xC9", "0xCA", "0xCB", "0xCC", "0xCD", "0xCE", "0xCF",
            "0xD0", "0xD1", "0xD2", "0xD3", "0xD4", "0xD5", "0xD6", "0xD7",
            "0xD8", "0xD9", "0xDA", "0xDB", "0xDC", "0xDD", "0xDE", "0xDF",
            "0xE0", "0xE1", "0xE2", "0xE3", "0xE4", "0xE5", "0xE6", "0xE7",
            "0xE8", "0xE9", "0xEA", "0xEB", "0xEC", "0xED", "0xEE", "0xEF",
            "0xF0", "0xF1", "0xF2", "0xF3", "0xF4", "0xF5", "0xF6", "0xF7",
            "0xF8", "0xF9", "0xFA", "0xFB", "0xFC", "0xFD", "0xFE", "0xFF",
        };

    static public String hexdump(String indent, Object value) {
        String result = "";
        if (value == null) {
            return result;
        } else if (value instanceof Byte[]) {
            Byte[] binary = (Byte[])value;
            if (binary.length == 0) {
                return result;
            }
            if (binary.length == 1) {
                return hexTable[binary[0].byteValue() & 0xFF];  // a byte
            }
            result = "\n" + indent + "  " +
                     hexTable[binary[0].byteValue() & 0xFF] + " ";
            for (int i = 1; i < binary.length; i++) {
                if ((i % 8) == 0) {
                    result += " \n";
                    result += indent + "  ";
                }
                result += hexTable[binary[i].byteValue() & 0xFF];
                result += " ";
            }
            result += "\n";
            result += indent;
        } else if (value instanceof ArrayList) {
            ArrayList list = (ArrayList)value;
            int aLength = list.size();
            if (aLength == 0) {
                return result;
            }
            result = "\n" + indent + "  " + list.get(0).toString() + " ";
            for (int i = 1; i < aLength; i++) {
                if ((i % 4) == 0) {
                    result += " \n";
                    result += indent + "  ";
                }
            }
            result += "\n";
            result += indent;
        } else {
            System.out.println("Illegal data type : Not Byte Array");
            result = indent + value.toString();
        }
        return result;
    }

}
