#include "stdafx.h"

#include "JavaVMMainThread.hpp"

#include "JVMThreadAttacher.hpp"

#include <assert.h>
#include <process.h>

namespace
{
	volatile LONG run_ = 1L;
	volatile LONG lock_ = 0L;

	JVMEventListener* pEventListener_;
}

///////////////

JavaVMMainThread::InvokeInfoRef::InvokeInfoData_::InvokeInfoData_( JavaVMMainThread* v_pThis, const StartupInfo& v_startupInfo, EventRef& v_hStarted ) throw()
	: pThis_( v_pThis )
	, startupInfo_( v_startupInfo )
	, hStarted_( v_hStarted )
{
	assert( v_pThis != NULL );
}

JavaVMMainThread::InvokeInfoRef::InvokeInfoRef( JavaVMMainThread* v_pThis, const StartupInfo& v_startupInfo, EventRef& v_hStarted ) throw()
{
	assert( v_pThis != NULL );

	pInvokeInfoData_ = new InvokeInfoData_( v_pThis, v_startupInfo, v_hStarted );
	while( InterlockedExchange( &lock_, 1L ) != 0L ) {
		Sleep( 0 );
	}
	pInvokeInfoData_->pThis_ = v_pThis;
	pInvokeInfoData_->startupInfo_ = v_startupInfo;
	pInvokeInfoData_->hStarted_ = v_hStarted;
	pInvokeInfoData_->ref_ = 1;
	InterlockedExchange( &lock_, 0L );
}

JavaVMMainThread::InvokeInfoRef::InvokeInfoRef( const InvokeInfoRef& v_other ) throw()
{
	while( InterlockedExchange( &lock_, 1L ) != 0L ) {
		Sleep( 0 );
	}
	__try{
		pInvokeInfoData_ = v_other.pInvokeInfoData_;
		pInvokeInfoData_->ref_++;
	}
	__finally {
		InterlockedExchange( &lock_, 0L );
	}
}

JavaVMMainThread::InvokeInfoRef& JavaVMMainThread::InvokeInfoRef::operator=( const InvokeInfoRef& v_other ) throw()
{
	if( this != &v_other ) {
		while( InterlockedExchange( &lock_, 1L ) != 0L ) {
			Sleep( 0 );
		}
		__try{
			if( --pInvokeInfoData_->ref_ == 0 ) {
				delete pInvokeInfoData_;
			}
			pInvokeInfoData_ = v_other.pInvokeInfoData_;
			pInvokeInfoData_->ref_++;
		}
		__finally {
			InterlockedExchange( &lock_, 0L );
		}
	}
	return *this;
}

JavaVMMainThread::InvokeInfoRef::~InvokeInfoRef() throw()
{
	while( InterlockedExchange( &lock_, 1L ) != 0L ) {
		Sleep( 0 );
	}
	__try{
		if( --pInvokeInfoData_->ref_ == 0 ) {
			delete pInvokeInfoData_;
		}
	}
	__finally {
		InterlockedExchange( &lock_, 0L );
	}
}

void JavaVMMainThread::InvokeInfoRef::setExceptionReason( const std::string& v_exceptionReason ) throw()
{
	pInvokeInfoData_->exceptionReason_ = v_exceptionReason;
}

const StartupInfo& JavaVMMainThread::InvokeInfoRef::getStartupInfo() const throw()
{
	return pInvokeInfoData_->startupInfo_;
}

const std::string& JavaVMMainThread::InvokeInfoRef::getExceptionReason() const throw()
{
	return pInvokeInfoData_->exceptionReason_;
}

JavaVMMainThread* JavaVMMainThread::InvokeInfoRef::getJavaVMMainThread() const throw()
{
	return pInvokeInfoData_->pThis_;
}

void JavaVMMainThread::InvokeInfoRef::signal() throw()
{
	pInvokeInfoData_->hStarted_.signal( true );
}

///////////////

JavaVMMainThread::JavaVMMainThread( JVMEventListener& v_eventListener, const JavaVMStartupInfo& v_jvmStartupInfo )  throw ( std::exception )
	: jvm_( NULL )
	, jvmStartupInfo_( v_jvmStartupInfo )
{
	if( InterlockedIncrement( &run_ ) < 0L ) {
		throw std::runtime_error( "JVMVɍ쐬邱Ƃł܂B" );
	}
	pEventListener_ = &v_eventListener;
}

JavaVMMainThread::~JavaVMMainThread() throw()
{
}

void JavaVMMainThread::start() throw ( std::exception )
{
	//required:
	assert( jvm_ == NULL && "łJVMNĂ܂B" );
	if( jvm_ != NULL ) {
		throw std::logic_error( "łJVMNĂ܂B" );
	}

	//do:
	JavaVMInitArgs vm_args;
	std::vector<JavaVMOption> options( jvmStartupInfo_.size() + 3 );
	int idx=0;
	for( StartupInfo::const_iterator p=jvmStartupInfo_.begin(), last=jvmStartupInfo_.end();
		p != last;
		++p, ++idx)
	{
		const std::string& option = *p;
		options[ idx ].optionString = (char*) option.c_str();
		options[ idx ].extraInfo = NULL;
	}

	options[ idx + 0 ].optionString = "exit";
	options[ idx + 0 ].extraInfo = jvm_exit;
	options[ idx + 1 ].optionString = "abort";
	options[ idx + 1 ].extraInfo = jvm_abort;
	options[ idx + 2 ].optionString = "vfprintf";
	options[ idx + 2 ].extraInfo = jvm_vfprintf;

	vm_args.version = JNI_VERSION_1_2;
	vm_args.options = &options[0];
	vm_args.nOptions = (int) options.size();
	vm_args.ignoreUnrecognized = jvmStartupInfo_.isIgnoreUnrecognized() ? JNI_TRUE : JNI_FALSE;

	JNIEnv* env;

	bool created = false;
	try{
		if( JNI_CreateJavaVM( &jvm_, (void **)&env, &vm_args) == 0 ) {
			created = true;
		}
	}
	catch( ... ) {
		created = false;
	}
	if( ! created ) {
		throw std::runtime_error( "JVM̋NɎs܂B" );
	}
}

bool JavaVMMainThread::invokeClassMethod( const StartupInfo& v_startupInfo, bool v_wait ) throw( std::exception )
{
	//required:
	assert( jvm_ != NULL && "JVMJnĂ܂B" );
	if( jvm_ == NULL ) {
		throw std::logic_error( "JVMJnĂ܂B" );
	}

	//do:
	JNIEnv* env;
	if( jvm_->GetEnv( (void**) &env, JNI_VERSION_1_2 ) != JNI_OK || env == NULL ) {
		throw std::runtime_error( "XbhJVMɃA^b`Ă܂B" );
	}

	EventRef hStarted;
	InvokeInfoRef invokeInfo( this, v_startupInfo, hStarted );

	unsigned int threadid = 0;
	HANDLE hThread = (HANDLE) _beginthreadex(
		NULL,
		0,
		invokeClassMethodProc,
		&invokeInfo,
		0,
		&threadid
		);
	if( hThread == NULL ) {
		throw std::runtime_error( "Xbh쐬ł܂B" );
	}

	hStarted.waitForSignal();

	bool result = true;
	if( v_wait ) {
		WaitForSingleObject( hThread, INFINITE );
		result = invokeInfo.getExceptionReason().empty();
	}
	CloseHandle( hThread );

	return result;
}

void JavaVMMainThread::waitForFinishedAll() throw ( std::exception )
{
	assert( jvm_ != NULL && "JVMJnĂ܂B" );
	if( jvm_ == NULL ) {
		throw std::logic_error( "JVMJnĂ܂B" );
	}

	jvm_->DestroyJavaVM();
}

void JavaVMMainThread::stopInterrupt() throw ( std::exception )
{
	//required:
	assert( jvm_ != NULL && "JVM쐬Ă܂B" );
	if( jvm_ == NULL ) {
		throw std::logic_error( "JVM쐬Ă܂B" );
	}

	//do:
	JVMThreadAttacher jvmAttachThread( jvm_ );
	JNIEnv* env = jvmAttachThread.getJNIEnv();

	jclass systemClass = env->FindClass( "java/lang/System" );
	JVMException::checkAndThrow( env );

	jmethodID exitMethod = env->GetStaticMethodID( systemClass, "exit", "(I)V");
	JVMException::checkAndThrow( env );

	env->CallStaticVoidMethod( systemClass, exitMethod, (jint)0 );
	JVMException::checkAndThrow( env );

	throw std::runtime_error( "stopInterruptɎs܂B" );
}

void __stdcall JavaVMMainThread::jvm_exit( jint v_exitCode )
{
	pEventListener_->notifyJVMExited( v_exitCode );
	exit( v_exitCode );
}

void __stdcall JavaVMMainThread::jvm_abort()
{
	pEventListener_->notifyJVMAborted();
	exit( 3 ); // abort
}

jint __stdcall JavaVMMainThread::jvm_vfprintf( FILE* v_fp, const char* v_format, va_list v_args )
{
	return pEventListener_->notifyJVMLog( v_fp, v_format, v_args );
}


std::string JavaVMMainThread::ConvertToJNIClassName( const std::string& v_className ) throw() {
	std::vector<char> buf( v_className.size() + 1 );
	strcpy( &buf[0], v_className.c_str() );
	for( std::vector<char>::iterator p=buf.begin(), last=buf.end();
		p != last;
		++p)
	{
		if( *p == '.' ) {
			*p = '/';
		}
	}
	return std::string( &buf[0] );
}

unsigned __stdcall JavaVMMainThread::invokeClassMethodProc( void* v_pData )
{
	//required:
	assert( v_pData != NULL );
	InvokeInfoRef invokeInfo = *reinterpret_cast<InvokeInfoRef*>( v_pData );
	JavaVMMainThread* me = invokeInfo.getJavaVMMainThread();
	assert( me != NULL );

	// Xbhҋ@CxgmɃVOiɂȂ邱Ƃۏ؂
	EventSentinel<InvokeInfoRef> sentinel( invokeInfo );

	//do:
	try{
		const StartupInfo& startupInfo = invokeInfo.getStartupInfo();

		JavaVM* jvm = me->jvm_;
		assert( jvm != NULL && "JVM쐬Ă܂B" );
		if( jvm == NULL ) {
			throw std::logic_error( "JVM쐬Ă܂B" );
		}

		// JVMɃXbhA^b`B
		JVMThreadAttacher jvmAttachThread( jvm );
		JNIEnv* env = jvmAttachThread.getJNIEnv();

		const std::string mainClassName = ConvertToJNIClassName( startupInfo.getMainClassName() );
		jclass mainClass = env->FindClass( mainClassName.c_str() );
		JVMException::checkAndThrow( env );

		const std::string mainMethodName = startupInfo.getMainMethodName();
		jmethodID mainMethodID = env->GetStaticMethodID( mainClass, mainMethodName.c_str(), "([Ljava/lang/String;)V");
		JVMException::checkAndThrow( env );

		// NIvV̍\z
		jclass stringClass = env->FindClass( "java/lang/String" );
		JVMException::checkAndThrow( env );

		jobjectArray args = env->NewObjectArray( (jsize) startupInfo.size(), stringClass, NULL );
		JVMException::checkAndThrow( env );

		int idx=0;
		for( StartupInfo::const_iterator p=startupInfo.begin(), last=startupInfo.end();
			p != last;
			++p, ++idx)
		{
			const std::string arg = *p;
			jstring arg1 = env->NewStringUTF( arg.c_str() );
			env->SetObjectArrayElement( args, idx, arg1 );
			JVMException::checkAndThrow( env );
		}

		// ĂяoƂ̑ҋ@
		me->notifyThreadEvent( ThreadEvent_Starting, invokeInfo );
		sentinel.signal();

		// \bh̋N
		env->CallStaticVoidMethod( mainClass, mainMethodID, args );
		JVMException::checkAndThrow( env );
	}
	catch( const std::exception& v_exception ) {
		invokeInfo.setExceptionReason( v_exception.what() );
	}
	catch( ... ) {
		invokeInfo.setExceptionReason( "sȗOɂXbhI܂B" );
	}

	me->notifyThreadEvent( ThreadEvent_Terminated, invokeInfo );
	return 0;
}

void JavaVMMainThread::notifyThreadEvent( ThreadEvent v_eventType, const InvokeInfoRef& v_invokeInfo ) throw()
{
	const std::string methodSig =
		v_invokeInfo.getStartupInfo().getMainClassName() + "#" + 
		v_invokeInfo.getStartupInfo().getMainMethodName() + "(java.lang.String[])";

	if( v_eventType == ThreadEvent_Starting ) {
		// do nothing.
	}
	else if( v_eventType == ThreadEvent_Terminated ) {
		const std::string& exceptionReason = v_invokeInfo.getExceptionReason();
		if( ! exceptionReason.empty() ) {
			pEventListener_->notifyJVMThreadAbnomalTerminated( methodSig, exceptionReason );
		}
		else {
			pEventListener_->notifyJVMThreadNormalyTerminated( methodSig );
		}
	}
}
