package org.maachang.bdb ;

import com.sleepycat.je.DatabaseEntry;

/**
 * Bdb キー要素オブジェクト.
 * 
 * @version 2010/01/05
 * @author masahito suzuki
 * @since Bdb 1.00
 */
public class BdbKeyValue {
    
    private final DatabaseEntry key = new DatabaseEntry() ;
    private final DatabaseEntry value = new DatabaseEntry() ;
    
    /**
     * コンストラクタ.
     */
    public BdbKeyValue() {
        
    }
    
    /**
     * 情報設定.
     * @param key キー情報を設定します.
     * @param value 要素情報を設定します.
     * @return BdbKeyValue オブジェクトが返されます.
     */
    public BdbKeyValue create( byte[] key,byte[] value )
        throws Exception {
        setKey( key ) ;
        setValue( value ) ;
        return this ;
    }
    
    /**
     * 情報クリア.
     * @return BdbKeyValue オブジェクトが返されます.
     */
    public BdbKeyValue clear() {
        this.key.setData( null,0,0 ) ;
        this.key.setPartial( 0,0,false ) ;
        this.value.setData( null,0,0 ) ;
        this.value.setPartial( 0,0,false ) ;
        return this ;
    }
    
    /**
     * キー情報を設定.
     * @param key キー情報を設定します.
     * @return BdbKeyValue オブジェクトが返されます.
     */
    public BdbKeyValue setKey( byte[] key ) {
        this.key.setPartial( 0,0,false ) ;
        if( key == null || key.length <= 0 ) {
            key = null ;
            this.key.setData( null,0,0 ) ;
            return this ;
        }
        this.key.setData( key,0,key.length ) ;
        return this ;
    }
    
    /**
     * 要素情報を設定.
     * @param value 要素情報を設定します.
     * @return BdbKeyValue オブジェクトが返されます.
     */
    public BdbKeyValue setValue( byte[] value ) {
        this.value.setPartial( 0,0,false ) ;
        if( value == null || value.length <= 0 ) {
            this.value.setData( null,0,0 ) ;
            return this ;
        }
        this.value.setData( value,0,value.length ) ;
        return this ;
    }
    
    /**
     * キー情報を取得.
     * @return byte[] キー情報を取得します.
     */
    public byte[] getKey() {
        return key.getData() ;
    }
    
    /**
     * 要素情報を取得.
     * @return byte[] 要素情報を取得します.
     */
    public byte[] getValue() {
        return value.getData() ;
    }
    
    /**
     * キーオブジェクトを取得.
     * @return DatabaseEntry キーオブジェクトが返されます.
     */
    protected DatabaseEntry keyEntry() {
        return key ;
    }
    
    /**
     * 要素オブジェクトを取得.
     * @return DatabaseEntry 要素オブジェクトが返されます.
     */
    protected DatabaseEntry valueEntry() {
        return value ;
    }
    
    /**
     * キー情報が存在するかチェック.
     * @return boolean [true]の場合、存在します.
     */
    public boolean isKey() {
        return key.getData() != null ;
    }
    
    /**
     * 要素情報が存在するかチェック.
     * @return boolean [true]の場合、存在します.
     */
    public boolean isValue() {
        return value.getData() != null ;
    }
    
    /**
     * キーと要素が存在するかチェック.
     * @return boolean [true]の場合、存在します.
     */
    public boolean isKeyValue() {
        return key.getData() != null &&
            value.getData() != null ;
    }
}
