package org.maachang.bdb ;

import java.util.Iterator;
import java.util.concurrent.locks.Lock;

import com.sleepycat.je.Database;

/**
 * Berkeley DB wrapper.
 *  
 * @version 2010/01/02
 * @author  masahito suzuki
 * @since   bdb 1.0.0
 */
public interface Bdb {
    
    /**
     * オブジェクトをクローズ.
     * @exception Exception 例外.
     */
    public void close() throws Exception ;
    
    /**
     * Berkeley DB データベースオブジェクトを取得.
     * @return Database Berkeley DB データベースオブジェクトが返されます.
     * @exception Exception 例外.
     */
    public Database database() throws Exception ;
    
    /**
     * データ追加.
     * @param o 追加条件を設定します.
     * @exception Exception 例外.
     */
    public void put( BdbKeyValue o ) throws Exception ;
    
    /**
     * データ部分書き換え.
     * @param o 追加条件を設定します.
     * @param offset 対象のオフセットを設定します.
     * @param length 対象の長さを設定します.
     * @exception Exception 例外.
     */
    public void put( BdbKeyValue o,int offset,int length ) throws Exception ;
    
    /**
     * データ削除.
     * @param o 削除条件を設定します.
     * @return boolean [true]の場合、データが削除できました.
     * @exception Exception 例外.
     */
    public boolean remove( BdbKeyValue o ) throws Exception ;
    
    /**
     * データ取得.
     * @param o 取得条件を設定します.
     * @return boolean [true]の場合、データが取得できました.
     * @exception Exception 例外.
     */
    public boolean get( BdbKeyValue o ) throws Exception ;
    
    /**
     * データ部分取得.
     * @param o 取得条件を設定します.
     * @param offset 対象のオフセットを設定します.
     * @param length 対象の長さを設定します.
     * @return boolean [true]の場合、データが取得できました.
     * @exception Exception 例外.
     */
    public boolean get( BdbKeyValue o,int offset,int length ) throws Exception ;
    
    /**
     * 情報を取得して削除.
     * @param o 取得条件を設定します.
     * @return boolean [true]の場合、正常に処理されました.
     * @exception Exception 例外.
     */
    public boolean getToRemove( BdbKeyValue o ) throws Exception ;
    
    /**
     * データ存在確認.
     * @param o チェック条件を設定します.
     * @return boolean [true]の場合、キー情報は存在します.
     * @exception Exception 例外.
     */
    public boolean containsKey( BdbKeyValue o ) throws Exception ;
    
    /**
     * キーイテレータを取得.
     * @return Iterator<BdbKeyValue> キーイテレータが返されます.
     * @exception Exception 例外.
     */
    public Iterator<byte[]> keys() throws Exception ;
    
    /**
     * キーイテレータを取得.
     * @param o 取得に対するオブジェクトを設定します.
     * @return Iterator<BdbKeyValue> キーイテレータが返されます.
     * @exception Exception 例外.
     */
    public Iterator<byte[]> keys( BdbKeyValue o ) throws Exception ;
    
    /**
     * キー要素イテレータを取得.
     * @return Iterator<BdbKeyValue> キーイテレータが返されます.
     * @exception Exception 例外.
     */
    public Iterator<BdbKeyValue> keyValues() throws Exception ;
    
    /**
     * キー要素イテレータを取得.
     * @param o 取得に対するオブジェクトを設定します.
     * @return Iterator<BdbKeyValue> キーイテレータが返されます.
     * @exception Exception 例外.
     */
    public Iterator<BdbKeyValue> keyValues( BdbKeyValue o ) throws Exception ;
    
    /**
     * データ数を取得.
     * @return long データ数が返されます.
     * @exception Exception 例外.
     */
    public long size() throws Exception ;
    
    /**
     * オープン名を取得.
     * @return String オープン名が返されます.
     * @exception Exception 例外.
     */
    public String getName() throws Exception ;
    
    /**
     * 読み込み用同期オブジェクトを取得.
     * @return Lock 読み込み用同期オブジェクトが返されます.
     */
    public Lock getRSync() ;
    
    /**
     * 書き込み用同期オブジェクトを取得.
     * @return Lock 書き込み用同期オブジェクトが返されます.
     */
    public Lock getWSync() ;
    
    /**
     * BdbKeyValueオブジェクトを取得.
     * <p>※このオブジェクトは、ThreadLocalから取得されているので、Threadセーフです.</p>
     * @return BdbKeyValue BdbKeyValueオブジェクトが返されます.
     */
    public BdbKeyValue getBdbKeyValue() ;
    
    /**
     * オブジェクトがクローズされているかチェック.
     * @return boolean [true]の場合、クローズされています.
     */
    public boolean isClose() ;
}

