package org.maachang.html;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

/**
 * HTML-Tag情報.
 * 
 * @version 2009/02/13
 * @author  masahito suzuki
 * @since   SimpleHtmlParser 1.0.0
 */
public class HtmlTag implements HtmlElement {
    private String tagName = null ;
    private List<HtmlTagElement> elements = null ;
    private boolean endFlag = false ;
    private boolean startEndFlag = false ;
    private String id = null ;
    private int listNo = -1 ;
    private Html parent = null ;
    
    /**
     * コンストラクタ.
     */
    private HtmlTag() {
        
    }
    
    /**
     * コンストラクタ.
     * @param startEndFlag 閉じタグを省略する場合は、trueを設定します.
     * @param endFlag このタグを閉じタグにする場合は、trueを設定します.
     * @param tagName 対象のタグ名を設定します.
     * @exception IOException I/O例外.
     */
    public HtmlTag( boolean startEndFlag,boolean endFlag,String tagName ) throws IOException {
        this( startEndFlag,endFlag,tagName,null ) ;
    }
    
    /**
     * コンストラクタ.
     * @param startEndFlag 閉じタグを省略する場合は、trueを設定します.
     * @param endFlag このタグを閉じタグにする場合は、trueを設定します.
     * @param tagName 対象のタグ名を設定します.
     * @param elements タグ要素群を設定します.
     * @exception IOException I/O例外.
     */
    public HtmlTag( boolean startEndFlag,boolean endFlag,String tagName,List<HtmlTagElement> elements )
        throws IOException {
        if( tagName == null || ( tagName = HtmlUtil.trim( tagName ) ).length() < 0 ) {
            throw new IOException( "不正なタグ名が設定されました" ) ;
        }
        if( endFlag ) {
            startEndFlag = false ;
        }
        this.tagName = tagName.toLowerCase() ;
        this.endFlag = endFlag ;
        this.startEndFlag = startEndFlag ;
        if( !endFlag ) {
            this.elements = elements ;
            checkId( elements ) ;
        }
    }
    
    /**
     * タグシンボル内容を追加.
     * @param simbol 対象のタグシンボル内容を追加します.
     */
    public void add( String simbol ) {
        if( !endFlag ) {
            if( simbol == null || ( simbol = HtmlUtil.trim( simbol ) ).length() <= 0 ) {
                return ;
            }
            add( new HtmlTagSimbol( simbol ) ) ;
        }
    }
    
    /**
     * タグ要素を追加.
     * @param key 対象のKey名を設定します.
     * @param value 対象の要素を追加します.
     */
    public void add( String key,String value ) {
        if( !endFlag ) {
            if( key == null || ( key = HtmlUtil.trim( key ) ).length() <= 0 ) {
                return ;
            }
            add( new HtmlTagKeyValue( key,value ) ) ;
        }
    }
    
    /**
     * HTMLタグ要素を追加.
     * @param element HTMLタグ要素を追加します.
     */
    public void add( HtmlTagElement element ) {
        if( !endFlag ) {
            if( element == null ) {
                return ;
            }
            if( elements == null ) {
                elements = new ArrayList<HtmlTagElement>() ;
            }
            if( element instanceof HtmlTagKeyValue ) {
                HtmlTagKeyValue kv = (HtmlTagKeyValue)element ;
                if( "id".equals( kv.getKey() ) ) {
                    String tagId = kv.getValue() ;
                    if( tagId != null && tagId.length() > 0 ) {
                        this.id = tagId ;
                    }
                }
                addTo( kv ) ;
            }
            else {
                elements.add( element ) ;
            }
        }
    }
    
    /**
     * KeyValue条件に対して、以前に存在する内容があった場合は、上書き(styleや、Javascriptイベント以外).
     */
    private void addTo( HtmlTagKeyValue tkv ) {
        int len = elements.size() ;
        String key = tkv.getKey() ;
        for( int i = 0 ; i < len ; i ++ ) {
            HtmlTagElement em = elements.get( i ) ;
            if( em instanceof HtmlTagKeyValue ) {
                HtmlTagKeyValue kv = (HtmlTagKeyValue)em ;
                if( tkv.equals( kv.getKey() ) ) {
                    if( key.equals( "style" ) || key.startsWith( "on" ) ) {
                        String v = kv.getValue() ;
                        kv.set( key,v+";"+tkv.getValue() ) ;
                    }
                    else {
                        kv.set( key,tkv.getValue() ) ;
                    }
                    return ;
                }
            }
        }
        // 同一条件が存在しない場合は、直接追加.
        elements.add( tkv ) ;
    }
    
    /**
     * 指定項番のHTMLタグ要素を取得.
     * @param no 対象の項番を設定します.
     * @return HtmlTagElement HTMLタグ要素が返されます.
     */
    public HtmlTagElement get( int no ) {
        if( !endFlag ) {
            if( elements == null || no <= -1 || no >= elements.size() ) {
                return null ;
            }
            return elements.get( no ) ;
        }
        return null ;
    }
    
    /**
     * 指定項番のHTMLタグ要素を取得.
     * @param name Key名または、シンボル名を設定します.
     * @return HtmlTagElement HTMLタグ要素が返されます.
     */
    public HtmlTagElement get( String name ) {
        return get( name,0 ) ;
    }
    
    /**
     * 指定項番のHTMLタグ要素を取得.
     * @param name Key名または、シンボル名を設定します.
     * @param off 検索開始オフセット値を設定します.
     * @return HtmlTagElement HTMLタグ要素が返されます.
     */
    public HtmlTagElement get( String name,int off ) {
        if( !endFlag ) {
            return get( search( name,off ) ) ;
        }
        return null ;
    }
    
    /**
     * 指定項番のHTMLタグ要素を削除.
     * @param no 対象の項番を設定します.
     * @return HtmlTagElement 削除されたHTMLタグ要素が返されます.
     */
    public HtmlTagElement remove( int no ) {
        if( !endFlag ) {
            if( elements == null || no <= -1 || no >= elements.size() ) {
                return null ;
            }
            return elements.remove( no ) ;
        }
        return null ;
    }
    
    /**
     * 指定項番のHTMLタグ要素を削除.
     * @param name Key名または、シンボル名を設定します.
     * @return HtmlTagElement 削除されたHTMLタグ要素が返されます.
     */
    public HtmlTagElement remove( String name ) {
        return remove( name,0 ) ;
    }
    
    /**
     * 指定項番のHTMLタグ要素を削除.
     * @param name Key名または、シンボル名を設定します.
     * @param off 検索開始オフセット値を設定します.
     * @return HtmlTagElement 削除されたHTMLタグ要素が返されます.
     */
    public HtmlTagElement remove( String name,int off ) {
        if( !endFlag ) {
            return remove( search( name,off ) ) ;
        }
        return null ;
    }
    
    /**
     * 子要素をHTMLで直接追加.
     * @param html 設定対象のHTML情報を設定します.
     * @exception IOException I/O例外.
     */
    public void addHTML( String html ) throws IOException {
        if( html == null || ( html = html.trim() ).length() <= 0 ) {
            return ;
        }
        if( listNo >= 0 && parent != null ) {
            HtmlTag t = parent.getEndTag( this ) ;
            if( t == null ) {
                return ;
            }
            parent.insert( t.getListNo(),html ) ;
        }
    }
    
    /**
     * 子要素をHTMLで直接設定.
     * @param html 設定対象のHTML情報を設定します.
     * @exception IOException I/O例外.
     */
    public void innerHTML( String html ) throws IOException {
        if( html == null || ( html = html.trim() ).length() <= 0 ) {
            return ;
        }
        if( listNo >= 0 && parent != null ) {
            HtmlTag t = parent.getEndTag( this ) ;
            if( t == null ) {
                return ;
            }
            parent.remove( this ) ;
            parent.insert( listNo+1,html ) ;
        }
    }
    
    /**
     * 子要素をHTMLで取得.
     * @return String 子要素情報が返されます.
     */
    public String innerHTML() {
        if( listNo >= 0 && parent != null ) {
            HtmlTag t = parent.getEndTag( this ) ;
            if( t == null ) {
                return "" ;
            }
            StringBuilder buf = new StringBuilder() ;
            int len = (t.getListNo() - listNo) - 1 ;
            int off = listNo + 1 ;
            for( int i = 0 ; i < len ; i ++ ) {
                parent.get( i+off ).toString( buf ) ;
            }
            return buf.toString() ;
        }
        return "" ;
    }
    
    /**
     * 現在のタグパスを取得.
     * @return String タグパスが返されます.
     */
    public String getPath() {
        if( listNo >= 0 && parent != null ) {
            int len = listNo ;
            StackObject lst = new StackObject() ;
            for( int i = 0 ; i < len ; i ++ ) {
                HtmlElement em = parent.get( i ) ;
                if( em instanceof HtmlTag ) {
                    HtmlTag tg = (HtmlTag)em ;
                    if( tg.isEndTag() ) {
                        HtmlTag st = ( HtmlTag )lst.peek() ;
                        if( parent.getEndTag( st ) == tg ) {
                            lst.pop() ;
                        }
                    }
                    else if( parent.getEndTag( tg ) != null ) {
                        lst.push( tg ) ;
                    }
                }
            }
            ArrayList<HtmlTag> ast = new ArrayList<HtmlTag>() ;
            while( true ) {
                Object o = lst.pop() ;
                if( o == null ) {
                    break ;
                }
                ast.add( (HtmlTag)o ) ;
            }
            lst = null ;
            if( ast.size() <= 0 ) {
                return "/"+tagName ;
            }
            StringBuilder buf = new StringBuilder() ;
            len = ast.size() ;
            buf.append( "/" ) ;
            for( int i = len-1 ; i >= 0 ; i -- ) {
                buf.append( ast.get( i ).getName() ).append( "/" ) ;
            }
            buf.append( tagName ) ;
            return buf.toString() ;
        }
        return "" ;
    }
    
    /**
     * HTMLタグ要素数を取得.
     * @return int HTMLタグ要素数が返されます.
     */
    public int size() {
        if( !endFlag ) {
            return ( elements == null ) ? 0 : elements.size() ;
        }
        return 0 ;
    }
    
    /**
     * タグ名を取得.
     * @return String タグ名が返されます.
     */
    public String getName() {
        return tagName ;
    }
    
    /**
     * タグIDを取得.
     * @return String タグIDが返されます.
     */
    public String getId() {
        return id ;
    }
    
    /**
     * このタグが閉じタグかチェック.
     * @return boolean [true]の場合、閉じタグです.
     */
    public boolean isEndTag() {
        return endFlag ;
    }
    
    /**
     * このタグが、閉じタグが省略されたタグかチェック.
     * @return boolean [true]の場合、閉じタグが省略されたタグです.
     */
    public boolean isStartEnd() {
        return startEndFlag ;
    }
    
    /**
     * 現在の格納位置を項番で取得.
     * @return int 現在の格納位置が返されます.
     */
    public int getListNo() {
        return listNo ;
    }
    
    /**
     * 現在の格納位置を項番で設定.
     * @param no 現在の格納位置を設定します.
     */
    public void setListNo( int no ) {
        listNo = no ;
    }
    
    /**
     * 親オブジェクトを設定.
     * @param parent 対象の親オブジェクトを設定します.
     */
    protected void setParent( Html parent ) {
        this.parent = parent ;
    }
    
    /**
     * 親オブジェクトを取得.
     * @return Html 親オブジェクトが返されます.
     */
    protected Html getParent() {
        return this.parent ;
    }
    
    /**
     * StringBuilderに内容を出力.
     * @param buf 対象のStringBuilderを設定します.
     */
    public void toString( StringBuilder buf ) {
        buf.append( "<" ) ;
        if( endFlag ) {
            buf.append( "/" ).append( tagName ).append( ">" ) ;
        }
        else {
            buf.append( tagName ) ;
            int len = elements.size() ;
            for( int i = 0 ; i < len ; i ++ ) {
                buf.append( " " ) ;
                elements.get( i ).toString( buf ) ;
            }
            if( startEndFlag ) {
                buf.append( " / " ) ;
            }
            buf.append( ">" ) ;
        }
    }
    
    /**
     * StringBuilderに内容を出力.
     * @param buf 対象のStringBuilderを設定します.
     */
    public void toSmart( StringBuilder buf ) {
        buf.append( "<" ) ;
        if( endFlag ) {
            buf.append( "/" ).append( tagName ).append( ">" ) ;
        }
        else {
            buf.append( tagName ) ;
            int len = elements.size() ;
            for( int i = 0 ; i < len ; i ++ ) {
                buf.append( " " ) ;
                HtmlTagElement em = elements.get( i ) ;
                if( em instanceof HtmlTagKeyValue ) {
                    ((HtmlTagKeyValue)em).toSmart( buf ) ;
                }
                else {
                    em.toString( buf ) ;
                }
            }
            if( startEndFlag ) {
                buf.append( " / " ) ;
            }
            buf.append( ">" ) ;
        }
    }
    
    /**
     * 文字列として出力.
     * @return String 文字列が返されます.
     */
    public String toString() {
        StringBuilder buf = new StringBuilder() ;
        toString( buf ) ;
        return buf.toString() ;
    }
    
    /**
     * 指定タグ内容をコンバート.
     * @param string １つのタグ内容を設定します.
     * @return HtmlTag 変換されたHTMLタグが返されます.
     * @exception IOException I/O例外.
     */
    public static final HtmlTag convert( String string ) throws IOException {
        if( string == null || ( string = HtmlUtil.trim( string ) ).length() <= 0 ) {
            throw new IOException( "不正なタグ条件が存在します" ) ;
        }
        if( string.startsWith( "<" ) && string.endsWith( ">" ) ) {
            string = string.substring( 1,string.length() - 1 ) ;
        }
        string = HtmlUtil.trim( string ) ;
        int[] oPos = new int[ 1 ] ;
        oPos[0] = 0 ;
        String tag = HtmlAnalysis.getTagName( oPos,string ) ;
        boolean end = false ;
        boolean startEnd = false ;
        int p = string.indexOf( "/" ) ;
        if( p != -1 && p < oPos[ 0 ] ) {
            end = true ;
        }
        if( string.endsWith( "/" ) ) {
            if( end ) {
                throw new IOException( "不正なタグ条件が存在します" ) ;
            }
            startEnd = true ;
            string = string.substring( 0,string.length()-1 ) ;
        }
        if( oPos[ 0 ] > 0 ) {
            string = string.substring( oPos[0] ) ;
        }
        oPos = null ;
        List<HtmlTagElement> em = new ArrayList<HtmlTagElement>() ;
        HtmlAnalysis.cutElement( em,string ) ;
        return new HtmlTag( startEnd,end,tag,em ) ;
    }
    
    private void checkId( List<HtmlTagElement> elements ) {
        if( elements != null && elements.size() > 0 ) {
            int len = elements.size() ;
            for( int i = 0 ; i < len ; i ++ ) {
                HtmlTagElement element = elements.get( i ) ;
                if( element instanceof HtmlTagKeyValue ) {
                    HtmlTagKeyValue kv = (HtmlTagKeyValue)element ;
                    if( "id".equals( kv.getKey() ) ) {
                        this.id = kv.getValue() ;
                    }
                }
            }
        }
    }
    
    private int search( String key,int off ) {
        if( elements == null || key == null || ( key = HtmlUtil.trim( key ) ).length() <= 0 ) {
            return -1 ;
        }
        if( off >= elements.size() ) {
            return -1 ;
        }
        if( off <= 0 ) {
            off = 0 ;
        }
        key = key.toLowerCase() ;
        int len = elements.size() ;
        for( int i = off ; i < len ; i ++ ) {
            if( key.equals( elements.get( i ).get() ) ) {
                return i ;
            }
        }
        return -1 ;
    }
}
