package org.maachang.html;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * HTML情報.
 * 
 * @version 2009/02/13
 * @author  masahito suzuki
 * @since   SimpleHtmlParser 1.0.0
 */
public class Html {
    private final Map<String,HtmlElement> tagId = new HashMap<String,HtmlElement>() ;
    private final HtmlElementList list = new HtmlElementList() ;
    
    /**
     * コンストラクタ.
     */
    public Html() {
        
    }
    
    /**
     * コンストラクタ.
     * @param html 解析対象のHTMLを設定します.
     * @exception IOException I/O例外.
     */
    public Html( String html ) throws IOException {
        this.create( html ) ;
    }
    
    /**
     * 空のHTML情報を生成.
     */
    public void create() {
        this.clear() ;
    }
    
    /**
     * 指定HTMLを解析して生成.
     * @param html 解析対象のHTMLを設定します.
     * @exception IOException I/O例外.
     */
    public void create( String html ) throws IOException {
        if( html == null || ( html = html.trim() ).length() <= 0 ) {
            this.create() ;
            return ;
        }
        this.clear() ;
        HtmlAnalysis.analysis( -1,this,tagId,list,html ) ;
    }
    
    /**
     * 情報クリア.
     */
    public void clear() {
        list.clear() ;
        tagId.clear() ;
    }
    
    /**
     * 要素追加.
     * @param element 対象の要素を追加します.
     */
    public void add( HtmlElement element ) {
        if( element == null ) {
            return ;
        }
        list.add( element ) ;
        addTag( element ) ;
    }
    
    /**
     * HTML定義を追加.
     * @param html 追加対象のHTML情報を設定します.
     * @exception IOException I/O例外.
     */
    public void add( String html )
        throws IOException {
        if( html == null || ( html = html.trim() ).length() <= 0 ) {
            return ;
        }
        HtmlAnalysis.analysis( -1,this,tagId,list,html ) ;
    }
    
    /**
     * 要素設定.
     * @param no 設定対象の項番を設定します.
     * @param element 対象の要素を設定します.
     */
    public void set( int no,HtmlElement element ) {
        if( no <= -1 || no >= list.size() || element == null ) {
            return ;
        }
        HtmlElement em = list.set( no,element ) ;
        addTag( element ) ;
        removeTag( em ) ;
    }
    
    /**
     * 要素設定.
     * @param tag 設定対象のタグを設定します.
     * @param element 対象の要素を設定します.
     */
    public void set( HtmlTag tag,HtmlElement element ) {
        if( tag == null ) {
            return ;
        }
        set( tag.getListNo(),element ) ;
    }
    
    /**
     * HTML定義を設定.
     * @param no 設定対象の項番を設定します.
     * @param html 設定対象のHTML情報を設定します.
     * @exception IOException I/O例外.
     */
    public void set( int no,String html )
        throws IOException {
        if( no <= -1 || no >= list.size() ||
            html == null || ( html = html.trim() ).length() <= 0 ) {
            remove( no ) ;
            return ;
        }
        remove( no ) ;
        HtmlAnalysis.analysis( no,this,tagId,list,html ) ;
    }
    
    /**
     * 要素設定.
     * @param tag 設定対象のタグを設定します.
     * @param html 設定対象のHTML情報を設定します.
     * @exception IOException I/O例外.
     */
    public void set( HtmlTag tag,String html )
        throws IOException {
        if( tag == null ) {
            return ;
        }
        set( tag.getListNo(),html ) ;
    }
    
    /**
     * 要素を指定位置の間に追加.
     * @param no 間に追加したい項番を設定します.
     * @param element 設定対象の要素情報を設定します.
     */
    public void insert( int no,HtmlElement element ) {
        if( no <= -1 || no >= list.size() || element == null ) {
            return ;
        }
        no ++ ;
        if( no >= list.size() ) {
            add( element ) ;
        }
        else {
            list.insert( no,element ) ;
            addTag( element ) ;
        }
    }
    
    /**
     * HTML定義を指定位置の間に追加.
     * @param tag 間に追加したいタグオブジェクトを設定します.
     * @param element 設定対象の要素情報を設定します.
     */
    public void insert( HtmlTag tag,HtmlElement element ) {
        if( tag == null ) {
            return ;
        }
        insert( tag.getListNo(),element ) ;
    }
    
    /**
     * HTML定義を指定位置の間に追加.
     * @param no 間に追加したい項番を設定します.
     * @param html 設定対象のHTML情報を設定します.
     * @exception IOException I/O例外.
     */
    public void insert( int no,String html )
        throws IOException {
        if( no <= -1 || no >= list.size() || 
            html == null || ( html = html.trim() ).length() <= 0 ) {
            return ;
        }
        no ++ ;
        if( no >= list.size() ) {
            add( html ) ;
        }
        else {
            HtmlAnalysis.analysis( no,this,tagId,list,html ) ;
        }
    }
    
    /**
     * HTML定義を指定位置の間に追加.
     * @param tag 間に追加したいタグオブジェクトを設定します.
     * @param html 設定対象のHTML情報を設定します.
     * @exception IOException I/O例外.
     */
    public void insert( HtmlTag tag,String html )
        throws IOException {
        if( tag == null ) {
            return ;
        }
        insert( tag.getListNo(),html ) ;
    }
    
    /**
     * 指定位置の要素を削除.
     * @param no 削除位置を設定します.
     */
    public void remove( int no ) {
        if( no <= -1 || no >= list.size() ) {
            return ;
        }
        HtmlElement em = list.remove( no ) ;
        removeTag( em ) ;
    }
    
    /**
     * 指定タグ内の子要素を削除.
     * @param tag 対象の開始タグを設定します.
     */
    public void remove( HtmlTag tag ) {
        HtmlTag t = getEndTag( tag ) ;
        if( t == null && t.getListNo() <= -1 ) {
            return ;
        }
        int len = (t.getListNo() - tag.getListNo()) - 1 ;
        int off = tag.getListNo() + 1 ;
        for( int i = 0 ; i < len ; i ++ ) {
            remove( off ) ;
        }
    }
    
    /**
     * 情報を取得.
     * @param no 対象の項番を設定します.
     * @return HtmlElement 対象の要素が返されます.
     */
    public HtmlElement get( int no ) {
        if( no <= -1 || no >= list.size() ) {
            return null ;
        }
        return list.get( no ) ;
    }
    
    /**
     * 指定IDでタグ要素を取得.
     * @param id 対象のIDを設定します.
     * @return HtmlTag 一致したタグ要素が返されます.
     */
    public HtmlTag getElementById( String id ) {
        if( id == null || id.length() <= 0 ) {
            return null ;
        }
        HtmlElement em = tagId.get( id ) ;
        if( em == null || !(em instanceof HtmlTag) ) {
            return null ;
        }
        return (HtmlTag)em ;
    }
    
    /**
     * 指定したタグ名内容を取得.
     * @param tagName 対象のタグ名を設定します.
     * @return HtmlTag 対象のHTMLタグが返されます.
     */
    public HtmlTag getElementsByTagName( String tagName ) {
        int n = searchTag( tagName,0 ) ;
        if( n <= -1 ) {
            return null ;
        }
        return ( HtmlTag )list.get( n ) ;
    }
    
    /**
     * 指定したタグ名内容を取得.
     * @param tagName 対象のタグ名を設定します.
     * @param off 対象のオフセット値を設定します.
     * @return HtmlTag 対象のHTMLタグが返されます.
     */
    public HtmlTag getElementsByTagName( String tagName,int off ) {
        int n = searchTag( tagName,off ) ;
        if( n <= -1 ) {
            return null ;
        }
        return ( HtmlTag )list.get( n ) ;
    }
    
    /**
     * 指定したタグ名内容を取得.
     * @param tag 前回取得したHtmlタグを設定します.
     * @return HtmlTag 対象のHTMLタグが返されます.
     */
    public HtmlTag getElementsByTagName( HtmlTag tag ) {
        if( tag == null ) {
            return null ;
        }
        int off = tag.getListNo() ;
        String tagName = tag.getName() ;
        if( off <= -1 ) {
            off = 0 ;
        }
        else {
            off ++ ;
        }
        int n = searchTag( tagName,off ) ;
        if( n <= -1 ) {
            return null ;
        }
        return ( HtmlTag )list.get( n ) ;
    }
    
    /**
     * 指定したタグの終了タグを取得.
     * @param tag 対象の開始タグを設定します.
     * @return HtmlTag 終了タグ位置が返されます.
     */
    public HtmlTag getEndTag( HtmlTag tag ) {
        if( tag == null ) {
            return null ;
        }
        int off = tag.getListNo() ;
        if( off <= -1 ) {
            return null ;
        }
        int n = searchEndTag( off ) ;
        if( n <= -1 ) {
            return null ;
        }
        return ( HtmlTag )list.get( n ) ;
    }
    
    /**
     * 要素数を取得.
     * @return int 要素数が返されます.
     */
    public int size() {
        return list.size() ;
    }
    
    /**
     * ID一覧を取得.
     * @return List<String> ID一覧が返されます.
     */
    public List<String> getIds() {
        if( tagId.size() <= 0 ) {
            return new ArrayList<String>() ;
        }
        Object[] o = tagId.keySet().toArray() ;
        int len ;
        if( o == null || ( len = o.length ) <= 0 ) {
            return new ArrayList<String>() ;
        }
        List<String> ret = new ArrayList<String>() ;
        for( int i = 0 ; i < len ; i ++ ) {
            ret.add( (String)o[ i ] ) ;
        }
        Object[] lst = ret.toArray() ;
        ret.clear() ;
        Arrays.sort( lst ) ;
        len = lst.length ;
        for( int i = 0 ; i < len ; i ++ ) {
            ret.add( (String)lst[ i ] ) ;
        }
        return ret ;
    }
    
    /**
     * ID一覧を取得.
     * @param head ヘッダ名を設定します.
     * @return List<String> ID一覧が返されます.
     */
    public List<String> getIds( String head ) {
        if( head == null || ( head = head.trim() ).length() <= 0 ) {
            return getIds() ;
        }
        if( tagId.size() <= 0 ) {
            return new ArrayList<String>() ;
        }
        Object[] o = tagId.keySet().toArray() ;
        int len ;
        if( o == null || ( len = o.length ) <= 0 ) {
            return new ArrayList<String>() ;
        }
        List<String> ret = new ArrayList<String>() ;
        for( int i = 0 ; i < len ; i ++ ) {
            String s = (String)o[ i ] ;
            if( s.startsWith( head ) ) {
                ret.add( s ) ;
            }
        }
        Object[] lst = ret.toArray() ;
        ret.clear() ;
        Arrays.sort( lst ) ;
        len = lst.length ;
        for( int i = 0 ; i < len ; i ++ ) {
            ret.add( (String)lst[ i ] ) ;
        }
        return ret ;
    }
    
    /**
     * Iteratorを取得.
     * @return Iterator Iteratorが返されます.
     */
    public Iterator iterator() {
        return tagId.keySet().iterator() ;
    }
    
    /**
     * HTML内容に変換.
     * @return String HTML内容が返されます.
     */
    public String toString() {
        int len = list.size() ;
        if( len <= 0 ) {
            return "" ;
        }
        StringBuilder buf = new StringBuilder() ;
        for( int i = 0 ; i < len ; i ++ ) {
            if( i != 0 ) {
                buf.append( "\n" ) ;
            }
            list.get( i ).toString( buf ) ;
        }
        return buf.toString() ;
    }
    
    /**
     * HTML内容を圧縮変換して出力.
     * @return String 圧縮変換された内容が返されます.
     */
    public String toSmart() {
        int len = list.size() ;
        if( len <= 0 ) {
            return "" ;
        }
        StringBuilder buf = new StringBuilder() ;
        int type = 0 ;
        int cnt = 0 ;
        for( int i = 0 ; i < len ; i ++ ) {
            HtmlElement em = list.get( i ) ;
            if( type == 0 ) {
                if( em instanceof HtmlView ) {
                    smart( buf,(HtmlView)em ) ;
                }
                else if( em instanceof HtmlTag ) {
                    ((HtmlTag)em).toSmart( buf ) ;
                    HtmlTag tg = (HtmlTag)em ;
                    if( !tg.isEndTag() && !tg.isStartEnd() ) {
                        String name = tg.getName() ;
                        if( "pre".equals( name ) ) {
                            type = 1 ;
                            cnt = 1 ;
                        }
                        else if( "textarea".equals( name ) ) {
                            type = 2 ;
                            cnt = 1 ;
                        }
                        else if( "script".equals( name ) ) {
                            type = 10 ;
                            cnt = 1 ;
                        }
                        else if( "style".equals( name ) ) {
                            type = 11 ;
                            cnt = 1 ;
                        }
                    }
                }
            }
            else {
                if( em instanceof HtmlTag ) {
                    ((HtmlTag)em).toSmart( buf ) ;
                    HtmlTag tg = (HtmlTag)em ;
                    int ap = 0 ;
                    if( !tg.isEndTag() && !tg.isStartEnd() ) {
                        ap = 1 ;
                    }
                    else if( tg.isEndTag() ) {
                        ap = -1 ;
                    }
                    boolean flg = false ;
                    String name = tg.getName() ;
                    if( type == 1 && "pre".equals( name ) ) {
                        cnt += ap ;
                        flg = true ;
                    }
                    else if( type == 2 && "textarea".equals( name ) ) {
                        cnt += ap ;
                        flg = true ;
                    }
                    else if( type == 10 && "script".equals( name ) ) {
                        cnt += ap ;
                        flg = true ;
                    }
                    else if( type == 11 && "style".equals( name ) ) {
                        cnt += ap ;
                        flg = true ;
                    }
                    if( flg && cnt <= 0 ) {
                        type = 0 ;
                    }
                }
                else if( em instanceof HtmlView || ( em instanceof HtmlComment && type >= 10 ) ) {
                    em.toString( buf ) ;
                }
            }
        }
        return buf.toString() ;
    }
    
    /**
     * HTML内容を整頓して出力.
     * @return String 整頓されたHTMLが返されます.
     */
    public String toTidy() {
        return toTidy( 4 ) ;
    }
    
    /**
     * HTML内容を整頓して出力.
     * @param indent スペースを付加するインデント数値を設定します.
     * @return String 整頓されたHTMLが返されます.
     */
    public String toTidy( int indent ) {
        if( indent <= 0 ) {
            return toString() ;
        }
        int len = list.size() ;
        if( len <= 0 ) {
            return "" ;
        }
        int spaceLen = 0 ;
        StringBuilder buf = new StringBuilder() ;
        for( int i = 0 ; i < len ; i ++ ) {
            int addLen = 0 ;
            if( i != 0 ) {
                buf.append( "\n" ) ;
            }
            HtmlElement em = list.get( i ) ;
            if( em instanceof HtmlTag ) {
                HtmlTag tg = ( HtmlTag )em ;
                if( tg.isEndTag() ) {
                    spaceLen -= indent ;
                    if( spaceLen <= 0 ) {
                        spaceLen = 0 ;
                    }
                }
                else if( tg.isStartEnd() == false && getEndTag( tg ) != null ) {
                    addLen = indent ;
                }
            }
            for( int j = 0 ; j < spaceLen ; j ++ ) {
                buf.append( " " ) ;
            }
            spaceLen += addLen ;
            em.toString( buf ) ;
        }
        return buf.toString() ;
    }
    
    /**
     * 指定タグ名を検索.
     * @param tagName 対象のタグ名を設定します.
     * @param off 対象のオフセット値を設定します.
     * @return int 一致した要素項番が返されます.<br>
     *      見つからない場合は、-1が返されます.
     */
    public int searchTag( String tagName,int off ) {
        if( tagName == null || ( tagName = HtmlUtil.trim( tagName ) ).length() <= 0 ) {
            return -1 ;
        }
        int len = list.size() ;
        if( off >= len ) {
            return -1 ;
        }
        if( off <= 0 ) {
            off = 0 ;
        }
        tagName = tagName.toLowerCase() ;
        for( int i = off ; i < len ; i ++ ) {
            HtmlElement e = list.get( i ) ;
            if( e instanceof HtmlTag ) {
                if( tagName.equals( ((HtmlTag)e).getName() ) ) {
                    return i ;
                }
            }
        }
        return -1 ;
    }
    
    /**
     * IDで検索.
     * @param id 対象のIDを設定します.
     * @return int 一致した要素項番が返されます.<br>
     *      見つからない場合は、-1が返されます.
     */
    public int searchId( String id ) {
        if( id == null || id.length() <= 0 ) {
            return -1 ;
        }
        HtmlElement em = tagId.get( id ) ;
        if( em == null ) {
            return -1 ;
        }
        return em.getListNo() ;
    }
    
    /**
     * 指定開始タグに対する終了タグを検索.
     * @param no 開始タグを示す項番を設定します.
     * @return int 終了タグ位置が返されます.<br>
     *      見つからない場合は、-1が返されます.
     */
    public int searchEndTag( int no ) {
        int len = list.size() ;
        if( no >= len ) {
            return -1 ;
        }
        if( no <= 0 ) {
            no = 0 ;
        }
        String target = null ;
        HtmlElement e = list.get( no ) ;
        if( e instanceof HtmlTag ) {
            HtmlTag tg = (HtmlTag)e ;
            if( tg.isEndTag() || tg.isStartEnd() ) {
                return -1 ;
            }
            target = tg.getName() ;
        }
        else {
            return -1 ;
        }
        int tNo = 1 ;
        for( int i = no+1 ; i < len ; i ++ ) {
            e = list.get( i ) ;
            if( e instanceof HtmlTag ) {
                HtmlTag tg = (HtmlTag)e ;
                if( target.equals( tg.getName() ) ) {
                    if( tg.isStartEnd() ) {
                        continue ;
                    }
                    else if( tg.isEndTag() ) {
                        tNo -- ;
                        if( tNo <= 0 ) {
                            return i ;
                        }
                    }
                    else {
                        tNo ++ ;
                    }
                }
            }
        }
        return -1 ;
    }
    
    private void addTag( HtmlElement em ) {
        if( em != null && em instanceof HtmlTag ) {
            HtmlTag tg = ( HtmlTag )em ;
            if( tg.getId() != null ) {
                tagId.put( tg.getId(),tg ) ;
            }
            tg.setParent( this ) ;
        }
    }
    
    private void removeTag( HtmlElement em ) {
        if( em != null && em instanceof HtmlTag ) {
            HtmlTag tg = ( HtmlTag )em ;
            if( tg.getId() != null ) {
                tagId.remove( tg.getId() ) ;
            }
            tg.setParent( null ) ;
        }
    }
    
    private void smart( StringBuilder buf,HtmlView view ) {
        String s = view.get() ;
        if( s == null || s.length() <= 0 ) {
            return ;
        }
        else if( s.startsWith( "<" ) && s.endsWith( ">" ) ) {
            buf.append( s ) ;
            return ;
        }
        int len = s.length() ;
        int b = 0 ;
        for( int i = 0 ; i < len ; i ++ ) {
            char c = s.charAt( i ) ;
            if( c == '\r' || c == '\n' || c == '\t' || ( c == ' ' && b == c ) ) {
                b = c ;
                continue ;
            }
            buf.append( c ) ;
            b = c ;
        }
    }
}
