package org.maachang.luxio ;

import java.io.IOException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

/**
 * トランザクション対応Lux&nbsp;IO.
 * <br>
 * Lux&nbsp;IOに対して、トランザクションに対応したオブジェクト.<br>
 * 利用方法は、トランザクション対応したいLuxIoImplをTransactionLuxIoオブジェクトに
 * 設定するだけで、あとは、通常のLuxIoと同じように利用可能.<br>
 * ただし、このオブジェクト自体から、LuxIoImplを破棄しないので、プログラム終了など
 * 対象のLux&nbsp;IOを利用しなくなった場合は、LuxIoImpl#destroy()を直接実行する必要がある.
 * <br>
 * 使い方としては、下記のように行うことで、トランザクション対応としてLux&nbsp;IOが利用できる.
 * <pre>
 * public static final void main( String[] args ) throws Exception {
 *     LuxIoImpl impl = new LuxIoImpl( "fileName" ) ;
 *     try {
 *         TransactionLuxIo tran = new TransactionLuxIo( impl ) ;
 *         tran.put( "test","hoge".getBytes() ) ;
 *         byte[] b = tran.get( "test" ) ;
 *         System.out.println( "[T]test=" + new String( b ) ) ;
 *         // ロールバック.
 *         tran.rollback() ;
 *         if( ( b = tran.get( "test" ) ) == null ) {
 *             System.out.println( "[R]test=なし" ) ;
 *         } else {
 *             System.out.println( "[R]test=" + new String( b ) ) ;
 *         }
 *         tran.clear() ;
 *     } finally {
 *         if( impl != null ) { impl.destroy() ; }
 *     }
 * }
 * </pre>
 * 実行結果は[LuxIoImpl impl = new LuxIoImpl( "fileName" ) ;]でbidxを新規作成した場合、
 * 以下の結果となる.<br>
 * <br>
 * >[T]test=hoge<br>
 * >[R]test=なし
 * 
 * @version 2008/12/05
 * @author  masahito suzuki
 * @since   Lux IO 0.1.0
 */
public class TransactionLuxIo implements LuxIo {
    
    private static final int TYPE_PUT = 0 ;
    private static final int TYPE_REMOVE = 1 ;
    
    /**
     * LuxIo.
     */
    private LuxIo luxio = null ;
    
    /**
     * トランザクション情報.
     */
    private final Map<String,TransElement> trans = new HashMap<String,TransElement>() ;
    
    /**
     * コンストラクタ.
     */
    public TransactionLuxIo() {
        
    }
    
    /**
     * コンストラクタ.
     * @param luxio LuxIoオブジェクトを設定します.
     * @exception Exception 例外.
     */
    public TransactionLuxIo( LuxIo luxio ) throws Exception {
        create( luxio ) ;
    }
    
    /**
     * デストラクタ.
     */
    protected void finalize() throws Exception {
        clear() ;
    }
    
    /**
     * オブジェクト生成.
     * @param luxio LuxIoオブジェクトを設定します.
     * @exception Exception 例外.
     */
    public void create( LuxIo luxio ) throws Exception {
        if( luxio == null || luxio.isUse() == false ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        if( luxio instanceof TransactionLuxIo ) {
            throw new IOException( "TransactionLuxIoはTransactionLuxIoに対して設定できません" ) ;
        }
        this.clear() ;
        this.luxio = luxio ;
    }
    
    /**
     * オブジェクトクリア.
     */
    public void clear() {
        trans.clear() ;
        luxio = null ;
    }
    
    /**
     * オブジェクトが有効かチェック.
     * @return boolean [true]の場合、有効です.
     */
    public boolean isUse() {
        return luxio != null && luxio.isUse() ;
    }
    
    /**
     * コミット処理.
     * @exception Exception 例外.
     */
    public void commit() throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        try {
            int len = trans.size() ;
            TransElement[] lst = new TransElement[ len ] ;
            Iterator it = trans.keySet().iterator() ;
            int cnt = 0 ;
            while( it.hasNext() ) {
                lst[ cnt ] = trans.get( ( String )it.next() ) ;
                cnt ++ ;
            }
            Arrays.sort( lst ) ;
            for( int i = 0 ; i < len ; i ++ ) {
                TransElement em = lst[ i ] ;
                if( em != null ) {
                    switch( em.getType() ) {
                        case TYPE_PUT :
                            luxio.put( em.getKey(),em.getValue() ) ;
                            break ;
                        case TYPE_REMOVE :
                            luxio.remove( em.getKey() ) ;
                            break ;
                    }
                }
            }
        } finally {
            trans.clear() ;
        }
    }
    
    /**
     * ロールバック処理.
     * @exception Exception 例外.
     */
    public void rollback() throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        trans.clear() ;
    }
    
    /**
     * 情報をセット.
     * @param key 対象のキーを設定します.
     * @param value 対象の要素を設定します.
     * @exception Exception 例外.
     */
    public void put( String key,byte[] value )
        throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        if( key == null || key.length() <= 0 ||
            value == null || value.length <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        TransElement em = new TransElement( trans.size(),TYPE_PUT,key,value ) ;
        trans.put( key,em ) ;
    }
    
    /**
     * 情報を取得.
     * @param key 対象のキーを設定します.
     * @return byte[] 対象の要素が返されます.
     * @exception Exception 例外.
     */
    public byte[] get( String key ) throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        if( key == null || key.length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        byte[] ret = null ;
        TransElement em = trans.get( key ) ;
        if( em != null ) {
            ret = em.getValue() ;
        }
        else {
            ret = luxio.get( key ) ;
        }
        return ret ;
    }
    
    /**
     * 情報を削除.
     * @param key 対象のキーを設定します.
     * @exception Exception 例外.
     */
    public void remove( String key ) throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        if( key == null || key.length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        TransElement em = new TransElement( trans.size(),TYPE_REMOVE,key,null ) ;
        trans.put( key,em ) ;
    }
    
}

/**
 * １つのトランザクション命令.
 */
class TransElement implements Comparable<TransElement> {
    
    int no = -1 ;
    int type = -1 ;
    private String key = null ;
    private byte[] value = null ;
    
    public TransElement( int no,int type,String key,byte[] value ) {
        this.no = no ;
        this.type = type ;
        this.key = key ;
        this.value = value ;
    }
    
    public int getType() {
        return type ;
    }
    
    public String getKey() {
        return key ;
    }
    
    public byte[] getValue() {
        return value ;
    }
    
    public int getNo() {
        return no ;
    }
    
    public int compareTo( TransElement o ) {
        // 低いほうを先頭に(降順).
        return no - o.no ;
    }
}
