package org.maachang.luxio ;

import java.io.IOException;
import java.io.Serializable;

/**
 * Lux&nbsp;IO操作オブジェクト.
 * <BR>
 * Lux&nbsp;IOを利用する場合は、主にこのオブジェクトを利用する.<br>
 * 基本的にLuxOpでは、<br>
 * <br>
 * &nbsp;&nbsp;・put/getでのboolean,int,long,float,double,String,Serializable,byte[]等が利用可能.
 * <br>
 * &nbsp;&nbsp;・再利用可能なオブジェクト方式.
 * <br>
 * &nbsp;&nbsp;・トランザクションサポート.
 * <br><br>
 * を基本構成として作成されたオブジェクトであり、Lux&nbsp;IOをJava用のDBMとして利用するときに適している.
 * 具体的なLuxOpの利用方法については、以下のサンプルを参考.
 * <pre>
 * public static final void main( String[] args ) throws Exception {
 *     int len = 1000000 ;
 *     LuxIoImpl lux = null ;
 *     LuxOp op = null ;
 *     try {
 *         lux = new LuxIoImpl( "test" ) ;
 *         op = new LuxOp( false,lux ) ;
 *         String[] keys = new String[len] ;
 *         System.out.println( "**** init ****" ) ;
 *         long time = System.currentTimeMillis() ;
 *         for( int i = 0 ; i < len ; i ++ ) {
 *             String k = new StringBuilder().
 *                 append( "00000000".substring( String.valueOf( i ).length() ) ).
 *                 append( String.valueOf( i ) ).toString() ;
 *             keys[ i ] = k ;
 *         }
 *         time = System.currentTimeMillis() - time ;
 *         //System.out.println( "init:" + time + "msec" ) ;
 *         System.out.println( "**** start ****" ) ;
 *         time = System.currentTimeMillis() ;
 *         for( int i = 0 ; i < len ; i ++ ) {
 *             op.put( keys[ i ],i ) ;
 *         }
 *         time = System.currentTimeMillis() - time ;
 *         System.out.println( "write:" + time + "msec" ) ;
 *         
 *         time = System.currentTimeMillis() ;
 *         for( int i = 0 ; i < len ; i ++ ) {
 *             //int x = op.getInteger( keys[ i ] ) ;
 *             //System.out.println( x ) ;
 *          op.getInteger( keys[ i ] ) ;
 *         }
 *         op.commit() ;
 *         op.clear() ;
 *         time = System.currentTimeMillis() - time ;
 *         System.out.println( "read:" + time + "msec" ) ;
 *     } finally {
 *         if( lux != null ) {
 *             lux.destroy() ;
 *         }
 *         lux = null ;
 *     }
 * }
 * </pre>
 * 
 * @version 2008/12/05
 * @author  masahito suzuki
 * @since   Lux IO 0.1.0
 */
public class LuxOp {
    
    /**
     * LuxIo.
     */
    private LuxIo luxio = null ;
    
    /**
     * transactionLuxIo.
     */
    private final TransactionLuxIo tran = new TransactionLuxIo() ;
    
    /**
     * コンストラクタ.
     */
    public LuxOp() {
        
    }
    
    /**
     * コンストラクタ.
     * @param luxio 対象のLuxIoオブジェクトを設定します.
     * @exception Exception 例外.
     */
    public LuxOp( LuxIo luxio ) throws Exception {
        this.create( luxio ) ;
    }
    
    /**
     * コンストラクタ.
     * @param transaction [true]の場合、トランザクションは有効になります.
     * @param luxio LuxIoオブジェクトを設定します.<br>
     *  また、transaction == trueの場合、luxioにTransactionLuxIoを設定した場合、
     *  IOExceptionが発生します.
     * @exception Exception 例外.
     */
    public LuxOp( boolean transaction,LuxIo luxio ) throws Exception {
        this.create( transaction,luxio ) ;
    }
    
    /**
     * デストラクタ.
     */
    protected void finalize() throws Exception {
        clear() ;
    }
    
    /**
     * オブジェクト生成.
     * @param luxio LuxIoオブジェクトを設定します.
     * @exception Exception 例外.
     */
    public void create( LuxIo luxio ) throws Exception {
        create( false,luxio ) ;
    }
    
    /**
     * オブジェクト生成.
     * @param transaction [true]の場合、トランザクションは有効になります.
     * @param luxio LuxIoオブジェクトを設定します.<br>
     *  また、transaction == trueの場合、luxioにTransactionLuxIoを設定した場合、
     *  IOExceptionが発生します.
     * @exception Exception 例外.
     */
    public void create( boolean transaction,LuxIo luxio ) throws Exception {
        if( luxio == null || luxio.isUse() == false ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        this.clear() ;
        if( transaction ) {
            if( luxio instanceof TransactionLuxIo ) {
                throw new IOException( "トランザクションモードONに対して、" +
                    "Transaction用Luxioは設定できません" ) ;
            }
            this.tran.create( luxio ) ;
            this.luxio = this.tran ;
        }
        else {
            this.luxio = luxio ;
        }
    }
    
    /**
     * オブジェクトクリア.
     */
    public void clear() {
        luxio = null ;
        if( tran.isUse() ) {
            try {
                tran.commit() ;
            } catch( Exception e ) {
            }
        }
        tran.clear() ;
    }
    
    /**
     * オブジェクトが有効かチェック.
     * @return boolean [true]の場合、有効です.
     */
    public boolean isUse() {
        return luxio != null && luxio.isUse() ;
    }
    
    /**
     * LuxIoオブジェクトを取得.
     * <BR>
     * @return LuxIo LuxIoオブジェクトが返されます.
     */
    public LuxIo luxIo() {
        return this.luxio ;
    }
    
    /**
     * コミット処理.
     * <BR>
     * @exception Exception 例外.
     */
    public void commit() throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        if( luxio instanceof TransactionLuxIo ) {
            ( ( TransactionLuxIo )luxio ).commit() ;
        }
    }
    
    /**
     * ロールバック処理.
     * <BR>
     * @exception Exception 例外.
     */
    public void rollback() throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        if( luxio instanceof TransactionLuxIo ) {
            ( ( TransactionLuxIo )luxio ).rollback() ;
        }
    }
    
    /**
     * 情報を設定.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @param value 対象の情報を設定します.
     * @exception Exception 例外.
     */
    public void put( String key,byte[] value ) throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        if( key == null || key.length() <= 0 ||
            value == null || value.length <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        this.luxio.put( key,value ) ;
    }
    
    /**
     * 情報を設定.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @param value 対象の情報を設定します.
     * @exception Exception 例外.
     */
    public void put( String key,Boolean value ) throws Exception {
        if( value == null ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        put( key,value.booleanValue() ) ;
    }
    
    /**
     * 情報を設定.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @param value 対象の情報を設定します.
     * @exception Exception 例外.
     */
    public void put( String key,boolean value ) throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        if( key == null || key.length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        byte[] b = new byte[]{(byte)((value)?1:0)} ;
        this.luxio.put( key,b ) ;
    }
    
    /**
     * 情報を設定.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @param value 対象の情報を設定します.
     * @exception Exception 例外.
     */
    public void put( String key,Integer value ) throws Exception {
        if( value == null ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        put( key,value.intValue() ) ;
    }
    
    /**
     * 情報を設定.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @param value 対象の情報を設定します.
     * @exception Exception 例外.
     */
    public void put( String key,int value ) throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        if( key == null || key.length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        byte[] b = new byte[]{
            (byte)(value&0x000000ff),
            (byte)((value&0x0000ff00)>>8),
            (byte)((value&0x00ff0000)>>16),
            (byte)((value&0xff000000)>>24) } ;
        this.luxio.put( key,b ) ;
    }
    
    /**
     * 情報を設定.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @param value 対象の情報を設定します.
     * @exception Exception 例外.
     */
    public void put( String key,Long value ) throws Exception {
        if( value == null ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        put( key,value.longValue() ) ;
    }
    
    /**
     * 情報を設定.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @param value 対象の情報を設定します.
     * @exception Exception 例外.
     */
    public void put( String key,long value ) throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        if( key == null || key.length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        byte[] b = new byte[]{
            (byte)(value&0x00000000000000ffL),
            (byte)((value&0x000000000000ff00L)>>8L),
            (byte)((value&0x0000000000ff0000L)>>16L),
            (byte)((value&0x00000000ff000000L)>>24L),
            (byte)((value&0x000000ff00000000L)>>32L),
            (byte)((value&0x0000ff0000000000L)>>40L),
            (byte)((value&0x00ff000000000000L)>>48L),
            (byte)((value&0xff00000000000000L)>>56L) } ;
        this.luxio.put( key,b ) ;
    }
    
    /**
     * 情報を設定.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @param value 対象の情報を設定します.
     * @exception Exception 例外.
     */
    public void put( String key,Float value ) throws Exception {
        if( value == null ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        put( key,value.floatValue() ) ;
    }
    
    /**
     * 情報を設定.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @param value 対象の情報を設定します.
     * @exception Exception 例外.
     */
    public void put( String key,float value ) throws Exception {
        put( key,Float.floatToIntBits( value ) ) ;
    }
    
    /**
     * 情報を設定.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @param value 対象の情報を設定します.
     * @exception Exception 例外.
     */
    public void put( String key,Double value ) throws Exception {
        if( value == null ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        put( key,value.doubleValue() ) ;
    }
    
    /**
     * 情報を設定.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @param value 対象の情報を設定します.
     * @exception Exception 例外.
     */
    public void put( String key,double value ) throws Exception {
        put( key,Double.doubleToLongBits( value ) ) ;
    }
    
    /**
     * 情報を設定.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @param value 対象の情報を設定します.
     * @exception Exception 例外.
     */
    public void put( String key,String value ) throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        if( key == null || key.length() <= 0 ||
            value == null || value.length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        this.luxio.put( key,value.getBytes() ) ;
    }
    
    /**
     * 情報を設定.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @param value 対象の情報を設定します.
     * @exception Exception 例外.
     */
    public void put( String key,Serializable value ) throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        if( key == null || key.length() <= 0 ||
            value == null ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        this.luxio.put( key,LuxioUtil.toBinary( value ) ) ;
    }
    
    /**
     * 情報を削除.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @exception Exception 例外.
     */
    public void remove( String key ) throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        if( key == null || key.length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        this.luxio.remove( key ) ;
    }
    
    /**
     * 情報を取得.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @return byte[] 対象の情報が返されます.
     * @exception Exception 例外.
     */
    public byte[] get( String key ) throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        if( key == null || key.length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        return this.luxio.get( key ) ;
    }
    
    /**
     * 情報を取得.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @return Boolean 対象の情報が返されます.
     * @exception Exception 例外.
     */
    public Boolean getBooleanObject( String key ) throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        if( key == null || key.length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        byte[] b = this.luxio.get( key ) ;
        if( b == null || b.length <= 0 ) {
            return null ;
        }
        return new Boolean( cbool( b ) ) ;
    }
    
    /**
     * 情報を取得.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @return boolean 対象の情報が返されます.
     * @exception Exception 例外.
     */
    public boolean getBoolean( String key ) throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        if( key == null || key.length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        byte[] b = this.luxio.get( key ) ;
        if( b == null || b.length <= 0 ) {
            return false ;
        }
        return cbool( b ) ;
    }
    
    /**
     * 情報を取得.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @return Integer 対象の情報が返されます.
     * @exception Exception 例外.
     */
    public Integer getIntegerObject( String key ) throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        if( key == null || key.length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        byte[] b = this.luxio.get( key ) ;
        if( b == null || b.length != 4 ) {
            return null ;
        }
        return new Integer( cint( b ) ) ;
    }
    
    /**
     * 情報を取得.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @return int 対象の情報が返されます.
     * @exception Exception 例外.
     */
    public int getInteger( String key ) throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        if( key == null || key.length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        byte[] b = this.luxio.get( key ) ;
        if( b == null || b.length != 4 ) {
            return -1 ;
        }
        return cint( b ) ;
    }
    
    /**
     * 情報を取得.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @return Long 対象の情報が返されます.
     * @exception Exception 例外.
     */
    public Long getLongObject( String key ) throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        if( key == null || key.length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        byte[] b = this.luxio.get( key ) ;
        if( b == null || b.length != 8 ) {
            return null ;
        }
        return new Long( clong( b ) ) ;
    }
    
    /**
     * 情報を取得.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @return long 対象の情報が返されます.
     * @exception Exception 例外.
     */
    public long getLong( String key ) throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        if( key == null || key.length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        byte[] b = this.luxio.get( key ) ;
        if( b == null || b.length != 8 ) {
            return -1L ;
        }
        return clong( b ) ;
    }
    
    /**
     * 情報を取得.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @return Float 対象の情報が返されます.
     * @exception Exception 例外.
     */
    public Float getFloatObject( String key ) throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        if( key == null || key.length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        byte[] b = this.luxio.get( key ) ;
        if( b == null || b.length != 4 ) {
            return null ;
        }
        return new Float( Float.intBitsToFloat( cint( b ) ) ) ;
    }
    
    /**
     * 情報を取得.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @return float 対象の情報が返されます.
     * @exception Exception 例外.
     */
    public float getFloat( String key ) throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        if( key == null || key.length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        byte[] b = this.luxio.get( key ) ;
        if( b == null || b.length != 4 ) {
            return -1.0f ;
        }
        return Float.intBitsToFloat( cint( b ) ) ;
    }
    
    /**
     * 情報を取得.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @return Double 対象の情報が返されます.
     * @exception Exception 例外.
     */
    public Double getDoubleObject( String key ) throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        if( key == null || key.length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        byte[] b = this.luxio.get( key ) ;
        if( b == null || b.length != 8 ) {
            return null ;
        }
        return new Double( Double.longBitsToDouble( clong( b ) ) ) ;
    }
    
    /**
     * 情報を取得.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @return double 対象の情報が返されます.
     * @exception Exception 例外.
     */
    public double getDouble( String key ) throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        if( key == null || key.length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        byte[] b = this.luxio.get( key ) ;
        if( b == null || b.length != 8 ) {
            return -1.0d ;
        }
        return Double.longBitsToDouble( clong( b ) ) ;
    }
    
    /**
     * 情報を取得.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @return String 対象の情報が返されます.
     * @exception Exception 例外.
     */
    public String getString( String key ) throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        if( key == null || key.length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        byte[] b = this.luxio.get( key ) ;
        if( b == null ) {
            return null ;
        }
        return new String( b ) ;
    }
    
    /**
     * 情報を取得.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @return Serializable 対象の情報が返されます.
     * @exception Exception 例外.
     */
    public Serializable getObject( String key ) throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        if( key == null || key.length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        byte[] b = this.luxio.get( key ) ;
        if( b == null ) {
            return null ;
        }
        return LuxioUtil.toObject( b ) ;
    }
    
    /**
     * このオブジェクトがトランザクション対応かチェック.
     * <BR>
     * @return boolean [true]の場合、トランザクションに対応しています.
     */
    public boolean isTransaction() {
        if( luxio instanceof TransactionLuxIo ) {
            return true ;
        }
        return false ;
    }
    
    private boolean cbool( byte[] b ) {
        return ( b[ 0 ] == 1 ) ;
    }
    
    private int cint( byte[] b ) {
        return ( int )(
            (b[ 0 ]&0x000000ff)|
            ((b[ 1 ] & 0x000000ff)<<8)|
            ((b[ 2 ] & 0x000000ff)<<16)|
            ((b[ 3 ] & 0x000000ff)<<24) ) ;
    }
    
    private long clong( byte[] b ) {
        return ( long )(
            (b[ 0 ]&0x00000000000000ffL)|
            ((b[ 1 ] & 0x00000000000000ffL)<<8L)|
            ((b[ 2 ] & 0x00000000000000ffL)<<16L)|
            ((b[ 3 ] & 0x00000000000000ffL)<<24L)|
            ((b[ 4 ] & 0x00000000000000ffL)<<32L)|
            ((b[ 5 ] & 0x00000000000000ffL)<<40L)|
            ((b[ 6 ] & 0x00000000000000ffL)<<48L)|
            ((b[ 7 ] & 0x00000000000000ffL)<<56L) ) ;
    }
}
