package org.maachang.luxio ;

import java.io.File;
import java.io.IOException;

/**
 * Lux&nbsp;IOオブジェクト.
 * <br>
 * Lux&nbsp;IO-0.1.0をJavaから利用する場合のコアオブジェクト.<br>
 * 利用方法としては、下記の通り.
 * <pre>
 * public static final void main( String[] args ) throws Exception {
 *     LuxIoImpl impl = new LuxIoImpl( "fileName" ) ;
 *     try {
 *         impl.put( "test","hoge".getBytes() ) ;
 *         byte[] b = impl.get( "test" ) ;
 *         System.out.println( "test=" + new String( b ) ) ;
 *     } finally {
 *         if( impl != null ) { impl.destroy() ; }
 *     }
 * }
 * </pre>
 * 実行結果は[LuxIoImpl impl = new LuxIoImpl( "fileName" ) ;]でbidxを新規作成した場合、
 * 以下の結果となる.<br>
 * <br>
 * >test=hoge
 *
 * @version 2008/12/05
 * @author  masahito suzuki
 * @since   Lux IO 0.1.0
 */
public class LuxIoImpl implements LuxIo {
    
    /**
     * LuxIoポインタ.
     */
    private final AtomicLONG fd = new AtomicLONG( -1L ) ;
    
    /**
     * キーロック.
     */
    private final KeyLock lock = new KeyLock() ;
    
    /**
     * コンストラクタ.
     */
    private LuxIoImpl() {
        
    }
    
    /**
     * コンストラクタ.
     * @param name 対象のファイル名を設定します.
     * @exception Exception 例外.
     */
    public LuxIoImpl( String name ) throws Exception {
        if( name == null || ( name = name.trim() ).length() <= 0 ) {
            throw new IllegalArgumentException( "ファイル名は指定されていません" ) ;
        }
        if( NativeLuxIO.useInit() == false ) {
            throw new IOException( "Lux IO はWindowsまたはLinux以外で動作しません" ) ;
        }
        name = new File(name).getCanonicalPath();
        long fd = NativeLuxIO.open( NativeLuxIO.nativeString( name ) ) ;
        if( fd == 0 ) {
            throw new IOException( "ファイルオープンに失敗しました" ) ;
        }
        this.fd.set( fd ) ;
    }
    
    /**
     * デストラクタ.
     */
    protected void finalize() throws Exception {
        destroy() ;
    }
    
    /**
     * オブジェクト破棄.
     */
    public void destroy() {
        long fd = this.fd.get() ;
        if( fd != -1L ) {
            this.fd.set( -1L ) ;
            NativeLuxIO.close( fd ) ;
        }
    }
    
    /**
     * オブジェクトが有効かチェック.
     * @return boolean [true]の場合、有効です.
     */
    public boolean isUse() {
        return this.fd.get() != -1L ;
    }
    
    /**
     * 情報をセット.
     * @param key 対象のキーを設定します.
     * @param value 対象の要素を設定します.
     * @exception Exception 例外.
     */
    public void put( String key,byte[] value )
        throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        if( key == null || key.length() <= 0 ||
            value == null || value.length <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        lock.lock( key ) ;
        try {
            if( NativeLuxIO.set( fd.get(),NativeLuxIO.nativeString( key ),value,value.length ) <= -1 ) {
                throw new IOException( "書き込み処理に失敗しました" ) ;
            }
        } finally {
            lock.unlock( key ) ;
        }
    }
    
    /**
     * 情報を取得.
     * @param key 対象のキーを設定します.
     * @return byte[] 対象の要素が返されます.
     * @exception Exception 例外.
     */
    public byte[] get( String key ) throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        if( key == null || key.length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        byte[] ret = null ;
        lock.lock( key ) ;
        try {
            ret = NativeLuxIO.get( fd.get(),NativeLuxIO.nativeString( key ) ) ;
        } finally {
            lock.unlock( key ) ;
        }
        return ret ;
    }
    
    /**
     * 情報を削除.
     * @param key 対象のキーを設定します.
     * @exception Exception 例外.
     */
    public void remove( String key ) throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        if( key == null || key.length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        lock.lock( key ) ;
        try {
            if( NativeLuxIO.remove( fd.get(),NativeLuxIO.nativeString( key ) ) <= -1 ) {
                throw new IOException( "削除処理に失敗しました" ) ;
            }
        } finally {
            lock.unlock( key ) ;
        }
    }
    
}
