#ifndef LUX_UTIL_H
#define LUX_UTIL_H

#include "types.h"
#include <sys/stat.h>
#include <fcntl.h>

#ifndef _WIN32
#include <sys/mman.h>
#else
#include "win32/mman.h"
#endif

#include <errno.h>
#include <string>
#include <iostream>

#ifdef _WIN32
#include <direct.h>
#endif

#ifdef DEBUG
#define error_log(msg) \
  std::cerr << "[error] " << msg \
            << " in " << __FILE__ << ":" << __LINE__ \
            << std::endl; 
#else
#define error_log(msg)
#endif

// for verbose info logging
#ifdef DEBUG_VINFO
#define vinfo_log(msg) \
  std::cerr << "[info] " << msg \
            << std::endl; 
#else
#define vinfo_log(msg)
#endif

namespace Lux {

  void _mkdir(const char *str)
  {
    std::string str_(str); 
    size_t n = -1; 
    while (1) {
      n = str_.find_first_of('/', n+1);
      if (n == std::string::npos) {
        break;  
      }
      std::string dir = str_.substr(0, n);
      // [TODO] error handling
#ifndef _WIN32
      ::mkdir(dir.c_str(), 0755);
#else
      ::mkdir(dir.c_str());
#endif
    }
  }

  int _open(const char *pathname, int flags, mode_t mode)
  {
      int ret ;
    int oflags = O_RDONLY;

    if (flags & DB_RDWR) {
      oflags |= O_RDWR;
    }
    if (flags & DB_CREAT) {
      oflags |= O_CREAT | O_RDWR;
      _mkdir(pathname);
    }
    if (flags & DB_TRUNC) {
      oflags |= O_TRUNC;
    }

#ifdef _WIN32
    oflags = oflags | (O_BINARY | O_RANDOM) ;
#else
    oflags = oflags | (O_NOATIME | O_LARGEFILE) ;
#endif
    if( (ret = ::open(pathname, oflags, mode)) <= -1 ) {
        perror( "open failed" ) ;
        return -1 ;
    }
#ifndef _WIN32
    if( posix_fadvise( ret, 0, 0, POSIX_FADV_RANDOM ) <= -1 ) {
        perror( "posix_fadvise failed" ) ;
        close( ret ) ;
        return -1 ;
    }
#endif
    return ret ;
  }
  
  ssize_t _read(int fd, void *buf, size_t count)
  {
    char *p = reinterpret_cast<char *>(buf);
    const char * const end_p = p + count;

    while (p < end_p) {
      const int num_bytes = read(fd, p, ( unsigned int )(end_p - p));
      if (num_bytes < 0) {
        if (errno == EINTR) {
          continue;
        }
        perror("read failed");
        break;
      }
      p += num_bytes;
    }

    if (p != end_p) {
      return -1;
    }
    return count;
  }

  ssize_t _write(int fd, const void *buf, size_t count)
  {
    const char *p = reinterpret_cast<const char *>(buf);
    const char * const end_p = p + count;

    while (p < end_p) {
      const int num_bytes = write(fd, p, ( unsigned int )(end_p - p));
      if (num_bytes < 0) {
        if (errno == EINTR) continue;
        perror("write failed");
        break;
      }
      p += num_bytes;
    }

    if (p != end_p) {
      return -1;
    }
    return count;
  }

  int _lseek( int fd,off_t seek ) {
#ifdef _WIN32
#ifndef _WIN64
    HANDLE handle ;
    unsigned long low,high,res ;
    if( seek <= -1 ) {
        return -1 ;
    }
    handle = (HANDLE) _get_osfhandle(fd) ;
    low = ( unsigned long )( seek & 0x00000000ffffffff ) ;
    high = ( unsigned long )( ( seek & 0xffffffff00000000 ) >> 32 ) ;
    res = 0xffffffff ;
    if( high == 0 ){
        res = SetFilePointer( handle,low,NULL,FILE_BEGIN ) ;
    }else{
        res = SetFilePointer( handle,low,(long*)&high,FILE_BEGIN ) ;
    }
    if( res == 0xffffffff ){
        if( GetLastError() != NO_ERROR ) {
            return -1 ;
        }
    }
    return 0 ;
#else
    return lseek(fd, seek, SEEK_SET) ;
#endif
#else
    return lseek(fd, seek, SEEK_SET) ;
#endif
  }

  int _setFileLength( int fd,off_t length ) {
#ifdef _WIN32
#ifndef _WIN64
    long low,high ;
    unsigned long res ;
    HANDLE handle = (HANDLE) _get_osfhandle(fd) ;
    if( handle == INVALID_HANDLE_VALUE ) {
        return -1 ;
    }
    low = ( unsigned long )( length & 0x00000000ffffffff ) ;
    high = ( unsigned long )( ( length & 0xffffffff00000000 ) >> 32 ) ;
    res = SetFilePointer( handle,low,&high,FILE_BEGIN ) ;
    if( res == 0xffffffff ) {
        if(  GetLastError() != NO_ERROR ) {
            return -1 ;
        }
    }
    if( SetEndOfFile( handle ) == FALSE ) {
        return -1 ;
    }
    return 0;
#else
    return ftruncate(fd, length) ;
#endif
#else
    return ftruncate(fd, length) ;
#endif
  }
  
  off_t _getFileLength( int fd ) {
#ifdef _WIN32
#ifndef _WIN64
    unsigned long sizeLow = 0 ;
    unsigned long sizeHigh = 0 ;
    HANDLE handle = (HANDLE) _get_osfhandle(fd) ;
    if( handle == INVALID_HANDLE_VALUE ) {
        return -1 ;
    }
    sizeLow = GetFileSize( handle,&sizeHigh ) ;
    if( sizeLow == 0xffffffff ) {
        if( GetLastError() != ERROR_SUCCESS ) {
            return -1 ;
        }
    }
    return ( off_t )( ( ( off_t )sizeLow & 0x00000000ffffffff ) |
        ( ( ( off_t )sizeHigh & 0x00000000ffffffff ) << 32 ) ) ; ;
#else
      struct stat stat_buf;
      if (fstat(fd, &stat_buf) == -1 || !S_ISREG(stat_buf.st_mode)) {
        return -1;
      }
      return stat_buf.st_size ;
#endif
#else
      struct stat stat_buf;
      if (fstat(fd, &stat_buf) == -1 || !S_ISREG(stat_buf.st_mode)) {
        return -1;
      }
      return stat_buf.st_size ;
#endif
  }

  bool _pwrite(int fd, const void *buf, size_t nbyte, off_t offset)
  {
#ifndef _WIN32
    if( pwrite(fd,buf,nbyte,offset ) <= 0 ) {
        return false ;
    }
#else
    if (_lseek(fd, offset) < 0) {
      return false;
    }
    if (_write(fd, buf, nbyte) < 0) {
      return false; 
    }
#endif
    return true;
  }

  bool _pread(int fd, void *buf, size_t nbyte, off_t offset)
  {
#ifndef _WIN32
    if( pread(fd,buf,nbyte,offset ) <= 0 ) {
        return false ;
    }
#else
    if (_lseek(fd, offset) < 0) {
      return false;
    }
    if (_read(fd, buf, nbyte) < 0) {
      return false; 
    } 
#endif
    return true;
  }

#ifndef _WIN32
  void *_mmap(int fd, size_t size, int flags)
#else
  void *_win32_mmap(int fd, size_t size, int flags, HANDLE *h)
#endif
  {
    int prot = PROT_READ;
    if (flags & DB_RDWR || flags & DB_CREAT) {
      prot |= PROT_WRITE;
    }

#ifndef _WIN32
    void *p = mmap(0, size, prot, MAP_SHARED, fd, 0);  
#else
    void *p = win32_mmap(0, size, prot, MAP_SHARED, fd, 0, h);  
#endif

    if (p == MAP_FAILED) {
      return NULL;
    }
    return p;
  }

}

#ifdef _WIN32
#define _mmap(fd, size, flags) _win32_mmap((fd), (size), (flags), &h_map_)
#endif
#endif
