package org.maachang.report ;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

/**
 * セルポジションリスト.
 * 
 * @version 2008/09/29
 * @author masahito suzuki
 * @since MaachangReport 1.00
 */
class CellPositionList {
    
    /**
     * セルポジションリスト.
     */
    private Map<String,CellPosition[]> map = null ;
    
    /**
     * セルスタイルリスト.
     */
    private CellStyleList csList = null ;
    
    /**
     * コンストラクタ.
     */
    private CellPositionList() {
        
    }
    
    /**
     * コンストラクタ.
     * @param csList セルスタイルリストを設定します.
     */
    public CellPositionList( CellStyleList csList ) {
        this.map = new HashMap<String,CellPosition[]>() ;
        this.csList = csList ;
    }
    
    /**
     * 情報クリア.
     */
    public void clear() {
        map.clear() ;
    }
    
    /** セルスタイル範囲チェック. **/
    private void pushCellStyle( CellPosition pos ) {
        if( csList == null || csList.size() <= 0 ) {
            return ;
        }
        pos.setCellStyleName( null ) ;
        Iterator it = csList.iterator() ;
        while( it.hasNext() ) {
            String k = ( String )it.next() ;
            CellStyle cs = csList.get( k ) ;
            if( cs != null ) {
                if( cs.isGroup( pos.getX(),pos.getY() ) ) {
                    pos.setCellStyleName( cs.getName() ) ;
                    break ;
                }
            }
        }
    }
    
    /**
     * セルポジションを追加.
     * @param pos 対象のセルポジションを設定します.
     */
    public void put( CellPosition pos ) {
        if( pos == null ) {
            return ;
        }
        String k = pos.getName() ;
        pushCellStyle( pos ) ;
        if( pos.getCellStyleName() != null ) {
            map.put( k,new CellPosition[]{ pos } ) ;
        }
        else {
            CellPosition[] bef = map.get( k ) ;
            if( bef != null && bef[0].getCellStyleName() == null ) {
                int len = bef.length ;
                CellPosition[] nw = new CellPosition[ len+1 ] ;
                System.arraycopy( bef,0,nw,0,len ) ;
                nw[ len ] = pos ;
                map.put( k,nw ) ;
                bef = null ;
            }
            else {
                map.put( k,new CellPosition[]{ pos } ) ;
            }
        }
    }
    
    /**
     * セルポジションを取得.
     * @param name 対象の名前を設定します.
     * @return CellPosition[] セルポジション群が返されます.
     */
    public CellPosition[] get( String name ) {
        if( name == null || ( name = name.trim() ).length() <= 0 ) {
            return null ;
        }
        return map.get( name ) ;
    }
    
    /**
     * 登録されているセルポジション数を取得.
     * @return int セルポジション数が返されます.
     */
    public int size() {
        return map.size() ;
    }
    
    /**
     * Iteratorを取得.
     * @return Iterator Iteratorが返されます.
     */
    public Iterator iterator() {
        return map.keySet().iterator() ;
    }
    
    /**
     * 指定セルスタイル名が記述されているシーケンス付加名を取得.
     * @param name セルスタイル名を設定します.
     * @return CellPosition 対象のセルポジションが返されます.
     */
    public CellPosition getSequence( String name ) {
        if( name == null || ( name = name.trim() ).length() <= 0 ) {
            return null ;
        }
        Iterator it = map.keySet().iterator() ;
        while( it.hasNext() ) {
            String key = ( String )it.next() ;
            if( key.startsWith( "seq:" ) ) {
                CellPosition[] p = map.get( key ) ;
                if( p != null ) {
                    int len = p.length ;
                    for( int i = 0 ; i < len ; i ++ ) {
                        if( name.equals( p[ i ].getCellStyleName() ) ) {
                            return p[ i ] ;
                        }
                    }
                }
            }
        }
        return null ;
    }
    
    /**
     * 文字列で取得.
     * @return String 文字列が返されます.
     */
    public String toString() {
        if( map.size() <= 0 ) {
            return "なし" ;
        }
        StringBuilder buf = new StringBuilder() ;
        buf.append( "[CellPosition] ->" ) ;
        Iterator it = map.keySet().iterator() ;
        while( it.hasNext() ) {
            String s = ( String )it.next() ;
            CellPosition[] lst = map.get( s ) ;
            buf.append( " name:" ).append( s ) ;
            int len = lst.length ;
            buf.append( "(" ).append( len ).append( ")=>" ) ;
            for( int i = 0 ; i < len ; i ++ ) {
                buf.append( " #" ).append( (i+1) ).append( "#" ).
                append( lst[ i ].toString() ) ;
            }
        }
        return buf.toString() ;
    }
}

