package org.maachang.comet.httpd.engine.script.cache;

import java.util.Map;

import javax.script.Bindings;
import javax.script.ScriptContext;
import javax.script.ScriptEngine;
import javax.script.SimpleScriptContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.comet.MaachangDef;
import org.maachang.comet.ServiceDef;
import org.maachang.comet.httpd.engine.script.DummyBaseModel;
import org.maachang.comet.httpd.engine.script.ScriptDef;
import org.maachang.comet.httpd.engine.script.SrcScript;
import org.maachang.comet.httpd.engine.script.js.JsDef;
import org.maachang.jsr.script.api.ApiManager;
import org.maachang.jsr.script.api.ExternalBindings;
import org.maachang.manager.GlobalManager;
import org.maachang.util.FileUtil;

/**
 * モデルAPIマネージャ.
 * 
 * @version 2008/08/11
 * @author masahito suzuki
 * @since MaachangComet 1.23
 */
class ModelApiManager {
    
    /**
     * LOG.
     */
    private static final Log LOG = LogFactory.getLog( ModelApiManager.class ) ;
    
    /**
     * スクリプトエンジン名.
     */
    protected static final String ENGINE_NAME = "js" ;
    
    /**
     * スクリプト実行名.
     */
    protected static final String SCRIPT_NAME = "[modelApi]" ;
    
    /**
     * モデルファイルフッダ名.
     */
    protected static final String MODEL_NAME_FOODER = "Model" ;
    
    /**
     * カスタムモデルスクリプト.
     */
    protected static final String MODEL_CUSTOM = "Custom" ;
    
    /**
     * 読み込み対象ディレクトリ.
     */
    private static final String READ_DIR = baseDir() ;
    
    /**
     * 再読み込み確認時間.
     */
    private static final long CHECK_TIME = 30000L ;
    
    /**
     * 前回再読み込み時間.
     */
    private long beforeTime = -1L ;
    
    /**
     * モデル管理コード.
     */
    private String modelCode = null ;
    
    /**
     * スクリプト内容.
     */
    private SrcScript srcScript = null ;
    
    /**
     * デフォルトパッケージ更新ID.
     */
    private int packageId = -1 ;
    
    /**
     * 同期オブジェクト.
     */
    private final Object sync = new Object() ;
    
    /**
     * コンストラクタ.
     */
    public ModelApiManager() throws Exception {
        try {
            if( GlobalManager.getValue( ServiceDef.MANAGER_BY_DBMS_POOL ) != null ) {
                synchronized( sync ) {
                    loadScript() ;
                }
            }
        } catch( Exception e ) {
            LOG.error( "## [modelApi] loadScriptError",e ) ;
            throw e ;
        }
    }
    
    /**
     * デストラクタ.
     */
    protected void finalize() throws Exception {
        this.clear() ;
    }
    
    /**
     * 基本ディレクトリを取得.
     */
    private static final String baseDir() {
        try {
            return FileUtil.getFullPath( MaachangDef.DIRECTORY_MODEL ) + FileUtil.FILE_SPACE ;
        } catch( Exception e ) {
        }
        return null ;
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    protected void clear() {
        synchronized( sync ) {
            beforeTime = -1L ;
            modelCode = null ;
            srcScript = null ;
            packageId = -1 ;
        }
    }
    
    /**
     * 再読み込み処理.
     * @exception Exception 例外.
     */
    public void reload() throws Exception {
        synchronized( sync ) {
            loadScript() ;
        }
    }
    
    /**
     * スクリプトキャッシュを有効にする.
     * この処理は、スクリプト呼び出しの前に必ず実行する必要があります.
     * @exception Exception 例外.
     */
    public void useScript() throws Exception {
        long time = System.currentTimeMillis() ;
        try {
            synchronized( sync ) {
                if( GlobalManager.getValue( ServiceDef.MANAGER_BY_DBMS_POOL ) != null ) {
                    if( this.packageId != -1 && this.packageId != JsDef.getDefaultPackageId() ) {
                        loadScript() ;
                    }
                    else if( beforeTime + CHECK_TIME <= time ) {
                        if( modelCode == null || modelCode.equals( createModelMonitor() ) == false ) {
                            loadScript() ;
                        }
                        else {
                            beforeTime = System.currentTimeMillis() ;
                        }
                    }
                }
            }
        } catch( Exception e ) {
            throw e ;
        }
    }
    
    /**
     * スクリプト内容を取得.
     * <BR><BR>
     * スクリプト内容を取得します.
     * <BR>
     * @return SrcScript スクリプト内容が返されます.
     */
    public SrcScript getSrcScript() {
        SrcScript ret = null ;
        synchronized( sync ) {
            ret = srcScript ;
        }
        return ret ;
    }
    
    /**
     * 指定パスがこのキャッシュ内の条件であるかチェック.
     * <BR><BR>
     * 指定パスがこのキャッシュ内の条件であるかチェックします.
     * <BR>
     * @param path チェック対象のパスを設定します.
     * @return boolean [true]の場合はこのキャッシュ内の条件です.
     */
    public boolean isCache( String path ) {
        if( path == null || ( path = path.trim() ).length() <= 0 ||
            path.equals( SCRIPT_NAME ) == false ) {
            return false ;
        }
        return true ;
    }
    
    /**
     * スクリプト内容を読み込む.
     */
    private void loadScript() throws Exception {
        int cnt = 0 ;
        String beforeManagerCode = this.modelCode ;
        int beforePackageId = this.packageId ;
        long beforeBeforeTime = this.beforeTime ;
        ScriptContext ctx = null ;
        try {
            // モデル内容を読み込み.
            String[] models = ScriptDef.getUseScript( READ_DIR ) ;
            // 対象となるモデル内容が存在する場合.
            if( models != null && models.length > 0 ) {
                StringBuilder buf = new StringBuilder() ;
                // 更新された内容を一時保存.
                this.modelCode = ScriptDef.getScriptManagerCode( models,READ_DIR ) ;
                this.packageId = JsDef.pushDefaultPackage( buf ) ;
                this.beforeTime = System.currentTimeMillis() ;
                
                // モデル内容を読み込む.
                int len = models.length ;
                for( int i = 0 ; i < len ; i ++ ) {
                    if( models[i].endsWith( ScriptDef.SCRIPT_PLUS ) ) {
                        models[i] = models[i].substring( 0,models[i].length()-ScriptDef.SCRIPT_PLUS.length() ) ;
                    }
                    if( models[i].endsWith( MODEL_CUSTOM ) ||
                        models[i].endsWith( MODEL_NAME_FOODER ) == false ) {
                        continue ;
                    }
                    else {
                        cnt ++ ;
                    }
                    // 本来のモデル名を取得して、付加処理.
                    String modelName = models[i].substring( 0,models[i].length()-MODEL_NAME_FOODER.length() ) ;
                    if( LOG.isDebugEnabled() ) {
                        LOG.debug( ">read[model] - " + modelName ) ;
                    }
                    putModel( modelName,buf ) ;
                }
                String script = buf.toString() ;
                buf = null ;
                this.srcScript = new SrcScript( script ) ;
                
                // 処理実行.
                Bindings bindings = ExternalBindings.getInstance() ;
                // dummyデータ設定.
                bindings.put( ScriptDef.SCRIPT_BY_MODEL,new DummyBaseModel() ) ;
                
                // エンジン実行.
                ScriptEngine engine = ApiManager.getInstance().getScriptEngine() ;
                ctx = new SimpleScriptContext() ;
                ctx.setBindings( bindings,ScriptContext.ENGINE_SCOPE ) ;
                engine.setContext( ctx ) ;
                engine.put( ScriptEngine.FILENAME,SCRIPT_NAME ) ;
                //engine.eval( script,bindings ) ;
                engine.eval( script ) ;
                
                // APIマネージャに登録.
                Map<String,Object> modelManager = ApiManager.getLocal() ;
                bindings.clear() ;
                ApiManager.getInstance().setModelMap( modelManager ) ;
                if( LOG.isDebugEnabled() ) {
                    LOG.debug( "** model-cacheを読み込み" ) ;
                }
            }
            // 対象となるモデル内容が存在しない場合は、内容をクリアする.
            if( cnt <= 0 ) {
                this.modelCode = null ;
                this.packageId = -1 ;
                ApiManager.getInstance().setModelMap( null ) ;
                this.beforeTime = System.currentTimeMillis() ;
            }
        } catch( Exception e ) {
            // 例外の場合は、ロールバック処理.
            this.modelCode = beforeManagerCode ;
            this.packageId = beforePackageId ;
            this.beforeTime = beforeBeforeTime ;
            throw e ;
        } finally {
            ApiManager.removeLocal() ;
        }
    }
    
    /**
     * Model管理コードを生成.
     */
    private String createModelMonitor()
        throws Exception {
        String[] models = ScriptDef.getUseScript( READ_DIR ) ;
        return ScriptDef.getScriptManagerCode( models,READ_DIR ) ;
    }
    
    /**
     * Model内容を追加.
     */
    private void putModel( String model,StringBuilder buf )
        throws Exception {
        buf.append( "\n\n" ) ;
        buf.append( "var " ).append( model ).append( " = baseModel(\"" ).append( model ).append( "\");\n" ) ;
        String modelScript = FileUtil.getFileByString( READ_DIR+"/"+model+MODEL_NAME_FOODER+ScriptDef.SCRIPT_PLUS,"UTF8" ) ;
        if( modelScript != null && ( modelScript = modelScript.trim() ).length() > 0 ) {
            buf.append( modelScript ).append( "\n" ) ;
        }
        modelScript = null ;
        // カスタムモデルスクリプトが存在する場合、それも含めて取り込む.
        String custom = READ_DIR+"/"+model+MODEL_CUSTOM+ScriptDef.SCRIPT_PLUS ;
        if( FileUtil.isFileExists( custom ) == true ) {
            modelScript = FileUtil.getFileByString( READ_DIR+"/"+custom,"UTF8" ) ;
            if( modelScript != null && ( modelScript = modelScript.trim() ).length() > 0 ) {
                buf.append( "\n" ).append( modelScript ).append( "\n" ) ;
            }
        }
    }
    
}
