/*
 * Joey and its relative products are published under the terms
 * of the Apache Software License.
 */
/*
 * Created on 2003/12/15
 */
package test.org.asyrinx.brownie.core.io;

import java.io.File;
import java.util.List;

import junit.framework.TestCase;

import org.asyrinx.brownie.core.io.FileNameUtils;

/**
 * @author akima
 */
public class FileNameUtilsTest extends TestCase {

	/**
	 * Constructor for FileNameUtilsTest.
	 * @param name
	 */
	public FileNameUtilsTest(String name) {
		super(name);
	}

	public static void main(String[] args) {
		junit.swingui.TestRunner.run(FileNameUtilsTest.class);
	}

	public void testHasZipExtension() {
	}

	public void testIsAbsolutePath() {
		assertTrue(!FileNameUtils.isAbsolutePath("test.txt"));
	}

	public void testToAbsolutePath() {
		assertEquals(
			"D:/a1/b1/c1/test.txt",
			FileNameUtils.toAbsolutePath("D:/a1/b1/c1", "test.txt"));
		assertEquals(
			"D:/a1/b1/c1/test.txt",
			FileNameUtils.toAbsolutePath("D:/a1/b1/c1", "./test.txt"));
		assertEquals(
			"D:/a1/b1/test.txt",
			FileNameUtils.toAbsolutePath("D:/a1/b1/c1", "../test.txt"));
		assertEquals(
			"D:/a1/b1/c2/test.txt",
			FileNameUtils.toAbsolutePath("D:/a1/b1/c1", "../c2/test.txt"));
	}

	public void testIsSeparatorTail() {
		assertTrue(FileNameUtils.isSeparatorTail("/a1/b1/"));
		assertTrue(FileNameUtils.isSeparatorTail("\\a1\\b1\\"));
		assertTrue(!FileNameUtils.isSeparatorTail("/a1/b1"));
		assertTrue(!FileNameUtils.isSeparatorTail("\\a1\\b1"));
	}

	public void testIsSeparatorHead() {
		assertTrue(FileNameUtils.isSeparatorHead("\\test.txt"));
		assertTrue(FileNameUtils.isSeparatorHead("/test.txt"));
		assertTrue(!FileNameUtils.isSeparatorHead("test.txt"));
		assertTrue(!FileNameUtils.isSeparatorHead("a/test.txt"));
		assertTrue(!FileNameUtils.isSeparatorHead("a\\test.txt"));
	}

	public void testAddSeparatorHead() {
		assertEquals(
			"/a/test.txt",
			FileNameUtils.addSeparatorHead("a/test.txt"));
		assertEquals(
			"\\a\\test.txt",
			FileNameUtils.addSeparatorHead("a\\test.txt"));
		assertEquals(
			"/a/test.txt",
			FileNameUtils.addSeparatorHead("/a/test.txt"));
		assertEquals(
			"\\a\\test.txt",
			FileNameUtils.addSeparatorHead("\\a\\test.txt"));
	}

	public void testAddSeparatorTail() {
		assertEquals("/a1/b1/", FileNameUtils.addSeparatorTail("/a1/b1/"));
		assertEquals(
			"\\a1\\b1\\",
			FileNameUtils.addSeparatorTail("\\a1\\b1\\"));
		assertEquals("/a1/b1/", FileNameUtils.addSeparatorTail("/a1/b1"));
		assertEquals("\\a1\\b1\\", FileNameUtils.addSeparatorTail("\\a1\\b1"));
	}

	public void testDeleteSeparatorHead() {
		assertEquals("a1/b1/", FileNameUtils.deleteSeparatorHead("/a1/b1/"));
		assertEquals("a1/b1/", FileNameUtils.deleteSeparatorHead("a1/b1/"));
		assertEquals(
			"a1\\b1\\",
			FileNameUtils.deleteSeparatorHead("\\a1\\b1\\"));
		assertEquals("a1\\b1\\", FileNameUtils.deleteSeparatorHead("a1\\b1\\"));
	}

	public void testDeleteSeparatorTail() {
		assertEquals("/a1/b1", FileNameUtils.deleteSeparatorTail("/a1/b1/"));
		assertEquals("/a1/b1", FileNameUtils.deleteSeparatorTail("/a1/b1"));
		assertEquals(
			"\\a1\\b1",
			FileNameUtils.deleteSeparatorTail("\\a1\\b1\\"));
		assertEquals("\\a1\\b1", FileNameUtils.deleteSeparatorTail("\\a1\\b1"));
	}

	public void testGetParentPath() {
		assertEquals(null, FileNameUtils.getParentPath(null));
		assertEquals(null, FileNameUtils.getParentPath(""));
		assertEquals(null, FileNameUtils.getParentPath("abc"));
		assertEquals(null, FileNameUtils.getParentPath("abc.txt"));
		assertEquals("abc", FileNameUtils.getParentPath("abc/def"));
		assertEquals(
			"abc",
			FileNameUtils.getParentPath("abc" + File.separator + "def"));
		assertEquals(
			"abc" + File.separator + "def",
			FileNameUtils.getParentPath("abc/def/ghi"));
		assertEquals(
			"abc" + File.separator + "def",
			FileNameUtils.getParentPath(
				"abc" + File.separator + "def" + File.separator + "ghi"));
	}

	/*
	 * Test for String replaceSeparator(String, String, String)
	 */
	public void testReplaceSeparatorStringStringString() {
		assertEquals(
			"abc" + File.separator + "def",
			FileNameUtils.replaceSeparator("abc/def", "/", File.separator));
		assertEquals(
			null,
			FileNameUtils.replaceSeparator(null, "/", File.separator));
	}

	/*
	 * Test for String replaceSeparator(String)
	 */
	public void testReplaceSeparatorString() {
		assertEquals(
			"abc" + File.separator + "def",
			FileNameUtils.replaceSeparatorSlash("abc/def"));
		assertEquals(
			"abc" + File.separator + "def",
			FileNameUtils.replaceSeparatorSlash(
				"abc" + File.separator + "def"));
		assertEquals(null, FileNameUtils.replaceSeparatorSlash(null));
	}

	public void testToRelativePath() {
		assertEquals(".", FileNameUtils.toRelativePath("/abc/def", "/abc/def"));
		assertEquals(
			"..",
			FileNameUtils.toRelativePath("/abc/def", "/abc/def/ghi"));
		assertEquals(
			"./ghi/jkl",
			FileNameUtils.toRelativePath("/abc/def/ghi/jkl", "/abc/def"));
		assertEquals(
			"../..",
			FileNameUtils.toRelativePath("/abc/def", "/abc/def/ghi/jkl"));
		assertEquals(
			"../../mno",
			FileNameUtils.toRelativePath("/abc/def/mno", "/abc/def/ghi/jkl"));
		assertEquals(
			"../mno/pqr",
			FileNameUtils.toRelativePath("/abc/def/mno/pqr", "/abc/def/ghi"));
	}

	public void testGetExtension() {
		assertEquals(null, FileNameUtils.getExtension(null));
		assertEquals(null, FileNameUtils.getExtension(""));
		assertEquals(null, FileNameUtils.getExtension("D:/temp/abc"));
		assertEquals("", FileNameUtils.getExtension("D:/temp/abc."));
		assertEquals("zip", FileNameUtils.getExtension("D:/temp/abc.zip"));
		assertEquals("bak", FileNameUtils.getExtension("D:/temp/abc.zip.bak"));
		assertEquals("zip", FileNameUtils.getExtension("D:/temp.jar/abc.zip"));
		assertEquals(
			"bak",
			FileNameUtils.getExtension("D:/temp.jar/abc.zip.bak"));

	}

	public void testRemoveExtension() {
		assertEquals(null, FileNameUtils.removeExtension(null));
		assertEquals("", FileNameUtils.removeExtension(""));
		assertEquals(
			"D:/temp/abc",
			FileNameUtils.removeExtension("D:/temp/abc"));
		assertEquals(
			"D:/temp/abc",
			FileNameUtils.removeExtension("D:/temp/abc."));
		assertEquals(
			"D:/temp/abc",
			FileNameUtils.removeExtension("D:/temp/abc.zip"));
		assertEquals(
			"D:/temp/abc.zip",
			FileNameUtils.removeExtension("D:/temp/abc.zip.bak"));
		assertEquals(
			"D:/temp.jar/abc",
			FileNameUtils.removeExtension("D:/temp.jar/abc.zip"));
		assertEquals(
			"D:/temp.jar/abc.zip",
			FileNameUtils.removeExtension("D:/temp.jar/abc.zip.bak"));
	}

	public void testChangeExtension() {
		assertEquals(null, FileNameUtils.changeExtension(null, ""));
		assertEquals(null, FileNameUtils.changeExtension(null, "aaa"));
		assertEquals("", FileNameUtils.changeExtension("", ""));
		assertEquals(".aaa", FileNameUtils.changeExtension("", "aaa"));
		assertEquals(
			"D:/temp/abc",
			FileNameUtils.changeExtension("D:/temp/abc", ""));
		assertEquals(
			"D:/temp/abc.aaa",
			FileNameUtils.changeExtension("D:/temp/abc", "aaa"));
		assertEquals(
			"D:/temp/abc",
			FileNameUtils.changeExtension("D:/temp/abc.", ""));
		assertEquals(
			"D:/temp/abc.aaa",
			FileNameUtils.changeExtension("D:/temp/abc.", "aaa"));
		assertEquals(
			"D:/temp/abc",
			FileNameUtils.changeExtension("D:/temp/abc.zip", null));
		assertEquals(
			"/temp/abc.jar",
			FileNameUtils.changeExtension("/temp/abc.zip", "jar"));
		assertEquals(
			"/temp/abc.zip",
			FileNameUtils.changeExtension("/temp/abc.zip.bak", ""));
		assertEquals(
			"/temp/abc.zip.jar",
			FileNameUtils.changeExtension("/temp/abc.zip.bak", "jar"));
	}

	public void testIsURL() {
		assertEquals(true, FileNameUtils.isURL("file://test/data"));
		assertEquals(true, FileNameUtils.isURL("http://www.asyrinx.org/data"));
		assertEquals(true, FileNameUtils.isURL("ftp://ftp.asyrinx.org/"));
		assertEquals(true, FileNameUtils.isURL("file://D:\\temp\\sample/"));
		assertEquals(false, FileNameUtils.isURL("/test/data"));
		assertEquals(false, FileNameUtils.isURL("www.asyrinx.org/data"));
		assertEquals(false, FileNameUtils.isURL("ftp.asyrinx.org/"));
		assertEquals(false, FileNameUtils.isURL("D:\\temp\\sample/"));
	}

	public void testParseFileNames() {
		List result;
		result = FileNameUtils.parseFileNames(null);
		assertEquals(0, result.size());
		result = FileNameUtils.parseFileNames("");
		assertEquals(0, result.size());
		result = FileNameUtils.parseFileNames("/abc/def");
		assertEquals(1, result.size());
		assertEquals("/abc/def", result.get(0));
		result =
			FileNameUtils.parseFileNames(
				"/abc/def" + File.pathSeparator + "foo.exe");
		assertEquals(2, result.size());
		assertEquals("/abc/def", result.get(0));
		assertEquals("foo.exe", result.get(1));
	}

	public void testToFile() {
		assertEquals(null, FileNameUtils.toFile(null));
		assertEquals(null, FileNameUtils.toFile(""));
		File file;
		file = FileNameUtils.toFile("/abc");
		assertEquals(File.separator + "abc", file.getPath());
		file = FileNameUtils.toFile(new File("/abc"));
		assertEquals(File.separator + "abc", file.getPath());
	}

}
