/*
 * Joey and its relative products are published under the terms
 * of the Apache Software License.
 */
package org.asyrinx.brownie.jdbc.measure;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.sql.Statement;

import org.asyrinx.brownie.core.log.MeasureLog;
import org.asyrinx.brownie.core.log.MeasureLogImpl;
import org.asyrinx.brownie.core.util.Wrapper;

/**
 * @author Akima
 */
public class StatementMeasure extends Wrapper implements Statement {

	/**
	 * Constructor for StatementMeasure.
	 */
	public StatementMeasure(Statement source, MeasureLog measureLog) {
		super(source);
		this.source = source;
		if (measureLog == null)
			measureLog = new MeasureLogImpl("sql");
		this.measureLog = measureLog;
	}

	/**
	 * Constructor for StatementMeasure.
	 */
	public StatementMeasure(Statement source) {
		this(source, null);
	}

	protected final MeasureLog measureLog;
	private final Statement source;

	protected static final String PREFIX = "[measure]";

	public static String format(String name, long t) {
		return PREFIX + " " + name + " -- " + String.valueOf(t);
	}

	/**
	 * @see java.sql.Statement#execute(String)
	 */
	public boolean execute(String sql) throws SQLException {
		measureLog.reset();
		final boolean result = source.execute(sql);
		measureLog.done(sql);
		return result;
	}

	/**
	 * @see java.sql.Statement#executeQuery(String)
	 */
	public ResultSet executeQuery(String sql) throws SQLException {
		measureLog.reset();
		final ResultSet result = source.executeQuery(sql);
		measureLog.done(sql);
		return result;
	}

	/**
	 * @see java.sql.Statement#executeUpdate(String)
	 */
	public int executeUpdate(String sql) throws SQLException {
		measureLog.reset();
		final int result = source.executeUpdate(sql);
		measureLog.done(sql);
		return result;
	}

	/**
	 * @see java.sql.Statement#addBatch(String)
	 */
	public void addBatch(String sql) throws SQLException {
		source.addBatch(sql);
	}

	/**
	 * @see java.sql.Statement#clearBatch()
	 */
	public void clearBatch() throws SQLException {
		source.clearBatch();
	}

	/**
	 * @see java.sql.Statement#executeBatch()
	 */
	public int[] executeBatch() throws SQLException {
		measureLog.reset();
		final int[] result = source.executeBatch();
		measureLog.done("batch");
		return result;
	}

	/**
	 * @see java.sql.Statement#close()
	 */
	public void close() throws SQLException {
		source.close();
	}

	/**
	 * @see java.sql.Statement#getMaxFieldSize()
	 */
	public int getMaxFieldSize() throws SQLException {
		return source.getMaxFieldSize();
	}

	/**
	 * @see java.sql.Statement#setMaxFieldSize(int)
	 */
	public void setMaxFieldSize(int max) throws SQLException {
		source.setMaxFieldSize(max);
	}

	/**
	 * @see java.sql.Statement#getMaxRows()
	 */
	public int getMaxRows() throws SQLException {
		return source.getMaxRows();
	}

	/**
	 * @see java.sql.Statement#setMaxRows(int)
	 */
	public void setMaxRows(int max) throws SQLException {
		source.setMaxRows(max);
	}

	/**
	 * @see java.sql.Statement#setEscapeProcessing(boolean)
	 */
	public void setEscapeProcessing(boolean enable) throws SQLException {
		source.setEscapeProcessing(enable);
	}

	/**
	 * @see java.sql.Statement#getQueryTimeout()
	 */
	public int getQueryTimeout() throws SQLException {
		return source.getQueryTimeout();
	}

	/**
	 * @see java.sql.Statement#setQueryTimeout(int)
	 */
	public void setQueryTimeout(int seconds) throws SQLException {
		source.setQueryTimeout(seconds);
	}

	/**
	 * @see java.sql.Statement#cancel()
	 */
	public void cancel() throws SQLException {
		source.cancel();
	}

	/**
	 * @see java.sql.Statement#getWarnings()
	 */
	public SQLWarning getWarnings() throws SQLException {
		return source.getWarnings();
	}

	/**
	 * @see java.sql.Statement#clearWarnings()
	 */
	public void clearWarnings() throws SQLException {
		source.clearWarnings();
	}

	/**
	 * @see java.sql.Statement#setCursorName(String)
	 */
	public void setCursorName(String name) throws SQLException {
		source.setCursorName(name);
	}

	/**
	 * @see java.sql.Statement#getResultSet()
	 */
	public ResultSet getResultSet() throws SQLException {
		return source.getResultSet();
	}

	/**
	 * @see java.sql.Statement#getUpdateCount()
	 */
	public int getUpdateCount() throws SQLException {
		return source.getUpdateCount();
	}

	/**
	 * @see java.sql.Statement#getMoreResults()
	 */
	public boolean getMoreResults() throws SQLException {
		return source.getMoreResults();
	}

	/**
	 * @see java.sql.Statement#setFetchDirection(int)
	 */
	public void setFetchDirection(int direction) throws SQLException {
		source.setFetchDirection(direction);
	}

	/**
	 * @see java.sql.Statement#getFetchDirection()
	 */
	public int getFetchDirection() throws SQLException {
		return source.getFetchDirection();
	}

	/**
	 * @see java.sql.Statement#setFetchSize(int)
	 */
	public void setFetchSize(int rows) throws SQLException {
		source.setFetchSize(rows);
	}

	/**
	 * @see java.sql.Statement#getFetchSize()
	 */
	public int getFetchSize() throws SQLException {
		return source.getFetchSize();
	}

	/**
	 * @see java.sql.Statement#getResultSetConcurrency()
	 */
	public int getResultSetConcurrency() throws SQLException {
		return source.getResultSetConcurrency();
	}

	/**
	 * @see java.sql.Statement#getResultSetType()
	 */
	public int getResultSetType() throws SQLException {
		return source.getResultSetType();
	}

	/**
	 * @see java.sql.Statement#getConnection()
	 */
	public Connection getConnection() throws SQLException {
		return source.getConnection();
	}

	/**
	 * @param sql
	 * @param autoGeneratedKeys
	 * @return
	 * @throws java.sql.SQLException
	 */
	public boolean execute(String sql, int autoGeneratedKeys)
		throws SQLException {
		return source.execute(sql, autoGeneratedKeys);
	}

	/**
	 * @param sql
	 * @param columnIndexes
	 * @return
	 * @throws java.sql.SQLException
	 */
	public boolean execute(String sql, int[] columnIndexes)
		throws SQLException {
		return source.execute(sql, columnIndexes);
	}

	/**
	 * @param sql
	 * @param columnNames
	 * @return
	 * @throws java.sql.SQLException
	 */
	public boolean execute(String sql, String[] columnNames)
		throws SQLException {
		return source.execute(sql, columnNames);
	}

	/**
	 * @param sql
	 * @param autoGeneratedKeys
	 * @return
	 * @throws java.sql.SQLException
	 */
	public int executeUpdate(String sql, int autoGeneratedKeys)
		throws SQLException {
		return source.executeUpdate(sql, autoGeneratedKeys);
	}

	/**
	 * @param sql
	 * @param columnIndexes
	 * @return
	 * @throws java.sql.SQLException
	 */
	public int executeUpdate(String sql, int[] columnIndexes)
		throws SQLException {
		return source.executeUpdate(sql, columnIndexes);
	}

	/**
	 * @param sql
	 * @param columnNames
	 * @return
	 * @throws java.sql.SQLException
	 */
	public int executeUpdate(String sql, String[] columnNames)
		throws SQLException {
		return source.executeUpdate(sql, columnNames);
	}

	/**
	 * @return
	 * @throws java.sql.SQLException
	 */
	public ResultSet getGeneratedKeys() throws SQLException {
		return source.getGeneratedKeys();
	}

	/**
	 * @param current
	 * @return
	 * @throws java.sql.SQLException
	 */
	public boolean getMoreResults(int current) throws SQLException {
		return source.getMoreResults(current);
	}

	/**
	 * @return
	 * @throws java.sql.SQLException
	 */
	public int getResultSetHoldability() throws SQLException {
		return source.getResultSetHoldability();
	}

}
