/*
 * Joey and its relative products are published under the terms
 * of the Apache Software License.
 */
/*
 * Created on 2004/03/01
 */
package org.asyrinx.brownie.jdbc.logger;

import java.io.InputStream;
import java.io.Reader;
import java.math.BigDecimal;
import java.net.URL;
import java.sql.Array;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.Date;
import java.sql.ParameterMetaData;
import java.sql.PreparedStatement;
import java.sql.Ref;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Time;
import java.sql.Timestamp;
import java.util.Calendar;
import java.util.HashMap;

import org.asyrinx.brownie.core.collection.IntegerKeyMap;
import org.asyrinx.brownie.core.collection.MapUtils;
import org.asyrinx.brownie.core.log.CascadeNamedLog;
import org.asyrinx.brownie.core.sql.SqlReplacer;

/**
 * @author akima
 */
public class LogPreparedStatement
	extends LogStatement
	implements PreparedStatement {
	/**
	 * @param wrapped
	 */
	public LogPreparedStatement(
		PreparedStatement wrapped,
		String sql,
		CascadeNamedLog parentLog) {
		super(wrapped, parentLog);
		this.wrapped = wrapped;
		this.sql = sql;
	}
	private final PreparedStatement wrapped;
	private final IntegerKeyMap parameters =
		MapUtils.toIntegerKeyMap(new HashMap());

	protected final String sql;
	protected final SqlReplacer replacer = new SqlReplacer();

	protected final void addParam(int paramIndex, Object value) {
		parameters.put(paramIndex, value);
	}

	protected void log() {
		this.log.log(formatSql());
	}

	protected void log(SQLException e) {
		this.log.log((Object) formatSql(), e);
	}

	protected String formatSql() {
		return replacer.execute(this.sql, this.parameters);
	}

	/**
	 * @throws java.sql.SQLException
	 */
	public void addBatch() throws SQLException {
		wrapped.addBatch();
		batches.add(formatSql());
	}

	/**
	 * @throws java.sql.SQLException
	 */
	public void clearParameters() throws SQLException {
		wrapped.clearParameters();
		parameters.clear();
	}

	/**
	 * @return
	 * @throws java.sql.SQLException
	 */
	public boolean execute() throws SQLException {
		try {
			final boolean reasult = wrapped.execute();
			log();
			return reasult;
		} catch (SQLException e) {
			log(e);
			throw e;
		}
	}

	/**
	 * @return
	 * @throws java.sql.SQLException
	 */
	public ResultSet executeQuery() throws SQLException {
		try {
			final ResultSet result = wrapped.executeQuery();
			log();
			return result;
		} catch (SQLException e) {
			log(e);
			throw e;
		}
	}

	/**
	 * @return
	 * @throws java.sql.SQLException
	 */
	public int executeUpdate() throws SQLException {
		try {
			final int result = wrapped.executeUpdate();
			log();
			return result;
		} catch (SQLException e) {
			log(e);
			throw e;
		}
	}

	/**
	 * @return
	 * @throws java.sql.SQLException
	 */
	public ResultSetMetaData getMetaData() throws SQLException {
		return wrapped.getMetaData();
	}

	/**
	 * @return
	 * @throws java.sql.SQLException
	 */
	public ParameterMetaData getParameterMetaData() throws SQLException {
		return wrapped.getParameterMetaData();
	}

	/**
	 * @param i
	 * @param x
	 * @throws java.sql.SQLException
	 */
	public void setArray(int parameterIndex, Array x) throws SQLException {
		wrapped.setArray(parameterIndex, x);
		addParam(parameterIndex, x);
	}

	/**
	 * @param parameterIndex
	 * @param x
	 * @param length
	 * @throws java.sql.SQLException
	 */
	public void setAsciiStream(int parameterIndex, InputStream x, int length)
		throws SQLException {
		wrapped.setAsciiStream(parameterIndex, x, length);
		//addParam(parameterIndex, x);
	}

	/**
	 * @param parameterIndex
	 * @param x
	 * @throws java.sql.SQLException
	 */
	public void setBigDecimal(int parameterIndex, BigDecimal x)
		throws SQLException {
		wrapped.setBigDecimal(parameterIndex, x);
		addParam(parameterIndex, x);
	}

	/**
	 * @param parameterIndex
	 * @param x
	 * @param length
	 * @throws java.sql.SQLException
	 */
	public void setBinaryStream(int parameterIndex, InputStream x, int length)
		throws SQLException {
		wrapped.setBinaryStream(parameterIndex, x, length);
		//addParam(parameterIndex, x);
	}

	/**
	 * @param i
	 * @param x
	 * @throws java.sql.SQLException
	 */
	public void setBlob(int parameterIndex, Blob x) throws SQLException {
		wrapped.setBlob(parameterIndex, x);
		addParam(parameterIndex, x);
	}

	/**
	 * @param parameterIndex
	 * @param x
	 * @throws java.sql.SQLException
	 */
	public void setBoolean(int parameterIndex, boolean x) throws SQLException {
		wrapped.setBoolean(parameterIndex, x);
		addParam(parameterIndex, new Boolean(x));
	}

	/**
	 * @param parameterIndex
	 * @param x
	 * @throws java.sql.SQLException
	 */
	public void setByte(int parameterIndex, byte x) throws SQLException {
		wrapped.setByte(parameterIndex, x);
		addParam(parameterIndex, new Byte(x));
	}

	/**
	 * @param parameterIndex
	 * @param x
	 * @throws java.sql.SQLException
	 */
	public void setBytes(int parameterIndex, byte[] x) throws SQLException {
		wrapped.setBytes(parameterIndex, x);
		addParam(parameterIndex, x);
	}

	/**
	 * @param parameterIndex
	 * @param reader
	 * @param length
	 * @throws java.sql.SQLException
	 */
	public void setCharacterStream(
		int parameterIndex,
		Reader reader,
		int length)
		throws SQLException {
		wrapped.setCharacterStream(parameterIndex, reader, length);
		//addParam(parameterIndex, reader);
	}

	/**
	 * @param i
	 * @param x
	 * @throws java.sql.SQLException
	 */
	public void setClob(int parameterIndex, Clob x) throws SQLException {
		wrapped.setClob(parameterIndex, x);
		addParam(parameterIndex, x);
	}

	/**
	 * @param parameterIndex
	 * @param x
	 * @throws java.sql.SQLException
	 */
	public void setDate(int parameterIndex, Date x) throws SQLException {
		wrapped.setDate(parameterIndex, x);
		addParam(parameterIndex, x);
	}

	/**
	 * @param parameterIndex
	 * @param x
	 * @param cal
	 * @throws java.sql.SQLException
	 */
	public void setDate(int parameterIndex, Date x, Calendar cal)
		throws SQLException {
		wrapped.setDate(parameterIndex, x, cal);
		addParam(parameterIndex, x);
	}

	/**
	 * @param parameterIndex
	 * @param x
	 * @throws java.sql.SQLException
	 */
	public void setDouble(int parameterIndex, double x) throws SQLException {
		wrapped.setDouble(parameterIndex, x);
		addParam(parameterIndex, new Double(x));
	}

	/**
	 * @param parameterIndex
	 * @param x
	 * @throws java.sql.SQLException
	 */
	public void setFloat(int parameterIndex, float x) throws SQLException {
		wrapped.setFloat(parameterIndex, x);
		addParam(parameterIndex, new Float(x));
	}

	/**
	 * @param parameterIndex
	 * @param x
	 * @throws java.sql.SQLException
	 */
	public void setInt(int parameterIndex, int x) throws SQLException {
		wrapped.setInt(parameterIndex, x);
		addParam(parameterIndex, new Integer(x));
	}

	/**
	 * @param parameterIndex
	 * @param x
	 * @throws java.sql.SQLException
	 */
	public void setLong(int parameterIndex, long x) throws SQLException {
		wrapped.setLong(parameterIndex, x);
		addParam(parameterIndex, new Long(x));
	}

	/**
	 * @param parameterIndex
	 * @param sqlType
	 * @throws java.sql.SQLException
	 */
	public void setNull(int parameterIndex, int sqlType) throws SQLException {
		wrapped.setNull(parameterIndex, sqlType);
		//addParam(parameterIndex, null);
	}

	/**
	 * @param paramIndex
	 * @param sqlType
	 * @param typeName
	 * @throws java.sql.SQLException
	 */
	public void setNull(int parameterIndex, int sqlType, String typeName)
		throws SQLException {
		wrapped.setNull(parameterIndex, sqlType, typeName);
		//addParam(parameterIndex, null);
	}

	/**
	 * @param parameterIndex
	 * @param x
	 * @throws java.sql.SQLException
	 */
	public void setObject(int parameterIndex, Object x) throws SQLException {
		wrapped.setObject(parameterIndex, x);
	}

	/**
	 * @param parameterIndex
	 * @param x
	 * @param targetSqlType
	 * @throws java.sql.SQLException
	 */
	public void setObject(int parameterIndex, Object x, int targetSqlType)
		throws SQLException {
		wrapped.setObject(parameterIndex, x, targetSqlType);
		addParam(parameterIndex, x);
	}

	/**
	 * @param parameterIndex
	 * @param x
	 * @param targetSqlType
	 * @param scale
	 * @throws java.sql.SQLException
	 */
	public void setObject(
		int parameterIndex,
		Object x,
		int targetSqlType,
		int scale)
		throws SQLException {
		wrapped.setObject(parameterIndex, x, targetSqlType, scale);
		addParam(parameterIndex, x);
	}

	/**
	 * @param i
	 * @param x
	 * @throws java.sql.SQLException
	 */
	public void setRef(int parameterIndex, Ref x) throws SQLException {
		wrapped.setRef(parameterIndex, x);
		addParam(parameterIndex, x);
	}

	/**
	 * @param parameterIndex
	 * @param x
	 * @throws java.sql.SQLException
	 */
	public void setShort(int parameterIndex, short x) throws SQLException {
		wrapped.setShort(parameterIndex, x);
		addParam(parameterIndex, new Short(x));
	}

	/**
	 * @param parameterIndex
	 * @param x
	 * @throws java.sql.SQLException
	 */
	public void setString(int parameterIndex, String x) throws SQLException {
		wrapped.setString(parameterIndex, x);
		addParam(parameterIndex, x);
	}

	/**
	 * @param parameterIndex
	 * @param x
	 * @throws java.sql.SQLException
	 */
	public void setTime(int parameterIndex, Time x) throws SQLException {
		wrapped.setTime(parameterIndex, x);
		addParam(parameterIndex, x);
	}

	/**
	 * @param parameterIndex
	 * @param x
	 * @param cal
	 * @throws java.sql.SQLException
	 */
	public void setTime(int parameterIndex, Time x, Calendar cal)
		throws SQLException {
		wrapped.setTime(parameterIndex, x, cal);
		addParam(parameterIndex, x);
	}

	/**
	 * @param parameterIndex
	 * @param x
	 * @throws java.sql.SQLException
	 */
	public void setTimestamp(int parameterIndex, Timestamp x)
		throws SQLException {
		wrapped.setTimestamp(parameterIndex, x);
		addParam(parameterIndex, x);
	}

	/**
	 * @param parameterIndex
	 * @param x
	 * @param cal
	 * @throws java.sql.SQLException
	 */
	public void setTimestamp(int parameterIndex, Timestamp x, Calendar cal)
		throws SQLException {
		wrapped.setTimestamp(parameterIndex, x, cal);
		addParam(parameterIndex, x);
	}

	/**
	 * @param parameterIndex
	 * @param x
	 * @param length
	 * @throws java.sql.SQLException
	 * @deprecated
	 */
	public void setUnicodeStream(int parameterIndex, InputStream x, int length)
		throws SQLException {
		wrapped.setUnicodeStream(parameterIndex, x, length);
		//addParam(parameterIndex, x);
	}

	/**
	 * @param parameterIndex
	 * @param x
	 * @throws java.sql.SQLException
	 */
	public void setURL(int parameterIndex, URL x) throws SQLException {
		wrapped.setURL(parameterIndex, x);
		addParam(parameterIndex, x);
	}

}
