/*
 * Joey and its relative products are published under the terms
 * of the Apache Software License.
 */
/*
 * Created on 2004/02/21
 */
package org.asyrinx.brownie.core.sql;

import java.util.Iterator;

import org.apache.commons.lang.StringUtils;

/**
 * @author akima
 */
public class BasicSqlBuilder implements IBuilder {

	/**
	 * 
	 */
	public BasicSqlBuilder() {
		super();
	}

	private char tablePrefixDelimiter = '.';
	private String aliasSigniture = "as";

	/**
	 * @see org.asyrinx.brownie.sql.IBuilder#add(org.asyrinx.brownie.sql.Fields, java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	public String buildField(
		String tableAlias,
		String fieldName,
		String aliasName,
		String functionName) {
		final StringBuffer val = new StringBuffer();
		if (StringUtils.isNotEmpty(functionName)) {
			val.append(functionName);
			val.append("(");
		}
		if (StringUtils.isNotEmpty(tableAlias)) {
			val.append(tableAlias);
			val.append(getTablePrefixDelimiter());
		}
		val.append(fieldName);
		if (StringUtils.isNotEmpty(functionName))
			val.append(")");
		if (StringUtils.isNotEmpty(aliasName)) {
			if (StringUtils.isNotEmpty(getAliasSigniture())) {
				val.append(" ");
				val.append(getAliasSigniture());
				val.append(" ");
			}
			val.append(aliasName);
		}
		return val.toString();
	}

	private String indexPrefix = "( INDEX ";
	private String indexSuffix = ")";

	/**
	 * @see org.asyrinx.brownie.sql.IBuilder#add(org.asyrinx.brownie.sql.Tables, java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	public String buildTable(
		String tablePrefix,
		String tableName,
		String tableAlias,
		String indexName) {
		final StringBuffer result = new StringBuffer();
		if (StringUtils.isNotEmpty(tablePrefix)) {
			if (tablePrefix.charAt(tablePrefix.length() - 1)
				!= getTablePrefixDelimiter())
				tablePrefix += getTablePrefixDelimiter();
			result.append(tablePrefix);
		}
		result.append(tableName);
		if (StringUtils.isNotEmpty(tableAlias)) {
			result.append(" ");
			result.append(tableAlias);
		}
		if (StringUtils.isNotEmpty(indexName)) {
			result.append(" ");
			result.append(getIndexPrefix() + indexName + getIndexSuffix());
		}
		return result.toString();
	}

	private char quote = '\'';

	/**
	 * @see org.asyrinx.brownie.sql.IBuilder#add(org.asyrinx.brownie.sql.Conditions, java.lang.String, java.lang.String, org.asyrinx.brownie.sql.Operator)
	 */
	public String buildCondition(
		String fieldName,
		String value,
		Operator operator,
		boolean quote) {
		if (operator == null)
			operator = Operator.EQUAL;
		if (quote)
			value = getQuote() + value + getQuote();
		return fieldName + " " + operator.toString() + " " + value;
	}

	/**
	 * @see org.asyrinx.brownie.sql.IBuilder#build(org.asyrinx.brownie.sql.DynamicSelect, java.lang.StringBuffer)
	 */
	public void build(DynamicSelect dynamicSelect, StringBuffer dest) {
		buildSelect(dest, dynamicSelect);
		buildFrom(dest, dynamicSelect.getFromTables());
		buildWhere(dest, dynamicSelect.getWhereConditions());
		buildGroupBy(dest, dynamicSelect.getGroupByFields());
		buildHaving(dest, dynamicSelect.getHavingConditions());
		buildOrderBy(dest, dynamicSelect.getOrderByFields());
	}

	protected void buildSelect(
		StringBuffer dest,
		DynamicSelect dynamicSelect) {
		if (dynamicSelect.getSelectFields().isEmpty())
			return;
		dest.append("select ");
		if (dynamicSelect.isDistinct())
			dest.append(" distinct ");
		buildFields(dest, dynamicSelect.getSelectFields());
		dest.append(" ");
	}

	protected void buildFrom(StringBuffer dest, Tables fromTables) {
		if (fromTables.isEmpty())
			return;
		dest.append("from ");
		buildTables(dest, fromTables);
		dest.append(" ");
	}

	protected void buildWhere(StringBuffer dest, Conditions conditions) {
		if (conditions.isEmpty())
			return;
		dest.append("where ");
		buildConditions(dest, conditions);
		dest.append(" ");
	}

	protected void buildGroupBy(StringBuffer dest, Fields groupByFields) {
		if (groupByFields.isEmpty())
			return;
		dest.append("group by ");
		buildFields(dest, groupByFields);
		dest.append(" ");
	}

	protected void buildHaving(
		StringBuffer dest,
		Conditions havingConditions) {
		if (havingConditions.isEmpty())
			return;
		dest.append("having ");
		buildConditions(dest, havingConditions);
		dest.append(" ");
	}

	protected void buildOrderBy(StringBuffer dest, Fields orderByFields) {
		if (orderByFields.isEmpty())
			return;
		dest.append("order by ");
		buildFields(dest, orderByFields);
		dest.append(" ");
	}

	protected void buildFields(StringBuffer dest, Fields fields) {
		final Iterator iterator = fields.iterator();
		boolean first = true;
		while (iterator.hasNext()) {
			if (first)
				first = false;
			else
				dest.append(fields.getDelimiter());
			final Object field = iterator.next();
			if (field instanceof Fields) {
				buildFields(dest, (Fields) field);
			} else {
				dest.append(field);
			}
		}
	}

	protected void buildTables(StringBuffer dest, Tables tables) {
		final Iterator iterator = tables.iterator();
		boolean first = true;
		while (iterator.hasNext()) {
			if (first)
				first = false;
			else
				dest.append(tables.getDelimiter());
			final Object table = iterator.next();
			if (table instanceof Tables) {
				buildTables(dest, (Tables) table);
			} else {
				dest.append(table);
			}
		}
	}

	protected void buildConditions(StringBuffer dest, Conditions conditions) {
		final Iterator iterator = conditions.iterator();
		boolean first = true;
		while (iterator.hasNext()) {
			if (first) {
				first = false;
			} else {
				dest.append(" ");
				dest.append(conditions.getConnection());
				dest.append(" ");
			}
			final Object condition = iterator.next();
			if (condition instanceof Conditions) {
				dest.append("(");
				buildConditions(dest, (Conditions) condition);
				dest.append(")");
			} else {
				dest.append(condition);
			}
		}
	}

	/**
	 * @return
	 */
	public String getIndexPrefix() {
		return indexPrefix;
	}

	/**
	 * @return
	 */
	public String getIndexSuffix() {
		return indexSuffix;
	}

	/**
	 * @return
	 */
	public char getTablePrefixDelimiter() {
		return tablePrefixDelimiter;
	}

	/**
	 * @param string
	 */
	public void setIndexPrefix(String string) {
		indexPrefix = string;
	}

	/**
	 * @param string
	 */
	public void setIndexSuffix(String string) {
		indexSuffix = string;
	}

	/**
	 * @param string
	 */
	public void setTablePrefixDelimiter(char string) {
		tablePrefixDelimiter = string;
	}

	/**
	 * @return
	 */
	public char getQuote() {
		return quote;
	}

	/**
	 * @param c
	 */
	public void setQuote(char c) {
		quote = c;
	}

	/**
	 * @return
	 */
	public String getAliasSigniture() {
		return aliasSigniture;
	}

	/**
	 * @param string
	 */
	public void setAliasSigniture(String string) {
		aliasSigniture = string;
	}

}
