/*
 * brownies and its relative products are published under the terms
 * of the Apache Software License.
 * 
 * Created on 2004/11/20 1:30:14
 */
package test.org.asyrinx.brownie.core.query;

import java.util.Date;

import junit.framework.TestCase;

import org.asyrinx.brownie.core.query.Select;
import org.asyrinx.brownie.core.query.BasicSelectBuilder;
import org.asyrinx.brownie.core.query.StatementType;
import org.asyrinx.brownie.core.query.exp.CompositeExpression;
import org.asyrinx.brownie.core.query.exp.ExpressionWrap;
import org.asyrinx.brownie.core.query.exp.FieldExpression;
import org.asyrinx.brownie.core.query.exp.IExpression;
import org.asyrinx.brownie.core.query.model.Field;
import org.asyrinx.brownie.core.query.model.FieldAlias;
import org.asyrinx.brownie.core.query.model.Table;
import org.asyrinx.brownie.core.query.model.TableAlias;

/**
 * @author takeshi
 */
public class SelectBuilderTest extends TestCase {

    public static void main(String[] args) {
        junit.swingui.TestRunner.run(SelectBuilderTest.class);
    }

    public static final Table TABLE1 = new Table("table1", "table1");

    public static final Field COLUMN1 = new Field(TABLE1, Long.class, "id", "id");

    public static final Field COLUMN2 = new Field(TABLE1, String.class, "name", "name");

    public static final Field COLUMN3 = new Field(TABLE1, Double.class, "price", "price");

    public static final Field COLUMN4 = new Field(TABLE1, Date.class, "updated", "updated");

    public void testBuild1() {
        final BasicSelectBuilder builder = new BasicSelectBuilder();
        builder.setStringQuote('\'');
        //
        final IExpression exp1 = new FieldExpression(COLUMN1, IExpression.EQUAL, "123456");
        assertEquals("id = 123456", builder.buildExpression(exp1));
        //
        final IExpression exp2 = new FieldExpression(COLUMN2, IExpression.LIKE, "a*");
        assertEquals("name like 'a*'", builder.buildExpression(exp2));
        //
        final IExpression exp3 = new FieldExpression(COLUMN3, IExpression.GREATER_EQUAL, "123.4567");
        assertEquals("price >= 123.4567", builder.buildExpression(exp3));
        //
        final IExpression exp4 = new ExpressionWrap(IExpression.NOT, exp3);
        assertEquals("not (price >= 123.4567)", builder.buildExpression(exp4));
        //
        final CompositeExpression compo1 = new CompositeExpression(IExpression.OR);
        compo1.add(exp1).add(exp2).add(exp4);
        builder.setBracketDepth(0);
        assertEquals("(id = 123456) or (name like 'a*') or (not (price >= 123.4567))", builder.buildExpression(compo1));
        //
        final IExpression exp5 = new FieldExpression(COLUMN4, IExpression.LESS, "2004/11/20 02:46:52");
        assertEquals("updated < '2004/11/20 02:46:52'", builder.buildExpression(exp5));
        //
        final CompositeExpression compo2 = new CompositeExpression(IExpression.AND);
        compo2.add(exp5).add(compo1);
        assertEquals(
                "(updated < '2004/11/20 02:46:52') and ((id = 123456) or (name like 'a*') or (not (price >= 123.4567)))",
                builder.buildExpression(compo2));
    }

    public void testBuild2() {
        final BasicSelectBuilder builder = new BasicSelectBuilder();
        builder.setStringQuote('\'');
        //
        final CompositeExpression root = new CompositeExpression(IExpression.AND);
        root.add(new FieldExpression(COLUMN4, IExpression.LESS, "2004/11/20 02:46:52")).addCompo(IExpression.OR).add(
                new FieldExpression(new FieldAlias(TABLE1, COLUMN1, "longId"), IExpression.EQUAL, "123456")).add(
                new FieldExpression(COLUMN2, IExpression.LIKE, "a*"))
                .add(
                        new ExpressionWrap(IExpression.NOT, new FieldExpression(COLUMN3, IExpression.GREATER_EQUAL,
                                "123.4567")));
        assertEquals(
                "updated < '2004/11/20 02:46:52' and (id = 123456) or (name like 'a*') or (not (price >= 123.4567))",
                builder.buildExpression(root));
        builder.setBracketDepth(1);
        assertEquals(
                "(updated < '2004/11/20 02:46:52') and ((id = 123456) or (name like 'a*') or (not (price >= 123.4567)))",
                builder.buildExpression(root));
    }

    public void testBuildSelect() {
        final BasicSelectBuilder sqlBuilder = new BasicSelectBuilder();
        sqlBuilder.setStringQuote('\'');
        //
        final BasicSelectBuilder hqlBuilder = new BasicSelectBuilder(StatementType.HQL);
        sqlBuilder.setStringQuote('\'');
        //
        final Table table_party = new Table("party", "org.asyrinx.brownie.sample.Party");
        final Field col_party_id = new Field(table_party, Long.class, "party_id", "partyId");
        final Field col_name = new Field(table_party, String.class, "name", "name");
        final Field col_updated = new Field(table_party, Date.class, "updated", "updated", "yyyy/MM/dd HH:mm:ss");
        final Table table_party_relation = new Table("party_relation", "org.asyrinx.brownie.sample.PartyRelation");
        final Field col_party_relation_id = new Field(table_party_relation, Long.class, "party_relation_id",
                "partyRelationId");
        final Field col_origin_party_id = new Field(table_party_relation, Long.class, "origin_party_id",
                "originPartyId");
        final Field col_dest_party_id = new Field(table_party_relation, Long.class, "dest_party_id", "destPartyId");
        //
        final TableAlias party1 = table_party.alias("p1");
        final TableAlias party2 = table_party.alias("p2");
        final TableAlias relation = table_party_relation.alias("r1");
        {
            final Select select1 = new Select();
            select1.getWhere().addField(table_party.getField("party_id").alias(), IExpression.EQUAL, "123");
            assertEquals("from party where party_id = 123", sqlBuilder.buildSelect(select1));
            assertEquals("from org.asyrinx.brownie.sample.Party as party where party.partyId = 123", hqlBuilder
                    .buildSelect(select1));
        }
        {
            final Select select2 = new Select();
            select2.getFroms().add(party1).add(party2).add(relation);
            select2.getWhere().addField(party1.getField(col_party_id), IExpression.EQUAL, "987");
            select2.getWhere().addCompare(party1.getField(col_party_id), relation.getField(col_origin_party_id));
            select2.getWhere().addCompare(party2.getField(col_party_id), relation.getField(col_dest_party_id));
            assertEquals("from party as p1, party as p2, party_relation as r1" //
                    + " where p1.party_id = 987" //
                    + " and p1.party_id = r1.origin_party_id" //
                    + " and p2.party_id = r1.dest_party_id", sqlBuilder.buildSelect(select2));
            assertEquals(
                    "from org.asyrinx.brownie.sample.Party as p1, org.asyrinx.brownie.sample.Party as p2, org.asyrinx.brownie.sample.PartyRelation as r1" //
                            + " where p1.partyId = 987" //
                            + " and p1.partyId = r1.originPartyId" //
                            + " and p2.partyId = r1.destPartyId", hqlBuilder.buildSelect(select2));
        }
        {
            final Select select3 = new Select();
            select3.getFroms().add(party1).add(party2).add(relation);
            select3.getWhere().addField(party1.getField(col_name), IExpression.LIKE, "a*");
            select3.getWhere().addCompare(party1.getField(col_party_id), relation.getField(col_origin_party_id));
            select3.getWhere().addCompare(party2.getField(col_party_id), relation.getField(col_dest_party_id));
            assertEquals("from party as p1, party as p2, party_relation as r1 " //
                    + "where p1.name like 'a*'" //
                    + " and p1.party_id = r1.origin_party_id" //
                    + " and p2.party_id = r1.dest_party_id", sqlBuilder.buildSelect(select3));
            assertEquals(
                    "from org.asyrinx.brownie.sample.Party as p1, org.asyrinx.brownie.sample.Party as p2, org.asyrinx.brownie.sample.PartyRelation as r1 " //
                            + "where p1.name like 'a*'" //
                            + " and p1.partyId = r1.originPartyId" //
                            + " and p2.partyId = r1.destPartyId", hqlBuilder.buildSelect(select3));
        }
        {
            final Select select4 = new Select();
            //select4.getFroms().add(party1).add(party2).add(relation);
            final CompositeExpression condition = new CompositeExpression(IExpression.AND);
            condition.addField(party1.getField(col_name), IExpression.LIKE, "a*");
            condition.addCompare(party1.getField(col_party_id), relation.getField(col_origin_party_id));
            condition.addCompare(party2.getField(col_party_id), relation.getField(col_dest_party_id));
            select4.getWhere().add(condition);
            select4.prepareTableAlias();
            assertEquals("from party as p1, party_relation as r1, party as p2 " //
                    + "where p1.name like 'a*'" //
                    + " and p1.party_id = r1.origin_party_id" //
                    + " and p2.party_id = r1.dest_party_id", sqlBuilder.buildSelect(select4));
            assertEquals(
                    "from org.asyrinx.brownie.sample.Party as p1, org.asyrinx.brownie.sample.PartyRelation as r1, org.asyrinx.brownie.sample.Party as p2 " //
                            + "where p1.name like 'a*'" //
                            + " and p1.partyId = r1.originPartyId" //
                            + " and p2.partyId = r1.destPartyId", hqlBuilder.buildSelect(select4));
        }
    }
    
    
    
    public void testJoin(){
        
        
        
        
        
        
    }

}