/*
 * brownies and its relative products are published under the terms
 * of the Apache Software License.
 * 
 * Created on 2004/05/27 3:20:55
 */
package org.asyrinx.brownie.swing.table;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.swing.table.TableModel;

import org.apache.commons.lang.ObjectUtils;
import org.asyrinx.brownie.swing.table.calc.CalcTableModel;
import org.asyrinx.brownie.swing.table.calc.CalcTableModelImpl;
import org.asyrinx.brownie.swing.table.key.KeyColumn;
import org.asyrinx.brownie.swing.table.key.KeyColumnTableModel;
import org.asyrinx.brownie.swing.table.key.RowKeyComparator;
import org.asyrinx.brownie.swing.table.key.SortableRowKey;
import org.asyrinx.brownie.swing.table.util.ArrayAccessibleTableModel;
import org.asyrinx.brownie.swing.table.util.InvisibleColumnTableModel;
import org.asyrinx.brownie.swing.table.util.ListAccessibleTableModel;
import org.asyrinx.brownie.swing.table.util.SimpleSortableTableModel;
import org.asyrinx.brownie.swing.table.wrapper.TableModelWrapper;

/**
 * @author akima
 */
public class TableUtils {

    public static TableModel toUneditableModel(TableModel model) {
        return new UneditableTableModel(model);
    }

    public static InvisibleColumnTableModel toInvisibleColumnModel(TableModel model, int invisibleColumnIndex,
            int invisibleColumnCount) {
        if (invisibleColumnCount < 1)
            return getNullModel(model);
        if (invisibleColumnIndex < 0)
            return getNullModel(model);
        return new InvisibleColumnTableModelImpl(model, invisibleColumnIndex, invisibleColumnCount);
    }

    public static InvisibleColumnTableModel getNullModel(TableModel model) {
        return new NullInvisibleColumnTableModel(model);
    }

    public static SimpleSortableTableModel toSortableModel(TableModel model) {
        return new SimpleSortableTableModelImpl(model);
    }

    public static CalcTableModel toCalculatable(TableModel model, int[] keyColumns) {
        return new CalcTableModelImpl(model, keyColumns);
    }

    public static ListAccessibleTableModel toListAccessible(TableModel tableModel) {
        return new ListAccessibleTableModelImpl(tableModel);
    }

    public static ArrayAccessibleTableModel toArrayAccessible(TableModel tableModel) {
        return new ArrayAccessibleTableModelImpl(tableModel);
    }

    public static TableModel toSortable(TableModel model, int[] keyColumns) {
        return new SortableTableModelImpl(model, keyColumns);
    }
}

class UneditableTableModel extends TableModelWrapper {
    /**
     * @param tableModel
     */
    public UneditableTableModel(TableModel model) {
        super(model);
    }

    public boolean isCellEditable(int rowIndex, int columnIndex) {
        return false;
    }
}

class InvisibleColumnTableModelImpl extends TableModelWrapper implements InvisibleColumnTableModel {
    /**
     * @param tableModel
     */
    public InvisibleColumnTableModelImpl(TableModel model, int invisibleColumnIndex, int invisibleColumnCount) {
        super(model);
        this.invisibleColumnIndex = invisibleColumnIndex;
        this.invisibleColumnCount = invisibleColumnCount;
    }

    private final int invisibleColumnIndex;

    private final int invisibleColumnCount;

    /*
     * (non-Javadoc)
     * 
     * @see org.asyrinx.brownie.swing.table.wrapper.TableModelWrapper#getColumnCount()
     */
    public int getColumnCount() {
        return model.getColumnCount() - invisibleColumnCount;
    }

    public int getOriginalColumnIndex(int columnIndex) {
        return (columnIndex < invisibleColumnIndex) ? columnIndex : columnIndex + invisibleColumnCount;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.asyrinx.brownie.swing.table.wrapper.TableModelWrapper#getColumnName(int)
     */
    public String getColumnName(int columnIndex) {
        return model.getColumnName(getOriginalColumnIndex(columnIndex));
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.asyrinx.brownie.swing.table.wrapper.TableModelWrapper#getValueAt(int,
     *        int)
     */
    public Object getValueAt(int rowIndex, int columnIndex) {
        return model.getValueAt(rowIndex, getOriginalColumnIndex(columnIndex));
    }

}

class SimpleSortableTableModelImpl extends TableModelWrapper implements SimpleSortableTableModel {
    /**
     * @param tableModel
     */
    public SimpleSortableTableModelImpl(TableModel model) {
        super(model);
    }

    private int[] rowIndexMap = null;

    class SortEntry implements Comparable {
        final String value;

        final int row;

        public SortEntry(String value, int row) {
            this.value = (value != null) ? value : "";
            this.row = row;
        }

        /*
         * (non-Javadoc)
         * 
         * @see java.lang.Comparable#compareTo(java.lang.Object)
         */
        public int compareTo(Object o) {
            return this.value.compareTo(((SortEntry) o).value);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.asyrinx.brownie.swing.table.SimpleSortableTableModel#sort(int)
     */
    public void sort(int columnIndex, boolean reverse) {
        if (model.getRowCount() < 2)
            return;
        final List list = new ArrayList();
        for (int row = 0; row < this.model.getRowCount(); row++)
            list.add(new SortEntry(ObjectUtils.toString(model.getValueAt(row, columnIndex)), row));
        Collections.sort(list);
        rowIndexMap = new int[list.size()];
        for (int i = 0; i < rowIndexMap.length; i++) {
            final int listIndex = (reverse) ? rowIndexMap.length - 1 - i : i;
            final SortEntry entry = (SortEntry) list.get(listIndex);
            rowIndexMap[i] = entry.row;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.asyrinx.brownie.swing.table.wrapper.TableModelWrapper#getValueAt(int,
     *        int)
     */
    public Object getValueAt(int rowIndex, int columnIndex) {
        return model.getValueAt(getOriginalRowIndex(rowIndex), columnIndex);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.asyrinx.brownie.swing.table.SimpleSortableTableModel#getOriginalRowIndex(int)
     */
    public int getOriginalRowIndex(int sortedRowIndex) {
        return (rowIndexMap == null) ? sortedRowIndex : rowIndexMap[sortedRowIndex];
    }

}

final class NullInvisibleColumnTableModel extends TableModelWrapper implements InvisibleColumnTableModel {
    /**
     * @param tableModel
     */
    public NullInvisibleColumnTableModel(TableModel model) {
        super(model);

    }

    public int getOriginalColumnIndex(int columnIndex) {
        return columnIndex;
    }
}

final class ListAccessibleTableModelImpl extends TableModelWrapper implements ListAccessibleTableModel {

    /**
     * @param model
     */
    public ListAccessibleTableModelImpl(TableModel model) {
        super(model);
    }

    public List getColumnList(int columnIndex) {
        final List list = new ArrayList();
        for (int r = 0; r < getRowCount(); r++)
            list.add(getValueAt(r, columnIndex));
        return list;
    }

    public List getRowList(int rowIndex) {
        final List list = new ArrayList();
        for (int c = 0; c < getColumnCount(); c++)
            list.add(getValueAt(rowIndex, c));
        return list;
    }
}

final class ArrayAccessibleTableModelImpl extends TableModelWrapper implements ArrayAccessibleTableModel {

    /**
     * @param model
     */
    public ArrayAccessibleTableModelImpl(TableModel model) {
        super(model);
    }

    public Object[] getColumnArray(int columnIndex) {
        final Object[] objects = new Object[getRowCount()];
        for (int r = 0; r < getRowCount(); r++)
            objects[r] = getValueAt(r, columnIndex);
        return objects;
    }

    public Object[] getRowArray(int rowIndex) {
        final Object[] objects = new Object[getColumnCount()];
        for (int c = 0; c < getColumnCount(); c++)
            objects[c] = getValueAt(rowIndex, c);
        return objects;
    }
}

final class SortableTableModelImpl extends KeyColumnTableModel {
    /**
     * @param tableModel
     */
    public SortableTableModelImpl(TableModel tableModel) {
        this(tableModel, null);
    }

    /**
     * @param tableModel
     * @param keyColumns
     */
    public SortableTableModelImpl(TableModel tableModel, int[] keyColumns) {
        super(tableModel, keyColumns);
        if (keyColumns != null) {
            initKeyColumns(keyColumns);
            refreshKeys();
        }
        initEventListener();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.asyrinx.brownie.swing.table.key.KeyColumnTableModel#refreshKeys()
     */
    public void refreshKeys() {
        clearKeyList();
        for (int r = 0; r < tableModel.getRowCount(); r++) {
            final Object[] values = new Object[getKeyColumnCount()];
            for (int i = 0; i < getKeyColumnCount(); i++) {
                final KeyColumn keyColumn = getKeyColumn(i);
                values[i] = tableModel.getValueAt(r, keyColumn.getOriginalColumnIndex());
            }
            addRowKey(new SortableRowKey(values, r));
        }
        sortKeyList(new RowKeyComparator());
    }

    /*
     * (non-Javadoc)
     * 
     * @see javax.swing.table.TableModel#getValueAt(int, int)
     */
    public Object getValueAt(int rowIndex, int columnIndex) {
        final SortableRowKey rowKey = (SortableRowKey) getKey(rowIndex);
        return tableModel.getValueAt(rowKey.getOriginalRowIndex(), columnIndex);
    }
}