/*
 * Joey and its relative products are published under the terms
 * of the Apache Software License.
 */
package org.asyrinx.brownie.core.log;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;

/**
 * @author Akima
 */
public abstract class LogLevel {

	/**
	 * Constructor for Level.
	 */
	protected LogLevel(int value, String name) {
		super();
		this.name = name;
		this.value = value;
		register(this);
	}

	private final int value;

	private final String name;

	/**
	 * Returns the name.
	 * 
	 * @return String
	 */
	public String getName() {
		return name;
	}

	/**
	 * Returns the value.
	 * 
	 * @return int
	 */
	public int getValue() {
		return value;
	}

	abstract public void write(Log log, Object msg, Throwable t);

	public void write(Log log, Object msg) {
		write(log, msg, null);
	}

	/**
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		return name;
	}

	static public Set getLevels() {
		return new HashSet(levels);
	}

	static private final Set levels = new HashSet();

	/**
	 * @see java.lang.Object#finalize()
	 */
	protected void finalize() throws Throwable {
		levels.remove(this);
		super.finalize();
	}

	static private Map name2Level = new HashMap();

	static public void register(LogLevel level) {
		levels.add(level);
		name2Level.put(level.name, level);
	}

	static public LogLevel findByName(String name) {
		return (LogLevel) name2Level.get(name);
	}

	static public LogLevel getLevel(String name) {
		return findByName(name);
	}

	static public final LogLevel TRACE = new LogLevel(0, "trace") {
		public void write(Log log, Object msg, Throwable t) {
			log.trace(msg, t);
		}
	};

	static public final LogLevel DEBUG = new LogLevel(1, "debug") {
		public void write(Log log, Object msg, Throwable t) {
			log.debug(msg, t);
		}
	};

	static public final LogLevel INFO = new LogLevel(2, "info") {
		public void write(Log log, Object msg, Throwable t) {
			log.info(msg, t);
		}
	};

	static public final LogLevel WARN = new LogLevel(3, "warn") {
		public void write(Log log, Object msg, Throwable t) {
			log.warn(msg, t);
		}
	};

	static public final LogLevel ERROR = new LogLevel(4, "error") {
		public void write(Log log, Object msg, Throwable t) {
			log.error(msg, t);
		}
	};

	static public final LogLevel FATAL = new LogLevel(5, "fatal") {
		public void write(Log log, Object msg, Throwable t) {
			log.fatal(msg, t);
		}
	};

	static public final LogLevel NONE = new LogLevel(5, "none") {
		public void write(Log log, Object msg, Throwable t) {
		}
	};

}