/*
 * Joey and its relative products are published under the terms
 * of the Apache Software License.
 */
/*
 * Created on 2004/03/01
 */
package org.asyrinx.brownie.jdbc.logger;

import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Savepoint;
import java.sql.Statement;

import org.asyrinx.brownie.core.lang.ArrayUtils;
import org.asyrinx.brownie.core.log.CascadeNamedLog;
import org.asyrinx.brownie.core.log.DispatchLog;
import org.asyrinx.brownie.jdbc.wrapper.ConnectionWrapper;

/**
 * @author akima
 */
public class LogConnection extends ConnectionWrapper {

    /**
     * @param wrapped
     */
    public LogConnection(Connection wrapped) {
        this(wrapped, wrapped.getClass().getName());
    }

    /**
     * @param wrapped
     */
    public LogConnection(Connection wrapped, String loggerName) {
        this(wrapped, loggerName, DispatchLog.DEFAULT_LEVEL);
    }

    /**
     * @param wrapped
     */
    public LogConnection(Connection wrapped, String loggerName, String logLevel) {
        this(wrapped, new CascadeNamedLog(loggerName, logLevel));
    }

    /**
     * @param wrapped
     */
    public LogConnection(Connection wrapped, CascadeNamedLog parentLog) {
        super(wrapped);
        this.log = parentLog.subLog(this.wrapped);
    }

    private final CascadeNamedLog log;

    /*
     * (non-Javadoc)
     * 
     * @see java.sql.Connection#createStatement()
     */
    public Statement createStatement() throws SQLException {
        try {
            final Statement result = super.createStatement();
            log.log("createStatement(): " + result);
            return new LogStatement(result, log);
        } catch (SQLException e) {
            log.error("failed to createStatement()", e);
            throw e;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.sql.Connection#createStatement(int, int, int)
     */
    public Statement createStatement(int resultSetType, int resultSetConcurrency, int resultSetHoldability)
            throws SQLException {
        try {
            final Statement result = super.createStatement(resultSetType, resultSetConcurrency, resultSetHoldability);
            log.log("createStatement(" + resultSetType + "," + resultSetConcurrency + "," + resultSetHoldability
                    + "): " + result);
            return new LogStatement(result, log);
        } catch (SQLException e) {
            log.error("failed to createStatement(" + resultSetType + "," + resultSetConcurrency + ","
                    + resultSetHoldability + ")", e);
            throw e;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.sql.Connection#createStatement(int, int)
     */
    public Statement createStatement(int resultSetType, int resultSetConcurrency) throws SQLException {
        try {
            final Statement result = super.createStatement(resultSetType, resultSetConcurrency);
            log.log("createStatement(" + resultSetType + "," + resultSetConcurrency + "): " + result);
            return new LogStatement(result, log);
        } catch (SQLException e) {
            log.error("failed to createStatement(" + resultSetType + "," + resultSetConcurrency + ")", e);
            throw e;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.asyrinx.brownie.jdbc.logger.ConnectionWrapper#nativeSQL(java.lang.String)
     */
    public String nativeSQL(String sql) throws SQLException {
        try {
            final String result = super.nativeSQL(sql);
            log.log("nativeSQL(\"" + sql + "\"): \"" + result + "\"");
            return result;
        } catch (SQLException e) {
            log.error(sql, e);
            throw e;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.sql.Connection#prepareCall(java.lang.String, int, int, int)
     */
    public CallableStatement prepareCall(String sql, int resultSetType, int resultSetConcurrency,
            int resultSetHoldability) throws SQLException {
        try {
            final CallableStatement result = super.prepareCall(sql, resultSetType, resultSetConcurrency,
                    resultSetHoldability);
            log.log("prepareCall(\"" + sql + "\"," + resultSetType + "," + resultSetConcurrency + ","
                    + resultSetHoldability + "): " + result);
            return new LogCallableStatement(result, sql, log.subLog(result));
        } catch (SQLException e) {
            log.error("failed to prepareCall(\"" + sql + "\"," + resultSetType + "," + resultSetConcurrency + ","
                    + resultSetHoldability + ")", e);
            throw e;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.sql.Connection#prepareCall(java.lang.String, int, int)
     */
    public CallableStatement prepareCall(String sql, int resultSetType, int resultSetConcurrency) throws SQLException {
        try {
            final CallableStatement result = super.prepareCall(sql, resultSetType, resultSetConcurrency);
            log.log("prepareCall(\"" + sql + "\"," + resultSetType + "," + resultSetConcurrency + "): " + result);
            return new LogCallableStatement(result, sql, log.subLog(result));
        } catch (SQLException e) {
            log.error("failed to prepareCall(\"" + sql + "\"," + resultSetType + "," + resultSetConcurrency + ")", e);
            throw e;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.sql.Connection#prepareCall(java.lang.String)
     */
    public CallableStatement prepareCall(String sql) throws SQLException {
        try {
            final CallableStatement result = super.prepareCall(sql);
            log.log("prepareCall(\"" + sql + "\"):" + result);
            return new LogCallableStatement(result, sql, log.subLog(result));
        } catch (SQLException e) {
            log.error("failed to prepareCall(\"" + sql + "\")", e);
            throw e;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.sql.Connection#prepareStatement(java.lang.String, int, int,
     *      int)
     */
    public PreparedStatement prepareStatement(String sql, int resultSetType, int resultSetConcurrency,
            int resultSetHoldability) throws SQLException {
        try {
            final PreparedStatement result = super.prepareStatement(sql, resultSetType, resultSetConcurrency,
                    resultSetHoldability);
            log.log("prepareStatement(\"" + sql + "\"," + resultSetType + "," + resultSetConcurrency + ","
                    + resultSetHoldability + "): " + result);
            return new LogPreparedStatement(result, sql, log.subLog(result));
        } catch (SQLException e) {
            log.error("failed to prepareStatement(\"" + sql + "\"," + resultSetType + "," + resultSetConcurrency + ","
                    + resultSetHoldability + ")", e);
            throw e;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.sql.Connection#prepareStatement(java.lang.String, int, int)
     */
    public PreparedStatement prepareStatement(String sql, int resultSetType, int resultSetConcurrency)
            throws SQLException {
        try {
            final PreparedStatement result = super.prepareStatement(sql, resultSetType, resultSetConcurrency);
            log.log("prepareStatement(\"" + sql + "\"," + resultSetType + "," + resultSetConcurrency + "): " + result);
            return new LogPreparedStatement(result, sql, log.subLog(result));
        } catch (SQLException e) {
            log.error("failed to prepareStatement(\"" + sql + "\"," + resultSetType + "," + resultSetConcurrency + ")",
                    e);
            throw e;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.sql.Connection#prepareStatement(java.lang.String, int)
     */
    public PreparedStatement prepareStatement(String sql, int autoGeneratedKeys) throws SQLException {
        try {
            final PreparedStatement result = super.prepareStatement(sql, autoGeneratedKeys);
            log.log("prepareStatement(\"" + sql + "\"," + autoGeneratedKeys + "): " + result);
            return new LogPreparedStatement(result, sql, log.subLog(result));
        } catch (SQLException e) {
            log.error("failed to prepareStatement(\"" + sql + "\"," + autoGeneratedKeys + ")", e);
            throw e;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.sql.Connection#prepareStatement(java.lang.String, int[])
     */
    public PreparedStatement prepareStatement(String sql, int[] columnIndexes) throws SQLException {
        try {
            final PreparedStatement result = super.prepareStatement(sql, columnIndexes);
            log.log("prepareStatement(\"" + sql + "\",[" + ArrayUtils.toString(columnIndexes) + "]): " + result);
            return new LogPreparedStatement(result, sql, log.subLog(result));
        } catch (SQLException e) {
            log.error("failed to prepareStatement(\"" + sql + "\",[" + ArrayUtils.toString(columnIndexes) + "])", e);
            throw e;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.sql.Connection#prepareStatement(java.lang.String,
     *      java.lang.String[])
     */
    public PreparedStatement prepareStatement(String sql, String[] columnNames) throws SQLException {
        try {
            final PreparedStatement result = super.prepareStatement(sql, columnNames);
            log.log("prepareStatement(\"" + sql + "\",[" + ArrayUtils.toString(columnNames) + "]): " + result);
            return new LogPreparedStatement(result, sql, log.subLog(result));
        } catch (SQLException e) {
            log.error("failed to prepareStatement(\"" + sql + "\",[" + ArrayUtils.toString(columnNames) + "])", e);
            throw e;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.sql.Connection#prepareStatement(java.lang.String)
     */
    public PreparedStatement prepareStatement(String sql) throws SQLException {
        try {
            final PreparedStatement result = super.prepareStatement(sql);
            log.log("prepareStatement(\"" + sql + "): " + result);
            return new LogPreparedStatement(result, sql, log.subLog(result));
        } catch (SQLException e) {
            log.error("failed to prepareStatement(\"" + sql + ")", e);
            throw e;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.asyrinx.brownie.jdbc.wrapper.ConnectionWrapper#commit()
     */
    public void commit() throws SQLException {
        try {
            super.commit();
            this.log.log("commit()");
        } catch (SQLException e) {
            this.log.error("failed to commit()", e);
            throw e;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.asyrinx.brownie.jdbc.wrapper.ConnectionWrapper#rollback()
     */
    public void rollback() throws SQLException {
        try {
            super.rollback();
            this.log.log("rollback()");
        } catch (SQLException e) {
            this.log.error("failed to rollback()", e);
            throw e;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.asyrinx.brownie.jdbc.wrapper.ConnectionWrapper#rollback(java.sql.Savepoint)
     */
    public void rollback(Savepoint savepoint) throws SQLException {
        try {
            super.rollback(savepoint);
            this.log.log("rollback(" + savepoint + ")");
        } catch (SQLException e) {
            this.log.error("failed to rollback(" + savepoint + ")", e);
            throw e;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.asyrinx.brownie.jdbc.wrapper.ConnectionWrapper#setAutoCommit(boolean)
     */
    public void setAutoCommit(boolean autoCommit) throws SQLException {
        try {
            super.setAutoCommit(autoCommit);
            this.log.log("setAutoCommit(" + autoCommit + ")");
        } catch (SQLException e) {
            this.log.error("failed to setAutoCommit(" + autoCommit + ")", e);
            throw e;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.asyrinx.brownie.jdbc.wrapper.ConnectionWrapper#setSavepoint()
     */
    public Savepoint setSavepoint() throws SQLException {
        try {
            final Savepoint result = super.setSavepoint();
            this.log.log("setSavepoint(): " + result);
            return result;
        } catch (SQLException e) {
            this.log.error("failed to setSavepoint()", e);
            throw e;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.asyrinx.brownie.jdbc.wrapper.ConnectionWrapper#setSavepoint(java.lang.String)
     */
    public Savepoint setSavepoint(String name) throws SQLException {
        try {
            final Savepoint result = super.setSavepoint(name);
            this.log.log("setSavepoint(" + name + "): " + result);
            return result;
        } catch (SQLException e) {
            this.log.error("failed to setSavepoint(" + name + ")", e);
            throw e;
        }
    }

}