/*
 * Joey and its relative products are published under the terms
 * of the Apache Software License.
 */
/*
 * Created on 2003/12/16
 */
package org.asyrinx.brownie.jdbc;

import java.io.IOException;
import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;

import org.asyrinx.brownie.core.io.sf.RelativeClassResourceStreamFactory;
import org.asyrinx.brownie.core.io.sf.StreamFactory;
import org.asyrinx.brownie.core.io.sf.StreamFactoryFacade;
import org.asyrinx.brownie.core.io.sf.StringLoader;
import org.asyrinx.brownie.core.lang.StringUtils;
import org.asyrinx.brownie.jdbc.wrapper.ConnectionWrapper;

/**
 * @author akima
 */
public final class SqlLoadableConnection extends ConnectionWrapper {

	/**
	 * @param source
	 */
	public SqlLoadableConnection(Connection source) {
		this(source, (String) null);
	}

	/**
	 * @param source
	 */
	public SqlLoadableConnection(Connection source, String classToBeCalled) {
		this(source, classToBeCalled, (String) null);
	}

	/**
	 * @param source
	 */
	public SqlLoadableConnection(Connection source, String classToBeCalled,
			String sqlResources) {
		super(source);
		this.baseFactoryFacade = newBaseFactoryFacade(sqlResources);
		this.streamFactory = newFactory(classToBeCalled, baseFactoryFacade);
		this.loader = new StringLoader(streamFactory);
	}

	private final StreamFactory streamFactory;

	private final StreamFactoryFacade baseFactoryFacade;

	private final StringLoader loader;

	protected final String load(String key) throws SQLException {
		try {
			return loader.load(key);
		} catch (IOException e) {
			throw new SQLException(e.getMessage());
		}
	}

	static StreamFactoryFacade newBaseFactoryFacade(String sqlResources) {
		final StreamFactoryFacade result = StreamFactoryFacade
				.newFacade(sqlResources);
		return result;
	}

	static StreamFactory newFactory(String classToBeCalled,
			StreamFactoryFacade baseFactoryFacade) {
		final StreamFactoryFacade result = baseFactoryFacade.copy();
		classToBeCalled = StringUtils.isEmpty(classToBeCalled) ? SqlLoadableConnection.class
				.getName()
				: classToBeCalled;
		result.add(1, new RelativeClassResourceStreamFactory(classToBeCalled));
		return result;
	}

	/**
	 * @see org.asyrinx.jdbc.wrapper.ConnectionWrapper#createStatement()
	 */
	public SqlLoadableStatement createSqlLoadableStatement()
			throws SQLException {
		return new SqlLoadableStatement(super.createStatement(),
				this.baseFactoryFacade);
	}

	/**
	 * @see org.asyrinx.jdbc.wrapper.ConnectionWrapper#createStatement(int, int)
	 */
	public SqlLoadableStatement createSqlLoadableStatement(int resultSetType,
			int resultSetConcurrency) throws SQLException {
		return new SqlLoadableStatement(super.createStatement(resultSetType,
				resultSetConcurrency), this.baseFactoryFacade);
	}

	/**
	 * @see org.asyrinx.jdbc.wrapper.ConnectionWrapper#createStatement(int, int,
	 *        int)
	 */
	public SqlLoadableStatement createSqlLoadableStatement(int resultSetType,
			int resultSetConcurrency, int resultSetHoldability)
			throws SQLException {
		return new SqlLoadableStatement(super.createStatement(resultSetType,
				resultSetConcurrency, resultSetHoldability),
				this.baseFactoryFacade);
	}

	/**
	 * @see java.sql.Connection#prepareCall(java.lang.String, int, int, int)
	 */
	public CallableStatement prepareCallFromFile(String sqlKey,
			int resultSetType, int resultSetConcurrency,
			int resultSetHoldability) throws SQLException {
		return super.prepareCall(load(sqlKey), resultSetType,
				resultSetConcurrency, resultSetHoldability);
	}

	/**
	 * @see java.sql.Connection#prepareCall(java.lang.String, int, int)
	 */
	public CallableStatement prepareCallFromFile(String sqlKey,
			int resultSetType, int resultSetConcurrency) throws SQLException {
		return super.prepareCall(load(sqlKey), resultSetType,
				resultSetConcurrency);
	}

	/**
	 * @see java.sql.Connection#prepareCall(java.lang.String)
	 */
	public CallableStatement prepareCallFromFile(String sqlKey)
			throws SQLException {
		return super.prepareCall(load(sqlKey));
	}

	/**
	 * @see java.sql.Connection#prepareStatement(java.lang.String, int, int,
	 *        int)
	 */
	public PreparedStatement prepareStatementFromFile(String sqlKey,
			int resultSetType, int resultSetConcurrency,
			int resultSetHoldability) throws SQLException {
		return super.prepareStatement(load(sqlKey), resultSetType,
				resultSetConcurrency, resultSetHoldability);
	}

	/**
	 * @see java.sql.Connection#prepareStatement(java.lang.String, int, int)
	 */
	public PreparedStatement prepareStatementFromFile(String sqlKey,
			int resultSetType, int resultSetConcurrency) throws SQLException {
		return super.prepareStatement(load(sqlKey), resultSetType,
				resultSetConcurrency);
	}

	/**
	 * @see java.sql.Connection#prepareStatement(java.lang.String, int)
	 */
	public PreparedStatement prepareStatementFromFile(String sqlKey,
			int autoGeneratedKeys) throws SQLException {
		return super.prepareStatement(load(sqlKey), autoGeneratedKeys);
	}

	/**
	 * @see java.sql.Connection#prepareStatement(java.lang.String, int[])
	 */
	public PreparedStatement prepareStatementFromFile(String sqlKey,
			int[] columnIndexes) throws SQLException {
		return super.prepareStatement(load(sqlKey), columnIndexes);
	}

	/**
	 * @see java.sql.Connection#prepareStatement(java.lang.String,
	 *        java.lang.String[])
	 */
	public PreparedStatement prepareStatementFromFile(String sqlKey,
			String[] columnNames) throws SQLException {
		return super.prepareStatement(load(sqlKey), columnNames);
	}

	/**
	 * @see java.sql.Connection#prepareStatement(java.lang.String)
	 */
	public PreparedStatement prepareStatementFromFile(String sqlKey)
			throws SQLException {
		return super.prepareStatement(load(sqlKey));
	}

}