/*
 * environment information
 *
 * Copyright(c) 2009 olyutorskii
 * $Id: EnvInfo.java 944 2009-12-05 16:31:41Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.io.File;
import java.text.NumberFormat;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

/**
 * 実行環境に関する各種情報。
 */
public final class EnvInfo{

    private static final Map<String, String> propertyMap =
            new TreeMap<String, String>();
    private static final String osName;
    private static final String osVersion;
    private static final String osArch;
    private static final String javaVendor;
    private static final String javaVersion;

    private static final String classpath;
    private static final String[] classpaths;

    private static final Map<String, String> environmentMap =
            new TreeMap<String, String>();
    private static final String envLANG;
    private static final String envDISPLAY;

    static{
        osName      = getSecureProperty("os.name");
        osVersion   = getSecureProperty("os.version");
        osArch      = getSecureProperty("os.arch");
        javaVendor  = getSecureProperty("java.vendor");
        javaVersion = getSecureProperty("java.version");
        classpath   = getSecureProperty("java.class.path");

        envLANG      = getSecureEnvironment("LANG");
        envDISPLAY   = getSecureEnvironment("DISPLAY");

        if(classpath != null){
            classpaths = classpath.split(File.pathSeparator);
        }else{
            classpaths = new String[0];
        }
    }

    /**
     * OS名を返す。
     * @return OS名
     */
    public static String getOsName(){
        return osName;
    }

    /**
     * アーキテクチャ名を返す。
     * @return アーキテクチャ名
     */
    public static String getOsArch(){
        return osArch;
    }

    /**
     * OSバージョン名を返す。
     * @return OSバージョン名
     */
    public static String getOsVersion(){
        return osVersion;
    }

    /**
     * Java実行系のベンダ名を返す。
     * @return Java実行系のベンダ名
     */
    public static String getJavaVendor(){
        return javaVendor;
    }

    /**
     * Java実行系のバージョン名を返す。
     * @return Java実行系のバージョン名
     */
    public static String getJavaVersion(){
        return javaVersion;
    }

    /**
     * クラスパス集合を返す。
     * @return クラスパス集合
     */
    public static String getClasspath(){
        return classpath;
    }

    /**
     * トークンで分解されたクラスパスを返す。
     * @return トークンで分解されたクラスパス
     */
    public static String[] getClasspaths(){
        return classpaths;
    }

    /**
     * 環境変数LANGの値を返す。
     * @return 環境変数LANGの値
     */
    public static String getEnvLANG(){
        return envLANG;
    }

    /**
     * 環境変数DISPLAYの値を返す。
     * @return 環境変数DISPLAYの値
     */
    public static String getEnvDISPLAY(){
        return envDISPLAY;
    }

    /**
     * 可能ならシステムプロパティを読み込む。
     * @param key キー
     * @return プロパティ値。セキュリティ上読み込み禁止の場合はnull。
     */
    private static String getSecureProperty(String key){
        String result;
        try{
            result = System.getProperty(key);
            if(result != null) propertyMap.put(key, result);
        }catch(SecurityException e){
            result = null;
        }
        return result;
    }

    /**
     * 可能なら環境変数を読み込む。
     * @param name 環境変数名
     * @return 環境変数値。セキュリティ上読み込み禁止の場合はnull。
     */
    private static String getSecureEnvironment(String name){
        String result;
        try{
            result = System.getenv(name);
            if(result != null) environmentMap.put(name, result);
        }catch(SecurityException e){
            result = null;
        }
        return result;
    }

    /**
     * VM詳細情報を文字列化する。
     * @return VM詳細情報
     */
    public static String getVMInfo(){
        StringBuilder result = new StringBuilder();
        NumberFormat nform = NumberFormat.getNumberInstance();

        result.append("最大ヒープメモリ量: "
                + nform.format(Jindolf.RUNTIME.maxMemory()) + " bytes\n");

        result.append("\n");

        result.append("起動時引数:\n");
        for(String arg : Jindolf.getOptionInfo().getInvokeArgList()){
            result.append("  ").append(arg).append("\n");
        }

        result.append("\n");

        result.append("主要システムプロパティ:\n");
        Set<String> propKeys = propertyMap.keySet();
        for(String propKey : propKeys){
            if(propKey.equals("java.class.path")) continue;
            String value = propertyMap.get(propKey);
            result.append("  ");
            result.append(propKey).append("=").append(value).append("\n");
        }

        result.append("\n");

        result.append("主要環境変数:\n");
        Set<String> envKeys = environmentMap.keySet();
        for(String envKey : envKeys){
            String value = environmentMap.get(envKey);
            result.append("  ");
            result.append(envKey).append("=").append(value).append("\n");
        }

        result.append("\n");

        result.append("クラスパス:\n");
        for(String path : classpaths){
            result.append("  ");
            result.append(path).append("\n");
        }

        result.append("\n");

        return result.toString();
    }

    /**
     * 隠れコンストラクタ。
     */
    private EnvInfo(){
        throw new Error();
    }

}
