/*
 * InputStream for Shift JIS encoded byte stream
 *
 * Copyright(c) 2009 olyutorskii
 * $Id: SjisInputStream.java 391 2009-02-13 03:59:35Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.io.IOException;
import java.io.InputStream;
import java.util.Collections;
import java.util.SortedMap;
import java.util.TreeMap;

/**
 * シフトJIS用JISX0208:1990バリデーションチェックを行うInputStream。
 *
 */
public class SjisInputStream extends InputStream{
    
    private static final int UNDEFCHAR = '?';

    static{
        assert 0x00 <= UNDEFCHAR && UNDEFCHAR <= 0xff;
    }

    private final InputStream in;
    private int pushChar;
    private long charCount;

    private final SortedMap<Long, byte[]> encodeError
            = new TreeMap<Long, byte[]>();

    /**
     * コンストラクタ
     * @param in 委譲先ストリーム
     */
    public SjisInputStream(InputStream in){
        this.in = in;
        init();
        return;
    }

    /**
     * {@inheritDoc}
     * シフトJIS2バイト目の先読み &amp; 検査を行う。
     * @return {@inheritDoc}
     * @throws java.io.IOException {@inheritDoc}
     */
    public int read() throws IOException{
        if(this.pushChar >= 0){
            int result = this.pushChar;
            this.pushChar = -1;
            this.charCount++;
            return result;
        }

        int ch1st = this.in.read();
        if(ch1st < 0) return -1;

        if( ! CodeX0208.isShiftJIS1stByte((byte)ch1st) ){
            this.charCount++;
            return ch1st;
        }

        int ch2nd = this.in.read();
        if(ch2nd < 0){
            pushEncodeError(new byte[]{ (byte)ch1st });
            this.charCount++;
            return UNDEFCHAR;
        }

        if( ! CodeX0208.isShiftJIS2ndByte((byte)ch2nd) ){
            pushEncodeError(new byte[]{ (byte)ch1st, (byte)ch2nd });
            this.charCount++;
            return UNDEFCHAR;
        }

        if( ! CodeX0208.isValidSJIS0208_1990(ch1st, ch2nd) ){
            pushEncodeError(new byte[]{ (byte)ch1st, (byte)ch2nd });
            this.charCount++;
            return UNDEFCHAR;
        }

        this.pushChar = ch2nd;

        return ch1st;
    }

    /**
     * {@inheritDoc}
     * @return {@inheritDoc}
     * @throws java.io.IOException {@inheritDoc}
     */
    @Override
    public int available() throws IOException{
        int result = this.in.available();
        if(this.pushChar >= 0) result++;
        return result;
    }

    /**
     * {@inheritDoc}
     * @throws java.io.IOException {@inheritDoc}
     */
    @Override
    public void close() throws IOException{
        this.in.close();
        return;
    }

    /**
     * プッシュバックおよび内部カウンタのクリア
     */
    protected void init(){
        this.pushChar = -1;
        this.charCount = 0;
        this.encodeError.clear();
        return;
    }

    /**
     * 読み込んだ文字数を返す。
     * JISX0208各キャラクタは1文字で数える。
     * @return 読み込んだ文字数。
     */
    public long getCharCount(){
        return this.charCount;
    }

    /**
     * エンコードエラーの記録
     * @param bdata エラーを引き起こしたバイト列。
     */
    protected void pushEncodeError(byte[] bdata){
        this.encodeError.put(this.charCount, bdata);
        return;
    }

    /**
     * エンコードエラーを返す。
     * SortedMapのキー値は出力中の代替文字のポジション、
     * 値はエラーの原因となった1つまたは2つのbyte値
     * @return エンコードエラー。
     */
    public SortedMap<Long, byte[]> getEncodeError(){
        SortedMap<Long, byte[]> result
                = Collections.unmodifiableSortedMap(this.encodeError);
        return result;
    }
}
