/*
 * JIS X0208:1990 文字集合に関する諸々
 *
 * Copyright(c) 2008 olyutorskii
 * $Id: CodeX0208.java 385 2009-02-09 17:23:25Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.io.IOException;
import java.util.Arrays;
import java.util.SortedSet;
import java.util.TreeSet;

/**
 * JIS X0208:1990 文字集合に関する諸々
 */
public class CodeX0208{

    private static final String RESOURCE_INVALIDCHAR =
            "resources/invalidX0208.txt";
    private static final char[] invalidCharArray = createInvalidCharArray();

    /**
     * ソートされた、禁止文字配列を生成する。
     * @return 禁止文字配列。
     */
    private static char[] createInvalidCharArray(){
        CharSequence source;
        try{
            source = Jindolf.loadResourceText(RESOURCE_INVALIDCHAR);
        }catch(IOException e){
            assert false;
            return null;
        }

        SortedSet<Character> charSet = new TreeSet<Character>();
        int sourceLength = source.length();
        for(int pos=0; pos < sourceLength; pos++){
            char ch = source.charAt(pos);
            if(Character.isWhitespace(ch)) continue;
            charSet.add(ch);
        }

        char[] result = new char[charSet.size()];
        int pos = 0;
        for(char ch : charSet){
            result[pos++] = ch;
        }

        Arrays.sort(result);

        return result;
    }

    /**
     * 禁止文字か否か判定する。
     * @param ch 判定対象文字
     * @return 禁止ならfalse
     */
    public static boolean isValid(char ch){
        int index = Arrays.binarySearch(invalidCharArray, ch);
        if(index < 0) return true;
        return false;
    }

    /**
     * 任意のバイト値がシフトJISの1バイト目でありうるか否か判定する。
     * @param ch バイト値
     * @return シフトJISの1バイト目でありうるならtrue
     */
    public static boolean isShiftJIS1stByte(byte ch){
        if(   (byte)0x81 <= ch && ch <= (byte)0x9f
           || (byte)0xe0 <= ch && ch <= (byte)0xfc){
            return true;
        }
        return false;
    }

    /**
     * 任意のバイト値がシフトJISの2バイト目でありうるか否か判定する。
     * @param ch バイト値
     * @return シフトJISの2バイト目でありうるならtrue
     */
    public static boolean isShiftJIS2ndByte(byte ch){
        if(   (byte)0x40 <= ch && ch <= (byte)0x7e
           || (byte)0x80 <= ch && ch <= (byte)0xfc){
            return true;
        }
        return false;
    }

    /**
     * シフトJISから区点コードへ変換する。
     *
     * @param chs 入力は[0]:上位8bit [1]:下位8bit、出力は[0]:区 [1]:点
     */
    public static void txSjisToKuten(int[] chs){
        int jis0;
        int jis1;

        jis0 = chs[0];
        if( jis0 <= 0x9f ) jis0 -= 0x71;
        else               jis0 -= 0xb1;

        jis0 <<= 1;
        jis0  |= 1;

        jis1 = chs[1];
        if( jis1 >= 0x7f ) jis1--;

        if( jis1 >= 0x9e ){
            jis0++;
            jis1 -= 0x7d;
        }else{
            jis1 -= 0x1f;
        }

        jis0 -= 0x20;
        jis1 -= 0x20;

        chs[0] = jis0;
        chs[1] = jis1;

        return;
    }

    /**
     * シフトJISエンコーディングがJISX0208:1990に適合しているか判定する。
     * @param high 上位8bit
     * @param low 下位8bit
     * @return 適合していればtrue
     */
    public static boolean isValidSJIS0208_1990(int high, int low){
        int jis0 = high;
        if( jis0 <= 0x9f ) jis0 -= 0x71;
        else               jis0 -= 0xb1;

        jis0 <<= 1;
        jis0  |= 1;

        int jis1 = low;
        if( jis1 >= 0x7f ) jis1--;

        if( jis1 >= 0x9e ){
            jis0++;
            jis1 -= 0x7d;
        }else{
            jis1 -= 0x1f;
        }

        int kuNo  = jis0 - 0x20;
        int tenNo = jis1 - 0x20;

        if(    1 <= kuNo && kuNo <=  8
           || 16 <= kuNo && kuNo <= 84){
            return true;
        }

        return false;
    }

    /**
     * ダミーコンストラクタ
     */
    private CodeX0208(){
        assert false;
        throw new Error();
    }

    // TODO アラビア語やハングルやも弾きたい。
}
