/*
 * characters in village
 *
 * Copyright(c) 2008 olyutorskii
 * $Id: Avatar.java 308 2008-11-29 15:54:21Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.io.IOException;
import java.io.InputStream;
import java.util.Collections;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeMap;
import java.util.TreeSet;

/**
 * Avatar またの名をキャラクター
 */
public class Avatar implements Comparable<Avatar>{

    private static final String AVATAR_DEF = "resources/avatar.properties";
    private static final String KEY_PREFIX = "avatar.serialNo.";

    private static final SortedSet<Avatar> predefinedAvatarSet;
    private static final Map<String, Avatar> avatarMap;

    static{
        predefinedAvatarSet = loadPredefSet();

        avatarMap = new TreeMap<String, Avatar>();
        for(Avatar avatar : predefinedAvatarSet){
            String fullName = avatar.getFullName();
            avatarMap.put(fullName, avatar);
        }
    }

    /**
     * リソース上のプロパティファイルからAvatar一覧を読み込む。
     */
    private static SortedSet<Avatar> loadPredefSet(){
        InputStream is = Jindolf.getResourceAsStream(AVATAR_DEF);
        return loadPredefSet(is);
    }

    /**
     * InputStreamからAvatar一覧を読み込む。
     */
    private static SortedSet<Avatar> loadPredefSet(InputStream is){
        Properties properties = new Properties();
        try{
            properties.load(is);
            is.close();
        }catch(IOException e){
            assert false;
            throw new Error();
        }

        return loadPredefSet(properties);
    }

    /**
     * プロパティから国一覧を読み込む。
     */
    private static SortedSet<Avatar> loadPredefSet(Properties properties){
        String codeCheck = properties.getProperty("codeCheck");
        if(   codeCheck == null
           || codeCheck.length() != 1
           || codeCheck.charAt(0) != '\u72fc'){  // 「狼」
            Jindolf.logger.severe("キャラクタ定義一覧プロパティファイルの"
                    +"文字コードがおかしいようです。"
                    +"native2ascii は正しく適用しましたか？");
            Jindolf.exit(1);
            throw new Error();
        }

        Set<Object> keySet = properties.keySet();

        Map<String, Integer> idSerialMap = new TreeMap<String, Integer>();
        for(Object keyObj : keySet){
            if(keyObj == null) continue;
            String key = keyObj.toString();
            if( ! key.startsWith(KEY_PREFIX) ) continue;
            String id = properties.getProperty(key);
            String serialNo = key.replace(KEY_PREFIX, "");
            Integer serialInt;
            try{
                serialInt = new Integer(serialNo);
            }catch(NumberFormatException e){
                continue;
            }
            idSerialMap.put(id, serialInt);
        }

        SortedSet<Avatar> result = new TreeSet<Avatar>();
        for(String id: idSerialMap.keySet()){
            String nameKey = id + ".name";
            String jobKey = id + ".job";

            Integer serialInt = idSerialMap.get(id);
            String name = properties.getProperty(nameKey);
            String job = properties.getProperty(jobKey);

            Avatar avatar = new Avatar(name, job, serialInt, id);
            result.add(avatar);
        }

        result = Collections.unmodifiableSortedSet(result);

        return result;
    }

    /**
     * 定義済みAvatar群のリストを返す。
     * @return Avatarのリスト
     */
    public static SortedSet<Avatar> getPredefinedAvatarSet(){
        return predefinedAvatarSet;
    }

    /**
     * 定義済みAvatarを返す。
     * @param fullName Avatarのフルネーム
     * @return Avatar
     */
    public static Avatar getPredefinedAvatar(String fullName){
        return avatarMap.get(fullName);
    }

    private final String name;
    private final String jobTitle;
    private final String fullName;
    private final int idNum;
    private final String identifier;

    /**
     * Avatarを生成する。
     * @param name 名前
     * @param jobTitle 職業名
     * @param idNum 通し番号
     * @param identifier 識別文字列
     */
    private Avatar(String name,
                    String jobTitle,
                    int idNum,
                    String identifier ){
        this.name = name.intern();
        this.jobTitle = jobTitle.intern();
        this.idNum = idNum;
        this.identifier = identifier.intern();

        this.fullName = (this.jobTitle + " " + this.name).intern();

        return;
    }

    /**
     * Avatarを生成する
     * @param fullName フルネーム
     */
    // TODO 当面は呼ばれないはず。Z国とか向け。
    public Avatar(String fullName){
        this.fullName = fullName.intern();
        this.idNum = -1;

        String[] tokens = this.fullName.split("\\p{Blank}+", 2);
        if(tokens.length == 1){
            this.jobTitle = null;
            this.name = this.fullName;
        }else if(tokens.length == 2){
            this.jobTitle = tokens[0].intern();
            this.name = tokens[1].intern();
        }else{
            this.jobTitle = null;
            this.name = null;
            assert false;
        }

        this.identifier = "???".intern();

        return;
    }

    /**
     * フルネームを取得する。
     * @return フルネーム
     */
    public String getFullName(){
        return this.fullName;
    }

    /**
     * 職業名を取得する。
     * @return 職業名
     */
    public String getJobTitle(){
        return this.jobTitle;
    }

    /**
     * 通常名を取得する。
     * @return 通常名
     */
    public String getName(){
        return this.name;
    }

    /**
     * 通し番号を返す。
     * @return 通し番号
     */
    public int getIdNum(){
        return this.idNum;
    }

    /**
     * {@inheritDoc}
     * @param obj {@inheritDoc}
     * @return {@inheritDoc}
     */
    @Override
    public boolean equals(Object obj){
        if(this == obj){
            return true;
        }
        if(obj == null){
            return false;
        }
        if( ! (obj instanceof Avatar) ){
            return false;
        }

        Avatar other = (Avatar) obj;

        boolean nameMatch = this.fullName.equals(other.fullName);
        boolean idMatch = this.idNum == other.idNum;

        if(nameMatch && idMatch) return true;

        return false;
    }

    /**
     * {@inheritDoc}
     * @return {@inheritDoc}
     */
    @Override
    public int hashCode(){
        return this.fullName.hashCode() ^ this.idNum;
    }

    /**
     * {@inheritDoc}
     * @param avatar {@inheritDoc}
     * @return {@inheritDoc}
     */
    public int compareTo(Avatar avatar){
        if(avatar == null) return +1;
        return this.idNum - avatar.idNum;
    }
}
